﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/database/olsc_PolicyInfoCache.h>
#include <nn/olsc/srv/olsc_Executor.h>
#include <nn/olsc/srv/olsc_PolicyInfoTask.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_SaveDataUtil.h"

#include <curl/curl.h>
#include <vector>



using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {
    struct Modules
    {
        Modules() :
            mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave),
            policyInfoCache(mountManager)
        {}
        ~Modules() {}
        srv::util::DefaultMountManager mountManager;
        srv::database::PolicyInfoCache policyInfoCache;
    };

    struct TestParam
    {
        ApplicationId appId;
        int launchRequiredVersion;
        int latestVersion;

        srv::database::PolicyInfoCacheEntry expectedEntry;
    };

    std::array<Bit8, 4 * 1024 * 1024> g_WorkBuffer;

    class OlscDataTransferPolicyTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData({}));
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nnt::olsc::Initialize();
        }

        static void TearDownTestCase()
        {
            curl_global_cleanup();
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }

    protected:
        static Result InitializeSaveData(fs::UserId userId)
        {
            fs::SystemSaveDataId toDelete[] = {
                nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSettingSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto& id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, userId))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }

        void TestDataTransferPolicy(TestParam& param)
        {
            nifm::NetworkConnection connection;
            auto& connectionEvent = connection.GetSystemEvent();
            connection.SubmitRequest();
            NN_ABORT_UNLESS(connectionEvent.TimedWait(TimeSpan::FromSeconds(10)));
            NN_ABORT_UNLESS(connection.IsAvailable());

            Modules m;
            auto er = reinterpret_cast<srv::ExecutionResource*>(g_WorkBuffer.data());
            er->curlHandle = curl_easy_init();
            NN_ABORT_UNLESS(er->curlHandle);

            srv::UpdateDataTransferPolicyTask task(param.appId, param.latestVersion, param.launchRequiredVersion, m.policyInfoCache);
            NN_ABORT_UNLESS_RESULT_SUCCESS(task.Initialize());
            task.Lock();
            task.Execute(g_WorkBuffer.data(), g_WorkBuffer.size());
            EXPECT_TRUE(task.HasDone());
            task.Unlock();


            auto actualEntry = m.policyInfoCache.Get(param.appId);
            EXPECT_EQ(param.expectedEntry, actualEntry);
        }
    };

    // このアプリは v0 で AllNg, v1 で AllOk のポリシーが登録されている
    const ApplicationId TestAppForDataTransferPolicy = { 0x010002500cc20000 };
}


// LaunchRequiredVersion も LatestVersion も AllNg の場合は、最終結果も AllNg
TEST_F(OlscDataTransferPolicyTest, NotRegistered)
{
    const ApplicationId NotRegisteredAppId = TestAppForDataTransferPolicy;
    int LaunchRequiredVersion = 0;
    int LatestVersion = 0;
    TestParam p = {
        NotRegisteredAppId,
        LaunchRequiredVersion,
        LatestVersion,

        {
            LaunchRequiredVersion,
            LatestVersion,
            {
                srv::PolicyType::AllNg
            }
        }
    };
    TestDataTransferPolicy(p);
}

// LatestVersion だけ AllOk の場合は、最終結果が AllNg
TEST_F(OlscDataTransferPolicyTest, Registered_Latest)
{
    const ApplicationId NotRegisteredAppId = TestAppForDataTransferPolicy;
    int LaunchRequiredVersion = 0;
    int LatestVersion = 1;
    TestParam p = {
        NotRegisteredAppId,
        LaunchRequiredVersion,
        LatestVersion,

        {
            LaunchRequiredVersion,
            LatestVersion,
            {
                srv::PolicyType::AllNg
            }
        }
    };
    TestDataTransferPolicy(p);
}

// LaunchRequiredVersion だけ AllOk の場合は、最終結果が AllNg
TEST_F(OlscDataTransferPolicyTest, Registered_Required)
{
    const ApplicationId NotRegisteredAppId = TestAppForDataTransferPolicy;
    int LaunchRequiredVersion = 1;
    int LatestVersion = 0;
    TestParam p = {
        NotRegisteredAppId,
        LaunchRequiredVersion,
        LatestVersion,

        {
            LaunchRequiredVersion,
            LatestVersion,
            {
                srv::PolicyType::AllNg
            }
        }
    };
    TestDataTransferPolicy(p);
}

// LatestVersion, LaunchRequiredVersion 共に AllOk の場合は、最終結果が AllOk
TEST_F(OlscDataTransferPolicyTest, Registered_All)
{
    const ApplicationId NotRegisteredAppId = TestAppForDataTransferPolicy;
    int LaunchRequiredVersion = 1;
    int LatestVersion = 1;
    TestParam p = {
        NotRegisteredAppId,
        LaunchRequiredVersion,
        LatestVersion,

        {
            LaunchRequiredVersion,
            LatestVersion,
            {
                srv::PolicyType::AllOk
            }
        }
    };
    TestDataTransferPolicy(p);
}
