﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/olsc/srv/util/olsc_AutonomyUploadWaitingList.h>
#include <nn/olsc/srv/util/olsc_SeriesPosition.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_SaveDataUtil.h"

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {
    class OlscRunningApplicationManager : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData({}));
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nnt::olsc::Initialize();
        }

        static void TearDownTestCase()
        {
            nnt::olsc::Finalize();
        }

    protected:
        static Result InitializeSaveData(fs::UserId userId)
        {
            fs::SystemSaveDataId toDelete[] = {
                nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSettingSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto& id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, userId))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }

    };

}


TEST_F(OlscRunningApplicationManager, Update)
{


    const int MaxWaitingAppCount = nn::olsc::srv::util::AutonomyUploadWaitingList::MaxApplicationCount;
    std::vector<ApplicationId> idList(MaxWaitingAppCount);
    for (int i = 0; i < MaxWaitingAppCount; ++i)
    {
        idList[i] = { static_cast<Bit64>(i + 1) };
    }

    // 初期状態
    {
        nn::olsc::srv::util::AutonomyUploadWaitingList m;
        EXPECT_EQ(0, m.GetCount());
        EXPECT_FALSE(m.GetFront());
        EXPECT_FALSE(m.PopFront());
    }

    // 上限
    {

        nn::olsc::srv::util::AutonomyUploadWaitingList m;
        m.Update(idList.data(), MaxWaitingAppCount);
        EXPECT_EQ(MaxWaitingAppCount, m.GetCount());
        for (int i = 0; i < MaxWaitingAppCount; ++i)
        {
            EXPECT_EQ(idList[i], m.GetFront());
            EXPECT_EQ(idList[i], m.PopFront());
        }
    }

    // 上限を超えた登録
    {
        nn::olsc::srv::util::AutonomyUploadWaitingList m;
        m.Update(idList.data(), MaxWaitingAppCount);
        ApplicationId overApp = { idList[MaxWaitingAppCount - 1].value + 1 };
        m.Update(&overApp, 1);

        // 先頭が追い出されていることを確認
        EXPECT_EQ(MaxWaitingAppCount, m.GetCount());
        for (int i = 0; i < MaxWaitingAppCount - 1; ++i)
        {
            EXPECT_EQ(idList[i + 1], m.GetFront());
            EXPECT_EQ(idList[i + 1], m.PopFront());
        }
        EXPECT_EQ(overApp, m.GetFront());
        EXPECT_EQ(overApp, m.PopFront());
    }

    // 重複は無視される
    {
        nn::olsc::srv::util::AutonomyUploadWaitingList m;
        m.Update(idList.data(), 1);
        EXPECT_EQ(1, m.GetCount());

        m.Update(idList.data(), 2);
        EXPECT_EQ(2, m.GetCount());
    }

}


TEST_F(OlscRunningApplicationManager, Remove)
{
    nn::olsc::srv::util::AutonomyUploadWaitingList m;

    const int MaxWaitingAppCount = nn::olsc::srv::util::AutonomyUploadWaitingList::MaxApplicationCount;
    std::vector<ApplicationId> idList(MaxWaitingAppCount);
    for (int i = 0; i < MaxWaitingAppCount; ++i)
    {
        idList[i] = { static_cast<Bit64>(i + 1) };
    }

    // 上限まで記録
    m.Update(idList.data(), MaxWaitingAppCount);
    EXPECT_EQ(MaxWaitingAppCount, m.GetCount());

    // 1番目, 0番目, MaxAppCount - 1番目の値の順で削除し
    // 2番目から MaxAppCount - 2 番目までが残ることを確認

    m.Remove(idList[1]);
    m.Remove(idList[0]);
    m.Remove(idList[MaxWaitingAppCount - 1]);
    EXPECT_EQ(MaxWaitingAppCount - 3, m.GetCount());

    for (int i = 0; i < MaxWaitingAppCount - 3; ++i)
    {
        EXPECT_EQ(idList[i + 2], m.GetFront());
        EXPECT_EQ(idList[i + 2], m.PopFront());
    }
    EXPECT_EQ(0, m.GetCount());
}
