﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/nsd/nsd_ApiForMiddleware.h> //設定削除に関連した Resolve 確認のため
#include <nn/nsd/nsd_ApiForTest.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#if NN_BUILD_CONFIG_OS_WIN

// Windows.h マクロと衝突しないかの確認

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <nn/nn_Windows.h>

#endif // NN_BUILD_CONFIG_OS_WIN

using namespace nn::nsd;

namespace
{
    uint8_t g_WorkBuffer[1024 * 256] = {};

    SaveData g_SaveData = {};
}

class NsdTest_ForMenu : public ::testing::Test
{
#ifndef NN_BUILD_CONFIG_OS_WIN
private:
    static void SetSettingsTestMode( bool isTestMode )
    {
        const char SettingsName[] = "nsd";
        const char SettingsKey[] = "test_mode";
        bool value = isTestMode;
        EXPECT_EQ( sizeof(bool), nn::settings::fwdbg::GetSettingsItemValueSize(SettingsName, SettingsKey) );
        nn::settings::fwdbg::SetSettingsItemValue( SettingsName, SettingsKey, &value, sizeof(value) );
    }

protected:
    static void SetUpTestCase()
    {
        SetSettingsTestMode( true );
    }

    static void TearDownTestCase()
    {
        SetSettingsTestMode( false );
    }
#endif
};

TEST_F(NsdTest_ForMenu, ImportSettings1)
{
    const char SampleInput[] =
        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJTZXR0aW5nTmFtZSI6IlNES-iHquWLleODhuOCueODiCBkZDEgWzIwMTYtMDktMDYgMDA6MTU6MDggVVRDXSBkZDEiLCJFbnZpcm9ubWVudCI6ImRkMSIsIkJhY2tib25lU2V0dGluZ3MiOnsiTmFzIjp7IlJlcXVlc3RGcWRuIjoiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiQXBpRnFkbiI6ImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLWFwaS5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJOeEFjY291bnQiOnsiQ2xpZW50SWQiOjEyODU1MjMwMDM0ODMyMzU5NzU5LCJSZWRpcmVjdFVyaSI6Im5pbnRlbmRvOi8vYWNjb3VudC5ueC5zeXMifSwiTnhTaG9wIjp7IkNsaWVudElkIjoxNTE2MDIwODQwODMwNDc3MzIyNywiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL3Nob3Aubnguc3lzIn0sIk54RnJpZW5kcyI6eyJDbGllbnRJZCI6MTc0Mjc5OTU1NTcwNDMzMzY1MDUsIlJlZGlyZWN0VXJpIjoibmludGVuZG86Ly9mcmllbmRzLm54LnN5cyJ9fSwiRnFkbkVudHJpZXMiOltbImUwZDY3YzUwOWZiMjAzODU4ZWJjYjJmZTNmODhjMmFhLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXSxbImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXV19LCJBcHBsaWNhdGlvblNldHRpbmdzIjp7IkZxZG5FbnRyaWVzIjpbXX0sIkV4cGlyZSI6MzI0NzIxNDQwMDAsIlZlcnNpb24iOjEsIkRldmljZUlkIjoiMCJ9.F-DS9hfdNQGRHBQ9P50bhcYPtpP83jbvkNR9ewhpm5Ik_8eDHyi84VI-GYMYVWqcZ9eHa3AkT_JPjLWCnl2bMa66RjwWn0KxoBWEPyqCL7Bo0gJ15MrCKVxLQJcXTWoJKQ8Nsa3gjAn9cLNL6yZsCgkoJngIXUyXfiGlbJEy9Fe0SQv6B7J0gy-O_pZZgbP09wc4kJAdoAUhpsjbb-4iPe7ZlSraX9DyHwCUv0zE1EC8tAZyqpMx_9yzJcKv1h9_i-PHbpDW1odJbq0HNCk_CHp4va7-hZWIL3AvHBa2z9j1g0DdPbkiaROa6gsQThoWkGzbDjju1cYnNSiFVDSOhw";
    // Version: 0x1
    // DeviceId: 0 (1 byte)
    // SettingName : SDK自動テスト dd1 [2016-09-06 00:15:08 UTC] dd1 (52 byte)
    // EnvironmentIdentifier: dd1 (3 byte)
    // ExpireTime: 0x78f7d9880
    // IsAvailable: true
    // NasServiseSettings: NasRequestServiceFqdn: e97b8a9d672e4ce4845ec6947cd66ef6-sb.accounts.nintendo.com (57 byte)
    // NasServiseSettings: NasApiServiceFqdn: e97b8a9d672e4ce4845ec6947cd66ef6-sb-api.accounts.nintendo.com (61 byte)
    // NasServiseSettings: Entry_0: Name: NxAccount (9 byte),  ClientId: b266f39b541d814f, RedirectUri: nintendo://account.nx.sys (25 byte)
    // NasServiseSettings: Entry_1: Name: NxShop (6 byte),  ClientId: d263e139a1e5686b, RedirectUri: nintendo://shop.nx.sys (22 byte)
    // NasServiseSettings: Entry_2: Name: NxFriends (9 byte),  ClientId: f1dcad9f24a77539, RedirectUri: nintendo://friends.nx.sys (25 byte)
    // BackboneSetting_000: Src: e0d67c509fb203858ebcb2fe3f88c2aa.baas.nintendo.com (50 byte), Dest: e97b8a9d672e4ce4845ec6947cd66ef6-sb.baas.nintendo.com (53 byte)
    // BackboneSetting_001: Src: e97b8a9d672e4ce4845ec6947cd66ef6-sb.baas.nintendo.com (53 byte), Dest: e97b8a9d672e4ce4845ec6947cd66ef6-sb.baas.nintendo.com (53 byte)

    // テストから fs を使うのでこれを呼んでおく必要がある
    nn::fs::SetResultHandledByApplication(false);

    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ImportSettings(
        static_cast<const void*>(SampleInput), sizeof(SampleInput),
        g_WorkBuffer, sizeof(g_WorkBuffer), nn::nsd::ImportMode_All) );

    //
    // セーブデータを直接 Read して中身の確認
    //
    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ReadSaveDataFromFsForTest(&g_SaveData));
    //EXPECT_STREQ("SDK自動テスト dd1 [2016-09-06 00:15:08 UTC] dd1", g_SaveData.settingName.value);
    EXPECT_STREQ("SDK\xe8\x87\xaa\xe5\x8b\x95\xe3\x83\x86\xe3\x82\xb9\xe3\x83\x88 dd1 [2016-09-06 00:15:08 UTC] dd1", g_SaveData.settingName.value);

    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);

    EXPECT_STREQ("e97b8a9d672e4ce4845ec6947cd66ef6-sb.accounts.nintendo.com",
        g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("e97b8a9d672e4ce4845ec6947cd66ef6-sb-api.accounts.nintendo.com",
        g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);

    EXPECT_STREQ("NxAccount",
        g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(12855230034832359759ULL,
        g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nintendo://account.nx.sys",
        g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("NxShop",
        g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceName.value);
    EXPECT_EQ(15160208408304773227ULL,
        g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.clientId);
    EXPECT_STREQ("nintendo://shop.nx.sys",
        g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("NxFriends",
        g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceName.value);
    EXPECT_EQ(17427995557043336505ULL,
        g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceSetting.clientId);
    EXPECT_STREQ("nintendo://friends.nx.sys",
        g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceSetting.redirectUri.value);

    EXPECT_EQ(0x78f7d9880, g_SaveData.expireUnixTime);
    EXPECT_EQ(1, g_SaveData.version);
    EXPECT_STREQ("0", g_SaveData.deviceId.value);

    DeleteSettings(nn::nsd::DeleteMode_All);
}

TEST_F(NsdTest_ForMenu, ImportSettings2)
{
    const char SampleInput[] =
        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJTZXR0aW5nTmFtZSI6IlNES-iHquWLleODhuOCueODiCBkZDEgWzIwMTYtMDktMDYgMDA6MTU6MDggVVRDXSBkZDEiLCJFbnZpcm9ubWVudCI6ImRkMSIsIkJhY2tib25lU2V0dGluZ3MiOnsiTmFzIjp7IlJlcXVlc3RGcWRuIjoiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiQXBpRnFkbiI6ImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLWFwaS5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJOeEFjY291bnQiOnsiQ2xpZW50SWQiOjEyODU1MjMwMDM0ODMyMzU5NzU5LCJSZWRpcmVjdFVyaSI6Im5pbnRlbmRvOi8vYWNjb3VudC5ueC5zeXMifSwiTnhTaG9wIjp7IkNsaWVudElkIjoxNTE2MDIwODQwODMwNDc3MzIyNywiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL3Nob3Aubnguc3lzIn0sIk54RnJpZW5kcyI6eyJDbGllbnRJZCI6MTc0Mjc5OTU1NTcwNDMzMzY1MDUsIlJlZGlyZWN0VXJpIjoibmludGVuZG86Ly9mcmllbmRzLm54LnN5cyJ9fSwiRnFkbkVudHJpZXMiOltbImUwZDY3YzUwOWZiMjAzODU4ZWJjYjJmZTNmODhjMmFhLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXSxbImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXV19LCJBcHBsaWNhdGlvblNldHRpbmdzIjp7IkZxZG5FbnRyaWVzIjpbXX0sIkV4cGlyZSI6MzI0NzIxNDQwMDAsIlZlcnNpb24iOjEsIkRldmljZUlkIjoiMCJ9.F-DS9hfdNQGRHBQ9P50bhcYPtpP83jbvkNR9ewhpm5Ik_8eDHyi84VI-GYMYVWqcZ9eHa3AkT_JPjLWCnl2bMa66RjwWn0KxoBWEPyqCL7Bo0gJ15MrCKVxLQJcXTWoJKQ8Nsa3gjAn9cLNL6yZsCgkoJngIXUyXfiGlbJEy9Fe0SQv6B7J0gy-O_pZZgbP09wc4kJAdoAUhpsjbb-4iPe7ZlSraX9DyHwCUv0zE1EC8tAZyqpMx_9yzJcKv1h9_i-PHbpDW1odJbq0HNCk_CHp4va7-hZWIL3AvHBa2z9j1g0DdPbkiaROa6gsQThoWkGzbDjju1cYnNSiFVDSOhw";
    // Version: 0x1
    // DeviceId: 0 (1 byte)
    // SettingName : SDK自動テスト dd1 [2016-09-06 00:15:08 UTC] dd1 (52 byte)
    // EnvironmentIdentifier: dd1 (3 byte)
    // ExpireTime: 0x78f7d9880
    // IsAvailable: true
    // NasServiseSettings: NasRequestServiceFqdn: e97b8a9d672e4ce4845ec6947cd66ef6-sb.accounts.nintendo.com (57 byte)
    // NasServiseSettings: NasApiServiceFqdn: e97b8a9d672e4ce4845ec6947cd66ef6-sb-api.accounts.nintendo.com (61 byte)
    // NasServiseSettings: Entry_0: Name: NxAccount (9 byte),  ClientId: b266f39b541d814f, RedirectUri: nintendo://account.nx.sys (25 byte)
    // NasServiseSettings: Entry_1: Name: NxShop (6 byte),  ClientId: d263e139a1e5686b, RedirectUri: nintendo://shop.nx.sys (22 byte)
    // NasServiseSettings: Entry_2: Name: NxFriends (9 byte),  ClientId: f1dcad9f24a77539, RedirectUri: nintendo://friends.nx.sys (25 byte)
    // BackboneSetting_000: Src: e0d67c509fb203858ebcb2fe3f88c2aa.baas.nintendo.com (50 byte), Dest: e97b8a9d672e4ce4845ec6947cd66ef6-sb.baas.nintendo.com (53 byte)
    // BackboneSetting_001: Src: e97b8a9d672e4ce4845ec6947cd66ef6-sb.baas.nintendo.com (53 byte), Dest: e97b8a9d672e4ce4845ec6947cd66ef6-sb.baas.nintendo.com (53 byte)

    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ImportSettings(
            static_cast<const void*>(SampleInput), sizeof(SampleInput),
            g_WorkBuffer, sizeof(g_WorkBuffer),
            nn::nsd::ImportMode_ApplicationSettings) );

    //
    // セーブデータを直接 Read して中身の確認
    //
    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ReadSaveDataFromFsForTest(&g_SaveData));
    //EXPECT_STREQ("SDK自動テスト dd1 [2016-09-06 00:15:08 UTC] dd1", g_SaveData.settingName.value);
    EXPECT_STREQ("SDK\xe8\x87\xaa\xe5\x8b\x95\xe3\x83\x86\xe3\x82\xb9\xe3\x83\x88 dd1 [2016-09-06 00:15:08 UTC] dd1", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);

    EXPECT_STREQ("e97b8a9d672e4ce4845ec6947cd66ef6-sb.accounts.nintendo.com",
        g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("e97b8a9d672e4ce4845ec6947cd66ef6-sb-api.accounts.nintendo.com",
        g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);

    EXPECT_STREQ("NxAccount",
        g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(12855230034832359759ULL,
        g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nintendo://account.nx.sys",
        g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("NxShop",
        g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceName.value);
    EXPECT_EQ(15160208408304773227ULL,
        g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.clientId);
    EXPECT_STREQ("nintendo://shop.nx.sys",
        g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("NxFriends",
        g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceName.value);
    EXPECT_EQ(17427995557043336505ULL,
        g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceSetting.clientId);
    EXPECT_STREQ("nintendo://friends.nx.sys",
        g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceSetting.redirectUri.value);

    EXPECT_EQ(0x78f7d9880, g_SaveData.expireUnixTime);
    EXPECT_EQ(1, g_SaveData.version);
    EXPECT_STREQ("0", g_SaveData.deviceId.value);

    DeleteSettings(nn::nsd::DeleteMode_All);
}

TEST_F(NsdTest_ForMenu, ImportSettings3)
{
    // dd1 注入
    const char Dd1Settings[] =
        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJTZXR0aW5nTmFtZSI6IlNES-iHquWLleODhuOCueODiCBkZDEgWzIwMTYtMDktMDYgMDA6MTU6MDggVVRDXSBkZDEiLCJFbnZpcm9ubWVudCI6ImRkMSIsIkJhY2tib25lU2V0dGluZ3MiOnsiTmFzIjp7IlJlcXVlc3RGcWRuIjoiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiQXBpRnFkbiI6ImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLWFwaS5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJOeEFjY291bnQiOnsiQ2xpZW50SWQiOjEyODU1MjMwMDM0ODMyMzU5NzU5LCJSZWRpcmVjdFVyaSI6Im5pbnRlbmRvOi8vYWNjb3VudC5ueC5zeXMifSwiTnhTaG9wIjp7IkNsaWVudElkIjoxNTE2MDIwODQwODMwNDc3MzIyNywiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL3Nob3Aubnguc3lzIn0sIk54RnJpZW5kcyI6eyJDbGllbnRJZCI6MTc0Mjc5OTU1NTcwNDMzMzY1MDUsIlJlZGlyZWN0VXJpIjoibmludGVuZG86Ly9mcmllbmRzLm54LnN5cyJ9fSwiRnFkbkVudHJpZXMiOltbImUwZDY3YzUwOWZiMjAzODU4ZWJjYjJmZTNmODhjMmFhLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXSxbImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXV19LCJBcHBsaWNhdGlvblNldHRpbmdzIjp7IkZxZG5FbnRyaWVzIjpbXX0sIkV4cGlyZSI6MzI0NzIxNDQwMDAsIlZlcnNpb24iOjEsIkRldmljZUlkIjoiMCJ9.F-DS9hfdNQGRHBQ9P50bhcYPtpP83jbvkNR9ewhpm5Ik_8eDHyi84VI-GYMYVWqcZ9eHa3AkT_JPjLWCnl2bMa66RjwWn0KxoBWEPyqCL7Bo0gJ15MrCKVxLQJcXTWoJKQ8Nsa3gjAn9cLNL6yZsCgkoJngIXUyXfiGlbJEy9Fe0SQv6B7J0gy-O_pZZgbP09wc4kJAdoAUhpsjbb-4iPe7ZlSraX9DyHwCUv0zE1EC8tAZyqpMx_9yzJcKv1h9_i-PHbpDW1odJbq0HNCk_CHp4va7-hZWIL3AvHBa2z9j1g0DdPbkiaROa6gsQThoWkGzbDjju1cYnNSiFVDSOhw";
    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ImportSettings(
            static_cast<const void*>(Dd1Settings), sizeof(Dd1Settings),
            g_WorkBuffer, sizeof(g_WorkBuffer),
            nn::nsd::ImportMode_All) );

    // ImportMode_ApplicationSettings で td1 注入 -> dd1 の書き換えができないので失敗するはず
    const char Td1Setting[] =
        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJTZXR0aW5nTmFtZSI6IlNES-iHquWLleODhuOCueODiCB0ZDEgWzIwMTYtMDktMDYgMDA6MTU6MDggVVRDXSB0ZDEiLCJFbnZpcm9ubWVudCI6InRkMSIsIkJhY2tib25lU2V0dGluZ3MiOnsiTmFzIjp7IlJlcXVlc3RGcWRuIjoiZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiQXBpRnFkbiI6ImUwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLWFwaS5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJOeEFjY291bnQiOnsiQ2xpZW50SWQiOjE0NTk4NTkyOTkyMDkwMzI4MTcwLCJSZWRpcmVjdFVyaSI6Im5pbnRlbmRvOi8vYWNjb3VudC5ueC5zeXMifSwiTnhTaG9wIjp7IkNsaWVudElkIjoxMzEyMzg5NDc4OTk5MzM1NTQ2MywiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL3Nob3Aubnguc3lzIn0sIk54RnJpZW5kcyI6eyJDbGllbnRJZCI6NjUxODQ3NjIzODkxMDE3Mzc2MSwiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL2ZyaWVuZHMubnguc3lzIn19LCJGcWRuRW50cmllcyI6W1siZTBkNjdjNTA5ZmIyMDM4NThlYmNiMmZlM2Y4OGMyYWEuYmFhcy5uaW50ZW5kby5jb20iLCJlMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvLmNvbSJdLFsiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iLCJlMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvLmNvbSJdXX0sIkFwcGxpY2F0aW9uU2V0dGluZ3MiOnsiRnFkbkVudHJpZXMiOltdfSwiRXhwaXJlIjozMjQ3MjE0NDAwMCwiVmVyc2lvbiI6MSwiRGV2aWNlSWQiOiIwIn0.MLHj47LBCSgXvolBFl5a78tdTfvbwwzRahn7BEBxamk5YqB8nQ6JJSeGtE4rNbcKrAl5naieoXxv2Ch24OFCqpobla77M8qq9vgtQWnXCRZPVs_o5WfZ0mj3pF20C7myRvmB81S47bDNyIGSTENaZdJv9mb2e0CgJ_3S5L1GLFzV1_fBYUcAUeAJ9OJCcNthpgcQKle1LmRM2mdmNCMejTPUSU1VIEi15rJD3x_16B90TI8MJ6ipfXTiHotUEh5k6hN1T3kHJtaBEnhN8ddG7UUp_AB7_m53aw1wAY-MWZ0eMzQFY4i1ESqxo8WAoKd6XdehBK7vZkC_1wj2z_f9UA";
    NNT_ASSERT_RESULT_FAILURE(ResultEnvironmentIdentifierNotUpdated, nn::nsd::ImportSettings(
            static_cast<const void*>(Td1Setting), sizeof(Td1Setting),
            g_WorkBuffer, sizeof(g_WorkBuffer),
            nn::nsd::ImportMode_ApplicationSettings) );

    DeleteSettings(nn::nsd::DeleteMode_All);
}

TEST_F(NsdTest_ForMenu, DeleteMode_All)
{
    DeleteSettings(nn::nsd::DeleteMode_All);

    NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultNotFound, nn::nsd::ReadSaveDataFromFsForTest(&g_SaveData));
}

TEST_F(NsdTest_ForMenu, DeleteMode_ApplicationSettings)
{
    const char SampleInput[] =
        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJTZXR0aW5nTmFtZSI6IlNES-iHquWLleODhuOCueODiCBkZDEgWzIwMTYtMDktMDYgMDA6MTU6MDggVVRDXSBkZDEiLCJFbnZpcm9ubWVudCI6ImRkMSIsIkJhY2tib25lU2V0dGluZ3MiOnsiTmFzIjp7IlJlcXVlc3RGcWRuIjoiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiQXBpRnFkbiI6ImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLWFwaS5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJOeEFjY291bnQiOnsiQ2xpZW50SWQiOjEyODU1MjMwMDM0ODMyMzU5NzU5LCJSZWRpcmVjdFVyaSI6Im5pbnRlbmRvOi8vYWNjb3VudC5ueC5zeXMifSwiTnhTaG9wIjp7IkNsaWVudElkIjoxNTE2MDIwODQwODMwNDc3MzIyNywiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL3Nob3Aubnguc3lzIn0sIk54RnJpZW5kcyI6eyJDbGllbnRJZCI6MTc0Mjc5OTU1NTcwNDMzMzY1MDUsIlJlZGlyZWN0VXJpIjoibmludGVuZG86Ly9mcmllbmRzLm54LnN5cyJ9fSwiRnFkbkVudHJpZXMiOltbImUwZDY3YzUwOWZiMjAzODU4ZWJjYjJmZTNmODhjMmFhLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXSxbImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXV19LCJBcHBsaWNhdGlvblNldHRpbmdzIjp7IkZxZG5FbnRyaWVzIjpbXX0sIkV4cGlyZSI6MzI0NzIxNDQwMDAsIlZlcnNpb24iOjEsIkRldmljZUlkIjoiMCJ9.F-DS9hfdNQGRHBQ9P50bhcYPtpP83jbvkNR9ewhpm5Ik_8eDHyi84VI-GYMYVWqcZ9eHa3AkT_JPjLWCnl2bMa66RjwWn0KxoBWEPyqCL7Bo0gJ15MrCKVxLQJcXTWoJKQ8Nsa3gjAn9cLNL6yZsCgkoJngIXUyXfiGlbJEy9Fe0SQv6B7J0gy-O_pZZgbP09wc4kJAdoAUhpsjbb-4iPe7ZlSraX9DyHwCUv0zE1EC8tAZyqpMx_9yzJcKv1h9_i-PHbpDW1odJbq0HNCk_CHp4va7-hZWIL3AvHBa2z9j1g0DdPbkiaROa6gsQThoWkGzbDjju1cYnNSiFVDSOhw";

    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ImportSettings(
            static_cast<const void*>(SampleInput), sizeof(SampleInput),
            g_WorkBuffer, sizeof(g_WorkBuffer),
            nn::nsd::ImportMode_ApplicationSettings) );

    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ReadSaveDataFromFsForTest(&g_SaveData));

    nn::util::Strlcpy(g_SaveData.applicationSettings.fqdnEntries[0].src.value, "hoge", static_cast<int>(Fqdn::Size));
    nn::util::Strlcpy(g_SaveData.applicationSettings.fqdnEntries[0].dst.value, "aho", static_cast<int>(Fqdn::Size));

    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::WriteSaveDataToFsForTest(g_SaveData));

    // ちゃんと設定が保存されたか確認
    std::memset(&g_SaveData, 0, sizeof(g_SaveData));
    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ReadSaveDataFromFsForTest(&g_SaveData));
    //EXPECT_STREQ("SDK自動テスト dd1 [2016-09-06 00:15:08 UTC] dd1", g_SaveData.settingName.value);
    EXPECT_STREQ("SDK\xe8\x87\xaa\xe5\x8b\x95\xe3\x83\x86\xe3\x82\xb9\xe3\x83\x88 dd1 [2016-09-06 00:15:08 UTC] dd1", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("hoge", g_SaveData.applicationSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("aho", g_SaveData.applicationSettings.fqdnEntries[0].dst.value);

    DeleteSettings(nn::nsd::DeleteMode_ApplicationSettings);
    // 保存されている設定が dd1 で、削除後も dd1 になるので
    // DeleteMode_ApplicationSettings 指定だが、全部削除される。
    NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultNotFound, nn::nsd::ReadSaveDataFromFsForTest(&g_SaveData));
}

TEST_F(NsdTest_ForMenu, IsChangeEnvironmentIdentifierDisabled1)
{
    bool isDisabledBackup = IsChangeEnvironmentIdentifierDisabled();
    NN_UTIL_SCOPE_EXIT
    {
        if(isDisabledBackup)
        {
            DisableChangeEnvironmentIdentifier();
        }
        else
        {
            EnableChangeEnvironmentIdentifier();
        }
    };

    DisableChangeEnvironmentIdentifier();
    EXPECT_TRUE(IsChangeEnvironmentIdentifierDisabled());

    EnableChangeEnvironmentIdentifier();
    EXPECT_FALSE(IsChangeEnvironmentIdentifierDisabled());

    DisableChangeEnvironmentIdentifier();
    EXPECT_TRUE(IsChangeEnvironmentIdentifierDisabled());

    EnableChangeEnvironmentIdentifier();
    EXPECT_FALSE(IsChangeEnvironmentIdentifierDisabled());
}

TEST_F(NsdTest_ForMenu, IsChangeEnvironmentIdentifierDisabled2)
{
    bool isDisabledBackup = IsChangeEnvironmentIdentifierDisabled();
    NN_UTIL_SCOPE_EXIT
    {
        if(isDisabledBackup)
        {
            DisableChangeEnvironmentIdentifier();
        }
        else
        {
            EnableChangeEnvironmentIdentifier();
        }
    };


    // 環境識別子を変更する設定インポートの有効化
    EnableChangeEnvironmentIdentifier();
    const char Td1Setting[] =
        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJTZXR0aW5nTmFtZSI6IlNES-iHquWLleODhuOCueODiCB0ZDEgWzIwMTYtMDktMDYgMDA6MTU6MDggVVRDXSB0ZDEiLCJFbnZpcm9ubWVudCI6InRkMSIsIkJhY2tib25lU2V0dGluZ3MiOnsiTmFzIjp7IlJlcXVlc3RGcWRuIjoiZTAzYTk3ODE5Yzk3MTFlNTk1MTBkODIwYTUyZjI5OGEtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiQXBpRnFkbiI6ImUwM2E5NzgxOWM5NzExZTU5NTEwZDgyMGE1MmYyOThhLXNiLWFwaS5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJOeEFjY291bnQiOnsiQ2xpZW50SWQiOjE0NTk4NTkyOTkyMDkwMzI4MTcwLCJSZWRpcmVjdFVyaSI6Im5pbnRlbmRvOi8vYWNjb3VudC5ueC5zeXMifSwiTnhTaG9wIjp7IkNsaWVudElkIjoxMzEyMzg5NDc4OTk5MzM1NTQ2MywiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL3Nob3Aubnguc3lzIn0sIk54RnJpZW5kcyI6eyJDbGllbnRJZCI6NjUxODQ3NjIzODkxMDE3Mzc2MSwiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL2ZyaWVuZHMubnguc3lzIn19LCJGcWRuRW50cmllcyI6W1siZTBkNjdjNTA5ZmIyMDM4NThlYmNiMmZlM2Y4OGMyYWEuYmFhcy5uaW50ZW5kby5jb20iLCJlMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvLmNvbSJdLFsiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iLCJlMDNhOTc4MTljOTcxMWU1OTUxMGQ4MjBhNTJmMjk4YS1zYi5iYWFzLm5pbnRlbmRvLmNvbSJdXX0sIkFwcGxpY2F0aW9uU2V0dGluZ3MiOnsiRnFkbkVudHJpZXMiOltdfSwiRXhwaXJlIjozMjQ3MjE0NDAwMCwiVmVyc2lvbiI6MSwiRGV2aWNlSWQiOiIwIn0.MLHj47LBCSgXvolBFl5a78tdTfvbwwzRahn7BEBxamk5YqB8nQ6JJSeGtE4rNbcKrAl5naieoXxv2Ch24OFCqpobla77M8qq9vgtQWnXCRZPVs_o5WfZ0mj3pF20C7myRvmB81S47bDNyIGSTENaZdJv9mb2e0CgJ_3S5L1GLFzV1_fBYUcAUeAJ9OJCcNthpgcQKle1LmRM2mdmNCMejTPUSU1VIEi15rJD3x_16B90TI8MJ6ipfXTiHotUEh5k6hN1T3kHJtaBEnhN8ddG7UUp_AB7_m53aw1wAY-MWZ0eMzQFY4i1ESqxo8WAoKd6XdehBK7vZkC_1wj2z_f9UA";
    NNT_ASSERT_RESULT_SUCCESS(nn::nsd::ImportSettings(
            static_cast<const void*>(Td1Setting), sizeof(Td1Setting),
            g_WorkBuffer, sizeof(g_WorkBuffer),
            nn::nsd::ImportMode_All) );

    // 環境識別子を変更する設定インポートの無効化
    DisableChangeEnvironmentIdentifier();

    const char Dd1Setting[] =
        "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiJ9.eyJTZXR0aW5nTmFtZSI6IlNES-iHquWLleODhuOCueODiCBkZDEgWzIwMTYtMDktMDYgMDA6MTU6MDggVVRDXSBkZDEiLCJFbnZpcm9ubWVudCI6ImRkMSIsIkJhY2tib25lU2V0dGluZ3MiOnsiTmFzIjp7IlJlcXVlc3RGcWRuIjoiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYWNjb3VudHMubmludGVuZG8uY29tIiwiQXBpRnFkbiI6ImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLWFwaS5hY2NvdW50cy5uaW50ZW5kby5jb20iLCJOeEFjY291bnQiOnsiQ2xpZW50SWQiOjEyODU1MjMwMDM0ODMyMzU5NzU5LCJSZWRpcmVjdFVyaSI6Im5pbnRlbmRvOi8vYWNjb3VudC5ueC5zeXMifSwiTnhTaG9wIjp7IkNsaWVudElkIjoxNTE2MDIwODQwODMwNDc3MzIyNywiUmVkaXJlY3RVcmkiOiJuaW50ZW5kbzovL3Nob3Aubnguc3lzIn0sIk54RnJpZW5kcyI6eyJDbGllbnRJZCI6MTc0Mjc5OTU1NTcwNDMzMzY1MDUsIlJlZGlyZWN0VXJpIjoibmludGVuZG86Ly9mcmllbmRzLm54LnN5cyJ9fSwiRnFkbkVudHJpZXMiOltbImUwZDY3YzUwOWZiMjAzODU4ZWJjYjJmZTNmODhjMmFhLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXSxbImU5N2I4YTlkNjcyZTRjZTQ4NDVlYzY5NDdjZDY2ZWY2LXNiLmJhYXMubmludGVuZG8uY29tIiwiZTk3YjhhOWQ2NzJlNGNlNDg0NWVjNjk0N2NkNjZlZjYtc2IuYmFhcy5uaW50ZW5kby5jb20iXV19LCJBcHBsaWNhdGlvblNldHRpbmdzIjp7IkZxZG5FbnRyaWVzIjpbXX0sIkV4cGlyZSI6MzI0NzIxNDQwMDAsIlZlcnNpb24iOjEsIkRldmljZUlkIjoiMCJ9.F-DS9hfdNQGRHBQ9P50bhcYPtpP83jbvkNR9ewhpm5Ik_8eDHyi84VI-GYMYVWqcZ9eHa3AkT_JPjLWCnl2bMa66RjwWn0KxoBWEPyqCL7Bo0gJ15MrCKVxLQJcXTWoJKQ8Nsa3gjAn9cLNL6yZsCgkoJngIXUyXfiGlbJEy9Fe0SQv6B7J0gy-O_pZZgbP09wc4kJAdoAUhpsjbb-4iPe7ZlSraX9DyHwCUv0zE1EC8tAZyqpMx_9yzJcKv1h9_i-PHbpDW1odJbq0HNCk_CHp4va7-hZWIL3AvHBa2z9j1g0DdPbkiaROa6gsQThoWkGzbDjju1cYnNSiFVDSOhw";

    // td1 がインポート済で、dd1 を Import しようとする.
    // ImportMode_All でも環境識別子が変わるものはインポートできない
    NNT_ASSERT_RESULT_FAILURE(
        ResultEnvironmentIdentifierNotUpdated,
        nn::nsd::ImportSettings(
            static_cast<const void*>(Dd1Setting), sizeof(Dd1Setting),
            g_WorkBuffer, sizeof(g_WorkBuffer), nn::nsd::ImportMode_All) );
}

TEST_F(NsdTest_ForMenu, Finish)
{
    // 最後に全削除しとく
    DeleteSettings(nn::nsd::DeleteMode_All);
}
