﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/nim/srv/nim_NetworkSystemUpdateTask.h>
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_CommandOptions.h"
#include "testNs_NcmStorageVerify.h"

#define LOG_OUT( ... )  NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace {

    //!--------------------------------------------------------------------------------------
    //! 配列要素数算出.
    //!--------------------------------------------------------------------------------------
    template< typename TYPE, size_t SIZE >
    inline static unsigned CountOf( const TYPE( & )[ SIZE ] ) {
        return SIZE;
    }

    //!--------------------------------------------------------------------------------------
    //! Result出力.
    //!--------------------------------------------------------------------------------------
    static void LogoutResultValue( const nn::Result& result, const char* const pTitle )
    {
        if ( nullptr != pTitle )
        {
            LOG_OUT( "%s\n", pTitle );
        }
        LOG_OUT( "-------- Result ---------\n" );
        LOG_OUT( "module = %u\n", result.GetModule() );
        LOG_OUT( "description = %u\n", result.GetDescription() );
        LOG_OUT( "inner value = 0x%08x\n", result.GetInnerValueForDebug() );
        LOG_OUT( "-------------------------\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! Case Setup
    //!--------------------------------------------------------------------------------------
    class NupCase : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
    private:
        NN_ALIGNAS(4096) static uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
#endif

    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ovln::PrepareSenderAndReceiverForDevelop();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
            nn::nim::InitializeForNetworkInstallManager();
#endif

            nn::ns::Initialize();

            auto& verifierInstance = nn::util::Get( m_NcmStorageVerifier );
            new ( &verifierInstance ) NcmStorageVerifier( nn::ncm::StorageId::BuildInSystem );
            verifierInstance.Initialize();
        }

        virtual void TearDown()
        {
            nn::util::Get( m_NcmStorageVerifier ).Finalize();
            nn::ns::Finalize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::nim::FinalizeForNetworkInstallManager();
            nn::ovln::FinalizeSenderLibraryForOverlay();
            nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
#endif

            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NNT_EXPECT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                                                             nn::socket::DefaultSocketMemoryPoolSize,
                                                             nn::socket::DefaultSocketAllocatorSize,
                                                             nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);
#endif

            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::nifm::Initialize() );
            NN_ABORT_UNLESS_RESULT_SUCCESS( testns::MountHostFileSystem( "Contents" ) );
            testns::InitializeCommandLineOptions();
        }

        static void TearDownTestCase()
        {
            testns::FinalizeCommandLineOptions();
            testns::UnmountHostFileSystem();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            curl_global_cleanup();
            nn::socket::Finalize();
#endif
        }

    protected:
        nn::util::TypedStorage< NcmStorageVerifier, sizeof( NcmStorageVerifier ), NN_ALIGNOF( NcmStorageVerifier ) > m_NcmStorageVerifier;
    };

#if defined( NN_BUILD_CONFIG_OS_WIN )
    NN_ALIGNAS(4096) uint8_t NupCase::s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize] = {0};
#endif



    // AsyncUtils
    template< typename THandle, typename TControl = nn::ns::SystemUpdateControl >
    class RequestSynchronizer
    {
    public:
        typedef TControl    ControlType;
        typedef THandle     HandleType;

        HandleType& GetHandle() NN_NOEXCEPT
        {
            return m_Handle;
        }

        const nn::TimeSpan GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

        nn::Result WaitResponse() NN_NOEXCEPT
        {
            const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            auto result = OnWaitResponse( m_Control, m_Handle );
            m_ElapsedTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            LOG_OUT( "Response wait : %lld ms.\n", m_ElapsedTime.GetMilliSeconds() );
            return result;
        }

    protected:
        explicit RequestSynchronizer( ControlType& control ) NN_NOEXCEPT
            : m_Control( control )
        {
        }

        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT = 0;

    private:
        ControlType&    m_Control;
        HandleType      m_Handle;
        nn::TimeSpan    m_ElapsedTime;
    };

    // CheckLatestUpdate
    class NewlyUpdateInquirer : public RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate >
    {
    public:
        static const nn::ns::LatestSystemUpdate INVALID_LATEST_VALUE = static_cast< nn::ns::LatestSystemUpdate >( -1 );
        typedef RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate > BaseType;

        explicit NewlyUpdateInquirer( ControlType& control ) NN_NOEXCEPT
            : BaseType( control ), m_LatestStatus( INVALID_LATEST_VALUE )
        {
        }

        const nn::ns::LatestSystemUpdate& GetLatestStatus() const NN_NOEXCEPT
        {
            return m_LatestStatus;
        }

        const char* GetLatestStatusString() const NN_NOEXCEPT
        {
            switch ( m_LatestStatus )
            {
            case nn::ns::LatestSystemUpdate::Downloaded:    return "Downloaded";
            case nn::ns::LatestSystemUpdate::NeedsDownload: return "NeedsDownload";
            case nn::ns::LatestSystemUpdate::UpToDate:      return "UpToDate";
            default:                                        return "Unknown";
            }
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            while ( !handle.TryWait() )
            {
                LOG_OUT( "Wating check latest update...\n" );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            return handle.Get( &m_LatestStatus );
        }

    private:
        nn::ns::LatestSystemUpdate m_LatestStatus;
    };

    // DownloadLatestUpdate
    class NewlyUpdateDownloader : public RequestSynchronizer< nn::ns::AsyncResult >
    {
    public:
        typedef RequestSynchronizer< nn::ns::AsyncResult > BaseType;

        explicit NewlyUpdateDownloader( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

        const char* GetBackStateString() const NN_NOEXCEPT
        {
            switch ( nn::ns::GetBackgroundNetworkUpdateState() )
            {
            case nn::ns::BackgroundNetworkUpdateState::None:        return "None";
            case nn::ns::BackgroundNetworkUpdateState::InProgress:  return "InProgress";
            case nn::ns::BackgroundNetworkUpdateState::Ready:       return "Ready";
            default:                                                return "Unknown";
            }
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            while ( !handle.TryWait() )
            {
                auto progress = control.GetDownloadProgress();
                LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            auto progress = control.GetDownloadProgress();
            LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
            return handle.Get();
        }
    };

    // ApplyDownloadedUpdate
    class NewlyUpdateInstaller : public RequestSynchronizer< int >
    {
    public:
        typedef RequestSynchronizer< int > BaseType;

        explicit NewlyUpdateInstaller( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            //  TODO 多分、これも ForDebug() 作らんといかんかも. ( エラー時はABORTするはず )
            control.ApplyDownloadedUpdate();
            NN_RESULT_SUCCESS;
        }
    };

    //!--------------------------------------------------------------------------------------
    //! Install procedure
    //!--------------------------------------------------------------------------------------
    static void ExecuteInstall( bool& outInstallSuccess, nn::ns::SystemUpdateControl& control, const bool bExpectLimitOver ) NN_NOEXCEPT
    {
        // インストール. ApplyDownloadedUpdate
        LOG_OUT( "Start install downloaded update... On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate]\n" );
        NewlyUpdateInstaller installer( control );
        const auto result = installer.WaitResponse();
        outInstallSuccess = result.IsSuccess();
        if ( false == bExpectLimitOver )
        {
            NNT_ASSERT_RESULT_SUCCESS( result );
        }
        else if ( outInstallSuccess )
        {
            outInstallSuccess = false;
            GTEST_FAIL() << "Unexpected success on [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate].";
        }
        else
        {
            outInstallSuccess = true;
            LogoutResultValue( result, "Happened an expected failure on [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate]." );
        }
        LOG_OUT( "Complete install downloaded update.\n" );
    }
}

TEST_F( NupCase, Simple )
{
    // インストール上限突破を目的としたテスト時は true.
    const bool bExpectLimitOver = testns::QueryCommandLineOptions( "--testns_ExpectLimitOver" );

    // ストレージ上のインストールコンテンツ及び本体更新対象コンテンツを含むコレクション生成
    auto& verifier = nn::util::Get( m_NcmStorageVerifier );
    verifier.PrepareUnexpectedContentOnStorage( bExpectLimitOver );

    // ネットワーク接続確立.
    LOG_OUT( "Request network connection ...\n" );
    const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
    nn::nifm::SubmitNetworkRequestAndWait();
    const nn::TimeSpan spanOnConnection = nn::os::GetSystemTick().ToTimeSpan() - before;
    LOG_OUT( "Response wait : %lld ms.\n", spanOnConnection.GetMilliSeconds() );
    const auto availableNetwork = nn::nifm::IsNetworkAvailable();
    LOG_OUT( "Network connection %s.\n", availableNetwork ? "succeeded" : "failed" );
    ASSERT_EQ( true, availableNetwork );

    // 本体更新制御の占有開始.
    nn::ns::SystemUpdateControl control;
    NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

    bool newlyInstalled = false;
    {
        nn::ns::LatestSystemUpdate latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
        {
            // 最新の更新の検出. RequestCheckLatestUpdate
            LOG_OUT( "Start latest update checking... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
            NewlyUpdateInquirer inquirer( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
            latest = inquirer.GetLatestStatus();
            EXPECT_EQ( nn::ns::LatestSystemUpdate::NeedsDownload, latest );
            LOG_OUT( "Complete latest update checking [ %s ].\n", inquirer.GetLatestStatusString() );
        }

        if ( nn::ns::LatestSystemUpdate::NeedsDownload == latest )
        {
            bool downloaded = false;
            {
                // ダウンロード. RequestDownloadLatestUpdate
                LOG_OUT( "Start latest update downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]\n" );
                NewlyUpdateDownloader downloader( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( downloader.WaitResponse() );
                LOG_OUT( "Complete latest update downloading.\n" );
                downloaded = control.HasDownloaded();
                EXPECT_EQ( true, downloaded );
            }
            if ( downloaded )
            {
                // インストール. ApplyDownloadedUpdate
                ExecuteInstall( newlyInstalled, control, bExpectLimitOver );
            }
        }
        else if ( nn::ns::LatestSystemUpdate::Downloaded == latest )
        {
            // インストール. ApplyDownloadedUpdate
            ExecuteInstall( newlyInstalled, control, bExpectLimitOver );
        }
    }
    if ( true == newlyInstalled )
    {
        if ( false == bExpectLimitOver )
        {
            nn::ns::LatestSystemUpdate latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
            {
                // インストール済確認. RequestCheckLatestUpdate == nn::ns::LatestSystemUpdate::UpToDate
                LOG_OUT( "Start verify latest update status... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
                NewlyUpdateInquirer inquirer( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
                latest = inquirer.GetLatestStatus();
                EXPECT_EQ( nn::ns::LatestSystemUpdate::UpToDate, latest );
                LOG_OUT( "Complete verify latest update status [ %s ].\n", inquirer.GetLatestStatusString() );
            }

            if ( nn::ns::LatestSystemUpdate::UpToDate == latest )
            {
                // NCMデータベースベリファイ
                verifier.VerifyHasContentOnStorage();
                verifier.VerifyUnnecessaryPlaceHolderOnStorage();
                verifier.VerifyUnexpectedContentOnStorage();

                // ダウンロード済コンテンツの消去確認. RequestDownloadLatestUpdate != nn::ns::ResultAlreadyUpToDate
                LOG_OUT( "Verify whether the downloaded contents be clean... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]\n" );
                NewlyUpdateDownloader downloader( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
                NNT_EXPECT_RESULT_FAILURE( nn::ns::ResultAlreadyUpToDate, downloader.WaitResponse() );
                LOG_OUT( "Complete, verify whether the downloaded contents be clean.\n" );
            }
        }
        else
        {
            // NCMデータベースベリファイ
            verifier.VerifyHasContentOnStorage();
            verifier.VerifyUnnecessaryPlaceHolderOnStorage();
            verifier.VerifyUnexpectedContentOnStorage();
        }
    }
} // NOLINT(impl/function_size)
