﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include "libraries/testNs_XmlParser.h"
#include <vector>

#include <functional>   // std::hash
#include <string>       // std::hash< std::string >
#include <unordered_map>

//!--------------------------------------------------------------------------------------
//! @brief コンテンツIDコレクション
//!--------------------------------------------------------------------------------------
class ContentIdCollection : public std::unordered_map< std::string, nn::ncm::ContentId >
{
    typedef std::unordered_map< std::string, nn::ncm::ContentId > CollectionType;

public:
    //!---------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!---------------------------------------------------------------------------------
    explicit ContentIdCollection( const unsigned reservedCapacity = 128 ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief デストラクタ
    //!---------------------------------------------------------------------------------
    ~ContentIdCollection() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief クリーン
    //!---------------------------------------------------------------------------------
    void Clean() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief 任意のコンテンツIDの追加
    //!---------------------------------------------------------------------------------
    void Entry( const nn::ncm::ContentId& contentId ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief ストレージ上のコンテンツIDを取得
    //!---------------------------------------------------------------------------------
    void LoadFrom( nn::ncm::ContentStorage& storage ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief 検索
    //!---------------------------------------------------------------------------------
    const bool Find( const nn::ncm::ContentId& contentId ) const NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief 比較
    //!---------------------------------------------------------------------------------
    const bool Compare( const ContentIdCollection& other ) const NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief コレクションダンプ
    //!---------------------------------------------------------------------------------
    void Dump( const char* pPrefix = nullptr ) const NN_NOEXCEPT;

protected:
    //!---------------------------------------------------------------------------------
    //! @brief 検索
    //!---------------------------------------------------------------------------------
    const bool Find( const std::string& contentId ) const NN_NOEXCEPT;
};

//!--------------------------------------------------------------------------------------
//! @brief ベリファイ対象メタキーオブジェクト
//!--------------------------------------------------------------------------------------
class VerifyContentMetaKey : public nn::ncm::ContentMetaKey, public std::vector< nn::ncm::ContentInfo >
{
    typedef std::vector< nn::ncm::ContentInfo >  CollectionType;

public:
    //!---------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!---------------------------------------------------------------------------------
    explicit VerifyContentMetaKey( const unsigned reservedCapacity = 32 ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief デストラクタ
    //!---------------------------------------------------------------------------------
    ~VerifyContentMetaKey() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief クリーン
    //!---------------------------------------------------------------------------------
    void Clean() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief 設定
    //!---------------------------------------------------------------------------------
    void EntryMetaKey( const nn::Bit64& metaId, const uint32_t metaVersion, const nn::ncm::ContentMetaType metaType,
        const nn::ncm::ContentInstallType metaInstallType = nn::ncm::ContentInstallType::Full ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief コンテンツID追加
    //!---------------------------------------------------------------------------------
    void AppendContent( const nn::ncm::ContentId& contentId, const nn::ncm::ContentType contentType ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief コンテンツIDが含まれるか
    //!---------------------------------------------------------------------------------
    const bool HasContent( const nn::ncm::ContentId& contentId ) const NN_NOEXCEPT;
};

//!--------------------------------------------------------------------------------------
//! @brief ベリファイ対象キーコレクション
//!--------------------------------------------------------------------------------------
class ContentMetaKeyCollection : public std::vector< VerifyContentMetaKey >, protected XmlParser::CallbackInterface
{
    NN_DISALLOW_COPY( ContentMetaKeyCollection );
    NN_DISALLOW_MOVE( ContentMetaKeyCollection );

    typedef std::vector< VerifyContentMetaKey >     ParentType;
    typedef XmlParser::Attribute::CollectionType    AttrListType;

public:
    //!---------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!---------------------------------------------------------------------------------
    explicit ContentMetaKeyCollection( const unsigned reservedCapacity = 32 ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief デストラクタ
    //!---------------------------------------------------------------------------------
    ~ContentMetaKeyCollection() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief ファイナライザ
    //!---------------------------------------------------------------------------------
    void Finalize() NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief コレクションローダー ( from File )
    //!---------------------------------------------------------------------------------
    void LoadFromPropertyFile( const char* pFilePath ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief コレクションダンプ
    //!---------------------------------------------------------------------------------
    void Dump( const char* pPrefix = nullptr ) const NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief メタキーコレクション取得
    //!---------------------------------------------------------------------------------
    void ObtainMetaKeys( std::vector< nn::ncm::ContentMetaKey >& outCollection ) const NN_NOEXCEPT;

public:
    //!---------------------------------------------------------------------------------
    //! @brief 文字列シンボルから ContentMetaType へ変換。
    //!---------------------------------------------------------------------------------
    static const nn::ncm::ContentMetaType LookupContentMetaType( const std::string& expectTypeString ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief ContentMetaType から文字列シンボルへ変換。
    //!---------------------------------------------------------------------------------
    static const char* ToStringFromContentMetaType( const nn::ncm::ContentMetaType& type ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief 文字列シンボルから ContentType へ変換。
    //!---------------------------------------------------------------------------------
    static const nn::ncm::ContentType LookupContentType( const std::string& expectTypeString ) NN_NOEXCEPT;

    //!---------------------------------------------------------------------------------
    //! @brief ContentType から文字列シンボルへ変換。
    //!---------------------------------------------------------------------------------
    static const char* ToStringFromContentType( const nn::ncm::ContentType& type ) NN_NOEXCEPT;

protected:
    //!---------------------------------------------------------------------------------
    //! @brief
    //!---------------------------------------------------------------------------------
    virtual void OnElementBegin( const XmlParser::StringType& name, const XmlParser::Attribute::CollectionType& attributes ) NN_NOEXCEPT NN_OVERRIDE;

    //!---------------------------------------------------------------------------------
    //! @brief
    //!---------------------------------------------------------------------------------
    virtual void OnText( const XmlParser::StringType& text ) NN_NOEXCEPT NN_OVERRIDE;

    //!---------------------------------------------------------------------------------
    //! @brief
    //!---------------------------------------------------------------------------------
    virtual void OnElementEnd( const XmlParser::StringType& name ) NN_NOEXCEPT NN_OVERRIDE;

private:
    XmlParser*              m_pXmlParser;
    VerifyContentMetaKey    m_FocusedKey;
    bool                    m_DetectedMetaKey;
};

//!--------------------------------------------------------------------------------------
//! @brief データベース/ストレージ整合性検証
//!--------------------------------------------------------------------------------------
class NcmStorageVerifier
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit NcmStorageVerifier( const nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInSystem ) NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 初期化
    //!
    //! @pre
    //!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
    //!     - NcmStorageVerifier::Initialize が呼び出されていない。
    //!--------------------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 終了
    //!
    //! @pre
    //!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
    //!     - NcmStorageVerifier::Initialize が呼び出されている。
    //!--------------------------------------------------------------------------------------
    void Finalize() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 検証実施
    //! @details
    //!     - 期待するContentMetaKeyが全てデータベースに存在しているかチェックする。
    //!     - 期待するContentMetaKeyが持つコンテンツがストレージ上に存在するかチェックする。
    //!     - 期待するContentMetaKeyが持つコンテンツがデータベース上に存在するかチェックする。
    //!
    //! @pre
    //!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
    //!     - NcmStorageVerifier::Initialize が呼び出されている。
    //!--------------------------------------------------------------------------------------
    void VerifyHasContentOnStorage() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 検証実施
    //! @details プレースホルダが残っていないかチェックする。
    //!
    //! @pre
    //!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
    //!     - NcmStorageVerifier::Initialize が呼び出されている。
    //!--------------------------------------------------------------------------------------
    void VerifyUnnecessaryPlaceHolderOnStorage() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 検証準備
    //! @details 事前のインストールコンテンツを除外コンテンツとしてリストアップする。
    //!          本APIの除外コンテンツには、本体更新時に新たにインストールされる期待コンテンツも含まれる。
    //!
    //! @param[in]  ignoreExpectedContents  trueを指定した場合、以降の期待対象コンテンツに依存した検証をスキップします。
    //!                                     これは、不整合が確定しているなどの状態が予測される場合に使用します。
    //!
    //! @pre
    //!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
    //!     - NcmStorageVerifier::Initialize が呼び出されている。
    //! @see
    //!     - void VerifyUnexpectedContentOnStorage()
    //!--------------------------------------------------------------------------------------
    void PrepareUnexpectedContentOnStorage( const bool ignoreExpectedContents = false ) NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 検証実施
    //! @details 予期しないコンテンツが存在しないかチェックする。
    //!          事前に PrepareUnexpectedContentOnStorage によって除外コンテンツをリストアップしておく。
    //!          除外コンテンツ以外が検出された場合、NGとなる。
    //!
    //! @pre
    //!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
    //!     - NcmStorageVerifier::Initialize が呼び出されている。
    //! @see
    //!     - void PrepareUnexpectedContentOnStorage()
    //!--------------------------------------------------------------------------------------
    void VerifyUnexpectedContentOnStorage() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief コレクション取得
    //!
    //! @pre
    //!     - NcmStorageVerifier::Initialize が呼び出されている。
    //!--------------------------------------------------------------------------------------
    const ContentMetaKeyCollection& GetContentMetaKeyCollection() const NN_NOEXCEPT
    {
        return m_ExpectContents;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief データベース管理ストレージID取得
    //!
    //! @pre
    //!     - NcmStorageVerifier::Initialize が呼び出されている。
    //!--------------------------------------------------------------------------------------
    const nn::ncm::StorageId& GetStorageId() const NN_NOEXCEPT
    {
        return m_StorageId;
    }

private:
    nn::ncm::ContentMetaDatabase    m_ContentMetaDatabase;      //!< メタデータベースハンドル
    nn::ncm::ContentStorage         m_ContentStorage;           //!< コンテンツストレージハンドル
    ContentMetaKeyCollection        m_ExpectContents;           //!< 検証対象コンテンツコレクション
    ContentIdCollection             m_PreContentCollection;     //!< 本体更新前ストレージコンテンツリスト
    const nn::ncm::StorageId        m_StorageId;                //!< データベース管理ストレージタイプ識別子
    bool                            m_IgnoreExpectedContents;   //!< 期待更新対象コンテンツ検証を無視
};
