﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <cstdlib>
#include <functional>
#include "libraries/testNs_MessageThread.h"

namespace testns {

//!--------------------------------------------------------------------------------------
//! @brief １キュー送受信メッセージスレッド
//!--------------------------------------------------------------------------------------
template< const size_t ThreadStackByteSize = nn::os::ThreadStackAlignment * 4 >
class InstantMessenger
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief 要求送信用メッセージ
    //!--------------------------------------------------------------------------------------
    class Request : public testns::ObjectRecyclerBase< Request >::BasedNode, public testns::ThreadMessage
    {
    public:
        static const unsigned MessageQueueCapacity = 1;
        typedef testns::StaticObjectRecycler< Request, MessageQueueCapacity > RecyclerType;
        typedef testns::MessageThread< ThreadStackByteSize, MessageQueueCapacity > ThreadBaseType;

        //!--------------------------------------------------------------------------------------
        //! @brief
        //!--------------------------------------------------------------------------------------
        explicit Request( RecyclerType* pOwnerRecycler ) NN_NOEXCEPT
            : RecyclerType::BasedNode( pOwnerRecycler ) {}
    };

    //!--------------------------------------------------------------------------------------
    //! @brief 結果受信用メッセージ
    //!--------------------------------------------------------------------------------------
    class Reply : public testns::ObjectRecyclerBase< Reply >::BasedNode
    {
    public:
        static const unsigned MessageQueueCapacity = 1;
        typedef testns::StaticObjectRecycler< Reply, MessageQueueCapacity > RecyclerType;
        typedef testns::MessageReceiver< MessageQueueCapacity > ReplyReceiver;

        //!--------------------------------------------------------------------------------------
        //! @brief
        //!--------------------------------------------------------------------------------------
        explicit Reply( RecyclerType* pOwnerRecycler ) NN_NOEXCEPT
            : RecyclerType::BasedNode( pOwnerRecycler ) {}
    };

    //!--------------------------------------------------------------------------------------
    //! @brief スレッドコンテキスト
    //!--------------------------------------------------------------------------------------
    class ThreadContext : public Request::ThreadBaseType
    {
    public:
        //!--------------------------------------------------------------------------------------
        //! @brief スレッド基底
        //!--------------------------------------------------------------------------------------
        typedef typename Request::ThreadBaseType ThreadBaseType;

        //!--------------------------------------------------------------------------------------
        //! @brief コンストラクタ
        //!--------------------------------------------------------------------------------------
        explicit ThreadContext() NN_NOEXCEPT : m_pActiveRequest( nullptr )
        {
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 初期化
        //!--------------------------------------------------------------------------------------
        void Initialize() NN_NOEXCEPT
        {
            m_RequestRecycler.Initialize();
            m_ReplyRecycler.Initialize();
            m_ReplyReceiver.Initialize();
            ThreadBaseType::InitializeWithRelativePriority( 0 );
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 終了
        //!--------------------------------------------------------------------------------------
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
        {
            CancelAndWait();
            ThreadBaseType::Finalize();
            m_ReplyReceiver.Finalize();
            m_ReplyRecycler.Finalize();
            m_RequestRecycler.Finalize();
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 要求開始
        //! @details Request::MessageQueueCapacity が 1 の場合のみ有効です。
        //!--------------------------------------------------------------------------------------
        void Start( ThreadMessage::RunnableFunctionType pRunnable, void* pUserContext = nullptr ) NN_NOEXCEPT
        {
            auto& request = m_RequestRecycler.Obtain();
            m_pActiveRequest = &request;
            request.Initialize( pRunnable, pUserContext );
            ThreadBaseType::Send( request );
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 要求をキャンセルして並列動作の完了を待ちます。
        //! @details Request::MessageQueueCapacity が 1 の場合のみ有効です。
        //!--------------------------------------------------------------------------------------
        void CancelAndWait() NN_NOEXCEPT
        {
            // マルチにリクエストを実施する場合には m_pActiveRequestは外部で保持し、
            // スレッド運用終了後のリサイクルに伴い破棄される必要があります。
            // 但し、その場合後述 OnMessageReceived でコメントしている
            // 「ハンドル返却とリサイクルの同期」について熟慮が必要です。
            Request* pRequest;
            if ( nullptr != ( pRequest = m_pActiveRequest ) )
            {
                pRequest->Cancel();
            }
            // もし要求の複数並列受理( Queue > 1 )を実施するなら、
            // Replyメッセージクリア及び、Recyclerによる同期の手法は行えません。
            ClearReceivedReplyMessage();        // Replyキューをクリーン
            m_RequestRecycler.WaitCanObtain();  // Requestコンテナ取得可能まで待機
            ClearReceivedReplyMessage();        // Request可能になったら、再度 Replyキューをクリーン
        }

        //!--------------------------------------------------------------------------------------
        //! @brief メッセージ受信通知
        //!
        //! 正しく Request::Recycle() と ユーザのリクエストハンドル保持を同期させるには、
        //! ハンドルのリサイクルではなく"使い捨て"にする必要があります。
        //! リサイクルの場合の同期は、要求側スレッドの ReplyReceiverが 0U を受け取った時に、
        //! 初めて Recycleとハンドル放棄( m_pActiveRequest = null )を行うのが簡単です。
        //!
        //! ですが、このクラスは単一リクエストハンドルである前提のコードとして、
        //! スレッド上でリサイクルとハンドル返却を実施しています。
        //! 本来はリクエストハンドルをユーザに提供し、完了時にユーザ側で不要になったリクエストハンドルを
        //! 明示的にリサイクル( == ハンドルの返却 )を呼び出す必要があります。
        //!
        //! なので、正しい実装としては以下( 所謂WeakReference )のような機構が必要です。
        //! - ハンドルインスタンス( ThreadMessage, Request )を直接ユーザに渡さず、参照保持コンテナクラスを用意する。
        //! - 保持コンテナインスタンスをユーザが用意して、コンテナ内と実ハンドルインスタンス相互に相互依存関係を持たせる。
        //! - ハンドル無効化時はコンテナからのハンドル参照削除、コンテナ破棄時はハンドルからのコンテナ参照を削除。
        //! - 勿論、上記相互依存の関係削除処理はスレッドセーフに構築する。
        //! - コンテナはユーザ責務で破棄。
        //! スレッドクラスから作成している状態であれば、使い方に特化したadhocなハンドルライフサイクル制御が出来るので、
        //! このクラスは adhoc にやってます。
        //!--------------------------------------------------------------------------------------
        virtual void OnMessageReceived( testns::ThreadMessage* pMessage ) NN_NOEXCEPT NN_OVERRIDE
        {
            if ( nullptr == pMessage )
            {
                return; // 終了要求
            }
            Request* pRequest;
            if ( nullptr != ( pRequest = static_cast< Request* >( pMessage ) ) )
            {
                testns::ThreadMessage::RunnableFunctionType pRunnable;
                if ( nullptr != ( pRunnable = pMessage->GetRunnable() ) )
                {
                    pRunnable( this, pMessage );
                }
                m_pActiveRequest = nullptr; // 単一リクエストハンドル放棄
                m_ReplyReceiver.Send( 0U ); // 処理終端メッセージ送信
                pRequest->Recycle();        // クエリ要求ハンドルの返却
            }
        }

    protected:
        //!--------------------------------------------------------------------------------------
        //! @brief リプライメッセージの共通処理
        //!--------------------------------------------------------------------------------------
        void DoActionReceivedReplyMessage( std::function< void( Reply* ) > doRunnable ) NN_NOEXCEPT
        {
            uintptr_t receive;
            while ( true == m_ReplyReceiver.TryReceive( receive ) )
            {
                if ( 0U != receive )
                {
                    Reply* const pReply = reinterpret_cast< Reply* >( receive );
                    if ( nullptr != doRunnable )
                    {
                        doRunnable( pReply );
                    }
                    pReply->Recycle();
                }
                else if ( nullptr != doRunnable )
                {
                    doRunnable( nullptr );
                }
            }
        }

        //!--------------------------------------------------------------------------------------
        //! @brief キュークリア
        //!--------------------------------------------------------------------------------------
        void ClearReceivedReplyMessage() NN_NOEXCEPT
        {
            DoActionReceivedReplyMessage( nullptr );
        }

        Request* volatile               m_pActiveRequest;    //!< アクティブなクエリ要求ハンドル ( Request::MessageQueueCapacity が 1 の場合のみ )
        typename Request::RecyclerType  m_RequestRecycler;   //!< クエリ要求ハンドルリサイクラ
        typename Reply::ReplyReceiver   m_ReplyReceiver;     //!< クエリ結果受信機
        typename Reply::RecyclerType    m_ReplyRecycler;     //!< クエリ結果受信ハンドルリサイクラ
    };

};

} // ::testns

