﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Log.h>
#include "testNs_FsUtilities.h"

namespace fsutil {

bool IsExistPath( const char* inPath ) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entryType;

    // 指定パスの存在確認
    const nn::Result result = nn::fs::GetEntryType( &entryType, inPath );
    if( result.IsSuccess() == true )
    {
        // 指定されたファイルまたはディレクトリのパスが存在する
        return true;
    }

    if( nn::fs::ResultPathNotFound::Includes( result ) == true )
    {
        // 指定パスが存在しない場合はこちらに入るはず。ログを出してもよいが・・
    }
    else
    {
        NN_LOG( "Unexpected result\n" );
    }

    // 存在していてもエラーとなった場合も存在しない扱いとする？
    // ひとまず存在しない(false)を返しておく
    return false;
}

// -------------------------------------------------------------

Directory::Directory() NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
}

Directory::Directory( const char* inDirPath ) NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
    this->Open( inDirPath );
}

Directory::~Directory() NN_NOEXCEPT
{
    this->Close();
}

bool Directory::IsOpened() NN_NOEXCEPT
{
    return ( m_Handle.handle != nullptr ) ? true : false;
}

nn::Result Directory::Open( const char* inPath, int mode ) NN_NOEXCEPT
{
    // すでにクラス内で開いているディレクトリがあれば閉じる(強制的に指定したパスをオープンする形となる)
    this->Close();

    //GLV_LOG( "Call nn::fs::OpenDirectory()\n" );
    const auto result = nn::fs::OpenDirectory( &m_Handle, inPath, mode );
    if( result.IsFailure() )
    {
        SetResult( result );
        // オープンに失敗した場合は、念のためハンドルにNULLを設定しておく
        m_Handle.handle = nullptr;
    }

    return result;
}

void Directory::Close() NN_NOEXCEPT
{
    if( IsOpened() == true )
    {
        nn::fs::CloseDirectory( m_Handle );
        m_Handle.handle = nullptr;
    }
}

nn::Result Directory::Read( int64_t* outValue, nn::fs::DirectoryEntry* entryBuffer, int64_t entryBufferCount ) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadDirectory( outValue, entryBuffer, m_Handle ,entryBufferCount );
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

int64_t Directory::GetEntryCount() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒ひとまず0を返しておく

    int64_t count = 0;
    const auto result = nn::fs::GetDirectoryEntryCount( &count, m_Handle );
    if( result.IsFailure() )
    {
        SetResult( result );
        return 0;
    }

    return count;
}

// -------------------------------------------------------------

File::File() NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
}

File::File( const char* inFilePath ) NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
    this->Open( inFilePath );
}

File::~File() NN_NOEXCEPT
{
    this->Close();
}

bool File::IsOpened() NN_NOEXCEPT
{
    return ( m_Handle.handle != nullptr ) ? true : false;
}

nn::Result File::Open( const char* inPath, int mode ) NN_NOEXCEPT
{
    // すでにクラス内で開いているディレクトリがあれば閉じる(強制的に指定したパスをオープンする形となる)
    this->Close();

    //GLV_LOG( "Call nn::fs::OpenFile()\n" );
    const auto result = nn::fs::OpenFile( &m_Handle, inPath, mode );
    if( result.IsFailure() )
    {
        SetResult( result );
        // オープンに失敗した場合は、念のためハンドルにNULLを設定しておく
        m_Handle.handle = nullptr;
    }

    return result;
}

void File::Close() NN_NOEXCEPT
{
    if( IsOpened() == true )
    {
        //GLV_LOG( "Call nn::fs::CloseFile()\n" );
        nn::fs::CloseFile( m_Handle );
        m_Handle.handle = nullptr;
    }
}

nn::Result File::Read( int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option ) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile( m_Handle, offset, buffer, size, option );
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

nn::Result File::Read( int64_t offset, void* buffer, size_t size ) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile( m_Handle, offset, buffer, size );
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

nn::Result File::Read( size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option ) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile( outValue, m_Handle, offset, buffer, size, option );
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

nn::Result File::Read( size_t* outValue, int64_t offset, void* buffer, size_t size ) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile( outValue, m_Handle, offset, buffer, size );
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

nn::Result File::Write( int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option ) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::WriteFile( m_Handle, offset, buffer, size, option );
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

nn::Result File::Flush() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::FlushFile(m_Handle);
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

nn::Result File::SetSize(int64_t size) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::SetFileSize( m_Handle, size );
    if( result.IsFailure() )
    {
        SetResult( result );
    }

    return result;
}

int64_t File::GetSize() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒ひとまず0を返しておく

    int64_t size = 0;
    const auto result = nn::fs::GetFileSize( &size, m_Handle );
    if( result.IsFailure() )
    {
        SetResult( result );
        return 0;
    }

    return size;
}

int File::GetFileOpenMode() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ (返り値はFSの仕様依存)

    return nn::fs::GetFileOpenMode( m_Handle );
}

} // fsutil
