﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted MakeTestContents.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the intermediate working folder path
    [string] $IntermediateDirectory,

    # Qualify the system update meta identifier.
    [string] $UpdateId = "01001A500005E080",

    # Qualify the system program meta identifier.
    [string] $ProgramId = "01001A500005E081",

    # Qualify the proxy configuration.
    [string] $Proxy = "http://proxy.nintendo.co.jp:8080",

    # Qualify the server environments.
    [string] $ReqEnv = "dev6"
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath           = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath  = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\Modules\PlatformUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\OptionUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"
Import-Module "${MyScriptDirectoryPath}\Modules\FileUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\XmlFileUtilities"

# -----------------------------------------------------------------------
# Make system program nsp.
# -----------------------------------------------------------------------
function Out-MakeSystemProgramNsp( [string]$defIntermediateDirectory, [string]$defArchSignature, [string]$defProgramId, [int]$defVersion )
{
    # system program meta ファイル生成
    [string] $defBaseName = $(Out-MakeSystemProgramMeta ${defIntermediateDirectory} ${defProgramId} ${defVersion} ${defArchSignature})
    [string] $defMetaFile = "${defIntermediateDirectory}\${defBaseName}.nmeta"

    # 入力メタファイルの存在確認
    $(Test-ExistFile ${defMetaFile})

    # system program nsp ファイルパス
    [string] $defNspFile = "${defIntermediateDirectory}\${defBaseName}.nsp"

    # system program nsp 生成
    [string] $defMakeTestApplication = "${NintendoSdkRootPath}\Tools\CommandLineTools\MakeTestApplication\MakeTestApplication.exe"
    [string] $defWorkDirectory = "${defIntermediateDirectory}\_NspWorkTemporary_"
    $(Edit-MakeDirectory ${defWorkDirectory})
    Invoke-Expression "& `"${defMakeTestApplication}`" --type SystemProgram --meta `"${defMetaFile}`" -o `"${defIntermediateDirectory}`" --output-file-name ${defBaseName} --work-directory `"${defWorkDirectory}`"" | Write-Host
    $(Test-ExistFile ${defNspFile})

    return ${defNspFile}
}

# -----------------------------------------------------------------------
# Make system update nsp.
# -----------------------------------------------------------------------
function Out-MakeSystemUpdateNsp( [string]$defIntermediateDirectory, [string]$defUpdateId, [int]$defVersion, [string]$defProgramId, [string]$defDataId )
{
    # system program meta ファイル生成
    [string] $defBaseName = $(Out-MakeSystemUpdateMeta ${defIntermediateDirectory} ${defUpdateId} ${defVersion} ${defProgramId} ${defDataId})
    [string] $defMetaFile = "${defIntermediateDirectory}\${defBaseName}.nmeta"

    # 入力メタファイルの存在確認
    $(Test-ExistFile ${defMetaFile})

    # system update nsp ファイルパス
    [string] $defNspFile = "${defIntermediateDirectory}\${defBaseName}.nsp"

    # system update nsp 生成
    [string] $defAuthoringTool = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.exe"
    [string] $defKeyConfig = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.ocean.keyconfig.xml"
    Invoke-Expression "& `"${defAuthoringTool}`" creatensp --type SystemUpdate --meta `"${defMetaFile}`" --keyconfig `"${defKeyConfig}`" -o `"${defNspFile}`"" | Write-Host
    $(Test-ExistFile ${defNspFile})

    return ${defNspFile}
}



# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    # エンコード情報表示
    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # 中間ファイル作成作業フォルダ存在確認
    [string] $IntermediateDirectory = $(Get-AbsoluteFullPath ${IntermediateDirectory})
    $(Edit-MakeDirectory ${IntermediateDirectory})

    Write-Host "Platform        : ${Platform}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"

    # 64/32 アーキテクトシグネチャ取得
    [string] $ArchSignature = $(Get-ArchitectSignature ${Platform})
    Write-Host "Architexture    : ${ArchSignature}"

    # ===== system program nsp making. =====
    ( [string]$defSystemProgramNspV0 = $(Out-MakeSystemProgramNsp ${IntermediateDirectory} ${ArchSignature} ${ProgramId} 0) )
    ( [string]$defSystemProgramNspV1 = $(Out-MakeSystemProgramNsp ${IntermediateDirectory} ${ArchSignature} ${ProgramId} 1) )

    # ===== system update meta nsp making. =====
    ( [string]$defSystemUpdateNspV0 = $(Out-MakeSystemUpdateNsp ${IntermediateDirectory} ${UpdateId} 0 ${ProgramId} "") )
    ( [string]$defSystemUpdateNspV1 = $(Out-MakeSystemUpdateNsp ${IntermediateDirectory} ${UpdateId} 1 ${ProgramId} "") )

    Write-Host "`n >>>>>>> Complete nsp contents making. <<<<<<< `n"

    # サーバー選択
    [string]$defEnvironment = ${ReqEnv}

    # ===== system program nsp uploading. =====
    $(Publish-RomByRops ${IntermediateDirectory} ${Proxy} ${defEnvironment} ${ProgramId} ${defSystemProgramNspV0} 0)
    $(Publish-RomByRops ${IntermediateDirectory} ${Proxy} ${defEnvironment} ${ProgramId} ${defSystemProgramNspV1} 1)

    # ===== system update meta nsp uploading. =====
    $(Publish-RomByRops ${IntermediateDirectory} ${Proxy} ${defEnvironment} ${UpdateId} ${defSystemUpdateNspV0} 0)
    $(Publish-RomByRops ${IntermediateDirectory} ${Proxy} ${defEnvironment} ${UpdateId} ${defSystemUpdateNspV1} 1)

    Write-Host "`n >>>>>>> Complete nsp uploading. <<<<<<< `n"

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode

