﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/nifm.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/nim/srv/nim_NetworkSystemUpdateTask.h>

namespace {

    class SystemUpdateTest : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
    private:
        NN_ALIGNAS(4096) static uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
#endif

    protected:
        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::ncm::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::InitializeDependenciesForDfc();
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::FinalizeDependenciesForDfc();
#endif
            nn::ncm::Finalize();
        }

        std::vector<nn::ncm::ContentMetaKey> GetSystemContentMetaList()
        {
            nn::ncm::ContentMetaKey array[] = {
                nn::ncm::ContentMetaKey::Make(0x0100394000058000, 0, nn::ncm::ContentMetaType::SystemUpdate),
                nn::ncm::ContentMetaKey::Make(0x0100394000058001, 0, nn::ncm::ContentMetaType::SystemData),
                nn::ncm::ContentMetaKey::Make(0x0100394000058002, 0, nn::ncm::ContentMetaType::SystemData)
            };

            return std::vector<nn::ncm::ContentMetaKey>(array, std::end(array));
        }

        void TestNeedsUpdate()
        {
            nn::ns::SystemUpdateControl control;
            NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

            {
                nn::ns::AsyncLatestSystemUpdate asyncLatest;
                NNT_EXPECT_RESULT_SUCCESS(control.RequestCheckLatestUpdate(&asyncLatest));
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                asyncLatest.Cancel();

                while (!asyncLatest.TryWait())
                {
                    NN_LOG("Wating check latest update canceled...\n");
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }

                nn::ns::LatestSystemUpdate latest;
                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultCanceled, asyncLatest.Get(&latest));
            }
            nn::ns::AsyncLatestSystemUpdate asyncLatest;
            NNT_EXPECT_RESULT_SUCCESS(control.RequestCheckLatestUpdate(&asyncLatest));
            while (!asyncLatest.TryWait())
            {
                NN_LOG("Wating check latest update...\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            nn::ns::LatestSystemUpdate latest;
            NNT_EXPECT_RESULT_SUCCESS(asyncLatest.Get(&latest));
            EXPECT_EQ(nn::ns::LatestSystemUpdate::NeedsDownload, latest);
        }

        void TestDownloadLatestAndApply()
        {
            nn::ns::SystemUpdateControl control;
            NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

            {
                nn::ns::AsyncResult asyncResult;
                NNT_EXPECT_RESULT_SUCCESS(control.RequestDownloadLatestUpdate(&asyncResult));
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                asyncResult.Cancel();

                while (!asyncResult.TryWait())
                {
                    auto progress = control.GetDownloadProgress();
                    NN_LOG("Waiting download canceled...\n");
                    NN_LOG("loaded %lld, total %lld\n", progress.loaded, progress.total);
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }

                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultCanceled, asyncResult.Get());
            }

            nn::ns::AsyncResult asyncResult;
            NNT_EXPECT_RESULT_SUCCESS(control.RequestDownloadLatestUpdate(&asyncResult));

            while (!asyncResult.TryWait())
            {
                auto progress = control.GetDownloadProgress();
                NN_LOG("loaded %lld, total %lld\n", progress.loaded, progress.total);
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            auto progress = control.GetDownloadProgress();
            NN_LOG("loaded %lld, total %lld\n", progress.loaded, progress.total);
            NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());

#ifdef NN_TEST_NS_TEST_EULA
            size_t eulaDataSize;
            NNT_EXPECT_RESULT_SUCCESS(control.GetDownloadedEulaDataSize(&eulaDataSize, "dummy.txt"));
            NN_LOG("eulaDataSize %zu\n", eulaDataSize);

            std::vector<char> eulaData;
            eulaData.resize(eulaDataSize);
            size_t outSize;
            NNT_EXPECT_RESULT_SUCCESS(control.GetDownloadedEulaData(&outSize, eulaData.data(), eulaDataSize, "dummy.txt"));
            EXPECT_EQ(eulaDataSize, outSize);
#endif

            control.ApplyDownloadedUpdate();
        }

        void TestUpToDate()
        {
            nn::ns::SystemUpdateControl control;
            NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

            {
                nn::ns::AsyncLatestSystemUpdate asyncLatest;
                NNT_EXPECT_RESULT_SUCCESS(control.RequestCheckLatestUpdate(&asyncLatest));
                while (!asyncLatest.TryWait())
                {
                    NN_LOG("Wating check latest update...\n");
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }

                nn::ns::LatestSystemUpdate latest;
                NNT_EXPECT_RESULT_SUCCESS(asyncLatest.Get(&latest));
                EXPECT_EQ(nn::ns::LatestSystemUpdate::UpToDate, latest);
            }

            {
                nn::ns::AsyncResult asyncResult;
                NNT_EXPECT_RESULT_SUCCESS(control.RequestDownloadLatestUpdate(&asyncResult));

                while (!asyncResult.TryWait())
                {
                    auto progress = control.GetDownloadProgress();
                    NN_LOG("loaded %lld, total %lld\n", progress.loaded, progress.total);
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }

                NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultAlreadyUpToDate, asyncResult.Get());
            }
        }
    };
}

TEST_F(SystemUpdateTest, SystemUpdate)
{
    auto list = GetSystemContentMetaList();
    for (auto& key : list)
    {
        DeleteAllContents(key.id, nn::ncm::StorageId::BuildInSystem);
    }

    nn::ns::Initialize();

    nn::nifm::SubmitNetworkRequestAndWait();

    nn::ns::ClearExFatDriverStatus();

    TestNeedsUpdate();
    TestDownloadLatestAndApply();
    TestUpToDate();

    nn::ns::NotifyExFatDriverRequired();

    TestNeedsUpdate();
    TestDownloadLatestAndApply();
    TestUpToDate();

    for (auto& key : list)
    {
        DeleteAllContents(key.id, nn::ncm::StorageId::BuildInSystem);
    }

    nn::ns::ClearExFatDriverStatus();

    nn::ns::Finalize();
} // NOLINT(impl/function_size)


TEST_F(SystemUpdateTest, Background)
{
    auto list = GetSystemContentMetaList();
    for (auto& key : list)
    {
        DeleteAllContents(key.id, nn::ncm::StorageId::BuildInSystem);
    }

    nn::ns::Initialize();

    nn::nifm::SubmitNetworkRequestAndWait();

    {
        nn::ns::SystemUpdateControl control;
        NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

        nn::ns::AsyncResult asyncResult;
        NNT_EXPECT_RESULT_SUCCESS(control.RequestDownloadLatestUpdate(&asyncResult));

        while (!asyncResult.TryWait())
        {
            NN_LOG("Wating update started...\n");

            if (control.GetDownloadProgress().loaded > 0)
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    while (NN_STATIC_CONDITION(true))
    {
        NN_LOG("Wating background update prepared...\n");

        if (nn::ns::GetBackgroundNetworkUpdateState() == nn::ns::BackgroundNetworkUpdateState::Ready)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    {
        nn::ns::SystemUpdateControl control;
        NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

        EXPECT_TRUE(control.HasDownloaded());
        control.ApplyDownloadedUpdate();
    }

    nn::ns::Finalize();
} // NOLINT(impl/function_size)

