﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstdlib>
#include "testNs_Common.h"
#include "libraries/testNs_XmlParser.h"

#include <functional>   // std::hash
#include <string>       // std::hash< std::string >
#include <unordered_map>

namespace testns {

    //!--------------------------------------------------------------------------------------
    //! @brief 文字列キーバリュー形式プロパティコレクション
    //!--------------------------------------------------------------------------------------
    class KeyValueCollection : public std::unordered_map< std::string, std::string >, protected XmlParser::CallbackInterface
    {
        NN_DISALLOW_COPY( KeyValueCollection );
        NN_DISALLOW_MOVE( KeyValueCollection );

        typedef std::unordered_map< std::string, std::string >  CollectionType;
        typedef XmlParser::Attribute::CollectionType            AttrListType;

    public:
        //!---------------------------------------------------------------------------------
        //! @brief コンストラクタ
        //!---------------------------------------------------------------------------------
        explicit KeyValueCollection( const unsigned reservedCapacity = 32 ) NN_NOEXCEPT;

        //!---------------------------------------------------------------------------------
        //! @brief デストラクタ
        //!---------------------------------------------------------------------------------
        ~KeyValueCollection() NN_NOEXCEPT;

        //!---------------------------------------------------------------------------------
        //! @brief ファイナライザ
        //!---------------------------------------------------------------------------------
        void Finalize() NN_NOEXCEPT;

        //!---------------------------------------------------------------------------------
        //! @brief コレクションローダー ( from File )
        //!---------------------------------------------------------------------------------
        bool LoadFromFile( const char* pFilePath ) NN_NOEXCEPT;

        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        const std::string GetValue( const std::string& key ) const NN_NOEXCEPT
        {
            const auto it = CollectionType::find( key );
            return ( it != CollectionType::cend() ) ? it->second : std::string();
        }

        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        template< typename TypeOfValue >
        const TypeOfValue GetValueAsInteger( const std::string& key, const TypeOfValue defaultValue = TypeOfValue() ) const NN_NOEXCEPT
        {
            const auto it = CollectionType::find( key );
            return ( it != CollectionType::cend() )
                ? static_cast< TypeOfValue >( std::strtoull( it->second.c_str(), nullptr, 0 ) )
                : defaultValue;
        }

    protected:
        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        virtual void OnElementBegin( const XmlParser::StringType& name, const XmlParser::Attribute::CollectionType& attributes ) NN_NOEXCEPT NN_OVERRIDE;

        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        virtual void OnText( const XmlParser::StringType& text ) NN_NOEXCEPT NN_OVERRIDE;

        //!---------------------------------------------------------------------------------
        //! @brief
        //!---------------------------------------------------------------------------------
        virtual void OnElementEnd( const XmlParser::StringType& name ) NN_NOEXCEPT NN_OVERRIDE;

    private:
        XmlParser*              m_pXmlParser;
        XmlParser::StringType   m_DetectedKey;
        XmlParser::StringType   m_DetectedValue;
        bool                    m_DetectRootElement;
    };

}
