﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Runtime.Serialization;

namespace CsTestAssistants
{
    /// <summary>
    /// ダウンロードタスクリストオブジェクト
    /// </summary>
    public class DownloadTaskList
    {
        /// <summary>
        /// ダウンロードタイトルオブジェクト
        /// </summary>
        public class Title : ContentMeta.Contexture.Basic
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="id"></param>
            /// <param name="version"></param>
            /// <param name="type"></param>
            public Title( ID64 id, int version, ContentMeta.Type type )
                : base ( id, version, type )
            {
            }

            /// <summary>
            /// コピーコンストラクタ
            /// </summary>
            /// <param name="other"></param>
            public Title( ContentMeta.Contexture.Basic other )
                : base( other )
            {
            }

            /// <summary>
            /// JSON用シリアライズ型
            /// </summary>
            [DataContract]
            internal class Json
            {
                [DataMember( Order = 0 )]
                public string id;
                [DataMember( Order = 1 )]
                public int version;
                [DataMember( Order = 2 )]
                public string type;

                public Json( Title title )
                {
                    id = "0x" + title.Identifier.ToString();
                    type = title.Type.ToString();
                    version = title.Version;
                }
            }
        }

        /// <summary>
        /// ダウンロードタスクオブジェクト
        /// </summary>
        public class Task
        {
            /// <summary>
            /// タスクオーナーアプリケーションID
            /// </summary>
            public ID64 OwnerApplication { get; }

            /// <summary>
            /// タスク識別子
            /// </summary>
            public string Uuid { get; }

            public List<Title> Titles { get; }

            public enum InstallConfig
            {
                ForceNone = 0,
                LatestAddOnContentsAuto = 0x1 << 1,
                SystemUpdateRecursive   = 0x1 << 2,
                RequiresExFatDriver     = 0x1 << 3,
                IgnoreTicket            = 0x1 << 4,
                SkipIndirectUpdateCheck = 0x1 << 5,
                Unused                  = 0x1 << 31
            };
            public InstallConfig Config { set; get; } = InstallConfig.Unused;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="ownerApplication"></param>
            /// <param name="uuid"></param>
            /// <param name="titleInitialCapacity"></param>
            public Task( ID64 ownerApplication, string uuid = null, int titleInitialCapacity = 4 )
            {
                Config = InstallConfig.Unused;
                OwnerApplication = ownerApplication;
                Uuid = ( null != uuid ) ? uuid : System.Guid.NewGuid().ToString( "D" );
                Titles = new List<Title>( titleInitialCapacity );
            }

            /// <summary>
            /// コピーコンストラクタ ( Titles は ShallowCopy です。 )
            /// </summary>
            /// <param name="other"></param>
            public Task( Task other )
            {
                if ( null == other )
                {
                    throw new System.ArgumentNullException();
                }
                Config = other.Config;
                OwnerApplication = other.OwnerApplication;
                Uuid = other.Uuid;
                Titles = new List<Title>( other.Titles );
            }

            public Task AddNewTitle( ID64 id, int version, ContentMeta.Type type )
            {
                Titles.Add( new Title( id, version, type ) );
                return this;
            }

            public Task AddNewTitle( ContentMeta.Contexture.Basic meta )
            {
                return AddNewTitle( meta.Identifier, meta.Version, meta.Type );
            }

            /// <summary>
            /// JSON用シリアライズ型
            /// </summary>
            [DataContract]
            [KnownType( typeof( JsonWithConfig ))]
            internal class Json
            {
                [DataMember( Order = 0 )]
                public string id;

                [DataMember( Order = 1 )]
                public string owner_application;

                [DataMember( Order = 2 )]
                public Title.Json[] titles;

                public Json( Task task )
                {
                    id = task.Uuid;
                    owner_application = "0x" + task.OwnerApplication.ToString();

                    int count;
                    Title.Json[] values = null;
                    List<Title> collection = task.Titles;
                    if ( ( count = collection.Count ) > 0 )
                    {
                        int index = 0;
                        values = new Title.Json[ count ];
                        foreach ( var title in collection )
                        {
                            values[ index++ ] = new Title.Json( title );
                        }
                    }
                    titles = values;
                }
            }

            [DataContract]
            internal class JsonWithConfig : Json
            {
                [DataMember( Order = 3 )]
                public uint config;

                public JsonWithConfig( Task task ) : base( task )
                {
                    config = (uint)task.Config;
                }
            }

            /// <summary>
            /// JSONオブジェクトからのコンストラクタ
            /// </summary>
            /// <param name="json"></param>
            internal Task( Json json )
                : this ( new ID64( json.owner_application, 16 ), json.id )
            {
                Title.Json[] jtitles;
                if ( null != ( jtitles = json.titles ) )
                {
                    foreach ( var jtitle in jtitles )
                    {
                        AddNewTitle( new ID64( jtitle.id, 16 ), jtitle.version,
                            CsExtension.ToEnumValue( jtitle.type, ContentMeta.Type.Unknown )
                        );
                    }
                }
            }

            /// <summary>
            /// JSONオブジェクトからのコンストラクタ
            /// </summary>
            /// <param name="json"></param>
            internal Task( JsonWithConfig json )
                : this( new ID64( json.owner_application, 16 ), json.id )
            {
                Config = ( InstallConfig )json.config;
                Title.Json[] jtitles;
                if ( null != ( jtitles = json.titles ) )
                {
                    foreach ( var jtitle in jtitles )
                    {
                        AddNewTitle( new ID64(jtitle.id, 16 ), jtitle.version,
                            CsExtension.ToEnumValue( jtitle.type, ContentMeta.Type.Unknown )
                        );
                    }
                }
            }
        }

        /// <summary>
        /// タスクIDキーとタスクのペアマップ.
        /// </summary>
        /// <remarks>
        /// OwnerApplicationId が重複しないなら、ID64 のマップ作ってもよいけど、
        /// 多分購入の都度タスク追加が想定じゃないかな.
        /// </remarks>
        public Dictionary<string, Task> Tasks { get; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="taskInitialCapacity"></param>
        public DownloadTaskList( int taskInitialCapacity = 64 )
        {
            Tasks = new Dictionary<string, Task>( taskInitialCapacity );
        }

        /// <summary>
        /// コピーコンストラクタ ( Title は ShallowCopy です。 )
        /// </summary>
        /// <param name="other"></param>
        public DownloadTaskList( DownloadTaskList other )
        {
            if ( null == other )
            {
                throw new System.ArgumentNullException();
            }
            Tasks = new Dictionary<string, Task>( other.Tasks.Count );
            foreach ( var kv in other.Tasks )
            {
                Tasks.Add( kv.Key, new Task( kv.Value ) );
            }
        }

        /// <summary>
        /// <see cref="GeneratedContentResult.TitleCategorizedCatalog"/> カタログからのコンストラクタ
        /// </summary>
        /// <param name="titles"></param>
        public DownloadTaskList( GeneratedContentResult.TitleCategorizedCatalog titles ) : this( titles.Count )
        {
            FromCatalog( titles );
        }

        /// <summary>
        /// タスクを生成してリストに登録します.
        /// </summary>
        /// <param name="ownerApplication">オーナーアプリケーションID</param>
        /// <param name="uuid">タスクID</param>
        /// <returns>生成されたタスクオブジェクトインスタンス</returns>
        public Task Create( ID64 ownerApplication, string uuid = null )
        {
            var t = new Task( ownerApplication, uuid );
            Tasks.Add( t.Uuid, t );

            return t;
        }

        /// <summary>
        /// タスクを生成してリストに登録します.
        /// </summary>
        /// <param name="ownerApplication">オーナーアプリケーションID</param>
        /// <param name="config">インストールコンフィグ( ncm )</param>
        /// <returns>生成されたタスクオブジェクトインスタンス</returns>
        public Task Create( ID64 ownerApplication, Task.InstallConfig config )
        {
            var t = new Task( ownerApplication );
            Tasks.Add( t.Uuid, t );
            t.Config = config;
            return t;
        }

        /// <summary>
        /// タスクID ( uuid )から登録されているタスクオブジェクトを検索します.
        /// </summary>
        /// <param name="uuid">タスクID ( "01234567-0123-0123-0123-0123456789ab" の 8-4-4-4-12 形式 )</param>
        /// <returns>見つかった場合は Task オブジェクト。見つからない場合は null が返されます</returns>
        public Task Find( string uuid )
        {
            Task value;
            return ( !string.IsNullOrEmpty( uuid ) && Tasks.TryGetValue( uuid, out value ) ) ? value : null;
        }

        /// <summary>
        /// JSON用シリアライズ型
        /// </summary>
        [DataContract]
        internal class Json
        {
            [DataMember]
            public Task.Json[] tasks;

            public Json( DownloadTaskList dtl )
            {
                int count;
                Task.Json[] values = null;
                Dictionary<string, Task>.ValueCollection collection = dtl.Tasks.Values;
                if ( ( count = collection.Count ) > 0 )
                {
                    int index = 0;
                    values = new Task.Json[ count ];
                    foreach ( var task in collection )
                    {
                        if ( task.Config == Task.InstallConfig.Unused )
                        {
                            values[ index++ ] = new Task.Json( task );
                        }
                        else
                        {
                            values[ index++ ] = new Task.JsonWithConfig( task );
                        }
                    }
                }
                tasks = values;
            }
        }

        /// <summary>
        /// JSONオブジェクトからのコンストラクタ
        /// </summary>
        /// <param name="json"></param>
        internal DownloadTaskList( Json json ) : this( 64 )
        {
            Task.Json[] jtasks;
            if ( null != json && null != ( jtasks = json.tasks ) )
            {
                foreach( var jtask in jtasks )
                {
                    Tasks.Add( jtask.id, new Task( jtask ) );
                }
            }
        }

        /// <summary>
        /// JSON文字列をデシリアライズして DownloadTaskList クラスインスタンスを生成します.
        /// </summary>
        /// <remarks>
        /// 本メソッドで生成された Task クラスインスタンスの Config 値は、全て Task.InstallConfig.Unused に設定されます。
        /// ( 入力 Json データの "config" 要素は無視されます )
        /// </remarks>
        /// <param name="stream">JSON文字列</param>
        public static DownloadTaskList FromJsonStream( string stream )
        {
            var settings = new System.Runtime.Serialization.Json.DataContractJsonSerializerSettings();
            settings.EmitTypeInformation = EmitTypeInformation.Never;
            return ( string.IsNullOrEmpty( stream ) )
                ? new DownloadTaskList()
                : new DownloadTaskList( stream.DeserializeFromJson< Json >( settings ) );
        }

        /// <summary>
        /// 指定の JSONオブジェクトを JSON文字列にシリアライズします。
        /// </summary>
        /// <param name="json">JSONオブジェクト</param>
        /// <param name="outFilePath">シリアライズされた文字列を出力するファイルパスを指定します。nullの場合、出力しません。</param>
        /// <param name="encoding">null の場合、UTF8 BOMなしエンコードが適用されます。</param>
        /// <param name="applyParagraphIndentFomatting">true の場合、シリアライズされた文字列を段落インデント整形します。</param>
        /// <returns>シリアライズされたJSON文字列が返されます</returns>
        internal static string ToJsonCore<TJson>( TJson json, string outFilePath = null, System.Text.Encoding encoding = null, bool applyParagraphIndentFomatting = true ) where TJson : class
        {
            // シリアライズ ( with indent fomatting ).
            var settings = new System.Runtime.Serialization.Json.DataContractJsonSerializerSettings();
            settings.EmitTypeInformation = EmitTypeInformation.Never;
            string serializedStream = ( applyParagraphIndentFomatting )
                ? JsonExtension.FormatParagraphIndent( json.SerializeToJson( settings ) )
                : json.SerializeToJson( settings );

            // ファイル書き込み
            if ( false == string.IsNullOrEmpty( outFilePath ) )
            {
                FileHelper.MakeDirectory( System.IO.Path.GetDirectoryName( outFilePath ) );
                System.IO.File.WriteAllText( outFilePath, serializedStream, ( null == encoding ) ? EncodeContext.UTF8_NoBOM : encoding );
            }
            return serializedStream;
        }

        /// <summary>
        /// DTL状態を JSON文字列にシリアライズします。
        /// </summary>
        /// <param name="outFilePath">シリアライズされた文字列を出力するファイルパスを指定します。nullの場合、出力しません。</param>
        /// <param name="encoding">null の場合、UTF8 BOMなしエンコードが適用されます。</param>
        /// <param name="applyParagraphIndentFomatting">true の場合、シリアライズされた文字列を段落インデント整形します。</param>
        /// <returns>シリアライズされたJSON文字列が返されます</returns>
        public string ToJson( string outFilePath = null, System.Text.Encoding encoding = null, bool applyParagraphIndentFomatting = true )
        {
            return ToJsonCore( new Json( this ), outFilePath, encoding, applyParagraphIndentFomatting );
        }

        /// <summary>
        /// JSONファイルをデシリアライズして DownloadTaskList クラスインスタンスを生成します.
        /// </summary>
        /// <param name="outFilePath">読み込み対象JSONファイルパス</param>
        /// <param name="encoding">null の場合、UTF8 BOMなしエンコードが適用されます。</param>
        /// <returns>読み込みに成功した場合クラスインスタンスが返されます。</returns>
        /// <remarks>
        /// 本メソッドで生成された Task クラスインスタンスの Config 値は、全て Task.InstallConfig.Unused に設定されます。
        /// ( 入力 Json データの "config" 要素は無視されます )
        /// </remarks>
        public static DownloadTaskList FromJson( string outFilePath, System.Text.Encoding encoding = null )
        {
            return FromJsonStream( System.IO.File.ReadAllText( outFilePath, ( null == encoding ) ? EncodeContext.UTF8_NoBOM : null ) );
        }

        /// <summary>
        /// 指定ID から指定数分だけ、タイトルを一つ持つタスクを生成します.
        /// </summary>
        /// <remarks>
        /// 各IDごとに1つのタスクを生成し、そのタスクに対して、同じIDの "Application" タイプのタイトルを登録します.
        /// 生成されたタスクは `Tasks` プロパティでペアマップが参照できます.
        /// </remarks>
        /// <param name="beginAppId">開始アプリケーションID</param>
        /// <param name="appCount">生成タスク数</param>
        /// <param name="version">タイトルバージョン</param>
        public DownloadTaskList MakeAsApplication( ID64 beginAppId, int appCount, int version )
        {
            for ( int index = 0; index < appCount; ++index )
            {
                ID64 nowAppId = beginAppId + ( ( ulong )index );
                Create( nowAppId ).AddNewTitle( nowAppId, version, ContentMeta.Type.Application );
            }
            return this;
        }

        /// <summary>
        /// 保持タスクに登録されているタイトルを一覧で取得します。
        /// 複数のタスクに跨った重複タイトルは除外されます。
        /// </summary>
        /// <returns>
        /// タイトル一覧リスト。
        /// </returns>
        public List<Title> ToTitleCatalog()
        {
            List<Title> values = new List<Title>( 64 );
            foreach ( var task in Tasks.Values )
            {
                foreach ( var title in task.Titles )
                {
                    if ( false == values.Contains( title ) )
                    {
                        values.Add( title );
                    }
                }
            }
            return values;
        }

        /// <summary>
        /// 指定のタイトルを持つタイトルを持つタスクを検索します。
        /// </summary>
        /// <param name="titleId"></param>
        /// <returns></returns>
        public Task FindTaskFrom( ID64 titleId )
        {
            foreach ( var task in Tasks.Values )
            {
                if ( null != task.Titles.Find( meta => { return ( meta.Identifier == titleId ); } ) )
                {
                    return task;
                }
            }
            return null;
        }

        /// <summary>
        /// <see cref="GeneratedContentResult.TitleCategorizedCatalog"/> カタログからDTLを作成します。
        /// </summary>
        /// <param name="titles"></param>
        /// <returns></returns>
        public DownloadTaskList FromCatalog( GeneratedContentResult.TitleCategorizedCatalog titles )
        {
            foreach ( var title in titles )
            {
                var types = title.Value;
                var task = Create( title.Key );
                GeneratedContentResult.Catalog catalog;
                if ( null != ( catalog = types.GetTypedCatalog( ContentMeta.Type.Application ) ) )
                {
                    foreach( var c in catalog )
                    {
                        task.AddNewTitle( c );
                    }
                }
                if ( null != ( catalog = types.GetTypedCatalog( ContentMeta.Type.Patch ) ) )
                {
                    foreach ( var c in catalog )
                    {
                        task.AddNewTitle( c );
                    }
                }
                if ( null != ( catalog = types.GetTypedCatalog( ContentMeta.Type.AddOnContent ) ) )
                {
                    foreach ( var c in catalog )
                    {
                        task.AddNewTitle( c );
                    }
                }
            }
            return this;
        }

        /// <summary>
        /// DTLから条件に合致するタイトルを全て削除します。
        /// </summary>
        /// <param name="remover">タイトルに対する削除条件式</param>
        /// <returns>削除した要素数</returns>
        public int RemoveTitleAll( System.Predicate<Title> remover )
        {
            int removed = 0;
            foreach ( var task in Tasks.Values )
            {
                removed += task.Titles.RemoveAll( remover );
            }
            return removed;
        }
    }

    public class DownloadTaskListFile : System.Tuple<string, DownloadTaskList>
    {
        public string Path { get { return Item1; } }
        public DownloadTaskList DTL { get { return Item2; } }

        public DownloadTaskListFile( string path, DownloadTaskList dtl ) : base( path, dtl ) { }

        public static DownloadTaskListFile Create( string path, DownloadTaskList dtl )
        {
            dtl.ToJson( path );
            return new DownloadTaskListFile( path, dtl );
        }
    }
}
