﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Net;
using System.Runtime.Serialization;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Xml;

namespace CsTestAssistants
{
    public static class D4cHelper
    {
        /// <summary>
        /// プロキシオプション生成ヘルパ.
        /// </summary>
        /// <remarks>
        /// 外部環境変数の状態に応じたプロキシオプションの生成を支援します.
        /// </remarks>
        public static class ProxyHelper
        {
            /// <summary>
            /// NCLプロキシアドレス。
            /// </summary>
            public const string NCL_PROXY_URI = "http://proxy.nintendo.co.jp:8080";

            /// <summary>
            /// WebRequest用のプロキシオブジェクトを作成します.
            /// </summary>
            /// <remarks>
            /// 環境変数 NN_TEST_SCRIPT_PROXY_CONFIGURATION が設定されていた場合、その内容が引数 uri よりも優先されて採用されます。
            /// 環境変数 NN_TEST_SCRIPT_PROXY_CONFIGURATION が "" 形式の空文字指定の場合、プロキシ設定は指定なしとして扱われます。
            /// 環境変数 NN_TEST_SCRIPT_PROXY_CONFIGURATION の扱いは、ローカル限定のプロキシ設定を有効化するためです。
            /// </remarks>
            /// <param name="uri">プロキシアドレスURI</param>
            /// <param name="username">プロキシアクセス用ユーザアカウント</param>
            /// <param name="password">プロキシアクセス用ユーザパスワード</param>
            /// <returns>
            /// 指定された uri などが不正、もしくは意図した null値であればプロキシは経由しないとして null を返します.
            /// </returns>
            /// <exception cref="System.UriFormatException">指定プロキシアドレスが不正な場合</exception>
            public static WebProxy Create( string uri = NCL_PROXY_URI, string username = null, string password = null )
            {
                string envValue;
                WebProxy handle = null;
                if ( null != ( envValue = System.Environment.GetEnvironmentVariable( "NN_TEST_SCRIPT_PROXY_CONFIGURATION" ) ) )
                {
                    // 環境変数が存在したので優先.
                    uri = envValue;
                    // "xxx" 形式文字列からダブルコーテーション外す
                    var matchResult = Regex.Match( envValue, "^\".*?\"$" );
                    if ( matchResult.Success )
                    {
                        uri = Regex.Replace( matchResult.Value, "^\"", "" );
                        uri = Regex.Replace( uri, "\"$", "" );
                    }
                }
                if ( false == string.IsNullOrEmpty( uri ) )
                {
                    handle = new WebProxy( uri );
                    if ( !string.IsNullOrEmpty( username ) || !string.IsNullOrEmpty( password ) )
                    {
                        handle.Credentials = new NetworkCredential( username, password );
                    }
                }
                return handle;
            }
        }

        /// <summary>
        /// D4Cサービスを利用する上でのコンフィギュレーション
        /// </summary>
        public static class Configuration
        {
            public class ServerEnvironment
            {
                public static readonly ServerEnvironment DEV1 = new ServerEnvironment( "dev1", "jd1", "jd1env" );
                public static readonly ServerEnvironment DEV6 = new ServerEnvironment( "dev6", "td1", "td1pass" );

                /// <summary>
                /// Service name. ( e.g. dev6, dev1 )
                /// </summary>
                public string Name { get; }
                /// <summary>
                /// Discovery alias name. ( e.g. td1, jd1 )
                /// </summary>
                public string Alias { get; }
                /// <summary>
                /// PassCode.
                /// </summary>
                public string PassCode { get; }

                private ServerEnvironment( string name, string alias, string pass )
                {
                    Name = name.ToLower();
                    Alias = alias.ToLower();
                    PassCode = pass;
                }

                public override string ToString()
                {
                    return "[ " + Name + ", " + Alias + " ]";
                }
            }

            /// <summary>
            /// CLIトークンユーティリティ
            /// ContentsUploader の -u / -p で同じことしてた。
            /// 未完成。
            /// </summary>
            public class CliToken
            {
                public static class Constants
                {
                    public const string FileName = "cli.auth.token.json";
                    public const string Uri = "https://star.debug.{0}.d4c.nintendo.net/v1/ndid/users/@me/token";
                }

                public string FilePath { get; }

                public CliToken( string filePath )
                {
                    FilePath = filePath;
                }

                public static CliToken QueryLatestValidateToken( string outPath, BasicAccount cliAccount, WebProxy proxy = null, ServerEnvironment server = null )
                {
                    // correct to full path string on the outPath.
                    var outDir = FileHelper.MakeDirectory( outPath );
                    string filePath = System.IO.Path.Combine( outDir.FullName, Constants.FileName );
                    server = ( null != server ) ? server : ServerEnvironment.DEV6;

                    HttpWebRequest request = WebRequest.CreateHttp( string.Format( Constants.Uri, server.Name ) );
                    request.Proxy = proxy;
                    request.Headers[ "Authorization" ] = "Basic " + cliAccount.ToAuthorizationBase64();
                    request.ServerCertificateValidationCallback = ( sender, cert, chain, sslPolicyErros ) =>
                    {
                        return true;    // --insecure, -k, 自己署名SSL許可。
                    };

                    WebExceptionStatus rc;
                    HttpExtension.ConnectionResult result;
                    if ( WebExceptionStatus.Success == ( rc = request.ConnectToServer( out result ) )
                        && null != result && HttpStatusCode.OK == result.StatusCode )
                    {
                        System.IO.File.WriteAllText( filePath, result.Text, EncodeContext.UTF8_NoBOM );
                        return new CliToken( filePath );
                    }
                    throw new UnexpectFailureException( $"CliTokenRequest: [ Method ={ request.Method }, URI ={ request.Address} ] => [ WebException={rc}, result={result} ]" );
                }

                [DataContract]
                public class Properties
                {
                    [DataMember( Order = 0 )]
                    public string access_token;

                    [DataMember( Order = 1 )]
                    public string token_type;

                    [DataMember( Order = 2 )]
                    public string refresh_token;

                    [DataMember( Order = 3 )]
                    public ulong expires_in;

                    public override string ToString()
                    {
                        StringBuilder b = new StringBuilder( 128 );
                        b.Append( "{" ).Append( System.Environment.NewLine );
                        b.Append( "    access_token: " ).Append( access_token ).Append( System.Environment.NewLine );
                        b.Append( "    token_type: " ).Append( token_type ).Append( System.Environment.NewLine );
                        b.Append( "    refresh_token: " ).Append( refresh_token ).Append( System.Environment.NewLine );
                        b.Append( "    expires_in: " ).Append( expires_in ).Append( System.Environment.NewLine );
                        b.Append( "}" ).Append( System.Environment.NewLine );
                        return b.ToString();
                    }
                }

                public Properties Deserialize( Encoding encoding = null )
                {
                    FileHelper.TestExistsFile( FilePath );
                    var text = System.IO.File.ReadAllText( FilePath, ( null == encoding ) ? EncodeContext.UTF8_NoBOM : null );
                    return text.DeserializeFromJson<Properties>();
                }
            }
        }

        public class Command
        {
            public class Result
            {
                public MatchCollection DetectCollection { get; }
                public bool HasError { get; }

                public Result( bool hasError = false, MatchCollection collection = null )
                {
                    HasError = hasError;
                    DetectCollection = collection;
                }
            }

            public string Name { get; }
            public string Signature { get; }
            public System.Func<OutputStreams, Result> ErrorDetector { get; }

            public Command( string name, System.Func<OutputStreams, Result> errorDetector )
            {
                Name = name;
                Signature = "";
                ErrorDetector = errorDetector;
            }

            public Command( string name, string signature, System.Func<OutputStreams, Result> errorDetector )
            {
                Name = name;
                Signature = signature;
                ErrorDetector = errorDetector;
            }
        }

        public class Rops
        {
            /// <summary>
            /// ROM 状態
            /// </summary>
            [System.Flags]
            public enum RomStatus
            {
                // success state
                NEW = 1 << 0,
                UPLOAD = 1 << 1,
                SCREENING = 1 << 2,
                SUBMITTBLE = 1 << 3,
                PREENCRYPT = 1 << 4,
                ENCRYPT = 1 << 5,
                PUBLISH = 1 << 6,
                CHECKING = 1 << 7,
                APPROVED = 1 << 8,
                PROMOTE = 1 << 9,

                // fail state
                FAILED = 1 << 10,
                UPLOAD_UNCOMPLETED = 1 << 11,
                SCREENING_REJECTED = 1 << 12,
                DELETED = 1 << 13,
                ABORTED = 1 << 14,
                REJECTED = 1 << 15,
            }

            public string ExecFile { get; }
            public string Intermediate { get; }
            public Configuration.CliToken Token { get; }
            public Configuration.ServerEnvironment Server { get; }
            public WebProxy Proxy { get; }
            public OutputStreams Output { get; }

            /// <summary>
            /// rops 実行ファイルパス取得
            /// </summary>
            /// <returns>rops 実行ファイルパス</returns>
            public static string GetExecutionFile()
            {
                return SigloHelper.ToolPath.FindTools( System.IO.Path.Combine( "CommandLineTools", "rops", "rops.exe" ) );
            }

            /// <summary>
            /// 配信設定ファイル作成
            /// </summary>
            /// <param name="path">ファイルパス</param>
            /// <param name="downloadable">ダウンロード可否</param>
            /// <param name="verbose">詳細ログ出力</param>
            public static void CreateDeliveryInfo( string path, bool downloadable, bool verbose = false )
            {
                XmlDocument xml = new XmlDocument();
                xml.AppendChild( xml.CreateXmlDeclaration( @"1.0", @"UTF-8", @"yes" ) );

                XmlElement delivery = xml.CreateElement( "update_title_delivery_info_entity" );
                xml.AppendChild( delivery );

                XmlElement entities = xml.CreateElement( "delivery_entities" );
                delivery.AppendChild( entities );

                // type: control
                {
                    XmlElement entity = xml.CreateElement( "delivery_entity" );
                    XmlElement type = xml.CreateElement( "content_type" );
                    XmlElement date = xml.CreateElement( "delivery_date" );
                    XmlElement download = xml.CreateElement( "is_downloadable" );
                    type.InnerText = "control";
                    date.InnerText = "2016-11-07 16:00:00";
                    download.InnerText = downloadable ? "true" : "false";
                    entity.AppendChild( type );
                    entity.AppendChild( date );
                    entity.AppendChild( download );
                    entities.AppendChild( entity );
                }
                // type: rom
                {
                    XmlElement entity = xml.CreateElement( "delivery_entity" );
                    XmlElement type = xml.CreateElement( "content_type" );
                    XmlElement date = xml.CreateElement( "delivery_date" );
                    XmlElement download = xml.CreateElement( "is_downloadable" );
                    type.InnerText = "rom";
                    date.InnerText = "2016-11-11 14:00:00";
                    download.InnerText = downloadable ? "true" : "false";
                    entity.AppendChild( type );
                    entity.AppendChild( date );
                    entity.AppendChild( download );
                    entities.AppendChild( entity );
                }
                xml.Save( path );

                // output verbose
                if ( verbose )
                {
                    string xmlData = System.IO.File.ReadAllText( path );
                    Log.WriteLine( $"delivery xml: {path}" );
                    Log.WriteLineAsIs( xmlData );
                }
            }

            public Rops( string intermediate, BasicAccount account, WebProxy proxy = null, Configuration.ServerEnvironment server = null )
            {
                ExecFile = GetExecutionFile();
                Intermediate = FileHelper.MakeDirectory( intermediate ).FullName;
                Token = Configuration.CliToken.QueryLatestValidateToken( intermediate, account, proxy, server );
                Server = ( null != server ) ? server : Configuration.ServerEnvironment.DEV6;
                Proxy = proxy;
                Output = new OutputStreams();
            }

            private bool Execute( Command command, string arguments, out Command.Result result )
            {
                var code = CommandLineExecutor.ExecuteOnProcess( ExecFile, arguments, Output );
                result = command.ErrorDetector( Output );
                return ( 0 == code && false == result.HasError );
            }

            private void Execute( Command command, string arguments, Retry.Watcher wather )
            {
                var executed = false;
                Command.Result result;
                while ( !( executed = Execute( command, arguments, out result ) ) && wather.TryContinue() )
                {
                    Log.WriteLine(
                        $"rops failed on [ {command.Name} ] => {command.Signature}, " +
                        $"retry after {wather.IntervalSeconds} seconds." );
                    wather.WaitInterval();
                }
                if ( !executed )
                {
                    throw new UnexpectFailureException( $"rops operation failed on [ {command.Name} ] => {command.Signature}." );
                }
                wather.RecordEndTime();
            }

            /// <summary>
            /// rops 引数文字列化
            /// </summary>
            /// <param name="command">コマンド情報</param>
            /// <param name="appendix">付録引数</param>
            /// <returns></returns>
            private string ToArguments( Command command, string appendix )
            {
                StringBuilder sb = new StringBuilder( 512 );

                // 定型部
                sb.Append( $"--insecure " );
                sb.Append( $"--suppress-warning http-fallback " );
                if ( null != Proxy )
                {
                    sb.Append( $"--proxy \"{Proxy.Address.AbsoluteUri}\" " );
                }
                sb.Append( $"{command.Name} " );
                sb.Append( $"--token \"{Token.FilePath}\" " );
                sb.Append( $"-e {Server.Name} " );

                // 付録部
                if ( !string.IsNullOrEmpty( appendix ) )
                {
                    sb.Append( $"{appendix}" );
                }
                return sb.ToString();
            }

            /// <summary>
            /// approve-title：タイトル設定承認
            /// </summary>
            /// <param name="id">コンテンツメタ ID</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ApproveTitle( ID64 id, int timeout = 60 )
            {
                var command = new Command( "approve-title", $"0x{id}", ( OutputStreams streams ) =>
                {
                    var match = Regex.Match(
                        streams.Standard.ToString(),
                        "Approve title has been completed successfully" );
                    return new Command.Result( !match.Success, null );
                } );
                Execute( command, ToArguments( command, $"-t {id}" ), new Retry.Watcher( 20, timeout ) );
            }

            /// <summary>
            /// approve-title：タイトル設定承認
            /// </summary>
            /// <param name="content">コンテンツ情報</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ApproveTitle( GeneratedContentResult content, int timeout = 60 )
            {
                ApproveTitle( content.Identifier, timeout );
            }

            /// <summary>
            /// delivery：コンテンツ配信のダウンロード可否設定
            /// </summary>
            /// <param name="id">コンテンツメタ ID</param>
            /// <param name="version">バージョン</param>
            /// <param name="downloadable">ダウンロード可否</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void Delivery( ID64 id, int version, bool downloadable, int timeout = 60 )
            {
                var path = System.IO.Path.Combine( Intermediate, "delivery.xml" );
                CreateDeliveryInfo( path, downloadable );
                var command = new Command( "delivery", $"0x{id}", ( OutputStreams streams ) =>
                {
                    var match = Regex.Match(
                        streams.Standard.ToString(),
                        "Update content delivery infos has been completed successfully" );
                    return new Command.Result( !match.Success, null );
                } );
                Execute( command, ToArguments( command, $"-t {id} -v {version} -i \"{path}\"" ), new Retry.Watcher( 20, timeout ) );
            }

            /// <summary>
            /// delivery：コンテンツ配信のダウンロード可否設定
            /// </summary>
            /// <param name="content">コンテンツ情報</param>
            /// <param name="downloadable">ダウンロード可否</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void Delivery( GeneratedContentResult content, bool downloadable, int timeout = 60 )
            {
                Delivery( content.Identifier, content.Version, downloadable, timeout );
            }

            /// <summary>
            /// delivery：コンテンツ配信開始
            /// </summary>
            /// <param name="content">コンテンツ情報</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void StartDelivery( GeneratedContentResult content, int timeout = 60 )
            {
                Delivery( content.Identifier, content.Version, true, timeout );
            }

            /// <summary>
            /// delivery：コンテンツ配信停止
            /// </summary>
            /// <param name="content">コンテンツ情報</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void StopDelivery( GeneratedContentResult content, int timeout = 60 )
            {
                Delivery( content.Identifier, content.Version, false, timeout );
            }

            /// <summary>
            /// list-roms：ROM 情報一覧取得（※ ApplicationId のみ指定可能）
            /// </summary>
            /// <param name="id">コンテンツメタ ID</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ListRoms( ID64 id, RomStatus status, int timeout = 60 )
            {
                var command = new Command( "list-roms", $"0x{id}", ( OutputStreams streams ) =>
                {
                    var match = Regex.Match(
                        streams.Standard.ToString(),
                        "Failed to list ROM details" );
                    return new Command.Result( match.Success, null );
                } );
                StringBuilder sb = new StringBuilder();
                sb.Append( $"-a {id}" );
                if( status != 0 )
                {
                    var arg = status.ToString().Replace( " ", "" );
                    sb.Append( $" --rom-status {arg}" );
                }
                Execute( command, ToArguments( command, sb.ToString() ), new Retry.Watcher( 20, timeout ) );
            }

            /// <summary>
            /// list-roms：ROM 情報一覧取得（※ ApplicationId のみ指定可能）
            /// </summary>
            /// <param name="content">コンテンツ情報</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ListRoms( GeneratedContentResult content, RomStatus status, int timeout = 60 )
            {
                if ( content.Type != ContentMeta.Type.Application )
                {
                    throw new UnexpectFailureException( $"Not support type \"{content.Type}\"" );
                }
                ListRoms( content.Identifier, status, timeout );
            }

            /// <summary>
            /// list-titles：タイトル情報一覧取得
            /// </summary>
            /// <param name="id">コンテンツメタ ID</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ListTitles( ID64 id, int timeout = 60 )
            {
                var command = new Command( "list-titles", $"0x{id}", ( OutputStreams streams ) =>
                {
                    var match = Regex.Match(
                        streams.Standard.ToString(),
                        "Failed to list title infos" );
                    return new Command.Result( match.Success, null );
                } );
                Execute( command, ToArguments( command, $"-t {id}" ), new Retry.Watcher( 20, timeout ) );
            }

            /// <summary>
            /// list-titles：タイトル情報一覧取得
            /// </summary>
            /// <param name="content">コンテンツ情報</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ListTitles( GeneratedContentResult content, int timeout = 60 )
            {
                ListTitles( content.Identifier, timeout );
            }

            /// <summary>
            /// list-contents：コンテンツ情報一覧取得
            /// </summary>
            /// <param name="id">コンテンツメタ ID</param>
            /// <param name="version">バージョン</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ListContents( ID64 id, int version, int timeout = 60 )
            {
                var command = new Command( "list-contents", $"0x{id}", ( OutputStreams streams ) =>
                {
                    var match = Regex.Match(
                        streams.Standard.ToString(),
                        "Failed to list contents" );
                    return new Command.Result( match.Success, null );
                } );
                Execute( command, ToArguments( command, $"-t {id} -v {version}" ), new Retry.Watcher( 20, timeout ) );
            }

            /// <summary>
            /// list-contents：コンテンツ情報一覧取得
            /// </summary>
            /// <param name="content">コンテンツ情報</param>
            /// <param name="timeout">タイムアウト秒</param>
            public void ListContents( GeneratedContentResult content, int timeout = 60 )
            {
                ListContents( content.Identifier, content.Version, timeout );
            }

            /// <summary>
            /// approve-rom コマンド --operation オプション指定子.
            /// </summary>
            public enum ApproveOperation
            {
                accept,
                reject,
                revoke
            }

            /// <summary>
            /// approve-rom コマンド
            /// </summary>
            /// <param name="romId"></param>
            /// <param name="operation"><see cref="ApproveOperation"/>参照。デフォルトは accept です。</param>
            /// <param name="timeout"></param>
            public void ApproveRom( string romId, ApproveOperation operation = ApproveOperation.accept, int timeout = 60 )
            {
                var command = new Command( "approve-rom", $"{romId}", ( OutputStreams streams ) =>
                {
                    var match = Regex.Match( streams.Standard.ToString(), "Approve ROM has been completed successfully." );
                    return new Command.Result( !match.Success, null );
                } );
                Execute( command, ToArguments( command, $"-r {romId} --operation {operation}" ), new Retry.Watcher( 20, timeout ) );
            }
        }

        /// <summary>
        /// ContentsUploaderによる upload / publish 支援。
        /// </summary>
        public class NspUploader : ParallelOptions
        {
            public static class SubCommands
            {
                public static readonly Command Upload = new Command( "upload", ( OutputStreams streams ) =>
                    {
                        // NSPパス特定可能エラーログ検索、マッチすればエラー
                        var patternSequencial = @"Result: [""']?[a-zA-Z]:\\.*\.nsp[""']?, [^,]*, Error:";
                        var patternParallel = @"[""']?[a-zA-Z]:\\.*\.nsp[""']?\s?[>,]? Result: [^,]*, Error:";
                        var matches = Regex.Matches( streams.Standard.ToString(), $"{patternParallel}|{patternSequencial}" );
                        var hasError = ( matches.Count > 0 );
                        if ( false == hasError )
                        {
                            // NSPパス特定可能エラーログが見つからないので、", Upload Done." を検索してマッチしなければエラー.
                            hasError = !Regex.Match( streams.Standard.ToString(), @"Upload Done\." ).Success;
                        }
                        return new Command.Result( hasError, matches );
                    }
                );
                public static readonly Command RegisterTitle = new Command( "register-title", ( OutputStreams streams ) =>
                    {
                        var match = Regex.Match( streams.Standard.ToString(), @"Register Title Done\." );
                        return new Command.Result( false == match.Success, null );
                    }
                );
                public static readonly Command RegisterDemo = new Command( "register-demo", ( OutputStreams streams ) =>
                    {
                        var match = Regex.Match( streams.Standard.ToString(), @"Register Demo Done\." );
                        return new Command.Result( false == match.Success, null );
                    }
                );
                public static readonly Command RejectRom = new Command( "revoke-rom", ( OutputStreams streams ) =>
                    {
                        var match = Regex.Match( streams.Standard.ToString(), @"Revoke Rom Done\." );
                        return new Command.Result( false == match.Success, null );
                    }
                );
                public static readonly Command RegisterVersion = new Command( "register-version", ( OutputStreams streams ) =>
                    {
                        var match = Regex.Match( streams.Standard.ToString(), @"Register Version Done\." );
                        return new Command.Result( false == match.Success, null );
                    }
                );
                public static readonly Command DeleteVersion = new Command( "delete-version", ( OutputStreams streams ) =>
                    {
                        var match = Regex.Match( streams.Standard.ToString(), @"Delete Version Done\." );
                        return new Command.Result( false == match.Success, null );
                    }
                );
            }
            public class UploadOptions
            {
                public enum Flag : uint
                {
                    /// <summary>
                    /// デフォルト
                    /// </summary>
                    None = 0,

                    /// <summary>
                    /// ContentsUploader --mastering-enabled オプション指定。
                    /// 有効指定の場合、製品化処理が必要なTicket所持コンテンツとしてアップロード処理を行います。
                    /// デフォルトは無効です。
                    /// </summary>
                    UseProductContents = 1 << 1,

                    /// <summary>
                    /// ContentsUploader --approve-accept-enabled オプション指定。
                    /// 有効指定の場合、Publishによるアップロード完了時のromステータスを「ロットチェック合格 (APPROVED)」にします。
                    /// デフォルトは無効です。
                    /// </summary>
                    UseApprovedUpload = 1 << 2,

                    /// <summary>
                    /// ContentsUploader --skip-revoke-enabled オプション指定。
                    /// 有効指定の場合、Publishによるアップロード時に「uploadコマンド前の対象rom のrevoke (承認取り消し)処理」をスキップします。
                    /// デフォルトは無効です。
                    /// </summary>
                    UseSkipPreRevokeUpload = 1 << 3,

                    /// <summary>
                    /// ContentsUploader --allow-non-mastered オプション指定。
                    /// 有効指定の場合、製品化（チケット付与）されていない ROM のアップロードを許可します。
                    /// デフォルトは無効です。
                    /// </summary>
                    UseAllowNonMastered = 1 << 4,

                    /// <summary>
                    /// ContentsUploader --server-checking-enabled オプション指定。
                    /// 有効指定の場合、サーバー上での rom のチェック行うようになります。rom のチェック行う場合、下記制限を受けます。
                    /// ・サーバー上でロットチェック合格 (status = APPROVED) になっている version よりも、低い version の rom をアップロードできない
                    /// ・サーバー上にロットチェック待ち (status = CHECKING) の rom がいる場合、その version の rom をアップロードできない
                    /// ・PATCH の rom は、その version よりも低い version の rom がサーバー上でロットチェック合格 (status = APPROVED) になっていないとアップロードできない。
                    /// デフォルトは無効です。
                    /// </summary>
                    UseServerCheckingEnabled = 1 << 5,

                    /// <summary>
                    /// ContentsUploader --polling-timeout オプション指定。
                    /// 有効指定の場合、rom ステータス遷移のタイムアウトを指定します。タイムアウト(分)は PollingTimeoutMinutes の値が用いられます。
                    /// PollingTimeoutMinutes の既定値は 5 分です。0 以下の値が指定された場合、タイムアウトしません。
                    /// </summary>
                    UsePollingTimeout = 1 << 6,

                    /// <summary>
                    /// ContentsUploader --use-index-for-aoc-archive-number オプション指定。
                    /// 有効指定の場合、`AoC アーカイブ No.` にアップロード対象nsp の index を考慮します。
                    /// 最終的に適用される `AoC アーカイブ No.` は、AocArchiveNumberBase に index を足した値になります。
                    /// - AoC を複数含む nsp の場合は、最も若い index が使用されれます。
                    /// - AoC 以外を含む nsp の場合は、このオプションは無視されます。
                    /// </summary>
                    UseIndexForAocArchiveNumber = 1 << 7,
                }

                /// <summary>
                /// 事前定義済オプション
                /// </summary>
                public static class Constants
                {
                    /// <summary>
                    /// 既定値 ( 通常 )
                    /// </summary>
                    public static readonly UploadOptions Normal = new UploadOptions( Flag.UseAllowNonMastered | Flag.UsePollingTimeout );

                    /// <summary>
                    /// 既定値 ( Patch upload 用 )
                    /// パッチは作成時に製品化（チケット付与）されるので UseAllowNonMastered が不要。
                    /// </summary>
                    public static readonly UploadOptions Patch = new UploadOptions( Flag.UseApprovedUpload | Flag.UseSkipPreRevokeUpload | Flag.UsePollingTimeout );

                    /// <summary>
                    /// 既定値 ( 承認完了用 )
                    /// </summary>
                    public static readonly UploadOptions WithApproved = new UploadOptions( Flag.UseApprovedUpload | Flag.UseAllowNonMastered | Flag.UsePollingTimeout );

                    /// <summary>
                    /// 承認完了＋サーバチェック有り＋製品化必須
                    /// </summary>
                    public static readonly UploadOptions WithCheckingAndApproved = new UploadOptions( Flag.UseApprovedUpload | Flag.UseServerCheckingEnabled | Flag.UsePollingTimeout );

                    /// <summary>
                    /// Aoc アーカイブ No. をアップロード対象Aoc Indexで代用.
                    /// チケット有無のどちらでも使えます。 ( --allow-non-mastered はチケットありなら無視される仕様の模様 )
                    /// --polling-timeout
                    /// --allow-non-mastered
                    /// --use-index-for-aoc-archive-number
                    /// </summary>
                    public static readonly UploadOptions AocArchiveAsIndex = new UploadOptions( Flag.UseIndexForAocArchiveNumber | Flag.UseAllowNonMastered | Flag.UsePollingTimeout );
                }

                /// <summary>
                /// 設定値
                /// </summary>
                public Flag Flags { get; private set; } = Flag.None;

                /// <summary>
                /// rom ステータス遷移のタイムアウト(分)です。
                /// </summary>
                public int PollingTimeoutMinutes { get; set; } = 5;

                /// <summary>
                /// `AoC アーカイブ No.` 基準値です。
                /// Flags.UseIndexForAocArchiveNumber 未指定時は、この設定値が `AoC アーカイブ No.` として適用されます。
                /// </summary>
                public int AocArchiveNumberBase { get; set; } = 0;

                /// <summary>
                /// コンストラクタ
                /// </summary>
                /// <param name="flags"></param>
                public UploadOptions( Flag flags = Flag.None )
                {
                    Flags = flags;
                }

                /// <summary>
                /// コピーコンストラクタ
                /// </summary>
                /// <param name="other"></param>
                public UploadOptions( UploadOptions other )
                {
                    if ( null != other )
                    {
                        Flags = other.Flags;
                        AocArchiveNumberBase = other.AocArchiveNumberBase;
                        PollingTimeoutMinutes = other.PollingTimeoutMinutes;
                    }
                }

                /// <summary>
                /// フラグ追加
                /// </summary>
                /// <param name="flag">source インスタンスに追加したいフラグ</param>
                /// <returns>指定のフラグを付与した新しい UploadOptions インスタンス</returns>
                public UploadOptions Append( Flag flag )
                {
                    return new UploadOptions( Flags | flag );
                }

                /// <summary>
                /// フラグ削除
                /// </summary>
                /// <param name="flag">source インスタンスから追加したいフラグ</param>
                /// <returns>指定のフラグを削除した新しい UploadOptions インスタンス</returns>
                public UploadOptions Remove( Flag flag )
                {
                    return new UploadOptions( Flags & ( ~flag ) );
                }

                /// <summary>
                /// コマンドラインオペランド支援.
                /// 追加されたオプションは本文字列の末尾に対して、"空白１文字" + "options" の形式で返されます。
                /// 末尾に空白が追加されない点に注意してください。
                /// </summary>
                /// <param name="b">追加先バッファ</param>
                /// <returns>追加後のバッファ。引数と同じインスタンス。</returns>
                public StringBuilder AppendCommandLineArguments( StringBuilder b )
                {
                    if ( Flags.HasFlag( Flag.UseProductContents ) )
                    {
                        b.Append( " -m" );
                    }
                    if ( Flags.HasFlag( Flag.UseApprovedUpload ) )
                    {
                        b.Append( " --approve-accept-enabled" );
                    }
                    if ( Flags.HasFlag( Flag.UseSkipPreRevokeUpload ) )
                    {
                        b.Append( " --skip-revoke-enabled" );
                    }
                    if ( Flags.HasFlag( Flag.UseAllowNonMastered ) )
                    {
                        b.Append( " --allow-non-mastered" );
                    }
                    if ( Flags.HasFlag( Flag.UseServerCheckingEnabled ) )
                    {
                        b.Append( " --server-checking-enabled" );
                    }
                    if ( Flags.HasFlag( Flag.UsePollingTimeout ) )
                    {
                        b.Append( $" --polling-timeout {PollingTimeoutMinutes}" );
                    }
                    if ( Flags.HasFlag( Flag.UseIndexForAocArchiveNumber ) )
                    {
                        b.Append( $" --use-index-for-aoc-archive-number" );
                    }
                    if ( AocArchiveNumberBase > 0 )
                    {
                        b.Append( $" --aoc-archive-number {AocArchiveNumberBase}" );
                    }
                    return b;
                }

                /// <summary>
                /// Version と Aoc Index に基づいた Archive Number でアップロードする UploadOption を作成します。
                /// </summary>
                /// <param name="version">アップロード対象バージョン</param>
                /// <returns></returns>
                public static UploadOptions MakeUniqueAocArchiveNumberBy( int version )
                {
                    var option = new UploadOptions( Constants.AocArchiveAsIndex );
                    option.AocArchiveNumberBase = version;
                    return option;
                }

                /// <summary>
                /// Version と Aoc Index に基づいた Archive Number でアップロードする UploadOption を作成します。
                /// </summary>
                /// <param name="nsp">アップロード対象コンテンツ</param>
                /// <returns></returns>
                public static UploadOptions MakeUniqueAocArchiveNumberBy( GeneratedContentResult nsp )
                {
                    return MakeUniqueAocArchiveNumberBy( nsp.Version );
                }
            }

            /// <summary>
            /// Web access proxy.
            /// </summary>
            public WebProxy Proxy { get; }

            /// <summary>
            /// Server environment.
            /// </summary>
            public Configuration.ServerEnvironment TargetServer { get; }

            /// <summary>
            /// 作業用ディレクトリ
            /// </summary>
            public string IntermediateDirectory { get; }

            /// <summary>
            /// 失敗と判断するリトライ回数です。
            /// タイムアウトとは別に同じ要求に対して本プロパティ回数のリトライを行っても成功しない場合は失敗したと判断します。
            /// デフォルト値は 5回です。
            /// </summary>
            public int FailureRetryCount { get; set; }

            /// <summary>
            /// アップロード系オプションコンテナ
            /// </summary>
            public UploadOptions BaseUploadOptions { get; }

            /// <summary>
            /// 詳細ログ出力の可否です。
            /// </summary>
            public bool IsVerboseLog { get; set; }

            /// <summary>
            /// 最後に出力したログ
            /// </summary>
            public string LastLog { get; set; }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="proxy">null の場合、明示的なプロキシなし</param>
            /// <param name="server">アップロード対象サーバーサービス</param>
            public NspUploader( string intermediateDirectory, WebProxy proxy = null, Configuration.ServerEnvironment server = null )
            {
                IntermediateDirectory = FileHelper.MakeDirectory( intermediateDirectory ).FullName;
                TargetServer = ( null != server ) ? server : Configuration.ServerEnvironment.DEV6;
                Proxy = proxy;
                MaxDegreeOfParallelism = 4;
                FailureRetryCount = 5;
                BaseUploadOptions = UploadOptions.Constants.Normal;
                IsVerboseLog = false;
            }

            /// <summary>
            /// ContentsUploader.exe パス検出.
            /// </summary>
            /// <returns></returns>
            public static string GetExecutionToolPath()
            {
                return SigloHelper.ToolPath.FindTools( System.IO.Path.Combine( "CommandLineTools", "ContentsUploader", "ContentsUploader.exe" ) );
            }

            /// <summary>
            /// ContentsUploader.exe プロセス実行.
            /// </summary>
            /// <param name="command">ContentsUploader サブコマンド</param>
            /// <param name="arguments">サブコマンドオペランド引数</param>
            /// <param name="outputs">実行結果標準出力バッファ</param>
            /// <param name="matchResult">指定サブコマンドのエラー検出マッチング結果</param>
            /// <exception cref="UnexpectFailureException">失敗した時に投げられます</exception>
            public bool Execute( Command command, string arguments, OutputStreams outputs, out Command.Result matchResult )
            {
                outputs = ( null == outputs ) ? new OutputStreams( 128 * 1024 ) : outputs;
                int result = CommandLineExecutor.ExecuteOnProcess( GetExecutionToolPath(), $"{command.Name} {arguments}", outputs );
                matchResult = command.ErrorDetector( outputs );
                LastLog = outputs.Standard.Buffer.ToString();
                return ( 0 == result && false == matchResult.HasError );
            }

            /// <summary>
            /// environment, proxy オプションの追加。
            /// </summary>
            /// <param name="b">オプション追加バッファ</param>
            /// <param name="appendix">任意追加オペランド</param>
            /// <returns></returns>
            private StringBuilder AppendConfiguration( StringBuilder b, string appendix = null )
            {
                if ( false == string.IsNullOrEmpty( appendix ) )
                {
                    b.Append( ' ' ).Append( appendix );
                }
                b.Append( " -e " ).Append( TargetServer.Alias );
                if ( null != Proxy )
                {
                    b.Append( " --proxy \"" ).Append( Proxy.Address.AbsoluteUri ).Append( '\"' );
                }
                if ( IsVerboseLog )
                {
                    b.Append( " -v" );
                }
                return b;
            }

            /// <summary>
            /// 要素単位リクエスト管理支援.
            /// </summary>
            private class RetryObserver : Retry.Observer<string>
            {
                public enum Result : byte
                {
                    /// <summary>
                    /// ログからリトライ対象の nsp特定に失敗した nspに依存しないケース.
                    /// Basic authentication failed など, カレントパスを再度リトライします.
                    /// </summary>
                    NoneMatched,
                    /// <summary>
                    /// ログからリトライ対象の nsp特定に成功し、何れかのリトライに成功したケース.
                    /// nsp単品アップロードとしてリトライキューに登録されます.
                    /// </summary>
                    NewRetryEntry,
                    /// <summary>
                    /// ログからリトライ対象の nsp特定に成功したが、対象全てがリトライ上限回数を超えてリトライしていた.
                    /// </summary>
                    NoneRetryEntry,
                }

                public RetryObserver( int leftoverTrialCount ) : base( leftoverTrialCount, 32 ) {}

                protected override string OnCorrectValue( string inValue )
                {
                    return System.IO.Path.GetFullPath( inValue );
                }

                /// <summary>
                /// ErrorDetector が検出した結果ログからリトライ対象のnspの特定を行い、リトライ要求します。
                /// </summary>
                /// <param name="matches"></param>
                /// <param name="suffix"></param>
                /// <returns></returns>
                public Result CollectMatchedPath( MatchCollection matches, string suffix = "nsp" )
                {
                    if ( null != matches && matches.Count > 0 )
                    {
                        Result result = Result.NoneRetryEntry;
                        var regex = new Regex( @"[a-zA-Z]:\\.*\." + suffix );
                        foreach ( Match match in matches )
                        {
                            var nspMatched = regex.Match( match.Value );
                            if ( nspMatched.Success && TryRetry( nspMatched.Value ) >= 0 )
                            {
                                result = Result.NewRetryEntry;
                            }
                        }
                        return result;
                    }
                    return Result.NoneMatched;
                }

                /// <summary>
                /// リトライキュー状態をレポートします。
                /// </summary>
                /// <param name="prefix"></param>
                /// <param name="b"></param>
                public void Reports( string prefix, StringBuilder b = null )
                {
                    b = ( null == b ) ? new StringBuilder( 512 ) : b.Clear();
                    foreach ( string item in Queue )
                    {
                        b.Append( "    " ).Append( item ).Append( '\n' );
                    }
                    Log.WriteLine( "{0} => {{\n{1}}}", prefix, b.ToString() );
                }
            }

            /// <summary>
            /// ContentsUploader upload 実行
            /// </summary>
            /// <param name="nspFilePath">フォルダパス or ファイルパス</param>
            /// <param name="options">任意追加オペランド</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            /// <param name="uploadOptions">アップロード機能制御オプション</param>
            /// <exception cref="UnexpectFailureException">失敗した時に投げられます</exception>
            public void Publish( string nspFilePath, string options, double timeoutSeconds = Timeout.Infinity, UploadOptions uploadOptions = null )
            {
                if ( string.IsNullOrEmpty( nspFilePath ) )
                {
                    throw new System.ArgumentNullException();
                }

                uploadOptions = ( null != uploadOptions ) ? uploadOptions : BaseUploadOptions;
                var timeoutObserver = Timeout.Observer.FromSeconds( timeoutSeconds );
                var retryObserver = new RetryObserver( FailureRetryCount );
                retryObserver.TryRetry( nspFilePath );

                while ( null != ( nspFilePath = retryObserver.QueryNext() ) )
                {
                    // check timeout
                    if ( timeoutObserver.IsTimeout() )
                    {
                        retryObserver.Reports( "Publish timeout, remain retry contents" );
                        throw new UnexpectFailureException( "Publish timeout, upload operation failed." );
                    }

                    // preapre --input.
                    StringBuilder b = new StringBuilder( 512 );
                    b.Append( "-s \"" ).Append( nspFilePath ).Append( '\"' );

                    // Will be check the specified path whether to a file or a directory.
                    if ( System.IO.File.Exists( nspFilePath ) )
                    {
                        // path is a file.
                    }
                    else if ( System.IO.Directory.Exists( nspFilePath ) )
                    {
                        // path is a directory, therefore will be using feature of parallel job.
                        b.Append( " -j " ).Append( MaxDegreeOfParallelism );
                    }
                    else
                    {
                        throw new System.ArgumentException( $"The specified path does not exists on storage. => [ {nspFilePath} ]" );
                    }
                    uploadOptions.AppendCommandLineArguments( AppendConfiguration( b, options ) );
                    Command.Result matchedResult;
                    if ( false == Execute( SubCommands.Upload, b.ToString(), null, out matchedResult ) )
                    {
                        var result = retryObserver.CollectMatchedPath( matchedResult.DetectCollection );
                        if ( RetryObserver.Result.NewRetryEntry  == result )
                        {
                            // ログから再試行対象のnsp特定成功＆再試行できた.
                            retryObserver.Reports( "Publish retry upload", b );
                        }
                        else if ( RetryObserver.Result.NoneRetryEntry == result || 0 > retryObserver.TryRetry( nspFilePath, true ) )
                        {
                            // - ログから再試行対象のnsp特定成功したが、対象全ての再試行に失敗( NoneRetryEntry )
                            // - Basic authentication failed などnsp特定失敗＆カレントパスの再試行に失敗( NoneMatched )
                            // -- 基本的に再試行失敗原因は、対象が再試行上限回数を超えて再試行済のためです.
                            throw new UnexpectFailureException( "Failure the retry becase has be over the count of retry, it would like be happened critical error of the ContentsUploader." );
                        }
                    }
                }
            }

            /// <summary>
            /// ContentsUploader --user / --password を用いたアップロードを行います。
            /// </summary>
            /// <param name="cliAccount">CLIトークン取得用ユーザアカウント情報</param>
            /// <param name="nspFilePath">アップロードするnspファイルパス</param>
            /// <param name="uploadOptions">アップロード機能制御オプション</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            /// <exception cref="UnexpectFailureException">失敗した時に投げられます</exception>
            public void Publish( BasicAccount cliAccount, string nspFilePath, UploadOptions uploadOptions, double timeoutSeconds = Timeout.Infinity )
            {
                if ( null == cliAccount )
                {
                    throw new System.ArgumentNullException();
                }
                Publish( nspFilePath, string.Format( "-u {0} -p {1}", cliAccount.ID, cliAccount.PASSWORD ), timeoutSeconds, uploadOptions );
            }

            /// <summary>
            /// ContentsUploader --user / --password を用いたアップロードを行います。
            /// </summary>
            /// <param name="cliAccount">CLIトークン取得用ユーザアカウント情報</param>
            /// <param name="nspFilePath">アップロードするnspファイルパス</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            /// <exception cref="UnexpectFailureException">失敗した時に投げられます</exception>
            public void Publish( BasicAccount cliAccount, string nspFilePath, double timeoutSeconds = Timeout.Infinity )
            {
                Publish( cliAccount, nspFilePath, null, timeoutSeconds );
            }

            /// <summary>
            /// ContentsUploader --token を用いたアップロードを行います。
            /// </summary>
            /// <param name="token">CLIトークンファイルパス( *.json )</param>
            /// <param name="nspFilePath">アップロードするnspファイルパス</param>
            /// <param name="uploadOptions">アップロード機能制御オプション</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            /// <exception cref="UnexpectFailureException">失敗した時に投げられます</exception>
            public void Publish( Configuration.CliToken token, string nspFilePath, UploadOptions uploadOptions, double timeoutSeconds = Timeout.Infinity )
            {
                if ( null == token )
                {
                    throw new System.ArgumentNullException();
                }
                Publish( nspFilePath, "-t " + token.FilePath, timeoutSeconds, uploadOptions );
            }

            /// <summary>
            /// ContentsUploader --token を用いたアップロードを行います。
            /// </summary>
            /// <param name="token">CLIトークンファイルパス( *.json )</param>
            /// <param name="nspFilePath">アップロードするnspファイルパス</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            /// <exception cref="UnexpectFailureException">失敗した時に投げられます</exception>
            public void Publish( Configuration.CliToken token, string nspFilePath, double timeoutSeconds = Timeout.Infinity )
            {
                Publish( token, nspFilePath, null, timeoutSeconds );
            }

            /// <summary>
            /// 複数nspアップロード( 並列対応版 )
            /// </summary>
            /// <param name="cliAccount">最新CLIトークン取得用BASIC認証アカウント</param>
            /// <param name="titles">アップロード対象タイトルのnspファイルパスリスト</param>
            /// <param name="uploadOptions">アップロード機能制御オプション</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            public void Publish( BasicAccount cliAccount, List<string> titles, UploadOptions uploadOptions, double timeoutSeconds = Timeout.Infinity )
            {
                if ( null == titles || titles.Count <= 0 )
                {
                    Log.WriteLine( "Could not found a contents." );
                    return;
                }

                // フォルダコピーステージ
                var tempFolder = FileHelper.CopyToTemporary( titles, IntermediateDirectory );

                // ContentsUploader アップロードステージ
                string formatBase = string.Format( "--- Publish the random identifier contents, ContentsUploader " );
                try
                {
                    Log.WriteLine( formatBase + "Begin ---" );
                    Publish( cliAccount, tempFolder.FullName, uploadOptions, timeoutSeconds );
                    Log.WriteLine( formatBase + "Finish ---" );
                }
                catch ( System.Exception e )
                {
                    Log.WriteLine( formatBase + "Finish by exception =>\n[exception]: {0}\n\n", e.Message );
                    throw new UnexpectFailureException( "NspUploader parallel upload operation failed." );
                }
                finally
                {
                    FileHelper.RemoveDirectory( tempFolder );
                }
            }

            /// <summary>
            /// 複数nspアップロード( 並列対応版 )
            /// </summary>
            /// <param name="cliAccount">最新CLIトークン取得用BASIC認証アカウント</param>
            /// <param name="titles">アップロード対象タイトルのnspファイルパスリスト</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            public void Publish( BasicAccount cliAccount, List<string> titles, double timeoutSeconds = Timeout.Infinity )
            {
                Publish( cliAccount, titles, null, timeoutSeconds );
            }

            /// <summary>
            /// 連番メタID( version=0, Type=Application, 固定 )によるnsp生成/アップロード( 並列対応版 )
            /// </summary>
            /// <param name="cliAccount">最新CLIトークン取得用BASIC認証アカウント</param>
            /// <param name="beginAppId">連番開始ID</param>
            /// <param name="appCount">連番総数</param>
            /// <param name="version">固定バージョン</param>
            /// <param name="platform">nsp生成対象プラットフォーム</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            public void Publish( BasicAccount cliAccount, ID64 beginAppId, int appCount, int version, string platform = SigloHelper.Configuration.DefaultPlatform, double timeoutSeconds = Timeout.Infinity )
            {
                // MakeTestApplication 生成ステージ
                Parallel.For( 0, appCount, this, ( int index, ParallelLoopState state ) =>
                {
                    ID64 nowAppId = beginAppId + ( ( ulong )index );
                    string formatBase = string.Format( "--- Publish the sequential identifier contents, MakeTestApplication [ {0} : {1} ] ", nowAppId, version );
                    Log.WriteLine( formatBase + "Begin ---" );
                    TestApplication.MakeOneAsStandard( IntermediateDirectory, nowAppId, version, platform );
                    Log.WriteLine( formatBase + "Finish ---" );
                } );

                // ContentsUploader アップロードステージ
                {
                    string formatBase = string.Format( "--- Publish the sequential identifier contents, ContentsUploader " );
                    try
                    {
                        Log.WriteLine( formatBase + "Begin ---" );
                        Publish( cliAccount, IntermediateDirectory, timeoutSeconds );
                        Log.WriteLine( formatBase + "Finish ---" );
                    }
                    catch ( System.Exception e )
                    {
                        Log.WriteLine( formatBase + "Finish by exception =>\n[exception]: {0}\n\n", e.Message );
                        throw new UnexpectFailureException( "NspUploader parallel upload operation failed." );
                    }
                }
            }

            /// <summary>
            /// 複数nspアップロード( <see cref="GeneratedContentResult.TitleCategorizedCatalog"/> 利用版 )
            /// カタログの各タイプ毎のリストはバージョンでソートされている必要があります。
            /// また、本メソッドでアップロード対象とするメタタイプは以下です。
            /// <see cref="ContentMeta.Type.Application"/>
            /// <see cref="ContentMeta.Type.Patch"/>
            /// <see cref="ContentMeta.Type.AddOnContent"/>
            /// </summary>
            /// <param name="cliAccount">最新CLIトークン取得用BASIC認証アカウント</param>
            /// <param name="titles">アップロード対象タイトルのnspカタログ</param>
            /// <param name="timeoutSeconds">1つのnspのアップロードタイムアウト秒数, 0 は無限</param>
            public void Publish( BasicAccount cliAccount, GeneratedContentResult.TitleCategorizedCatalog titles, double timeoutSecondsAsOneNsp = Timeout.Infinity )
            {
                // 対象ステージへの登録
                System.Func<List<GeneratedContentResult.Catalog>, int, GeneratedContentResult, bool> functionRegisterStage =
                ( List<GeneratedContentResult.Catalog> stages, int stageIndex, GeneratedContentResult content ) =>
                {
                    GeneratedContentResult.Catalog catalog;
                    if ( stages.Count > stageIndex )
                    {
                        catalog = stages[ stageIndex ];
                    }
                    else
                    {
                        stages.Add( catalog = new GeneratedContentResult.Catalog( 64 ) );
                    }
                    catalog.Add( content );
                    return true;
                };

                // ステージ単位での並列( patch はバージョン )
                var stageApps = new GeneratedContentResult.Catalog( titles.Count );
                var stageAocs = new GeneratedContentResult.Catalog( titles.Count );
                var stagePatches = new List<GeneratedContentResult.Catalog>( 16 );
                foreach ( var title in titles )
                {
                    var types = title.Value;
                    GeneratedContentResult.Catalog catalog;
                    if ( null != ( catalog = types.GetTypedCatalog( ContentMeta.Type.Application ) ) )
                    {
                        stageApps.AddRange( catalog );
                    }
                    if ( null != ( catalog = types.GetTypedCatalog( ContentMeta.Type.AddOnContent ) ) )
                    {
                        stageAocs.AddRange( catalog );
                    }
                    if ( null != ( catalog = types.GetTypedCatalog( ContentMeta.Type.Patch ) ) )
                    {
                        int index = 0;
                        foreach ( var c in catalog )
                        {
                            functionRegisterStage( stagePatches, index, c );
                            ++index;
                        }
                    }
                }

                // revoke-rom 並列ステージ
                Log.WriteLine( "Publish with 'GeneratedContentResult.TitleCategorizedCatalog' parameters => Reject roms... " );
                Parallel.ForEach( titles, ( title, state ) =>
                {
                    Log.WriteLine( $"Publish with 'GeneratedContentResult.TitleCategorizedCatalog' parameters => Reject rom [ {title.Key} ]" );
                    RejectRoms( cliAccount, title.Key );
                } );

                // Application for parallel
                {
                    Log.WriteLine( "Publish with 'GeneratedContentResult.TitleCategorizedCatalog' parameters => Application uploading... " );
                    var timeout = ( Timeout.Infinity == timeoutSecondsAsOneNsp ) ? Timeout.Infinity : timeoutSecondsAsOneNsp * stageApps.Count;
                    Publish( cliAccount, stageApps.ToNspPaths(), UploadOptions.Constants.WithApproved, timeout );
                }
                // AddOnContents for parallel
                {
                    Log.WriteLine( "Publish with 'GeneratedContentResult.TitleCategorizedCatalog' parameters => AddOnContents uploading... " );
                    var timeout = ( Timeout.Infinity == timeoutSecondsAsOneNsp ) ? Timeout.Infinity : timeoutSecondsAsOneNsp * stageAocs.Count;
                    Publish( cliAccount, stageAocs.ToNspPaths(), null, timeout );
                }
                // Patch for parallel
                {
                    Log.WriteLine( "Publish with 'GeneratedContentResult.TitleCategorizedCatalog' parameters => Patch uploading... " );
                    int index = 0;
                    foreach ( var patches in stagePatches )
                    {
                        var timeout = ( Timeout.Infinity == timeoutSecondsAsOneNsp ) ? Timeout.Infinity : timeoutSecondsAsOneNsp * patches.Count;
                        Publish( cliAccount, patches.ToNspPaths(), UploadOptions.Constants.Patch, timeout );
                        ++index;
                    }
                }
            }

            /// <summary>
            /// タイムアウトまでの再試行付き、サブコマンドの実行。
            /// </summary>
            /// <param name="command"></param>
            /// <param name="arguments"></param>
            /// <param name="signature">進捗通知用タグシグネチャ</param>
            /// <param name="timeoutSeconds"></param>
            private void ExecuteWithRetry( Command command, string arguments, string signature, double timeoutSeconds = 5 * 60 )
            {
                bool response;
                var timeoutObserver = Timeout.Observer.FromSeconds( timeoutSeconds );
                var retryMessage = $"ContentsUploader failed on [ {command.Name} ] => {signature}, Retry after 10 seconds.";
                Command.Result matchedResult;
                while ( false == ( response = Execute( command, arguments, null, out matchedResult ) ) &&
                    false == timeoutObserver.IsTimeout() )
                {
                    Log.WriteLine( retryMessage );
                    System.Threading.Thread.Sleep( 10000 ); // 10秒待機
                }
                if ( false == response )
                {
                    throw new UnexpectFailureException( $"ContentsUploader operation failed on [ {command.Name} ] => {signature}." );
                }
            }

            /// <summary>
            /// アップロード済コンテンツを商品登録する
            /// </summary>
            /// <param name="cliAccount"></param>
            /// <param name="nspFilePath"></param>
            /// <param name="initialCode"></param>
            /// <param name="timeoutSeconds"></param>
            public void RegisterTitleContent( BasicAccount cliAccount, string nspFilePath, string initialCode, double timeoutSeconds = 5 * 60 )
            {
                if ( null == cliAccount || string.IsNullOrEmpty( nspFilePath ) || string.IsNullOrEmpty( initialCode ) )
                {
                    throw new System.ArgumentNullException();
                }
                var arguments = AppendConfiguration( new StringBuilder(
                    $"-s \"{nspFilePath}\" --initial-code {initialCode} -u {cliAccount.ID} -p {cliAccount.PASSWORD}", 512 ) ).ToString();
                ExecuteWithRetry( SubCommands.RegisterTitle, arguments, nspFilePath, timeoutSeconds );
            }

            /// <summary>
            /// アップロード済コンテンツを商品登録する
            /// </summary>
            /// <param name="cliAccount"></param>
            /// <param name="contentMetaId"></param>
            /// <param name="initialCode"></param>
            /// <param name="timeoutSeconds"></param>
            public void RegisterTitleContent( BasicAccount cliAccount, ID64 contentMetaId, string initialCode, double timeoutSeconds = 5 * 60 )
            {
                if ( null == cliAccount || null == contentMetaId || string.IsNullOrEmpty( initialCode ) )
                {
                    throw new System.ArgumentNullException();
                }
                string arguments = AppendConfiguration( new StringBuilder(
                    $"--application-id 0x{contentMetaId} --initial-code {initialCode} -u {cliAccount.ID} -p {cliAccount.PASSWORD}", 512 ) ).ToString();
                ExecuteWithRetry( SubCommands.RegisterTitle, arguments, $"0x{contentMetaId}", timeoutSeconds );
            }

            /// <summary>
            /// アップロード済コンテンツを予約販売商品として商品登録する
            /// </summary>
            /// <param name="cliAccount"></param>
            /// <param name="contentMetaId"></param>
            /// <param name="initialCode"></param>
            /// <param name="priceControl"></param>
            /// <param name="preorderControl"></param>
            /// <param name="timeoutSeconds"></param>
            public void RegisterTitleContent(BasicAccount cliAccount, ID64 contentMetaId, string initialCode, string priceControl, string preorderControl, string targetCountry = null, double timeoutSeconds = 5 * 60)
            {
                if (null == cliAccount || null == contentMetaId || string.IsNullOrEmpty(initialCode))
                {
                    throw new System.ArgumentNullException();
                }
                string priceControlArgument = preorderControl == null ? "" : $"--price-control {priceControl}";
                string preorderControlArgument = preorderControl == null ? "" : $"--preorder-control {preorderControl}";
                string targetCountryArgument = targetCountry == null ? "" : $"--target-country {targetCountry}";
                string arguments = AppendConfiguration(new StringBuilder(
                    $"--application-id 0x{contentMetaId} --initial-code {initialCode} -u {cliAccount.ID} -p {cliAccount.PASSWORD} {priceControlArgument} {preorderControlArgument} {targetCountryArgument}", 512)).ToString();
                ExecuteWithRetry(SubCommands.RegisterTitle, arguments, $"0x{contentMetaId}", timeoutSeconds);
            }

            /// <summary>
            /// アップロード済コンテンツを体験版登録する( nspファイル指定 )
            /// </summary>
            /// <param name="nspFilePath"></param>
            /// <param name="timeoutSeconds"></param>
            public void RegisterDemoContent( string nspFilePath, double timeoutSeconds = 5 * 60 )
            {
                if ( string.IsNullOrEmpty( nspFilePath ) )
                {
                    throw new System.ArgumentNullException();
                }

                StringBuilder b = new StringBuilder( 512 );
                string arguments = AppendConfiguration( b.Append( "-s \"" ).Append( nspFilePath ).Append( '\"' ) ).ToString();
                ExecuteWithRetry( SubCommands.RegisterDemo, arguments, nspFilePath, timeoutSeconds );
            }

            /// <summary>
            /// アップロード済コンテンツを体験版登録する( タイトルID指定, Application固定 )
            /// </summary>
            /// <param name="contentMetaId"></param>
            /// <param name="timeoutSeconds">デフォルト30分</param>
            public void RegisterDemoContent( ID64 contentMetaId, double timeoutSeconds = 5 * 60 )
            {
                if ( null == contentMetaId )
                {
                    throw new System.ArgumentNullException();
                }
                StringBuilder b = new StringBuilder( 512 );
                string arguments = AppendConfiguration( b.Append( "--application-id 0x" ).Append( contentMetaId ) ).ToString();
                ExecuteWithRetry( SubCommands.RegisterDemo, arguments, $"0x{contentMetaId}", timeoutSeconds );
            }

            /// <summary>
            /// アップロード済コンテンツを PMSに登録する
            /// </summary>
            /// <param name="contentMetaId">コンテンツメタID</param>
            /// <param name="type">コンテンツメタタイプ</param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            public void RegisterRoms( ID64 contentMetaId, ContentMeta.Type type, double timeoutSeconds = 5 * 60 )
            {
                if ( null == contentMetaId )
                {
                    throw new System.ArgumentNullException();
                }
                var pms = new NintendoServices.PmsEditor( Proxy, TargetServer );

                bool success = false;
                var timeoutObserver = Timeout.Observer.FromSeconds( timeoutSeconds );
                var retryMessage = $"ContentsUploader failed on [ register-roms ] => 0x{contentMetaId}, {type.ToString()}, Retry after 10 seconds.";
                do
                {
                    try
                    {
                        pms.RegisterRoms( contentMetaId, type );
                        success = true;
                        break;
                    }
                    catch ( System.Exception e )
                    {
                        Log.WriteLine( e.Message );
                    }
                    Log.WriteLine( retryMessage );
                    System.Threading.Thread.Sleep( 10000 ); // 10秒待機
                } while ( false == timeoutObserver.IsTimeout() );

                if ( false == success )
                {
                    throw new UnexpectFailureException( $"ContentsUploader operation failed on [ register-roms ] => 0x{contentMetaId}, {type.ToString()}." );
                }
            }

            /// <summary>
            /// アップロード済パッチコンテンツを PMSに登録する
            /// </summary>
            /// <param name="contentMetaId"></param>
            /// <param name="timeoutSeconds">タイムアウト秒数, 0 は無限</param>
            public void RegisterPatch( string nspFilePath, double timeoutSeconds = 5 * 60 )
            {
                if ( string.IsNullOrEmpty( nspFilePath ) )
                {
                    throw new System.ArgumentNullException();
                }

                StringBuilder b = new StringBuilder( 512 );
                string arguments = AppendConfiguration( b.Append( "-s \"" ).Append( nspFilePath ).Append( '\"' ) ).ToString();
                ExecuteWithRetry( SubCommands.RegisterDemo, arguments, nspFilePath, timeoutSeconds );
            }

            /// <summary>
            /// 指定メタID( アプリケーション )に紐づく rom ステータスを REJECTED にします。
            /// </summary>
            /// <param name="contentMetaId">アプリケーションID</param>
            /// <param name="timeoutSeconds"></param>
            public void RejectRoms( BasicAccount cliAccount, ID64 contentMetaId, double timeoutSeconds = 5 * 60 )
            {
                if ( null == contentMetaId )
                {
                    throw new System.ArgumentNullException();
                }
                StringBuilder b = new StringBuilder( 512 );
                b.Append( "--application-id 0x" ).Append( contentMetaId );
                string arguments = AppendConfiguration( b, string.Format( "-u {0} -p {1}", cliAccount.ID, cliAccount.PASSWORD ) ).ToString();
                ExecuteWithRetry( SubCommands.RejectRom, arguments, $"0x{contentMetaId}", timeoutSeconds );
            }

            /// <summary>
            /// バージョン設定オプション
            /// </summary>
            public class VersionOptions
            {
                public static class Constants
                {
                    /// <summary>
                    /// 未指定値
                    /// </summary>
                    public const int Unspecified = -1;
                }
                public int DeliverVersion { get; set; }
                public int NotifyVersion { get; set; }

                /// <summary>
                /// コンストラクタ
                /// </summary>
                /// <param name="deliver">配信設定バージョン値, Constants.Unspecified は未指定扱い</param>
                /// <param name="notify">通知設定バージョン値, Constants.Unspecified は未指定扱い</param>
                public VersionOptions( int deliver, int notify )
                {
                    DeliverVersion = deliver;
                    NotifyVersion = notify;
                }

                /// <summary>
                /// デフォルトコンストラクタ
                /// </summary>
                /// <param name="version">配信/通知設定バージョン値, Constants.Unspecified は未指定扱い</param>
                public VersionOptions( int version = Constants.Unspecified ) : this( version, version )
                {
                }

                /// <summary>
                /// コマンドラインオペランド支援.
                /// 追加されたオプションは本文字列の末尾に対して、"空白１文字" + "options" の形式で返されます。
                /// 末尾に空白が追加されない点に注意してください。
                /// </summary>
                /// <param name="b">追加先バッファ</param>
                /// <returns>追加後のバッファ。引数と同じインスタンス。</returns>
                public StringBuilder AppendCommandLineArguments( StringBuilder b )
                {
                    if ( DeliverVersion > Constants.Unspecified )
                    {
                        b.Append( " --content-meta-version " ).Append( DeliverVersion );
                    }
                    if ( NotifyVersion > Constants.Unspecified )
                    {
                        b.Append( " --content-meta-notify-version " ).Append( NotifyVersion );
                    }
                    return b;
                }
            }

            /// <summary>
            /// アップロードコンテンツの最新バージョンを superfly に設定する。
            /// </summary>
            /// <param name="nspFilePath">アップロードNSPファイルパス。</param>
            /// <param name="options">配信/通知バージョン設定, null及び未指定時は nsp設定値が採用されます。</param>
            /// <param name="timeoutSeconds">タイムアウト</param>
            public void RegisterVersion( string nspFilePath, VersionOptions options = null, double timeoutSeconds = 5 * 60 )
            {
                if ( string.IsNullOrEmpty( nspFilePath ) )
                {
                    throw new System.ArgumentNullException();
                }
                StringBuilder b = new StringBuilder( 512 );
                b.Append( "-s \"" ).Append( nspFilePath ).Append( '\"' );
                options = ( null != options ) ? options : new VersionOptions();
                string arguments = AppendConfiguration( options.AppendCommandLineArguments( b ) ).ToString();
                ExecuteWithRetry( SubCommands.RegisterVersion, arguments, nspFilePath, timeoutSeconds );
            }

            /// <summary>
            /// アップロードコンテンツの最新バージョンを superfly に設定する。
            /// </summary>
            /// <param name="contentMetaId">設定対象コンテンツの ContentMetaId</param>
            /// <param name="ownerApplicationId">設定対象コンテンツのオーナーアプリケーションメタID</param>
            /// <param name="type">コンテンツタイプ</param>
            /// <param name="options">配信/通知バージョン設定, 配信バージョンは指定必須です。</param>
            /// <param name="timeoutSeconds">タイムアウト</param>
            public void RegisterVersion( ID64 contentMetaId, ID64 ownerApplicationId, ContentMeta.Type type, VersionOptions options, double timeoutSeconds = 5 * 60 )
            {
                if ( null == contentMetaId )
                {
                    throw new System.ArgumentNullException();
                }
                StringBuilder b = new StringBuilder( 512 );
                b.Append( "--content-meta-id 0x" ).Append( contentMetaId );
                b.Append( " --application-id 0x" ).Append( ( null == ownerApplicationId ) ? contentMetaId : ownerApplicationId );
                b.Append( " --type " ).Append( type.ToString() );
                options = ( null != options ) ? options : new VersionOptions();
                string arguments = AppendConfiguration( options.AppendCommandLineArguments( b ) ).ToString();
                ExecuteWithRetry( SubCommands.RegisterVersion, arguments, $"0x{contentMetaId}", timeoutSeconds );
            }

            /// <summary>
            /// アップロードコンテンツの最新バージョンを superfly に設定する。
            /// </summary>
            /// <param name="contentMetaId">設定対象コンテンツの ContentMetaId</param>
            /// <param name="ownerApplicationId">設定対象コンテンツのオーナーアプリケーションメタID</param>
            /// <param name="type">コンテンツタイプ</param>
            /// <param name="version">配信配信バージョン設定値, 通知バージョンは配信バージョンと同じになります。</param>
            /// <param name="timeoutSeconds">タイムアウト</param>
            public void RegisterVersion( ID64 contentMetaId, ID64 ownerApplicationId, ContentMeta.Type type, int version, double timeoutSeconds = 5 * 60 )
            {
                RegisterVersion( contentMetaId, ownerApplicationId, type, new VersionOptions( version, VersionOptions.Constants.Unspecified ), timeoutSeconds );
            }

            /// <summary>
            /// アップロードコンテンツの最新バージョンを superfly から削除する。
            /// </summary>
            /// <param name="nspFilePath">アップロードNSPファイルパス。</param>
            /// <param name="timeoutSeconds">タイムアウト</param>
            public void DeleteVersion( string nspFilePath, double timeoutSeconds = 5 * 60 )
            {
                if ( string.IsNullOrEmpty( nspFilePath ) )
                {
                    throw new System.ArgumentNullException();
                }
                StringBuilder b = new StringBuilder( 512 );
                b.Append( "-s \"" ).Append( nspFilePath ).Append( '\"' );
                string arguments = AppendConfiguration( b ).ToString();
                ExecuteWithRetry( SubCommands.DeleteVersion, arguments, nspFilePath, timeoutSeconds );
            }

            /// <summary>
            /// アップロードコンテンツの最新バージョンを superfly から削除する。
            /// </summary>
            /// <param name="contentMetaId">設定対象コンテンツの ContentMetaId</param>
            /// <param name="timeoutSeconds">タイムアウト</param>
            public void DeleteVersion( ID64 contentMetaId, double timeoutSeconds = 5 * 60 )
            {
                if ( null == contentMetaId )
                {
                    throw new System.ArgumentNullException();
                }
                StringBuilder b = new StringBuilder( 512 );
                b.Append( "--content-meta-id 0x" ).Append( contentMetaId );
                string arguments = AppendConfiguration( b ).ToString();
                ExecuteWithRetry( SubCommands.DeleteVersion, arguments, $"0x{contentMetaId}", timeoutSeconds );
            }
        }

        /// <summary>
        /// ショップサーバーのDTLをクリアします。
        /// </summary>
        /// <remarks>
        /// 以下の手法によるクリアです。
        /// 1. 対象デバイスの機器認証状態を全て解除する。 "shop unlink-device-all"
        /// 2. スレッドストールによるクリア待ち。"application request-download-task-list-data"
        /// </remarks>
        /// <param name="executor">DevVMenuCommand実行コンテキスト</param>
        /// <param name="timeoutSeconds">サーバ管理タイムアウト時間, デフォルト 5分</param>
        public static bool ClearDownloadTaskListOnShopServer( SigloHelper.CommodityExecutor.Context executor, double timeoutSeconds = 5 * 60 )
        {
            ThrowFrameworks.SkipThrow( () => { executor.RunDevMenuCommandSystem( "shop unlink-device-all" ); } );
            System.Func<string> PollingServer = () =>
            {
                if ( false == executor.RunDevMenuCommandSystem( "application request-download-task-list-data" ) )
                {
                    throw new UnexpectFailureException( "[request-download-task-list-data] command failed." );
                }
                return executor.OutputStream.Standard.ToString();
            };

            try
            {
                // サーバ監視ループ
                bool response;
                var timeoutObserver = Timeout.Observer.FromSeconds( timeoutSeconds );
                var result = PollingServer();
                while ( false == ( response = Regex.Match( result, @"\s+""tasks"": \[\]," ).Success ) &&
                    false == timeoutObserver.IsTimeout() )
                {
                    System.Threading.Thread.Sleep( 10000 ); // 10秒待機
                    result = PollingServer();
                };
                return response;
            }
            catch( System.Exception e )
            {
                Log.WriteLine( e.ToString() );
            }
            return false;
        }

        /// <summary>
        /// バージョンリストから更新されるまで待機します。
        /// </summary>
        /// <param name="executor">実行コンテキスト</param>
        /// <param name="judge">更新判定処理</param>
        /// <param name="request">更新要求</param>
        /// <param name="timeout">タイムアウト秒</param>
        /// <returns>バージョンリストが更新されたら true を返す</returns>
        public static bool WaitUntilVersionListUpdated(
            SigloHelper.CommodityExecutor.Context executor,
            System.Func<string, bool> judge,
            bool request,
            int timeout )
        {
            // superfly -> tagaya の取り込みは分間隔なので 30 秒待機
            var watcher = new Retry.Watcher( 30, timeout );
            while ( watcher.TryContinue() )
            {
                watcher.WaitInterval();
                if ( request && !executor.RunDevMenuCommandSystem( "application request-version-list" ) )
                {
                    Log.WriteLine( "[request-version-list] failed." );
                    break;
                }
                if ( !executor.RunDevMenuCommandSystem( "application version-list" ) )
                {
                    Log.WriteLine( "[version-list] failed." );
                    break;
                }
                if ( judge( executor.OutputStream.Standard.ToString() ) )
                {
                    Log.WriteLine( "[version-list] updated. (ElapsedTime: {0})", watcher.ElapsedTime );
                    return true;
                }
                Log.WriteLine( "[version-list] retry." );
            }
            Log.WriteLine( "[version-list] not updated. (ElapsedTime: {0})", watcher.ElapsedTime );
            return false;
        }

        /// <summary>
        /// バージョンリストから対象コンテンツが削除されるまで待機します。
        /// </summary>
        /// <param name="executor">実行コンテキスト</param>
        /// <param name="content">対象コンテンツ</param>
        /// <param name="request">更新要求</param>
        /// <param name="timeout">タイムアウト秒</param>
        /// <returns></returns>
        public static bool WaitUntilVersionListDeleted(
            SigloHelper.CommodityExecutor.Context executor,
            GeneratedContentResult content,
            bool request = false,
            int timeout = 5 * 60 )
        {
            Log.WriteLine( $"[version-list] wait untill 0x{content.Identifier} deleted." );

            System.Func<string, bool> judge = (string output) =>
            {
                return !Regex.Match( output, $@"\[target\] 0x{content.Identifier}\s+[0-9]+\s+[0-9]+" ).Success;
            };
            return WaitUntilVersionListUpdated( executor, judge, request, timeout );
        }

        /// <summary>
        /// バージョンリストから対象コンテンツが登録されるまで待機します。
        /// </summary>
        /// <param name="executor">実行コンテキスト</param>
        /// <param name="content">対象コンテンツ</param>
        /// <param name="request">更新要求</param>
        /// <param name="timeout">タイムアウト秒</param>
        /// <returns></returns>
        public static bool WaitUntilVersionListRegistered(
            SigloHelper.CommodityExecutor.Context executor,
            GeneratedContentResult content,
            bool request = false,
            int timeout = 5 * 60 )
        {
            Log.WriteLine( $"[version-list] wait untill 0x{content.Identifier} version {content.Version} registered." );

            System.Func<string, bool> judge = ( string output ) =>
            {
                var expect = $"0x{content.Identifier} +{content.Version} +{content.Version}";
                return Regex.Match( output, expect ).Success;
            };
            return WaitUntilVersionListUpdated( executor, judge, request, timeout );
        }
    }
}
