﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using CsTestAssistants;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace testNs_GameCardRegistration
{
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    /// <summary>
    /// テストクラス
    /// </summary>
    [TestClass]
    public class UnitMain
    {
        // `TestContext` プロパティを定義しておけば、UnitTestフレームワークで勝手にインスタンスが提供される.
        public TestContext TestContext { get; set; }

        private const int ReinitAfterInterval = 15000;
        private const string LogPrefix = "[GameCardRegistration]";
        private const string TestMethodLogBegin = "=== Begin the test of [ {0} ] ===";
        private const string TestMethodLogFinish = "=== Finish the test of [ {0} ] ===";
        private const string TestMethodLogFinally = "=== Finally the test of [ {0} ] ===";
        private static System.Net.WebProxy ProxyConfiguration;
        private static EncodeContext ActiveEncode = new EncodeContext();
        private static SigloHelper.Configuration.Context ActiveConfiguration = null;
        private static readonly BasicAccount CliTokenAccount = new BasicAccount("sasaki_yu", "Y2ZkMmEyOTAtMjk3");

        private static readonly string DefaultNintendoAccountId = "psg_gamecard_test";
        private static readonly string DefaultNintendoAccountPassword = "Password01";
        private static BasicAccount NintendoAccount = null;
        private static readonly string NetworkSettingJsonFilePath = System.IO.Path.Combine(SigloHelper.Path.GetTestsRoot(), "Ns", "Sources", "Tests", "GameCardRegistration", "NetworkSettingsForCI.txt");

        /// <summary>
        /// 実行環境に応じたプロキシ設定の取得.
        /// </summary>
        /// <param name="context">テストコンテキスト</param>
        /// <returns>プロキシ設定が無い場合は null, その他は対象の proxy instance.</returns>
        private static System.Net.WebProxy PrepareProxyConfiguration(TestContext context)
        {
            try
            {
                if (true == context.Properties.Contains("EnableRunTestOnIDE_FromPUX") &&
                    true == bool.Parse((string)context.Properties["EnableRunTestOnIDE_FromPUX"]))
                {
                    return D4cHelper.ProxyHelper.Create(null);
                }
            }
            catch (System.Exception e)
            {
                Log.WriteLine($"Critical failure of prepare proxy configuration => [ {e.Message} ]");
            }
            return D4cHelper.ProxyHelper.Create();
        }

        public class TestMethodLog : CsTestAssistants.ScopeLog
        {
            public TestMethodLog([System.Runtime.CompilerServices.CallerMemberName] string testName = "")
                : base(TestMethodLogBegin, TestMethodLogFinish, TestMethodLogFinally, testName) { }
        }

        private static void SetupNintendoAccount(TestContext context)
        {
            var id = context.Properties.Contains("NintendoAccountId") ? (string)context.Properties["NintendoAccountId"] : DefaultNintendoAccountId;
            var password = context.Properties.Contains("NintendoAccountPassword") ? (string)context.Properties["NintendoAccountPassword"] : DefaultNintendoAccountPassword;
            NintendoAccount = new BasicAccount(id, password);
        }

        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            ActiveEncode.Apply(EncodeContext.UTF8);
            Log.Verbose = true;
            Log.Prefix = LogPrefix;

            ProxyConfiguration = PrepareProxyConfiguration(context);

            string targetName = (string)context.Properties["TargetName"];
            string platform = (context.Properties.Contains("Platform")) ? (string)context.Properties["Platform"] : SigloHelper.Configuration.DefaultPlatform;
            string buildType = (context.Properties.Contains("BuildType")) ? (string)context.Properties["BuildType"] : SigloHelper.Configuration.DefaultBuildType;
            ActiveConfiguration = new SigloHelper.Configuration.Context(targetName, platform, buildType);

            SetupNintendoAccount(context);

            Log.WriteLine("=== Test Context ===");
            Log.WriteLine("TestDir                 : {0}", context.TestDir);
            Log.WriteLine("TestLogsDir             : {0}", context.TestLogsDir);
            Log.WriteLine("TestRunDirectory        : {0}", context.TestRunDirectory);
            Log.WriteLine("TestResultsDirectory    : {0}", context.TestResultsDirectory);
            Log.WriteLine("TestRunResultsDirectory : {0}", context.TestRunResultsDirectory);

            Log.WriteLine("=== Configuration ===");
            Log.WriteLine("TargetName      : {0}", ActiveConfiguration.TargetName);
            Log.WriteLine("TargetPlatform  : {0}", ActiveConfiguration.TargetPlatform);
            Log.WriteLine("TargetBuildType : {0}", ActiveConfiguration.TargetBuildType);
            Log.WriteLine("HostWebProxy    : {0}", (null != ProxyConfiguration) ? ProxyConfiguration.Address.ToString() : "None");
        }

        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ActiveEncode.Restore();
        }

        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">参照対象TestProperty属性を保持するメソッド名</param>
        /// <returns></returns>
        private string GenerateIntermediateDirectoryAsMethod(SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null)
        {
            // ベース作業用ディレクトリ
            string outPath = context.GetOutputTestDirectory("testNs_GameCardRegistration");
            if (false == string.IsNullOrEmpty(methodName))
            {
                string ticket = this.GetTestPropertyValue(methodName, "JIRA");
                if (!string.IsNullOrEmpty(ticket))
                {
                    outPath = System.IO.Path.Combine(outPath, ticket);
                }
            }
            Log.WriteLine("intermediate directory : {0}", outPath);
            return outPath;
        }

        /// <summary>
        /// SDEV再初期化のためにネットワーク設定シーケンスをメソッド化
        /// </summary>
        /// <param name="executor"></param>
        private void SetupSdevEnvironments(SigloHelper.CommodityExecutor.Context executor)
        {
            // SettingsManager import `network`
            Assert.IsTrue(executor.RunSettingsManager("Import \"" + NetworkSettingJsonFilePath + "\""));
            // reset sdev
            Assert.IsTrue(executor.ResetTarget());
            // サービスディスカバリ: TD1
            Assert.IsTrue(executor.RunDevMenuCommandSystem("servicediscovery import-all td1pass"));
            // reset sdev
            Assert.IsTrue(executor.ResetTarget());
        }

        class TestParameter
        {
            interface IConverter<T>
            {
                T Convert(string s);
            }

            private T GetOrDefault<T>(System.Collections.IDictionary dict, string key, T defaultValue, IConverter<T> converter)
            {
                if (dict.Contains(key))
                {
                    return converter.Convert((string)dict[key]);
                }
                return defaultValue;
            }

            class ID64Converter : IConverter<ID64>
            {
                public ID64 Convert(string s)
                {
                    return new ID64(System.Convert.ToUInt64(s, 16));
                }
            }

            private ID64 GetOrDefault(System.Collections.IDictionary dict, string key, ID64 defaultValue)
            {
                return GetOrDefault<ID64>(dict, key, defaultValue, new ID64Converter());
            }

            public ID64 ApplicationIdForSuccess { get; private set; } = new ID64(0x0100a620023e8000);
            public ID64 ApplicationIdForNotReleased { get; private set; } = new ID64(0x010031b0023ea000);

            public ID64 ApplicationIdForExpired { get; private set; } = new ID64(0x01004840028fa000);

            public ID64 ApplicationIdForInvalid { get; private set; } = new ID64(0x010052e002b94000);

            public ID64 ApplicationIdForInvalidCountry { get; private set; } = new ID64(0x01000000000020EC);

            public ID64 ApplicationIdForMultiApplicationCard { get; private set; } = new ID64(0x0100a620023e8100);

            public int MultiApplicationCount { get; private set; } = 4;
            public TestParameter(TestContext context)
            {
                var prop = context.Properties;
                ApplicationIdForSuccess = GetOrDefault(prop, "ApplicationIdForSuccess", ApplicationIdForSuccess);
                ApplicationIdForNotReleased = GetOrDefault(prop, "ApplicationIdForNotReleased", ApplicationIdForNotReleased);
                ApplicationIdForExpired = GetOrDefault(prop, "ApplicationIdForExpired", ApplicationIdForExpired);
                ApplicationIdForInvalid = GetOrDefault(prop, "ApplicationIdForInvalid", ApplicationIdForInvalid);
                ApplicationIdForMultiApplicationCard = GetOrDefault(prop, "ApplicationIdForMultiApplicationCard", ApplicationIdForMultiApplicationCard);
            }

        }

        /// <summary>
        /// SdCard クリーンアップ.
        /// </summary>
        /// <param name="executor"></param>
        private void CleanupSdCard(SigloHelper.CommodityExecutor.Context executor)
        {
            // format
            Assert.IsTrue(executor.RunDevMenuCommand("sdcard format"));
            // cleanup
            Assert.IsTrue(executor.RunDevMenuCommand("sdcard cleanup"));
            // reset sdev
            Assert.IsTrue(executor.ResetTarget());
            // status check
            Assert.IsTrue(executor.RunDevMenuCommand("sdcard status"));
        }

        private bool DoUpload()
        {
            var doUpload = TestContext.Properties["DoUpload"];
            return doUpload != null && bool.Parse((string)doUpload);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: SetupSequence
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [Priority(1)]
        public void SetupSequence()
        {
            Log.WriteLine(TestMethodLogBegin, "SetupSequence");

            var context = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // CLIトークン( td1, DEV6 )事前参照.
            // 並列要求時に有効期限切れの場合、取得競合で失敗するため.
            var intermediate = GenerateIntermediateDirectoryAsMethod(context);
            D4cHelper.Configuration.CliToken.QueryLatestValidateToken(intermediate, CliTokenAccount, ProxyConfiguration);

            // td1( dev6 ), ネットワーク設定インポート.
            SetupSdevEnvironments(context);

            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(
                "gamecard erase"));

            CleanupSdCard(context);

            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(
                "application uninstall --all"
                ));

            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(
                "application clear-task-status-list"
                ));

            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(
                "shop unlink-device-all AND " +
                "shop unregister-device-account AND " +
                "shop register-device-account AND " +
                "account clear_all AND " +
                "account add"
                ));
            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(string.Format("account link --index 0 --id {0} --password {1}", account.ID, account.PASSWORD)));

            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(
                "shop delete-all-rights AND " +
                "shop shop-account-status AND " +
                "shop link-device"
                ));

            Log.WriteLine(TestMethodLogFinish, "SetupSequence");
        }

        private void WriteMultiApplicationCard(string workPath, ID64 bundleId, SigloHelper.CommodityExecutor.Context executor, params ID64[] idList)
        {
            var requests = new List<TestApplication.GenerateParameter<int>>();
            foreach (var id in idList)
            {
                var appParam = new TestApplication.ApplicationParameter(id, 0);
                requests.Add(appParam);
            }
            var nsp = TestApplication.MakeContents(workPath, requests);
            var nspPathes = nsp.ConvertAll(n => n.NspPath).ToArray();

            using (var temp = new CsTestAssistants.TemporaryFileHolder("GameCardTest"))
            {
                var listFileInfo = temp.CreateTemporaryFilePath("nspList", ".txt");
                temp.AddTemporaryFile(listFileInfo);

                var outputFileInfo = temp.CreateTemporaryFilePath("output", ".nsp");
                temp.AddTemporaryFile(outputFileInfo);

                System.IO.File.WriteAllLines(listFileInfo.FullName, nspPathes);
                var authoringExecutor = new AuthoringExecutor();
                ExpectEvaluator.IsEqual(0, CommandLineExecutor.ExecuteOnProcess(authoringExecutor.ExecutorPath,
                    "bundleup --id " + bundleId.ToString() + " --nsp-list-file \"" + listFileInfo.FullName + "\" -o \"" + outputFileInfo.FullName + "\""));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard write --multi-application-card " + outputFileInfo.FullName + " --nsp-list-file " + listFileInfo.FullName
                    ));
            }
        }

        private void WriteGameCard(string workPath, SigloHelper.CommodityExecutor.Context executor, ID64 id)
        {
            var requests = new List<TestApplication.GenerateParameter<int>>();
            var appParam = new TestApplication.ApplicationParameter(id, 0);
            requests.Add(appParam);

            var nsp = TestApplication.MakeContents(workPath, requests);
            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                "gamecard write " + nsp[0].NspPath
                ));
        }

        private void Unregister(SigloHelper.CommodityExecutor.Context executor, int retryCount)
        {
            bool result = false;
            for (int i = 0; !result && i < retryCount; ++i)
            {
                result = executor.RunDevMenuCommandSystem(
                    "gamecard unregister");
            }
        }

        [TestMethod]
        [Priority(3)]
        [TestProperty("JIRA", "SIGLO-49607")]
        public void TestForGameCardRegistrationSuccess()
        {
            Log.WriteLine(TestMethodLogBegin, "GameCardRegistrationSuccess");

            // Parameter
            var p = new TestParameter(TestContext);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestForGameCardRegistrationSuccess");

            try
            {
                WriteGameCard(intermediate, executor, p.ApplicationIdForSuccess);

                Unregister(executor, 20);

                // 登録前
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard check-registration", "\"Registration: True\""));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard registration-gold-point", "\"Registration gold point: 5\""));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard register", "[SUCCESS]"));

                // 登録済み
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard check-registration", "\"Registration: False\""));

                CheckFailure("gamecard registration-gold-point", "result = 0x00032210");
                CheckFailure("gamecard register", "result = 0x00032210");
            }
            finally
            {
                // 登録成功してる場合でも失敗の場合でも必ず登録解除を呼んでおく
                Unregister(executor, 20);
            }

            Log.WriteLine(TestMethodLogFinish, "GameCardRegistrationSuccess");
        }

        private void CheckFailure(string command, string failurePattern)
        {
            Log.WriteLine("=== Checking failure pattern ===");
            Log.WriteLine("=== Command: " + command);
            Log.WriteLine("=== Pattern: " + failurePattern);
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            ExpectEvaluator.IsFalse(executor.RunDevMenuCommandSystem(command));
            ExpectEvaluator.IsGreater(executor.OutputStream.Standard.Buffer.ToString().IndexOf(failurePattern), -1);
        }

        [TestMethod]
        [Priority(4)]
        [TestProperty("JIRA", "SIGLO-49607")]
        public void TestForGameCardRegistrationNotReleased()
        {
            Log.WriteLine(TestMethodLogBegin, "GameCardRegistrationNotReleased");
            // Parameter
            var p = new TestParameter(TestContext);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestForGameCardRegistrationNotReleased");

            WriteGameCard(intermediate, executor, p.ApplicationIdForNotReleased);

            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                "gamecard check-registration", "\"Registration: True\""));

            CheckFailure("gamecard registration-gold-point", "result = 0x00032410");
            CheckFailure("gamecard register", "result = 0x00032410");

            Log.WriteLine(TestMethodLogFinish, "GameCardRegistrationNotReleased");
        }

        [TestMethod]
        [Priority(5)]
        [TestProperty("JIRA", "SIGLO-49607")]
        public void TestForGameCardRegistrationExpired()
        {
            Log.WriteLine(TestMethodLogBegin, "GameCardRegistrationNotExpired");
            // Parameter
            var p = new TestParameter(TestContext);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestForGameCardRegistrationExpired");

            WriteGameCard(intermediate, executor, p.ApplicationIdForExpired);

            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                "gamecard check-registration", "\"Registration: True\""));

            CheckFailure("gamecard registration-gold-point", "result = 0x00032610");
            CheckFailure("gamecard register", "result = 0x00032610");

            Log.WriteLine(TestMethodLogFinish, "GameCardRegistrationNotExpired");
        }

        [TestMethod]
        [Priority(6)]
        [TestProperty("JIRA", "SIGLO-49607")]
        public void TestForGameCardRegistrationInvalid()
        {
            Log.WriteLine(TestMethodLogBegin, "GameCardRegistrationInvalid");

            // Parameter
            var p = new TestParameter(TestContext);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestForGameCardRegistrationInvalid");

            WriteGameCard(intermediate, executor, p.ApplicationIdForInvalid);

            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                "gamecard check-registration", "\"Registration: True\""));

            CheckFailure("gamecard registration-gold-point", "result = 0x00032c10");
            CheckFailure("gamecard register", "result = 0x00032c10");

            Log.WriteLine(TestMethodLogFinish, "GameCardRegistrationInvalid");
        }

        [TestMethod]
        [Priority(7)]
        [TestProperty("JIRA", "SIGLO-51158")]
        public void TestForGameCardRegistrationInvalidCountry()
        {
            Log.WriteLine(TestMethodLogBegin, "GameCardRegistrationInvalidCountry");

            // Parameter
            var p = new TestParameter(TestContext);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestForGameCardRegistrationInvalidCountry");

            WriteGameCard(intermediate, executor, p.ApplicationIdForInvalidCountry);

            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                "gamecard check-registration", "\"Registration: True\""));

            CheckFailure("gamecard registration-gold-point", "result = 0x00033e10");
            CheckFailure("gamecard register", "result = 0x00033e10");

            Log.WriteLine(TestMethodLogFinish, "GameCardRegistrationInvalidCountry");
        }

        [TestMethod]
        [Priority(8)]
        [TestProperty("JIRA", "SIGLO-78669")]
        public void TestForMultiApplicationGameSuccess()
        {
            Log.WriteLine(TestMethodLogBegin, "MultiApplicationGameSuccess");

            // Parameter
            var p = new TestParameter(TestContext);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            // 作業用ディレクトリ
            string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestForGameCardRegistrationSuccess");

            // 複数入りのカードを作る
            var appIdList = new List<ID64>();
            for(var i = 0; i < p.MultiApplicationCount; ++i)
            {
                appIdList.Add(new ID64(p.ApplicationIdForSuccess.Value + (ulong)i));
            }

            try
            {
                WriteMultiApplicationCard(intermediate, p.ApplicationIdForMultiApplicationCard, executor, appIdList.ToArray());

                Unregister(executor, 20);

                // 登録前
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard check-registration", "\"Registration: True\""));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard registration-gold-point", "\"Registration gold point: 10\""));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard register", "[SUCCESS]"));

                // 登録済み
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard check-registration", "\"Registration: False\""));

                CheckFailure("gamecard registration-gold-point", "result = 0x00032210");
                CheckFailure("gamecard register", "result = 0x00032210");
            }
            finally
            {
                // 登録成功してる場合でも失敗の場合でも必ず登録解除を呼んでおく
                Unregister(executor, 20);
            }

            Log.WriteLine(TestMethodLogFinish, "MultiApplicationGameSuccess");
        }

    }
}
