﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <random>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Log.h>
#include <nn/ncm/ncm_ContentMetaType.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_DownloadTaskSystemApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/util/util_ScopeExit.h>

#define RAPIDJSON_NO_INT64DEFINE
#define RAPIDJSON_NAMESPACE             nne::rapidjson
#define RAPIDJSON_NAMESPACE_BEGIN       namespace nne { namespace rapidjson {
#define RAPIDJSON_NAMESPACE_END         }}
#define RAPIDJSON_ASSERT(x)             NN_SDK_ASSERT(x)
#define RAPIDJSON_HAS_CXX11_RVALUE_REFS 1 // NOLINT(readability/define)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 // NOLINT(readability/define)
#include <rapidjson/document.h>
#include <rapidjson/prettywriter.h>
#include <rapidjson/stringbuffer.h>
#include <rapidjson/writer.h>

using namespace nn;
using namespace nne;

namespace {
    rapidjson::StringBuffer GenerateRandomDownloadTaskList(std::mt19937_64* mt, int count)
    {
        rapidjson::Document document;
        document.SetObject();

        rapidjson::Value taskList(rapidjson::kArrayType);
        for (int i = 0; i < count; i++)
        {
            Bit64 randomApplicationId = (*mt)();

            rapidjson::Value task(rapidjson::kObjectType);

            util::Uuid uuid = util::GenerateUuid();
            char uuidStr[util::Uuid::StringSize];
            task.AddMember("id", rapidjson::Value(uuid.ToString(uuidStr, sizeof(uuidStr)), document.GetAllocator()), document.GetAllocator());
            char appIdStr[20];
            util::TSNPrintf(appIdStr, sizeof(appIdStr), "0x%016llx", randomApplicationId);
            task.AddMember("owner_application", rapidjson::Value(appIdStr, document.GetAllocator()), document.GetAllocator());

            rapidjson::Value title(rapidjson::kObjectType);
            char cmetaIdStr[20];
            util::TSNPrintf(cmetaIdStr, sizeof(cmetaIdStr), "0x%016llx", randomApplicationId);
            title.AddMember("id", rapidjson::Value(cmetaIdStr, document.GetAllocator()), document.GetAllocator());
            title.AddMember("version", rapidjson::Value(0), document.GetAllocator());
            title.AddMember("type", rapidjson::StringRef("Application"), document.GetAllocator());

            rapidjson::Value titleList(rapidjson::kArrayType);
            titleList.PushBack(title, document.GetAllocator());

            task.AddMember("titles", titleList, document.GetAllocator());

            taskList.PushBack(task, document.GetAllocator());
        }

        document.AddMember("tasks", taskList, document.GetAllocator());

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        return buffer;
    }
}

TEST(DownloadTaskLocal, CreateOutOfMaxDownloadTask)
{
    ns::Initialize();
    NN_UTIL_SCOPE_EXIT{ ns::Finalize(); };

    // 現存する環境をクリアする
    {
        const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
        nn::ns::ApplicationRecord recordList[128];
        auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
        for (int i = 0; i < recordCount; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletelyForDebug(recordList[i].id, flags));
        }
    }

    std::mt19937_64 mt;

    for (int i = 0; i < 100; i++)
    {
        auto buffer = GenerateRandomDownloadTaskList(&mt, 256);
        NN_LOG("%s\n", buffer.GetString());
        NNT_EXPECT_RESULT_SUCCESS(ns::PushDownloadTaskList(buffer.GetString(), buffer.GetSize()));

        {
            std::vector<ns::DownloadTaskStatus> list(256);
            auto count = ns::ListDownloadTaskStatus(list.data(), static_cast<int>(list.size()));
            list.resize(static_cast<size_t>(count));

            ASSERT_TRUE(std::all_of(list.begin(), list.end(), [](const ns::DownloadTaskStatus& status)
            {
                auto result = result::detail::ConstructResult(status.result);
                return result.IsSuccess() || result <= nim::ResultOutOfMaxTask();
            }));
        }
    }

    {
        std::vector<ns::ApplicationRecord> list(2048);
        auto count = ns::ListApplicationRecord(list.data(), static_cast<int>(list.size()), 0);
        list.resize(static_cast<size_t>(count));

        for (auto& record : list)
        {
            NNT_EXPECT_RESULT_SUCCESS(ns::DeleteApplicationCompletely(record.id));
        }
    }
}
