﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>
#include <map>

#include <nn/ns/ns_ApplicationManagerApi.h>

// アプリケーション管理データに該当する定義
struct ApplicationControlData
{
    struct AppTitle
    {
        std::string name;
        std::string publisher;
    };

    std::map<nn::settings::Language, AppTitle> titleList;
    std::string                          isbn;
    std::string                          displayVersion;
    nn::ns::StartupUserAccount           startupUserAccount;
    nn::ns::AddOnContentRegistrationType aocRegistrationType;
    nn::Bit32                            attributeFlag;
    std::vector<nn::settings::Language>  supportedLanguageList;
    nn::Bit32                            parentalControlFlag;
    nn::ns::Screenshot                   screenShot;
    nn::ns::VideoCapture                 videoCapture;
    nn::ns::DataLossConfirmation         dataLossConfirmation;
    nn::ns::PlayLogPolicy                playLogPolicy;
    nn::ns::PresenceGroupId              presenceGroupId;
    nn::Bit64                            addOnContentBaseId;
    nn::Bit64                            saveDataOwnerId;
    int64_t                              userAccountSaveDataSize;
    int64_t                              userAccountSaveDataJournalSize;
    int64_t                              deviceSaveDataSize;
    int64_t                              deviceSaveDataJournalSize;
    int64_t                              bcatDeliveryCacheStorageSize;
    int64_t                              userAccountSaveDataSizeMax;
    int64_t                              userAccountSaveDataJournalSizeMax;
    int64_t                              deviceSaveDataSizeMax;
    int64_t                              deviceSaveDataJournalSizeMax;
    int64_t                              temporaryStorageSize;
    int64_t                              cacheStorageSize;
    int64_t                              cacheStorageJournalSize;
    std::string                          applicationErrorCodeCategory;
    std::vector<nn::Bit64>               localCommunicationId;
    nn::ns::LogoType                     logoType;
    nn::ns::LogoHandling                 logoHandling;
    nn::ns::RuntimeAddOnContentInstall   runtimeAddOnContentInstall;
    nn::ns::CrashReport                  crashReport;
    nn::ns::Hdcp                         hdcp;
    std::map<nn::ns::RatingOrganization, int8_t> ratingList;
    nn::Bit64                            seedForPseudoDeviceId;
    std::string                          bcatPassphrase;
    nn::ns::PlayLogQueryCapability       playLogQueryCapability;
    std::vector<nn::Bit64>               playLogQueryableApplicationId;
    nn::ns::RepairFlag                   repairFlag;
    uint32_t                             programIndex;
    nn::Bit8                             requiredNetworkServiceLicenseOnLaunchFlag;

    // 設定されているかどうかを判定するため文字列型で保持する
    std::string releaseVersion;
    std::string privateVersion;
};

struct AddOnContentMeta
{
    size_t index;
    uint32_t releaseVersion;
    nn::ncm::ApplicationId id;
    uint32_t requiredAppVersion;
    std::string tag;
    std::string dataPath;

    // インストール先 (チェック処理用の情報として保持しておく)
    std::string installStorage;
};

struct AddOnContentProperty
{
    std::vector<AddOnContentMeta> aocList;
};

// 本テスト用のテストデータレコード定義
struct TestDataRecord
{
    std::string type;
    std::string installFilePath;

    // インストール先
    std::string installStorage;
    // 移動先のストレージ設定
    std::string moveDestinationStorage;
    // 削除処理実行フラグ
    bool isDeleteFlag;

    nn::ncm::ApplicationId id;
    nn::ns::ApplicationView view;
    //std::vector<std::pair<bool, nn::ncm::StorageId> > movableList;
    nn::ns::ApplicationOccupiedSize size;
    //bool isLaunchRights;

    ApplicationControlData controlData;

    AddOnContentProperty aocProperty;

    //std::string iconFilePath;
};

class TestDataContainer
{
public:
    TestDataContainer() NN_NOEXCEPT;
    virtual ~TestDataContainer() NN_NOEXCEPT;

    // テストデータの元になる一覧データの読み込み
    // m_TestDataList に展開する
    nn::Result ReadFromCSV(const std::string& inFilePath) NN_NOEXCEPT;

    // テストデータリストのクリア処理
    void Clear() NN_NOEXCEPT;

    std::vector<TestDataRecord>& GetDataList() NN_NOEXCEPT
    {
        return m_TestDataList;
    }

private:
    typedef std::vector<std::string> CsvRecord;
    typedef std::vector<CsvRecord> CsvRecordTable;
    typedef std::vector<CsvRecordTable> CsvDataTable;

    CsvDataTable ParseTable(const std::string& inData, const std::string& delim) NN_NOEXCEPT;

    void ToTestDataContainer(CsvDataTable& inCSVData) NN_NOEXCEPT;
    void SetCsvRecordData(TestDataRecord& outRecord, CsvRecordTable& inCSVRecordData) NN_NOEXCEPT;
    void SetCsvRecordData_Title(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT;
    void SetCsvRecordData_Rating(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT;
    void SetCsvRecordData_SupportedLanguage(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT;
    void SetCsvRecordData_LocalCommunicationId(ApplicationControlData& outData, CsvRecordTable& inRecordTable, nn::Bit64 inAppId) NN_NOEXCEPT;
    void SetCsvRecordData_PlayLogQueryCapability(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT;
    void SetCsvRecordData_Size(TestDataRecord& outRecord, CsvRecord& inRecord) NN_NOEXCEPT;
    void SetCsvRecordData_Aoc(TestDataRecord& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT;
    void SetCsvRecordData_RequiredNetworkServiceLicenseOnLaunch(ApplicationControlData& outData, CsvRecordTable& inRecordTable) NN_NOEXCEPT;

    nn::Bit32 GetFlagValue(CsvRecordTable& inRecordTable, const std::string& inTitleName) NN_NOEXCEPT;
    nn::Bit64 GetDefaultAppId(CsvRecord& inFindList, const std::string& inFindValue, nn::Bit64 inAppId) NN_NOEXCEPT;

    int FindIndexVector(CsvRecord& inFindList, const std::string& inFindValue) NN_NOEXCEPT;

    std::string SafeAt(CsvRecord& vec, const std::string& titleStr) NN_NOEXCEPT;

private:
    std::vector<TestDataRecord> m_TestDataList;

    std::map<std::string, int> m_TitleIndexMap;
    void InitializeTitleIndexMap() NN_NOEXCEPT;
};

