﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/settings/system/settings_Language.h>
#include <nn/ns/ns_Result.h>

#include "ApplicationManagerTestTool_Common.h"

#include "ApplicationManagerTestTool_AppOperation.h"
#include "ApplicationManagerTestTool_Stopwatch.h"

#include "ApplicationManagerTestTool_CheckProcess.h"
#include "ApplicationManagerTestTool_NsTypeConverter.h"
#include "ApplicationManagerTestTool_TestDataContainer.h"

#include "ApplicationManagerTestTool_Option.h"
#include "ApplicationManagerTestTool_ApplicationViewChecker.h"

static std::vector<nn::settings::Language> s_AllLangList;

struct ContentMetaStatusData
{
    TestDataRecord* pAppRecord;
    TestDataRecord* pPatchRecord;
    std::map<size_t, AddOnContentMeta*> aocRecordMap; // index番号検索用
    std::vector<AddOnContentMeta*> aocList; // インストール順保持用
    ApplicationViewChecker viewChecker;

    NN_IMPLICIT ContentMetaStatusData(TestDataRecord* inAppRecordPtr) NN_NOEXCEPT
        : pAppRecord(inAppRecordPtr), pPatchRecord(nullptr) {}

    int GetContentCount()
    {
        int count = static_cast<int>(aocRecordMap.size());
        if (pPatchRecord != nullptr)
        {
            ++count;
        }
        if (pAppRecord != nullptr && pAppRecord->installFilePath != "")
        {
            ++count;
        }
        return count;
    }
};

void InitializeAllLanguageList() NN_NOEXCEPT
{
    s_AllLangList.clear();
    s_AllLangList.push_back(nn::settings::Language::Language_Japanese);
    s_AllLangList.push_back(nn::settings::Language::Language_AmericanEnglish);
    s_AllLangList.push_back(nn::settings::Language::Language_German);
    s_AllLangList.push_back(nn::settings::Language::Language_French);
    s_AllLangList.push_back(nn::settings::Language::Language_Italian);
    s_AllLangList.push_back(nn::settings::Language::Language_Spanish);
    s_AllLangList.push_back(nn::settings::Language::Language_SimplifiedChinese);
    s_AllLangList.push_back(nn::settings::Language::Language_Korean);
    s_AllLangList.push_back(nn::settings::Language::Language_Dutch);
    s_AllLangList.push_back(nn::settings::Language::Language_Portuguese);
    s_AllLangList.push_back(nn::settings::Language::Language_Russian);
    s_AllLangList.push_back(nn::settings::Language::Language_TraditionalChinese);
    s_AllLangList.push_back(nn::settings::Language::Language_BritishEnglish);
    s_AllLangList.push_back(nn::settings::Language::Language_CanadianFrench);
    s_AllLangList.push_back(nn::settings::Language::Language_LatinAmericanSpanish);
}

void InitializeCheckProcess() NN_NOEXCEPT
{
    InitializeAllLanguageList();
}

#if 0
// デフォルト値設定用関数
void SetDefaultTestDataRecord(TestDataRecord& inRecord)
{
    inRecord.view.version = 0;

    auto& ctlData = inRecord.controlData;
    ctlData.isbn[0] = '\0';
    ctlData.startupUserAccount = NsTypeConv::ToStartupUserAccount("");
    ctlData.attributeFlag = static_cast<nn::Bit32>(NsTypeConv::ToAttributeFlag(""));
    ctlData.supportedLanguageFlag = 0;
    ctlData.parentalControlFlag = static_cast<nn::Bit32>(NsTypeConv::ToParentalControlFlag(""));
    ctlData.screenShot = NsTypeConv::ToScreenshot("");
    ctlData.videoCapture = NsTypeConv::ToVideoCapture("");
    ctlData.autoSave = NsTypeConv::ToAutoSave("");
    ctlData.playLogPolicy = NsTypeConv::ToPlayLogPolicy("");
    ctlData.presenceGroupId.value = 0;
}
#endif

void PrintApplicationView(nn::ns::ApplicationView& inView)
{
    NN_LOG(" [View]\n");
    NN_LOG("    ApplicationID                : 0x%016llx\n", inView.id);
    NN_LOG("    Version                      : %d (%d)\n", (inView.version >> 16), inView.version);
    NN_LOG("    HasRecord()                  : %s\n", StringUtil::ToString(inView.HasRecord()));
    NN_LOG("    HasMainRecord()              : %s\n", StringUtil::ToString(inView.HasMainRecord()));
    NN_LOG("    HasPatchRecord()             : %s\n", StringUtil::ToString(inView.HasPatchRecord()));
    NN_LOG("    HasAddOnContentRecord()      : %s\n", StringUtil::ToString(inView.HasAddOnContentRecord()));
    NN_LOG("    HasMainInstallRecord()       : %s\n", StringUtil::ToString(inView.HasMainInstallRecord()));
    NN_LOG("    IsDownloading()              : %s\n", StringUtil::ToString(inView.IsDownloading()));
    NN_LOG("    IsGameCard()                 : %s\n", StringUtil::ToString(inView.IsGameCard()));
    NN_LOG("    HasGameCardEntity()          : %s\n", StringUtil::ToString(inView.HasGameCardEntity()));
    NN_LOG("    IsLaunchable()               : %s\n", StringUtil::ToString(inView.IsLaunchable()));
    NN_LOG("    HasAllEntity()               : %s\n", StringUtil::ToString(inView.HasAllEntity()));
    NN_LOG("    HasMainEntity()              : %s\n", StringUtil::ToString(inView.HasMainEntity()));
    NN_LOG("    HasAllAddOnContentEntity()   : %s\n", StringUtil::ToString(inView.HasAllAddOnContentEntity()));
    NN_LOG("    HasPatchEntity()             : %s\n", StringUtil::ToString(inView.HasPatchEntity()));
    NN_LOG("    MaybeCorrupted()             : %s\n", StringUtil::ToString(inView.MaybeCorrupted()));
    NN_LOG("    IsWaitingCommit()            : %s\n", StringUtil::ToString(inView.IsWaitingCommit()));
    NN_LOG("    IsWaitingApplicationCommit() : %s\n", StringUtil::ToString(inView.IsWaitingApplicationCommit()));
    NN_LOG("    IsWaitingAocCommit()         : %s\n", StringUtil::ToString(inView.IsWaitingAocCommit()));
    NN_LOG("    IsWaitingPatchInstall()      : %s\n", StringUtil::ToString(inView.IsWaitingPatchInstall()));
    NN_LOG("    IsAutoDeleteDisabled()       : %s\n", StringUtil::ToString(inView.IsAutoDeleteDisabled()));
    NN_LOG("    IsApplyingDelta()            : %s\n", StringUtil::ToString(inView.IsApplyingDelta()));
}

// 主にデバック用途向けのアプリケーションプロパティ一覧の出力関数
void PrintApplicationTitleValue(nn::ns::ApplicationControlProperty& inCtrlProperty)
{
    NN_LOG("  - ApplicationTitle\n");
    bool isHit = false;
    for (auto& lang : s_AllLangList)
    {
        auto& title = inCtrlProperty.GetTitle(nn::settings::LanguageCode::Make(lang));
        if (std::strcmp(title.name, "") != 0 || std::strcmp(title.publisher, "") != 0)
        {
            NN_LOG("     [%s]\n", NsTypeConv::ToString(lang));
            NN_LOG("       name : %s\n", title.name);
            NN_LOG("       publisher : %s\n", title.publisher);
            isHit = true;
        }
    }
    if (isHit == false)
    {
        NN_LOG("      * None *\n");
    }

    if (isHit == true)
    {
        NN_LOG("  - DefaultTitle\n");
        auto& defTitle = inCtrlProperty.GetDefaultTitle();
        NN_LOG("      name : %s\n", defTitle.name);
        NN_LOG("      publisher : %s\n", defTitle.publisher);
    }
}

void PrintSupportedLanguageValue(nn::ns::ApplicationControlProperty& inCtrlProperty)
{
    NN_LOG("  - SupportedLanguage\n");
    for (auto& lang : s_AllLangList)
    {
        auto isSupported = inCtrlProperty.SupportsLanguage(nn::settings::LanguageCode::Make(lang));
        NN_LOG("      %s : %s\n", NsTypeConv::ToString(lang), StringUtil::ToString(isSupported));
    }
}

void PrintSizeValue(nn::ns::ApplicationControlProperty& inCp)
{
    NN_LOG("  - UserAccountSaveDataSize           : %lld (0x%016llx)\n", inCp.userAccountSaveDataSize, inCp.userAccountSaveDataSize);
    NN_LOG("  - UserAccountSaveDataJournalSize    : %lld (0x%016llx)\n", inCp.userAccountSaveDataJournalSize, inCp.userAccountSaveDataJournalSize);
    NN_LOG("  - DeviceSaveDataSize                : %lld (0x%016llx)\n", inCp.deviceSaveDataSize, inCp.deviceSaveDataSize);
    NN_LOG("  - DeviceSaveDataJournalSize         : %lld (0x%016llx)\n", inCp.deviceSaveDataJournalSize, inCp.deviceSaveDataJournalSize);
    NN_LOG("  - BcatDeliveryCacheStorageSize      : %lld (0x%016llx)\n", inCp.bcatDeliveryCacheStorageSize, inCp.bcatDeliveryCacheStorageSize);
    NN_LOG("  - UserAccountSaveDataSizeMax        : %lld (0x%016llx)\n", inCp.userAccountSaveDataSizeMax, inCp.userAccountSaveDataSizeMax);
    NN_LOG("  - UserAccountSaveDataJournalSizeMax : %lld (0x%016llx)\n", inCp.userAccountSaveDataJournalSizeMax, inCp.userAccountSaveDataJournalSizeMax);
    NN_LOG("  - DeviceSaveDataSizeMax             : %lld (0x%016llx)\n", inCp.deviceSaveDataSizeMax, inCp.deviceSaveDataSizeMax);
    NN_LOG("  - DeviceSaveDataJournalSizeMax      : %lld (0x%016llx)\n", inCp.deviceSaveDataJournalSizeMax, inCp.deviceSaveDataJournalSizeMax);
    NN_LOG("  - TemporaryStorageSize              : %lld (0x%016llx)\n", inCp.temporaryStorageSize, inCp.temporaryStorageSize);
    NN_LOG("  - CacheStorageSize                  : %lld (0x%016llx)\n", inCp.cacheStorageSize, inCp.cacheStorageSize);
    NN_LOG("  - CacheStorageJournalSize           : %lld (0x%016llx)\n", inCp.cacheStorageJournalSize, inCp.cacheStorageJournalSize);
}

void PrintRatingValue(nn::ns::ApplicationControlProperty& inCtrlProperty)
{
    std::vector<nn::ns::RatingOrganization> ratingList;
    ratingList.push_back(nn::ns::RatingOrganization::CERO);
    ratingList.push_back(nn::ns::RatingOrganization::GRACGCRB);
    ratingList.push_back(nn::ns::RatingOrganization::GSRMR);
    ratingList.push_back(nn::ns::RatingOrganization::ESRB);
    ratingList.push_back(nn::ns::RatingOrganization::ClassInd);
    ratingList.push_back(nn::ns::RatingOrganization::USK);
    ratingList.push_back(nn::ns::RatingOrganization::PEGI);
    ratingList.push_back(nn::ns::RatingOrganization::PEGIPortugal);
    ratingList.push_back(nn::ns::RatingOrganization::PEGIBBFC);
    ratingList.push_back(nn::ns::RatingOrganization::Russian);
    ratingList.push_back(nn::ns::RatingOrganization::ACB);
    ratingList.push_back(nn::ns::RatingOrganization::OFLC);

    NN_LOG("  - RatingAge\n");
    for (auto& rating : ratingList)
    {
        const auto age = inCtrlProperty.GetAge(rating);
        NN_LOG("      %s : %d\n", NsTypeConv::ToString(rating), age);
    }
}

void PrintPlayLogQueryCapability(nn::ns::ApplicationControlProperty& inCtrlProperty)
{
    NN_LOG("  - PlayLogQueryCapability         : %s\n", NsTypeConv::ToString(inCtrlProperty.playLogQueryCapability));
    for (int i = 0; i < nn::ns::PlayLogQueryableApplicationCountMax; ++i)
    {
        NN_LOG("  - playLogQueryableApplicationId[%d] : 0x%016llx\n",
            i, inCtrlProperty.playLogQueryableApplicationId[i].value);
    }
}

void PrintTestProperty(TestProperty& inProperty)
{
    PrintApplicationView(inProperty.view);

    NN_LOG("   LauchRight : %s\n", StringUtil::ToString(inProperty.isLaunchRights));

    const char* StorageName[] =
    {
        "None",
        "Host", // 将来無くなる
        "Card",
        "BuildInSystem",
        "BuildInUser",
        "SdCard"
    };
    NN_LOG(" [Size]\n");
    auto& refSize = inProperty.size;
    for (auto& refStorageSize : refSize.storage)
    {
        if (refStorageSize.storageId == nn::ncm::StorageId::None)
        {
            break;
        }
        NN_LOG("   - %13s : App=%lld, Patch=%lld Aoc=%lld\n", StorageName[static_cast<int>(refStorageSize.storageId)], refStorageSize.appSize, refStorageSize.patchSize, refStorageSize.aocSize);
    }

#if 0
    auto& refMList = inProperty.movableList;
    NN_LOG(" [Movable]\n");
    for (auto& movPair : refMList)
    {
        NN_LOG(" - Storage(%s) : %s\n", movPair.first, StringUtil::ToString(movPair.second));
    }
#endif

    auto& refCtl = inProperty.controlProperty;
    NN_LOG(" [ControlProperty]\n");
    PrintApplicationTitleValue(refCtl);

    NN_LOG("  - ISBN                  : %s\n", refCtl.isbn);
    NN_LOG("  - DisplayVersion        : %s\n", refCtl.displayVersion);
    NN_LOG("  - StartupUserAccount    : %s\n", NsTypeConv::ToString(refCtl.startupUserAccount));
    NN_LOG("  - AddOnContentRegistrationType : %s\n", NsTypeConv::ToString(refCtl.addOnContentRegistrationType));
    NN_LOG("  - Screenshot            : %s\n", NsTypeConv::ToString(refCtl.screenShot));
    NN_LOG("  - VideoCapture          : %s\n", NsTypeConv::ToString(refCtl.videoCapture));
    NN_LOG("  - DataLossConfirmation  : %s\n", NsTypeConv::ToString(refCtl.dataLossConfirmation));
    NN_LOG("  - PlayLogPolicy         : %s\n", NsTypeConv::ToString(refCtl.playLogPolicy));
    NN_LOG("  - AttributeFlag         : 0x%08x\n", refCtl.attributeFlag);
    NN_LOG("  - ParentalControlFlag   : 0x%08x\n", refCtl.parentalControlFlag);

    PrintSupportedLanguageValue(refCtl);

    NN_LOG("  - PresenceGroupId       : 0x%016llx\n", refCtl.presenceGroupId.value);

    PrintRatingValue(refCtl);

    NN_LOG("  - AddOnContentBaseId             : 0x%016llx\n", refCtl.addOnContentBaseId);
    NN_LOG("  - SaveDataOwnerId                : 0x%016llx\n", refCtl.saveDataOwnerId);

    PrintSizeValue(refCtl);

    NN_LOG("  - ApplicationErrorCodeCategory   : %s\n", refCtl.applicationErrorCodeCategory.value);
    for (int i = 0; i < 8; ++i)
    {
        NN_LOG("  - LocalCommunicationId[%d]        : 0x%016llx\n", i, refCtl.localCommunicationId[i]);
    }
    NN_LOG("  - LogoType                       : %s\n", NsTypeConv::ToString(refCtl.logoType));
    NN_LOG("  - LogoHandling                   : %s\n", NsTypeConv::ToString(refCtl.logoHandling));
    NN_LOG("  - RuntimeAddOnContentInstall     : %s\n", NsTypeConv::ToString(refCtl.runtimeAddOnContentInstall));
    NN_LOG("  - CrashReport                    : %s\n", NsTypeConv::ToString(refCtl.crashReport));
    NN_LOG("  - Hdcp                           : %s\n", NsTypeConv::ToString(refCtl.hdcp));
    NN_LOG("  - SeedForPseudoDeviceId          : 0x%016llx\n", refCtl.seedForPseudoDeviceId);
    NN_LOG("  - BcatPassphrase                 : %s\n", refCtl.bcatPassphrase);

    PrintPlayLogQueryCapability(refCtl);

    NN_LOG("  - RepairFlag                     : %s\n", NsTypeConv::ToString(static_cast<nn::ns::RepairFlag>(refCtl.repairFlag)));

    NN_LOG("  - ProgramIndex                   : %d\n", refCtl.programIndex);

    NN_LOG("  - RequiredNetworkServiceLicenseOnLaunchFlag : %s \n",
        NsTypeConv::RequiredNetworkServiceLicenseOnLaunchFlagToString(refCtl.requiredNetworkServiceLicenseOnLaunchFlag).c_str());
}

void PrintStorageInfo()
{
    int64_t total = 0;
    int64_t free = 0;
    nn::Result result;

    Stopwatch watch;

    auto printProcess = [&] {
        if (nn::ns::ResultStorageAccessFailed::Includes(result) == true)
        {
            NN_LOG("Access Failed : (result = 0x%08x)\n", result.GetInnerValueForDebug());
        }
        else
        {
            const int64_t div = 1024 * 1024;
            NN_LOG("%lld [MB] / %lld [MB]\n", free / div, total / div);
        }
    };

    {
        ScopedWatch sw(&watch);
        result = AppOperation::GetHostStorageSize(free, total);
    }
    NN_LOG(" [Trace] Host Storage (time=%lld [ms]) : ", watch.GetMilliSecond());
    printProcess();

// Generic 版では以下のサイズを取得時に未実装のAbortとなるため、実装分岐させておく
#ifndef NN_BUILD_CONFIG_OS_WIN
    {
        ScopedWatch sw(&watch);
        result = AppOperation::GetBuiltInStorageSize(free, total);
    }
    NN_LOG(" [Trace] BuiltIn Storage (time=%lld [ms]) : ", watch.GetMilliSecond());
    printProcess();

    // SDカードストレージのマウント状態を確認する (基本成功するはず)
    result = nn::ns::CheckSdCardMountStatus();
    NNT_EXPECT_RESULT_SUCCESS(result);
    {
        ScopedWatch sw(&watch);
        result = AppOperation::GetSdCardStorageSize(free, total);
    }
    NN_LOG(" [Trace] SdCard Storage (time=%lld [ms]) : ", watch.GetMilliSecond());
    printProcess();
#endif // NN_BUILD_CONFIG_OS_WIN
}

int InstallNspFile(std::vector<TestDataRecord>& inDataList, const std::string& inFilterType)
{
    Stopwatch watch;
    int installCount = 0;

    // データリスト内の指定タイプをインストールする
    for (auto& record : inDataList)
    {
        if (record.type != inFilterType || record.installFilePath == "")
        {
            // inFilterType に該当しないタイプは除外する
            // また、インストールパスが空文字の場合はインストール処理を"明示的に実施しない"とみなす
            continue;
        }

        watch.Start();
        NN_LOG(" [Trace] Install (%s): path = %s, storage = %s",
            inFilterType.c_str(), record.installFilePath.c_str(), record.installStorage.c_str());
        auto result = AppOperation::Install(record.installFilePath, record.installStorage);
        watch.Stop();
        NNT_EXPECT_RESULT_SUCCESS(result);
        NN_LOG(", time = %lld [ms]\n", watch.GetMilliSecond());
        ++installCount;
    }

    return installCount;
}

uint32_t GetExpectedVersionValue(TestDataRecord& inRecord)
{
    uint32_t retValue = inRecord.view.version;

    auto& cd = inRecord.controlData;
    if (cd.releaseVersion != "" || cd.privateVersion != "")
    {
        // ReleaseVersion または PrivateVersion が設定されている場合はこちらの設定値を優先
        int32_t rVer = StringUtil::ToBit32(cd.releaseVersion);
        if (rVer < 0)
        {
            rVer = 0;
        }
        int32_t pVer = StringUtil::ToBit32(cd.privateVersion);
        if (pVer < 0)
        {
            pVer = 0;
        }

        retValue = ((rVer << 16) | pVer);
    }

    return retValue;
}

void CheckTestRecordForSizeValue(TestDataRecord& inRecord, TestProperty& inProperty)
{
    auto& refCtl = inProperty.controlProperty;
    auto& refExpCtl = inRecord.controlData;

    EXPECT_EQ(refExpCtl.userAccountSaveDataSize, refCtl.userAccountSaveDataSize);
    EXPECT_EQ(refExpCtl.userAccountSaveDataJournalSize, refCtl.userAccountSaveDataJournalSize);
    EXPECT_EQ(refExpCtl.deviceSaveDataSize, refCtl.deviceSaveDataSize);
    EXPECT_EQ(refExpCtl.deviceSaveDataJournalSize, refCtl.deviceSaveDataJournalSize);
    EXPECT_EQ(refExpCtl.bcatDeliveryCacheStorageSize, refCtl.bcatDeliveryCacheStorageSize);

    // (SIGLO-62037) 一時データ保存領域に関するチェックの追加
    EXPECT_EQ(refExpCtl.userAccountSaveDataSizeMax, refCtl.userAccountSaveDataSizeMax);
    EXPECT_EQ(refExpCtl.userAccountSaveDataJournalSizeMax, refCtl.userAccountSaveDataJournalSizeMax);
    EXPECT_EQ(refExpCtl.deviceSaveDataSizeMax, refCtl.deviceSaveDataSizeMax);
    EXPECT_EQ(refExpCtl.deviceSaveDataJournalSizeMax, refCtl.deviceSaveDataJournalSizeMax);
    EXPECT_EQ(refExpCtl.temporaryStorageSize, refCtl.temporaryStorageSize);
    EXPECT_EQ(refExpCtl.cacheStorageSize, refCtl.cacheStorageSize);
    EXPECT_EQ(refExpCtl.cacheStorageJournalSize, refCtl.cacheStorageJournalSize);
}

void CheckTestRecordForPlayLogQueryCapability(TestDataRecord& inRecord, TestProperty& inProperty)
{
    auto& refCtl = inProperty.controlProperty;
    auto& refExpCtl = inRecord.controlData;

    EXPECT_EQ(refExpCtl.playLogQueryCapability, refCtl.playLogQueryCapability);
    int i = 0;
    for (auto& appId : refExpCtl.playLogQueryableApplicationId)
    {
        EXPECT_EQ(appId, refCtl.playLogQueryableApplicationId[i].value);
        ++i;
    }
}

void CheckTestRecord(TestDataRecord& inRecord, TestProperty& inProperty)
{
    // ApplicationView に関連するチェック
    auto& refView = inProperty.view;
    EXPECT_EQ(inRecord.id.value, refView.id.value);
    // SIGLO-39535 Version値のmeta設定仕様変更対応
    EXPECT_EQ(GetExpectedVersionValue(inRecord), refView.version);

    // アプリケーションのサイズに関連するチェック
    auto& refSize = inProperty.size;
    if (inRecord.type == CONTENT_TYPE_APPLICATION
        && inRecord.installStorage == INSTALL_STORAGE_BUILTIN)
    {
        EXPECT_EQ(inRecord.size.storage[0].storageId, nn::ncm::StorageId::BuildInUser);
        EXPECT_EQ(refSize.storage[0].storageId, nn::ncm::StorageId::BuildInUser);
        EXPECT_NE(0, refSize.storage[0].appSize);
        EXPECT_LT(inRecord.size.storage[0].appSize, refSize.storage[0].appSize);
    }

    // ApplicationControlData に関連するチェック
    auto& refCtl = inProperty.controlProperty;
    auto& refExpCtl = inRecord.controlData;

    // MakeTestApplication のデフォルト設定追加対応より、暫定対応として一旦チェック処理をコメントアウト
#if 0
    for (auto titlePair : refExpCtl.titleList)
    {
        auto& title = refCtl.GetTitle(nn::settings::LanguageCode::Make(titlePair.first));
        EXPECT_STREQ(titlePair.second.name.c_str(), title.name);
        EXPECT_STREQ(titlePair.second.publisher.c_str(), title.publisher);
    }
#endif

    EXPECT_STREQ(refExpCtl.isbn.c_str(), refCtl.isbn);
    // MakeTestApplication のデフォルト設定追加対応より、暫定対応として一旦チェック処理をコメントアウト
    //EXPECT_STREQ(refExpCtl.displayVersion.c_str(), refCtl.displayVersion);
    EXPECT_EQ(refExpCtl.startupUserAccount, refCtl.startupUserAccount);
    EXPECT_EQ(refExpCtl.attributeFlag, refCtl.attributeFlag);

    // MakeTestApplication のデフォルト設定追加対応より、暫定対応として一旦チェック処理をコメントアウト
#if 0
    {
        auto& setList = refExpCtl.supportedLanguageList;
        bool isAllLanguageSupported = setList.empty();
        for (auto& lang : s_AllLangList)
        {
            auto isSupported = refCtl.SupportsLanguage(nn::settings::LanguageCode::Make(lang));
            if (isAllLanguageSupported)
            {
                EXPECT_TRUE(isSupported);
                continue;
            }

            auto iter = std::find(setList.begin(), setList.end(), lang);
            if (iter == setList.end())
            {
                EXPECT_FALSE(isSupported);
                if (isSupported == true)
                {
                    NN_LOG(" Failed Language : %s\n", NsTypeConv::ToString(lang));
                }
            }
            else
            {
                EXPECT_TRUE(isSupported);
                if (isSupported == false)
                {
                    NN_LOG(" Failed Language : %s\n", NsTypeConv::ToString(lang));
                }
            }
        }
    }
#endif
    EXPECT_EQ(refExpCtl.parentalControlFlag, refCtl.parentalControlFlag);
    EXPECT_EQ(refExpCtl.screenShot, refCtl.screenShot);
    EXPECT_EQ(refExpCtl.videoCapture, refCtl.videoCapture);
    EXPECT_EQ(refExpCtl.dataLossConfirmation, refCtl.dataLossConfirmation);
    EXPECT_EQ(refExpCtl.playLogPolicy, refCtl.playLogPolicy);
    EXPECT_EQ(refExpCtl.presenceGroupId.value, refCtl.presenceGroupId.value);
    EXPECT_EQ(refExpCtl.saveDataOwnerId, refCtl.saveDataOwnerId);

    EXPECT_STREQ(refExpCtl.applicationErrorCodeCategory.c_str(), refCtl.applicationErrorCodeCategory.value);
    EXPECT_EQ(refExpCtl.logoType, refCtl.logoType);
    EXPECT_EQ(refExpCtl.logoHandling, refCtl.logoHandling);
    // (SIGLO-61528) Hdcp のチェック追加
    EXPECT_EQ(refExpCtl.hdcp, refCtl.hdcp);
    // SIGLO-46420, SIGLO-46752 対応
    EXPECT_EQ(refExpCtl.seedForPseudoDeviceId, refCtl.seedForPseudoDeviceId);
    EXPECT_STREQ(refExpCtl.bcatPassphrase.c_str(), refCtl.bcatPassphrase);
    // (SIGLO-51085) AddOnContentRegistrationType のチェック追加
    EXPECT_EQ(refExpCtl.aocRegistrationType, refCtl.addOnContentRegistrationType);
    // (SIGLO-64934) CrashReport のチェック追加
    EXPECT_EQ(refExpCtl.crashReport, refCtl.crashReport);
    // (SIGLO-67759) RuntimeAddOnContent のチェック追加
    EXPECT_EQ(refExpCtl.runtimeAddOnContentInstall, refCtl.runtimeAddOnContentInstall);

    // (SIGLO-73968) プレイ情報の取得権限設定 のチェック追加
    CheckTestRecordForPlayLogQueryCapability(inRecord, inProperty);

    // (SIGLO-73969) 修理ツールの動作指定設定 のチェック追加
    EXPECT_EQ(refExpCtl.repairFlag, static_cast<nn::ns::RepairFlag>(refCtl.repairFlag));

    // (SIGLO-82907) ProgramIndex のチェック追加
    EXPECT_EQ(refExpCtl.programIndex, refCtl.programIndex);

    // (SIGLO-83936) 起動に必要なネットワークライセンス設定のチェック追加
    EXPECT_EQ(refExpCtl.requiredNetworkServiceLicenseOnLaunchFlag,
        refCtl.requiredNetworkServiceLicenseOnLaunchFlag);

    // サイズに関連するチェック項目は別関数に分けておく(checkCppCode対策)
    CheckTestRecordForSizeValue(inRecord, inProperty);

    for (auto& ratingPair : refExpCtl.ratingList)
    {
        auto age = refCtl.GetAge(ratingPair.first);
        EXPECT_EQ(ratingPair.second, age);
    }

    {
        int i = 0;
        for (auto lcId : refExpCtl.localCommunicationId)
        {
            EXPECT_EQ(lcId, refCtl.localCommunicationId[i]);
            ++i;
        }

        const int arraySize = sizeof(refCtl.localCommunicationId) / sizeof(refCtl.localCommunicationId[0]);
        for (int j = i; j < arraySize; ++j)
        {
            // 残りは全て アプリケーションID  に設定されているはず
            EXPECT_EQ(inRecord.id.value, refCtl.localCommunicationId[j]);
        }
    }
}

void GetContentMetaStatusList(std::vector<TestDataRecord>& inDataList, std::map<nn::Bit64, ContentMetaStatusData>& outCmsList)
{
    outCmsList.clear();

    // まずはアプリケーションのレコードを詰め込んで、大元となるリストを組み上げる
    for (auto& record : inDataList)
    {
        if (record.type != CONTENT_TYPE_APPLICATION)
        {
            continue;
        }

        ContentMetaStatusData cms(&record);
        outCmsList.insert(std::make_pair(record.id.value, cms));
    }

    // パッチレコードを詰め込む
    for (auto& record : inDataList)
    {
        if (record.type != CONTENT_TYPE_PATCH)
        {
            continue;
        }

        auto iter = outCmsList.find(record.id.value);
        if (std::end(outCmsList) == iter)
        {
            // テストデータ的にはあり得ないはずだが、該当するアプリIDが存在しない場合は無視する
            // (パッチの元となるオリジナルアプリは存在するはず)
            continue;
        }

        auto& cms = iter->second;
        if (cms.pPatchRecord != nullptr)
        {
            // ID が被っている場合
            auto recordVersion = GetExpectedVersionValue(record);
            auto settingVersion = GetExpectedVersionValue(*(cms.pPatchRecord));
            if (recordVersion <= settingVersion)
            {
                // 現状設定されているレコードとバージョンの値を比較して
                // 小さい場合は次のレコードへ (パッチデータ的に書き換えられるため・・)
                continue;
            }
        }

        cms.pPatchRecord = &record;

        // ApplicationView の期待値設定
        cms.viewChecker.SetPatchExpectedFilterValue();
        if (cms.pAppRecord->installFilePath == "")
        {
            // アプリ本体をインストールしていない場合を考慮
            cms.viewChecker.SetPatchOnlyExpectedFilterValue();
        }
    }

    // アドオンコンテンツレコードを詰め込む
    for (auto& record : inDataList)
    {
        if (record.type != CONTENT_TYPE_ADD_ON_CONTENT)
        {
            continue;
        }

        for (auto& aoc : record.aocProperty.aocList)
        {
            auto iter = outCmsList.find(aoc.id.value);
            if (std::end(outCmsList) == iter)
            {
                // アプリケーションなしのAocはありえるため、新たに追加する
                ContentMetaStatusData cms(nullptr);
                auto ret = outCmsList.insert(std::make_pair(aoc.id.value, cms));
                iter = ret.first;
            }

            auto& refAocMap = iter->second.aocRecordMap;
            auto& refAocList = iter->second.aocList;
            auto indexIter = refAocMap.find(aoc.index);
            if (std::end(refAocMap) != indexIter)
            {

                if (aoc.releaseVersion <= indexIter->second->releaseVersion)
                {
                    // インデックス番号が同じで、バージョンが同じまたは小さい場合はなにもせずに次の要素へ
                    continue;
                }

                // 同じインデックス番号でバージョンが大きい場合、レコードを書き換えておく
                {
                    auto listIter = std::find(std::begin(refAocList), std::end(refAocList), indexIter->second);
                    if (listIter != std::end(refAocList))
                    {
                        *listIter = &aoc;
                    }
                }
                indexIter->second = &aoc;
            }
            else
            {
                // インデックス番号が登録されていない場合は無条件に挿入
                refAocMap.insert(std::make_pair(aoc.index, &aoc));
                refAocList.push_back(&aoc);

                // ApplicationView の期待値設定
                iter->second.viewChecker.SetAocExpectedFilterValue();
                if (iter->second.pAppRecord == nullptr)
                {
                    // アプリ本体をインストールしていない場合を考慮
                    iter->second.viewChecker.SetAocOnlyExpectedFilterValue();
                }
            }
        }
    }
}

void MoveCheckProcess(std::vector<TestDataRecord>& inDataList)
{
    Stopwatch watch;
    nn::Result result;

    // テスト前の移動テスト対象となるアプリケーションの抽出
    struct MoveTestData
    {
        TestDataRecord& refRecord;
        bool isMovable;

        explicit MoveTestData(TestDataRecord& inRecord, bool inIsMovable) NN_NOEXCEPT
            : refRecord(inRecord), isMovable(inIsMovable) {}
    };

    std::map<nn::Bit64, MoveTestData> moveTestList;

    for (auto& tdr : inDataList)
    {
        if (tdr.type != CONTENT_TYPE_APPLICATION)
        {
            // Type がアプリケーションでない場合はテスト対象としない
            continue;
        }

        MoveTestData mtd(tdr, false);
        if (tdr.moveDestinationStorage != "")
        {
            //NN_LOG(" [Trace][MoveProcess] Call IsMovable(ID=0x%016llx) : dst = %s\n", tdr.id.value, tdr.moveDestinationStorage.c_str());
            // 移動先のストレージが設定されている場合
            // 移動可能かどうかの確認を念のため実施する
            mtd.isMovable = AppOperation::IsMovable(tdr.id, tdr.moveDestinationStorage);
        }

        moveTestList.insert(std::make_pair(tdr.id.value, mtd));
    }

    bool isPropertyCheck = false;
    for (auto& mt : moveTestList)
    {
        auto& mtd = mt.second;
        if (mtd.isMovable == false)
        {
            // Move処理を実施しない場合は次のアプリに行く
            continue;
        }

        // 移動処理を実施
        {
            ScopedWatch sw(&watch);
            result = AppOperation::MoveApplication(
                mtd.refRecord.id, mtd.refRecord.moveDestinationStorage);
        }
        // 必ず成功するはず
        NNT_EXPECT_RESULT_SUCCESS(result);
        NN_LOG(" [Trace][MoveProcess] MoveApplication(ID=0x%016llx) : time = %lld [ms], src = %s, dst = %s\n", mt.first, watch.GetMilliSecond(), mtd.refRecord.installStorage.c_str(), mtd.refRecord.moveDestinationStorage.c_str());
        // 1回でも移動処理を実施した場合はフラグをONにしておく
        isPropertyCheck = true;
    }

    if (isPropertyCheck == false)
    {
        // 一度も移動処理をしていない場合は以下のプロパティチェックは意味がないので戻る
        return;
    }

    std::vector<TestProperty> testList;
    {
        ScopedWatch sw(&watch);
        result = AppOperation::GetTestPropertyList(testList);
    }
    NNT_EXPECT_RESULT_SUCCESS(result);
    EXPECT_EQ(moveTestList.size(), testList.size());
    NN_LOG(" [Trace][MoveProcess] GetTestProperty : time = %lld [ms], count = %d\n", watch.GetMilliSecond(), testList.size());

    for (auto& tp : testList)
    {
        NN_LOG(" [Trace][MoveProcess] --- MoveProcess : ID = 0x%016llx ---\n", tp.view.id.value);

        auto iter = moveTestList.find(tp.view.id.value);
        if (iter != moveTestList.end())
        {
            CheckTestRecord(iter->second.refRecord, tp);
        }
        else
        {
            // ここに来ることはありえないと思うが・・
            NN_LOG(" [Warning][MoveProcess] MoveProcess : ID(0x%016llx) not Found..\n", tp.view.id.value);
        }
    }
}

void DeleteCheckProcess(std::map<nn::Bit64, ContentMetaStatusData>& inCmsList)
{
    Stopwatch watch;
    nn::Result result;

    size_t deleteCount = 0;
    for (auto& tdr : inCmsList)
    {
        auto appRecordPtr = tdr.second.pAppRecord;
        if (appRecordPtr == nullptr)
        {
            // ひとまずアプリがインストールされていない場合は対象外とする (Aocのみインストールされている場合など)
            continue;
        }

        // 削除処理を実施
        if (appRecordPtr->isDeleteFlag == true)
        {
            {
                ScopedWatch sw(&watch);
                result = AppOperation::Uninstall(appRecordPtr->id);
            }
            // 基本的に削除処理は成功するはず
            NNT_EXPECT_RESULT_SUCCESS(result);
            NN_LOG(" [Trace][DeleteProcess] UninstallApplication(ID=0x%016llx) : time = %lld [ms]\n", appRecordPtr->id.value, watch.GetMilliSecond());
            ++deleteCount;
        }
    }

    if (deleteCount == 0)
    {
        // 一度も削除処理を実施していない場合は以下のチェック処理は意味がないので戻る
        return;
    }

    // 削除処理した後のインストール済アプリの有無とIDのチェック
    {
        std::set<nn::Bit64> idList;
        {
            ScopedWatch sw(&watch);
            result = AppOperation::GetInstalledIdList(idList);
        }
        NNT_EXPECT_RESULT_SUCCESS(result);
        EXPECT_EQ((inCmsList.size() - deleteCount), idList.size());
        NN_LOG(" [Trace][DeleteProcess] GetInstalledIdList : time = %lld [ms], count = %d\n", watch.GetMilliSecond(), idList.size());

        for (auto& dt : inCmsList)
        {
            auto iter = idList.find(dt.first);
            if (dt.second.pAppRecord != nullptr && dt.second.pAppRecord->isDeleteFlag == true)
            {
                // 削除されているので存在しないはず
                EXPECT_EQ(std::end(idList), iter);
            }
            else
            {
                // 存在はするはず
                EXPECT_NE(std::end(idList), iter);
            }
        }
    }

    // 念のため残っているアプリケーションのプロパティ値チェックを実施
    {
        std::vector<TestProperty> testList;
        {
            ScopedWatch sw(&watch);
            result = AppOperation::GetTestPropertyList(testList);
        }
        NNT_EXPECT_RESULT_SUCCESS(result);
        EXPECT_EQ((inCmsList.size() - deleteCount), testList.size());
        NN_LOG(" [Trace][DeleteProcess] GetTestProperty : time = %lld [ms], count = %d\n", watch.GetMilliSecond(), testList.size());

        for (auto& tp : testList)
        {
            auto iter = inCmsList.find(tp.view.id.value);
            if (std::end(inCmsList) != iter)
            {
                if (iter->second.pPatchRecord != nullptr)
                {
                    CheckTestRecord(*(iter->second.pPatchRecord), tp);
                }
                else
                {
                    if (iter->second.pAppRecord != nullptr)
                    {
                        CheckTestRecord(*(iter->second.pAppRecord), tp);
                    }
                }
            }
            else
            {
                // ここに来ることはありえないと思うが・・
                NN_LOG(" [Warning][DeleteProcess] PropertyCheck : ID(0x%016llx) not Found..\n", tp.view.id.value);
            }
        }
    }
}

void PatchAndAocCheckProcess(std::map<nn::Bit64, ContentMetaStatusData>& inCmsList)
{
    Stopwatch watch;
    nn::Result result;

    // インストール済みのアプリに対して管理データなどのプロパティ値を取得
    std::vector<TestProperty> testList;
    watch.Start();
    result = AppOperation::GetTestPropertyList(testList);
    watch.Stop();
    NNT_EXPECT_RESULT_SUCCESS(result);
    NN_LOG(" [Trace][PatchAndAocCheckProcess] GetTestProperty : time = %lld [ms], count = %d\n", watch.GetMilliSecond(), testList.size());

    // プロパティ値チェックを実施
    for (auto& tp : testList)
    {
        auto iter = inCmsList.find(tp.view.id.value);
        if (std::end(inCmsList) == iter)
        {
            NN_LOG(" [Trace][PatchAndAocCheckProcess] Not Found inCmsList : 0x%016x\n", tp.view.id.value);
            continue;
        }

        auto& cms = iter->second;
        if (cms.pPatchRecord != nullptr)
        {
            if (Option::HasKey("-p") == true)
            {
                // デバック用に -p 引数が指定された場合、取得したプロパティ値をログ出力する
                PrintTestProperty(tp);
            }

            // パッチ適用済のアプリのプロパティ値が期待値と同じかどうかのチェック
            CheckTestRecord(*(cms.pPatchRecord), tp);
            // (SIGLO-42684) ApplicationView の詳細なチェック処理を追加
            EXPECT_EQ(true, cms.viewChecker.DoCheck(tp.view));
        }
        else
        {
            // パッチが適用されていない場合
            if (cms.pAppRecord != nullptr)
            {
                if (cms.aocRecordMap.empty() == false)
                {
                    if (Option::HasKey("-p") == true)
                    {
                        // デバック用に -p 引数が指定された場合、取得したプロパティ値をログ出力する
                        PrintApplicationView(tp.view);
                    }
                    // アドオンコンテンツ等がインストールされてもアプリのプロパティ値が変わらずに取得できることを確認
                    CheckTestRecord(*(cms.pAppRecord), tp);
                    // (SIGLO-42684) ApplicationView の詳細なチェック処理を追加
                    EXPECT_EQ(true, cms.viewChecker.DoCheck(tp.view));
                }
            }
            else
            {
                // Aocのみインストールされている状態を想定
                if (Option::HasKey("-p") == true)
                {
                    // デバック用に -p 引数が指定された場合、取得したプロパティ値をログ出力する
                    PrintApplicationView(tp.view);
                }
                // (SIGLO-42684) ApplicationView の詳細なチェック処理を追加
                EXPECT_EQ(true, cms.viewChecker.DoCheck(tp.view));
            }
        }
    }
}

void ContentMetaStatusCheckProcess(std::map<nn::Bit64, ContentMetaStatusData>& inCmsList)
{
    Stopwatch watch;
    nn::Result result;

    for (auto& cms : inCmsList)
    {
        auto& cmsData = cms.second;
        const auto expectedCount = cmsData.GetContentCount();
        const nn::ncm::ApplicationId id = { cms.first };
        const auto getCount = AppOperation::CountContentMeta(id);
        EXPECT_EQ(expectedCount, getCount);
        if (getCount <= 0)
        {
            // ありえないと思うが1つも取れない場合は次のIDへ
            continue;
        }

        ContentMetaStatusList list;
        result = AppOperation::GetContentMetaStatus(id, list);
        NNT_EXPECT_RESULT_SUCCESS(result);
        EXPECT_EQ(expectedCount, list.size());

        if (Option::HasKey("-p") == true)
        {
            // デバック用のログ出力処理
            int index = 0;
            for (auto& status : list)
            {
                NN_LOG("  [ContentMetaStatus : appId=0x%016llx, index=%d]\n", id.value, ++index);
                NN_LOG("    Type    : %s\n", NsTypeConv::ToString(status.type));
                NN_LOG("    Strage  : %s\n", NsTypeConv::ToString(status.installedStorage));
                NN_LOG("    Version : %d (%d)\n", (status.version >> 16), status.version);
                NN_LOG("    ID      : 0x%016llx\n", status.id);
            }
            NN_LOG("\n");
        }

        struct AocStatus
        {
            uint32_t version;
            nn::ncm::StorageId installedStorage;
        };

        std::vector<AocStatus> aocStatusList;
        for (auto& status : list)
        {
            if(nn::ncm::ContentMetaType::Application == status.type)
            {
                // Application のチェック
                EXPECT_NE(cmsData.pAppRecord, nullptr);
                if (cmsData.pAppRecord->installStorage != INSTALL_STORAGE_AUTO)
                {
                    EXPECT_EQ(NsTypeConv::ToStorageIdForInstallStorage(cmsData.pAppRecord->installStorage), status.installedStorage);
                }
                EXPECT_EQ(GetExpectedVersionValue(*(cmsData.pAppRecord)), status.version);
            }
            else if (nn::ncm::ContentMetaType::Patch == status.type)
            {
                // Patch のチェック
                EXPECT_NE(cmsData.pPatchRecord, nullptr);
                if (cmsData.pPatchRecord->installStorage != INSTALL_STORAGE_AUTO)
                {
                    EXPECT_EQ(NsTypeConv::ToStorageIdForInstallStorage(cmsData.pPatchRecord->installStorage), status.installedStorage);
                }
                EXPECT_EQ(GetExpectedVersionValue(*(cmsData.pPatchRecord)), status.version);
            }
            else if (nn::ncm::ContentMetaType::AddOnContent == status.type)
            {
                // AddOnContent のチェックは後からまとめて行う形とする
                AocStatus aocStatus;
                aocStatus.version = status.version;
                aocStatus.installedStorage = status.installedStorage;

                aocStatusList.push_back(aocStatus);
            }
        }

        EXPECT_EQ(cmsData.aocList.size(), aocStatusList.size());
        {
            // AddOnContentのチェック
            int pos = 0;
            for (auto& aoc : cmsData.aocList)
            {
                auto& aocStatus = aocStatusList.at(pos);
                if (aoc->installStorage != INSTALL_STORAGE_AUTO)
                {
                    EXPECT_EQ(NsTypeConv::ToStorageIdForInstallStorage(aoc->installStorage), aocStatus.installedStorage);
                }
                // ReleaseVersionであるため左16ビットシフトする必要あり
                EXPECT_EQ((aoc->releaseVersion << 16), aocStatus.version);
                ++pos;
            }
        }
    }
}

void NormalCheckProcess(std::vector<TestDataRecord>& inDataList)
{
    Stopwatch watch;
    nn::Result result;

    // テスト前のテスト対象となるアプリケーションデータの抽出
    struct NormalTestData
    {
        TestDataRecord& refRecord;

        explicit NormalTestData(TestDataRecord& inRecord) NN_NOEXCEPT
            : refRecord(inRecord) {}
    };
    std::vector<NormalTestData> normalTestList;

    for (auto& tdr : inDataList)
    {
        if (tdr.type != CONTENT_TYPE_APPLICATION || tdr.installFilePath == "")
        {
            // ひとまずアプリケーションでないレコードはテスト対象としない
            // また、インストール先が空文字指定のデータもテスト対象としない
            continue;
        }

        normalTestList.push_back(NormalTestData(tdr));
    }

    std::vector<TestProperty> testList;
    {
        ScopedWatch sw(&watch);
        result = AppOperation::GetTestPropertyList(testList);
    }
    NNT_EXPECT_RESULT_SUCCESS(result);
    EXPECT_EQ(normalTestList.size(), testList.size());
    NN_LOG(" [Trace][NormalProcess] GetTestProperty : time = %lld [ms], count = %d\n", watch.GetMilliSecond(), testList.size());

    ApplicationViewChecker viewCheck;
    size_t count = 0;
    // テスト用にインストールされたアプリケーションは逆順に取得できるはず
    auto rIter = normalTestList.rbegin();
    for (auto& tp : testList)
    {
        ++count;
        NN_LOG(" [Trace][NormalProcess] --- Check %d : ID = 0x%016llx ---\n", count, rIter->refRecord.id.value);
        if (Option::HasKey("-p") == true)
        {
            // デバック用に -p 引数が指定された場合、取得したプロパティ値をログ出力する
            PrintTestProperty(tp);
        }

        CheckTestRecord(rIter->refRecord, tp);
        // (SIGLO-42684) ApplicationView の詳細なチェック処理を追加
        EXPECT_EQ(true, viewCheck.DoCheck(tp.view));

        ++rIter;
    }
}

bool SpecialOptionProcess()
{
    bool isSpecialOption = false;

    // SDカードの管理情報をクリアするためのオプションコマンドを用意しておく
    if (Option::HasKey("--cleanupsd"))
    {
        NN_LOG(" [Trace] *** Cleanup SD Card ***\n");
        AppOperation::CleanupSdCard();
        NN_LOG(" [Trace] *** Cleanup SD Card, Done ***\n");
        isSpecialOption = true;
    }
    // Format を実行するオプションも新たに追加
    else if (Option::HasKey("--formatsd"))
    {
        NN_LOG(" [Trace] *** Format SD Card ***\n");
        AppOperation::FormatSdCard();
        NN_LOG(" [Trace] *** Format SD Card, Done ***\n");
        isSpecialOption = true;
    }

    return isSpecialOption;
}

TEST(ApplicationManagerTestTool, BasicTest)
{
    Stopwatch watch;
    nn::Result result;

    if (SpecialOptionProcess())
    {
        // 特殊オプション処理に該当する場合はすぐに抜ける
        return;
    }

    TestDataContainer testData;
    result = testData.ReadFromCSV(Option::GetInputFilePath());
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsFailure())
    {
        // ここでコケたら以降の処理は実行しても意味がないので返る
        NN_LOG(" [Error] Invalid Input CsvFile\n");
        return;
    }

    // テスト前準備としてすべてのアプリケーションを削除しておく
    result = AppOperation::UninstallAll();
    NNT_EXPECT_RESULT_SUCCESS(result);

    // アプリが存在しないことを確認
    {
        std::vector<nn::ns::ApplicationView> viewList;
        result = AppOperation::GetViewList(viewList);
        NNT_EXPECT_RESULT_SUCCESS(result);
        EXPECT_EQ(0, viewList.size());
    }

    // 念のためストレージ情報を表示させておく
    PrintStorageInfo();
    // CSVファイルから得られたのテストデータリスト情報を取得
    auto& dataList = testData.GetDataList();

    // データリスト内の指定ファイルをインストールする
    // ひとまず Application のみ
    const auto installCount = InstallNspFile(dataList, CONTENT_TYPE_APPLICATION);
    if (installCount <= 0)
    {
        // アプリケーションがインストールされていなければ実行しても意味がないので返る
        NN_LOG(" [Error] No Application Installed\n");
        return;
    }

    AppOperation::ClearCache();

    // テスト用のアプリケーションが全てインストールされた直後のプロパティ値の確認処理
    NormalCheckProcess(dataList);

    std::map<nn::Bit64, ContentMetaStatusData> cmsList;
    GetContentMetaStatusList(dataList, cmsList);

    // Patch データをインストール
    const auto patchCount = InstallNspFile(dataList, CONTENT_TYPE_PATCH);

    // AddOnContent データをインストール
    const auto aocCount = InstallNspFile(dataList, CONTENT_TYPE_ADD_ON_CONTENT);

    // パッチとアドオンコンテンツ適用後のプロパティ値チェック処理
    if (patchCount > 0 || aocCount > 0)
    {
        // パッチかアドオンコンテンツのどちらかがインストールされればチェック処理を実施
        PatchAndAocCheckProcess(cmsList);
    }

    // コンテンツメタAPIに関するチェック処理
    ContentMetaStatusCheckProcess(cmsList);

    // 移動処理を実施するテスト処理
    // 移動処理はまだ未実装なようなので、コメントアウトしておく
    //MoveCheckProcess(dataList);

    // ApplicationManager API の各処理の処理時間を計測する
    result = AppOperation::MeasureApplicationManagerTime();
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (Option::HasKey("--no-uninstall") == false)
    {
        // 削除処理を実施するテスト処理
        DeleteCheckProcess(cmsList);

        // テストプロセスの後処理
        // 念のため全てのアプリをアンインストールする
        result = AppOperation::UninstallAll();
        NNT_EXPECT_RESULT_SUCCESS(result);

        // アプリが存在しないことを確認
        {
            std::vector<nn::ns::ApplicationView> viewList;
            result = AppOperation::GetViewList(viewList);
            NNT_EXPECT_RESULT_SUCCESS(result);
            EXPECT_EQ(0, viewList.size());
        }
    }
}
