﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/ncmutil/ncmutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ns/ns_Result.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ec/system/ec_DeviceAuthenticationApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>

#include "../Common/testLocalCommunication_Common.h"

namespace {

    class LocalCommunicationReceiveSystemUpdateTaskTest : public testing::Test, public nnt::ncmutil::SubmissionPackageFileInstaller
    {
    protected:
        virtual void SetUp()
        {

#if defined( NN_BUILD_CONFIG_OS_WIN )
            SetupBisWorkingDirectory(nnt::GetHostArgv()[1]);
#endif
            nn::ncm::Initialize();
            m_NspDirectory = std::string(nnt::GetHostArgv()[2]);
        }

        virtual void TearDown()
        {
            nn::ncm::Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ABORT_UNLESS_RESULT_SUCCESS(TearDownBisWorkingDirectory());
#endif
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ASSERT(nnt::GetHostArgc() > 2);
            nn::ovln::PrepareSenderAndReceiverForDevelop();
            nn::ovln::InitializeSenderLibraryForOverlay();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
            static NN_ALIGNAS(4096) uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                                                                  nn::socket::DefaultSocketMemoryPoolSize,
                                                                  nn::socket::DefaultSocketAllocatorSize,
                                                                  nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            curl_global_cleanup();
            nn::socket::Finalize();
            nn::fs::UnmountHostRoot();
            nn::ovln::FinalizeSenderLibraryForOverlay();
            nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
#endif
            nn::nifm::CancelNetworkRequest();
        }

        nn::Result InstallAllNsp(nn::ncm::StorageId storage)
        {
            std::vector<std::string> outList;
            return InstallAll(&outList, GetNspDirectory(), storage);
        }

    private:
        const char* GetNspDirectory() const NN_NOEXCEPT
        {
            return m_NspDirectory.c_str();
        }

        std::string m_NspDirectory;
    };

    nn::Result DeleteContentByKey(const nn::ncm::ContentMetaKey& key, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        nn::ncm::ContentMetaDatabase db;
        NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));
        nn::ncm::ContentStorage storage;
        NN_RESULT_DO(nn::ncm::OpenContentStorage(&storage, storageId));
        nn::ncm::ContentInfo contentInfoList[16];
        int offset = 0;
        while(NN_STATIC_CONDITION(true))
        {
            int infoCount;
            NN_RESULT_DO(db.ListContentInfo(&infoCount, contentInfoList, NN_ARRAY_SIZE(contentInfoList), key, offset));
            for (int j = 0; j < infoCount; j++)
            {
                storage.Delete(contentInfoList[j].GetId());
            }
            if (infoCount < NN_ARRAY_SIZE(contentInfoList))
            {
                break;
            }
            offset += infoCount;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType metaType) NN_NOEXCEPT
    {
        const int ListCount = 16;
        nn::ncm::ContentMetaKey keyList[ListCount];

        nn::ncm::ContentMetaDatabase db;
        NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuiltInSystem));
        auto count = db.ListContentMeta(keyList, ListCount, metaType);
        for (int i = 0; i < count.listed; i++)
        {
            NN_RESULT_DO(DeleteContentByKey(keyList[i], nn::ncm::StorageId::BuiltInSystem));
            NN_RESULT_DO(db.Remove(keyList[i]));
        }
        NN_RESULT_DO(db.Commit());

        NN_RESULT_SUCCESS;
    }

    nn::Result ExistsContent(bool* outValue, const nn::ncm::ContentMetaKey& key, nn::ncm::ContentMetaDatabase* pDb, nn::ncm::ContentStorage* pStorage) NN_NOEXCEPT
    {
        int offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            const int ListCount = 16;
            nn::ncm::ContentInfo infoList[ListCount];

            int count;
            NN_RESULT_DO(pDb->ListContentInfo(&count, infoList, ListCount, key, offset));
            for (int i = 0; i < count; i++)
            {
                bool hasContent;
                NN_RESULT_DO(pStorage->Has(&hasContent, infoList[i].GetId()));
                if (!hasContent)
                {
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            }
            if (count != ListCount)
            {
                break;
            }
            offset += count;
        }

        NN_RESULT_SUCCESS;
    }
}

TEST_F(LocalCommunicationReceiveSystemUpdateTaskTest, ReceiveSystemUpdate)
{
    nn::nim::InitializeForNetworkInstallManager();

    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInSystem;
    nn::ncm::ContentMetaDatabase db;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storageId));
    nn::ncm::ContentStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));

    // 一旦インストールされている SystemUpdate と SystemData を消す
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemUpdate));
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemData));

    // ディレクトリ内の nsp をインストール
    NNT_ASSERT_RESULT_SUCCESS(InstallAllNsp(storageId));

    // SystemUpdate と SystemData を検索
    // SystemData は SystemUpdate に含まれている前提
    nn::ncm::ContentMetaKey systemUpdateMetaKey;
    nn::ncm::ContentMetaKey systemDataMetaKey;
    {
        auto count = db.ListContentMeta(&systemUpdateMetaKey, 1, nn::ncm::ContentMetaType::SystemUpdate);
        ASSERT_TRUE(count.listed > 0);

        count = db.ListContentMeta(&systemDataMetaKey, 1, nn::ncm::ContentMetaType::SystemData);
        ASSERT_TRUE(count.listed > 0);
    }

    // 再度インストールされている SystemUpdate と SystemData を消す
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemUpdate));
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemData));

    // ダウンロード
    nn::nim::LocalCommunicationReceiveSystemUpdateTaskId id;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::CreateLocalCommunicationReceiveSystemUpdateTask(&id, 0, TestPort, systemUpdateMetaKey, 0));

        nn::nim::LocalCommunicationReceiveSystemUpdateTaskInfo info;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetLocalCommunicationReceiveSystemUpdateTaskInfo(&info, id));
        EXPECT_TRUE(info.key == systemUpdateMetaKey);
        EXPECT_EQ(info.progress.installedSize, 0LL);
        EXPECT_EQ(info.progress.state, nn::ncm::InstallProgressState::NotPrepared);

        nn::nim::AsyncResult asyncResult;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestLocalCommunicationReceiveSystemUpdateTaskRun(&asyncResult, id));

        asyncResult.Wait();
        NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());

        NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetLocalCommunicationReceiveSystemUpdateTaskInfo(&info, id));
        EXPECT_TRUE(info.key == systemUpdateMetaKey);
        EXPECT_GT(info.progress.installedSize, 0LL);
        EXPECT_EQ(info.progress.state, nn::ncm::InstallProgressState::Downloaded);

        // タスク実行中は呼び出し出来ない
        nn::nim::AsyncResult testResult;
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::RequestLocalCommunicationReceiveSystemUpdateTaskRun(&testResult, id));
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::CommitLocalCommunicationReceiveSystemUpdateTask(id));
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::DestroyLocalCommunicationReceiveSystemUpdateTask(id));

        // List
        nn::nim::LocalCommunicationReceiveSystemUpdateTaskId idList[2];
        EXPECT_EQ(1, nn::nim::ListLocalCommunicationReceiveSystemUpdateTask(idList, NN_ARRAY_SIZE(idList)));
        EXPECT_EQ(idList[0], id);
    }

    // コミット
    NNT_EXPECT_RESULT_SUCCESS(nn::nim::CommitLocalCommunicationReceiveSystemUpdateTask(id));

    // インストール出来たことの確認
    {
        bool has;
        NNT_EXPECT_RESULT_SUCCESS(db.Has(&has, systemUpdateMetaKey));
        EXPECT_TRUE(has);
        NNT_EXPECT_RESULT_SUCCESS(ExistsContent(&has, systemUpdateMetaKey, &db, &storage));

        NNT_EXPECT_RESULT_SUCCESS(db.Has(&has, systemDataMetaKey));
        EXPECT_TRUE(has);
        NNT_EXPECT_RESULT_SUCCESS(ExistsContent(&has, systemDataMetaKey, &db, &storage));
    }

    // 後処理の確認
    {
        nn::nim::LocalCommunicationReceiveSystemUpdateTaskInfo info;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetLocalCommunicationReceiveSystemUpdateTaskInfo(&info, id));
        EXPECT_GT(info.progress.installedSize, 0LL);
        EXPECT_EQ(info.progress.state, nn::ncm::InstallProgressState::Commited);

        NNT_EXPECT_RESULT_SUCCESS(nn::nim::DestroyLocalCommunicationReceiveSystemUpdateTask(id));

        nn::nim::LocalCommunicationReceiveSystemUpdateTaskId idList[2];
        EXPECT_EQ( 0, nn::nim::ListLocalCommunicationReceiveSystemUpdateTask(idList, NN_ARRAY_SIZE(idList)));
    }

    nn::nim::FinalizeForNetworkInstallManager();
} // NOLINT(impl/function_size)
