﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiProxy.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>

#include <nn/nifm/nifm_Result.h>
#include <nn/nifm/nifm_ResultPrivate.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include <algorithm>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;
    static const int BasePriority = 100;
}

class WlanDisabledCaseTest : public ::testing::Test
{
protected:
    virtual void TearDown() { }

    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(true);

        // 接続制御が許可されているとしてテスト実行
        m_IsCommunicationControlEnabledBackup = nn::nifm::test::IsCommunicationControlEnabled();
        nn::nifm::test::SetCommunicationControlEnabled(true);
    }

    static void TearDownTestCase()
    {
        // 接続制御許可を元に戻す
        nn::nifm::test::SetCommunicationControlEnabled(m_IsCommunicationControlEnabledBackup);

        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }

private:
    static bool m_IsCommunicationControlEnabledBackup;
};
bool WlanDisabledCaseTest::m_IsCommunicationControlEnabledBackup = false;


TEST_F(WlanDisabledCaseTest, GlobalNetworkConnectionSetting)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::NetworkConnection* pNetworkConnection = nn::nifm::GetGlobalNetworkConnectionPointer();
    ASSERT_NE(nullptr, pNetworkConnection);
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(pNetworkConnection->GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
}

TEST_F(WlanDisabledCaseTest, NetworkInterfaceList)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::NetworkInterfaceInfo networkInterfaceInfo[8];
    int bufferSize;
    int outCount;

    // 最初の scan 終了後の NIC 登録を待つ workaround
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(15));

    // 利用対象の NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 利用対象の無線 NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 利用対象の有線 NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 無線 NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 有線 NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // フィルタを指定しなければ何も取れない
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            0
        )
    );
    ASSERT_EQ(outCount, 0);

    // 定義外のフィルタビットを立てると失敗
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_FAILURE(
        nn::nifm::ResultInvalidArgument,
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            ~static_cast<nn::Bit32>(0)
        )
    );
}

// 実際の接続が行われるテストの中で最初に実行される
TEST_F(WlanDisabledCaseTest, Basic)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    {
        ASSERT_TRUE(nn::nifm::test::Disconnect());

        // 非接続状態では現在接続中のアクセスポイント情報を取得できない
        nn::nifm::AccessPointData accessPointData;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentAccessPoint(&accessPointData));
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 接続 ～ 切断
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());
        request1a.Cancel();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersNone);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(request1a.GetHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 接続 ～ 切断
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());
        request1a.Cancel();
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, request1a.GetResult());
    }
}

TEST_F(WlanDisabledCaseTest, Recycle)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceInTrouble, nn::nifm::Scan());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        // 接続 ～ 切断
        for (int i = 0; i < 10; ++i)
        {
            NN_LOG("Try %d\n", i);
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());
            request1a.Cancel();
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultCanceled, request1a.GetResult());
        }
    }
}

TEST_F(WlanDisabledCaseTest, TakeOverConnection)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceInTrouble, nn::nifm::Scan());

    {
        nn::nifm::Request request0a(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request0b(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request1b(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request1c(nn::nifm::test::RequestParametersNone);
        nn::nifm::Request request1d(nn::nifm::test::RequestParametersNone);

        // 接続

        // requirement0
        nn::nifm::detail::RequestClient *pRequestClient = nn::nifm::GetRequestClientPointer(request0a.GetHandle());
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_LocalGeneric));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRejectable(true));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request0a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request0a.GetResult());

        // ローカル通信接続では，現在接続中のアクセスポイント情報を取得できない
        nn::nifm::AccessPointData accessPointData;
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLocalConnection, nn::nifm::GetCurrentAccessPoint(&accessPointData));

        // ローカル通信接続では，現在利用中の接続設定を取得できない
        nn::nifm::NetworkProfileData networkProfileData;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLocalConnection, nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

        // 優先度の高い異種要求で乗っ取り（Local:100 -> Internet:99）
        // requirement1
        pRequestClient = nn::nifm::GetRequestClientPointer(request1b.GetHandle());
        ASSERT_NE(nullptr, pRequestClient);
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1b.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 1));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1b, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1b.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, request0a.GetRequestState());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request0a.GetResult());

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersNone);

            // 優先度の高い異種要求で乗っ取り
            // requirement2(Internet:99 -> Local:98)
            pRequestClient = nn::nifm::GetRequestClientPointer(request2a.GetHandle());
            ASSERT_NE(nullptr, pRequestClient);
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_LocalGeneric));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 2));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRejectable(true));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());
            EXPECT_EQ(nn::nifm::RequestState_Free, request1b.GetRequestState());
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request1b.GetResult());

            // 優先度の低い同種要求で相乗り（相乗り可能なローカル通信要求）
            // requirement0(Local), requirement2(Local)
            pRequestClient = nn::nifm::GetRequestClientPointer(request0b.GetHandle());
            ASSERT_NE(nullptr, pRequestClient);
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_LocalGeneric));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRejectable(true));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetGreedy(false));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetSharable(true));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request0b, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request0b.GetResult());
            EXPECT_EQ(nn::nifm::RequestState_Accepted, request2a.GetRequestState());
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());

            // 優先度の低い異種要求で乗っ取り失敗
            // requirement0(Local), requirement2(Local)
            pRequestClient = nn::nifm::GetRequestClientPointer(request1c.GetHandle());
            ASSERT_NE(nullptr, pRequestClient);
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_InternetGeneric));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1c.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 1));
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1c, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request1c.GetResult());
            EXPECT_EQ(nn::nifm::RequestState_Accepted, request0b.GetRequestState());
            NNT_ASSERT_RESULT_SUCCESS(request0b.GetResult());
            EXPECT_EQ(nn::nifm::RequestState_Accepted, request2a.GetRequestState());
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());
        }

        // 優先度の高い要求が取り下げられたことで、以前却下された異種要求が乗っ取り成功
        // requirement1(Internet)
        pRequestClient = nn::nifm::GetRequestClientPointer(request1d.GetHandle());
        ASSERT_NE(nullptr, pRequestClient);
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirementPreset(nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1d.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRawPriority(BasePriority - 1));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1d, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1d.GetResult());
        EXPECT_EQ(nn::nifm::RequestState_Free, request0b.GetRequestState());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request0b.GetResult());
    }
}

TEST_F(WlanDisabledCaseTest, TemporaryConnectWifi)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceInTrouble, nn::nifm::Scan());

    {
        ASSERT_TRUE(nn::nifm::test::Disconnect());

        {
            // 現在の接続情報の取得（接続前）
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

            nn::nifm::ProxySetting proxySetting;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentProxySetting(&proxySetting));
        }

        {
            // imatake-opennone-stealth
            const nn::nifm::WirelessSettingData wirelessSetting0 = {
                {  // ssidConfig
                    {  // ssid
                        24,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x6e,0x6f,0x6e,0x65,0x2d,0x73,0x74,0x65,0x61,0x6c,0x74,0x68 }  // hex
                    },
                true  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_None  // encryption
                    },
                    {  // sharedKey
                        7,  // length
                        "NotUsed"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting0 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    true,  // isEnabled
                    8080,  // port
                    "dummy.proxy.jp",  // proxy
                    {  // authentication
                        true,  // isEnabled
                        "dummy_user",  // username
                        "dummy_password"  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile0 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting0
                },
                ipSetting0
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求0
            nn::nifm::NetworkConnection networkConnection0;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection0.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection0.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection0.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection0.SubmitRequest();
            EXPECT_TRUE(networkConnection0.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection0.GetResult()); // wlan 動作抑制中

            // テンポラリ接続要求1
            nn::nifm::NetworkConnection networkConnection1;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), BasePriority - 1));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection1.SubmitRequest();
            EXPECT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection1.GetResult()); // wlan 動作抑制中

            // 同一のテンポラリ接続設定を用いた、優先度だけが異なる要求は相乗りする
            EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection0.GetRequestState());
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection0.GetResult()); // wlan 動作抑制中

            // 現在の接続情報の取得
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

            // 現在のプロキシ設定情報の取得
            nn::nifm::ProxySetting proxySetting;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentProxySetting(&proxySetting));
        }

        {
            // imatake-wpa2aes
            const nn::nifm::WirelessSettingData wirelessSetting1 = {
                {  // ssidConfig
                    {  // ssid
                        15,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Wpa2Psk,  // authentication
                        nn::nifm::Encryption_Aes  // encryption
                    },
                    {  // sharedKey
                        11,  // length
                        "Shi2iTaiZen"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting1 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile1 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true,  // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting1
                },
                ipSetting1
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());
            nn::util::Uuid id1 = temporaryNetworkProfile.GetId();

            // テンポラリ接続要求2
            nn::nifm::NetworkConnection networkConnection2;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), BasePriority));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection2.SubmitRequest();
            EXPECT_TRUE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection2.GetResult()); // wlan 動作抑制中

            // テンポラリ接続要求3
            nn::nifm::NetworkConnection networkConnection3;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection3.GetRequestHandle(), BasePriority - 2));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection3.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection3.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection3.SubmitRequest();
            EXPECT_TRUE(networkConnection3.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection3.GetResult()); // wlan 動作抑制中

            // 現在の接続情報の取得
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

            // imatake-mixed
            const nn::nifm::WirelessSettingData wirelessSetting2 = {
                {  // ssidConfig
                    {  // ssid
                        13,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6d,0x69,0x78,0x65,0x64 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Wpa2Psk,  // authentication
                        nn::nifm::Encryption_Aes  // encryption
                    },
                    {  // sharedKey
                        11,  // length
                        "Shi2iTaiZen"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting2 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile2 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting2
                },
                ipSetting2
            };

            //  テンポラリ接続オブジェクトを使いまわすと以前の接続設定は消える
            NNT_ASSERT_RESULT_SUCCESS(temporaryNetworkProfile.Update(networkProfile2));
            ASSERT_NE(id1, temporaryNetworkProfile.GetId());
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求4
            nn::nifm::NetworkConnection networkConnection4;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection4.GetRequestHandle(), BasePriority - 3));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection4.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection4.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection4.SubmitRequest();
            EXPECT_TRUE(networkConnection4.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection4.GetResult()); // wlan 動作抑制中

            // 異なるテンポラリ接続を用いた利用要求が accept されたので、以前の要求は reject
            EXPECT_EQ(nn::nifm::RequestState_Free, networkConnection3.GetRequestState());
        }

        {
            // "imatake-openwep104"
            const nn::nifm::WirelessSettingData wirelessSetting3 = {
                {  // ssidConfig
                    {  // ssid
                        18,  // length
                        { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x31,0x30,0x34 }  // hex
                    },
                false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_Wep  // encryption
                    },
                    {  // sharedKey
                        13,  // length
                        "Shi2iTaiZenTo"  // keyMaterial
                    }
                }
            };
            const nn::nifm::IpSettingData ipSetting3 = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkProfileData networkProfile3 = {
                nn::util::InvalidUuid,  // id
                {},  // name
                nn::nifm::NetworkProfileType_Temporary, // networkProfileType
                nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
                true, // isAutoConnect
                true, // isLargeCapacity
                {
                    wirelessSetting3
                },
                ipSetting3
            };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile3);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // テンポラリ接続要求5
            nn::nifm::NetworkConnection networkConnection5;

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection5.GetRequestHandle(), BasePriority - 4));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection5.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection5.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
            networkConnection5.SubmitRequest();
            EXPECT_TRUE(networkConnection5.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection5.GetResult()); // wlan 動作抑制中
        }

        // 残ったりリクエストが暗黙的に取り下げ
    }


    // TODO: APから切断されたことを検証
    // ...

} // NOLINT(impl/function_size)

TEST_F(WlanDisabledCaseTest, GetCurrentPrimaryIpAddress)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::socket::InAddr inAddr;

    ASSERT_TRUE(nn::nifm::test::Disconnect());
    inAddr.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentPrimaryIpAddress(&inAddr).IsFailure());

    {
        nn::nifm::SubmitNetworkRequestAndWait();

        EXPECT_TRUE(nn::nifm::IsNetworkAvailable());

        inAddr.S_addr = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentPrimaryIpAddress(&inAddr));
        EXPECT_TRUE(inAddr.S_addr != 0);
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());
    inAddr.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentPrimaryIpAddress(&inAddr).IsFailure());
}

TEST_F(WlanDisabledCaseTest, GetCurrentIpStatus)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::socket::InAddr ipAddress;
    nn::socket::InAddr subnetMask;
    nn::socket::InAddr defaultGateway;
    nn::socket::InAddr preferredDns;
    nn::socket::InAddr alternateDns;

    ASSERT_TRUE(nn::nifm::test::Disconnect());
    ipAddress.S_addr = 0;
    subnetMask.S_addr = 0;
    defaultGateway.S_addr = 0;
    preferredDns.S_addr = 0;
    alternateDns.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns).IsFailure());

    {
        nn::nifm::SubmitNetworkRequestAndWait();

        ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns));

        EXPECT_TRUE(ipAddress.S_addr != 0);
        EXPECT_TRUE(subnetMask.S_addr != 0);
        EXPECT_TRUE(defaultGateway.S_addr != 0);
        EXPECT_TRUE(preferredDns.S_addr != 0);
        //EXPECT_TRUE(alternateDns.S_addr != 0);

        nn::nifm::test::PrintInAddr(ipAddress);
        nn::nifm::test::PrintInAddr(subnetMask);
        nn::nifm::test::PrintInAddr(defaultGateway);
        nn::nifm::test::PrintInAddr(preferredDns);
        nn::nifm::test::PrintInAddr(alternateDns);
    }

    EXPECT_TRUE(nn::nifm::test::Disconnect());
    ipAddress.S_addr = 0;
    subnetMask.S_addr = 0;
    defaultGateway.S_addr = 0;
    preferredDns.S_addr = 0;
    alternateDns.S_addr = 0;
    EXPECT_TRUE(nn::nifm::GetCurrentIpConfigInfo(&ipAddress, &subnetMask, &defaultGateway, &preferredDns, &alternateDns).IsFailure());
}

TEST_F(WlanDisabledCaseTest, WirelessCommunicationControl)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(initializer.GetResult());

    nn::nifm::SetWirelessCommunicationEnabled(true);
    ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());

    {
        // 無線インターネット通信のオフ

        // imatake-wpa2aes
        nn::util::Uuid id1 = { { 0x93,0x18,0x84,0x40,0xb6,0xd5,0x14,0xb7,0x05,0x6b,0x08,0xcd,0x57,0x91,0x09,0x60 } };

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), id1));

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, request1a.GetResult()); // wlan 動作抑制中

        // 無線通信をオフに
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        // 無線オフ状態で接続要求を再提出しても拒否される
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, request1a.GetResult()); // wlan 動作抑制中

        nn::nifm::SetWirelessCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }

    {
        // ローカル通信のオフ

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersNone);
        nn::nifm::SetRequestRequirementPreset(request1a.GetHandle(), nn::nifm::RequirementPreset_LocalGeneric);
        nn::nifm::SetRequestRejectable(request1a.GetHandle(), true);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult()); // wlan 動作抑制中でもローカル通信要求は通る

        // 無線通信をオフに
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        // 無線オフ後、すこし遅延して切られる
        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, request1a.GetResult());

        // 無線オフ状態で接続要求を再提出しても拒否される
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNotAvailable, request1a.GetResult()); // wlan 動作抑制中＋無線オフでは失敗する

        nn::nifm::SetWirelessCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }

    {
        // 有線に影響を与えない確認

        // ethernet
        nn::util::Uuid id1 = { { 0x93,0x18,0x84,0x40,0xb6,0xd5,0x14,0xb7,0x05,0x6b,0x08,0xcd,0x57,0x91,0x09,0x64 } };

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), id1));

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_TRUE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(TIME_OUT_IN_SECONDS_FOR_SUCCESS)));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        // 無線通信をオフに
        nn::nifm::SetWirelessCommunicationEnabled(false);
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        // 無線オフされても影響はない
        EXPECT_FALSE(request1a.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        ASSERT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }

    {
        // テスト用操作

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetWirelessCommunicationEnabledForTest(false));
        EXPECT_FALSE(nn::nifm::IsWirelessCommunicationEnabled());

        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetWirelessCommunicationEnabledForTest(true));
        EXPECT_TRUE(nn::nifm::IsWirelessCommunicationEnabled());
    }
} // NOLINT(impl/function_size)


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
