﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_Result.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_FormatString.h>

#include "../Common/nifm_TestUtility.h"

// 1 for wifi, 0 for ethernet
#define NN_NIFM_CONFIG_TEST_TEMPORARY_CONNECT_USING_WIFI 1

namespace
{
    static const int BasePriority = 128;

    NN_ALIGNAS(4096) char g_StackOfThread1[4 * 1024];
    NN_ALIGNAS(4096) char g_StackOfThread2[4 * 1024];
    NN_ALIGNAS(4096) char g_StackOfThread3[4 * 1024];
}

class StressTest : public ::testing::Test
{
protected:
    virtual void TearDown() { }
};

// テストスレッド
// requirement を指定して接続要求
void ThreadFunc(void* p)
{
    nn::os::ThreadType* thread = reinterpret_cast<nn::os::ThreadType*>(p);
    const char* name = nn::os::GetThreadNamePointer( thread );

    while( NN_STATIC_CONDITION(1) )
    {
        // TODO: 接続要求の偏りは適当
        uint8_t priority = nn::nifm::test::Random() % 3 + BasePriority;
        bool isRejectable = true;
        bool isPersistent = false;
        bool isInstant = nn::nifm::test::Random() % 2 ? true : false;
        bool isSustainable = false;
        nn::nifm::NetworkType networkType = nn::nifm::test::Random() % 3 < 2 ? nn::nifm::NetworkType_Internet : nn::nifm::NetworkType_Local;
        bool isGreedy;
        bool isSharable;
        if (networkType == nn::nifm::NetworkType_Infrastructure)
        {
            isGreedy = false;
            isSharable = true;
        }
        else
        {
            isGreedy = true;
            isSharable = false;
        }
        bool isForeground = true;
        bool isAutomaticSwitchProhibited = true;
        bool isKeptInSleep = false;
        bool isInternalContextUpdatable = true;
        nn::nifm::ConnectionConfirmationOption options[] = {
            nn::nifm::ConnectionConfirmationOption_Invalid,
            nn::nifm::ConnectionConfirmationOption_Prohibited,
            nn::nifm::ConnectionConfirmationOption_NotRequired,
            nn::nifm::ConnectionConfirmationOption_Preferred,
            nn::nifm::ConnectionConfirmationOption_Required,
            nn::nifm::ConnectionConfirmationOption_Forced
        };
        nn::nifm::ConnectionConfirmationOption option = networkType == nn::nifm::NetworkType_Internet ? options[nn::nifm::test::Random() % 5 + 1] : options[0];

        nn::Result result;
        NN_LOG("%s[%s] --> Scan\n", name, nn::nifm::test::ElapsedTime() );
        result = nn::nifm::Scan();
        NN_LOG("%s[%s] --> ScanResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription() );

        nn::nifm::Requirement requirement = { priority, isRejectable, isPersistent, isInstant, isSustainable, isGreedy, isSharable, isForeground, isAutomaticSwitchProhibited, isKeptInSleep, isInternalContextUpdatable, networkType, nn::util::InvalidUuid, option };
        NN_LOG("%s[%s] --> ", name, nn::nifm::test::ElapsedTime());
        nn::nifm::test::PrintRequirement(requirement);

        nn::nifm::Request request(nn::nifm::test::RequestParametersNone);
        NN_LOG("%s[%s] --> CreateRequest\n", name, nn::nifm::test::ElapsedTime());
        nn::nifm::detail::RequestClient *pRequestClient = nn::nifm::GetRequestClientPointer(request.GetHandle());
        ASSERT_NE(nullptr, pRequestClient);
        ASSERT_NE(nullptr, pRequestClient);
        NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirement(requirement));
        nn::nifm::test::SubmitRequestAndWait(&request, 5);
        NN_LOG("%s[%s] --> TimedWait\n", name, nn::nifm::test::ElapsedTime());
        result = request.GetResult();
        NN_LOG("%s[%s] --> GetResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription() );

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( nn::nifm::test::Random() % 5000 + 1000 ) );

        request.Cancel();
        NN_LOG("%s[%s] --> Cancel\n", name, nn::nifm::test::ElapsedTime());
        result = request.GetResult();
        NN_LOG("%s[%s] --> GetResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription() );

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( nn::nifm::test::Random() % 3000 + 1000 ) );
    }
}

// テストスレッド
// 接続設定を直接指定して接続要求（テンポラリ接続）
void ThreadFuncTemporarySetting(void* p)
{
    nn::os::ThreadType* thread = reinterpret_cast<nn::os::ThreadType*>(p);
    const char* name = nn::os::GetThreadNamePointer( thread );

    while( NN_STATIC_CONDITION(1) )
    {
        nn::Result result;
        NN_LOG("%s[%s] --> Scan\n", name, nn::nifm::test::ElapsedTime() );
        result = nn::nifm::Scan();
        NN_LOG("%s[%s] --> ScanResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription() );

        {
#if NN_NIFM_CONFIG_TEST_TEMPORARY_CONNECT_USING_WIFI
            // imatake-openwep40
            nn::nifm::WirelessSettingData wirelessSetting = {
                {  // ssidConfig
                    {  // ssid
                        17,  // length
                        {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x34,0x30}  // hex
                    },
                    false  // nonBroadcast
                },
                {  // security
                    {  //authEncryption
                        nn::nifm::Authentication_Open,  // authentication
                        nn::nifm::Encryption_Wep  // encryption
                    },
                    {  // sharedKey
                        5,  // length
                        "Shi2i"  // keyMaterial
                    }
                }
            };
            nn::nifm::IpSettingData ipSetting = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            nn::nifm::NetworkInterfaceType networkInterface = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211;
#else
            nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
            nn::nifm::IpSettingData ipSetting = {
                {  // ip
                    true,  // isAuto
                    {},  // ipAddress
                    {},  // subnetMask
                    {}  // defaultGateway
                },
                {  // dns
                    true,  // isAuto
                    {},  // preferredDns
                    {}  // alternateDns
                },
                {  // proxy
                    false,  // isEnabled
                    0,  // port
                    "",  // proxy
                    {  // authentication
                        false,  // isEnabled
                        "",  // username
                        ""  // password
                    }
                },
                1400  //mtu
            };
            const nn::nifm::NetworkInterfaceType networkInterface = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet;
#endif
            const nn::nifm::NetworkProfileType profileType = nn::nifm::NetworkProfileType_Temporary;
            nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid, {}, profileType, networkInterface, true, true, { wirelessSetting }, ipSetting };

            nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
            ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

            // TODO: 接続要求の偏りは適当
            uint8_t priority = nn::nifm::test::Random() % 3 + BasePriority;
            bool isRejectable = true;
            bool isPersistent = false;
            bool isInstant = nn::nifm::test::Random() % 2 ? true : false;
            bool isSustainable = false;
            nn::nifm::NetworkType networkType = nn::nifm::test::Random() % 3 < 2 ? nn::nifm::NetworkType_Infrastructure : nn::nifm::NetworkType_Local;
            bool isGreedy;
            bool isSharable;
            if (networkType == nn::nifm::NetworkType_Infrastructure)
            {
                isGreedy = false;
                isSharable = true;
            }
            else
            {
                isGreedy = true;
                isSharable = false;
            }
            bool isForeground = true;
            bool isAutomaticSwitchProhibited = true;
            bool isKeptInSleep = false;
            bool isInternalContextUpdatable = true;
            nn::nifm::ConnectionConfirmationOption options[] = {
                nn::nifm::ConnectionConfirmationOption_Invalid,
                nn::nifm::ConnectionConfirmationOption_Prohibited,
                nn::nifm::ConnectionConfirmationOption_NotRequired,
                nn::nifm::ConnectionConfirmationOption_Preferred,
                nn::nifm::ConnectionConfirmationOption_Required,
                nn::nifm::ConnectionConfirmationOption_Forced
            };
            nn::nifm::ConnectionConfirmationOption option = networkType == nn::nifm::NetworkType_Internet ? options[nn::nifm::test::Random() % 5 + 1] : options[0];

            nn::nifm::Requirement requirement = { priority, isRejectable, isPersistent, isInstant, isSustainable, isGreedy, isSharable, isForeground, isAutomaticSwitchProhibited, isKeptInSleep, isInternalContextUpdatable, networkType, temporaryNetworkProfile.GetId(), option };
            NN_LOG("%s[%s] --> ", name, nn::nifm::test::ElapsedTime());
            nn::nifm::test::PrintRequirement(requirement);

            nn::nifm::Request request(nn::nifm::test::RequestParametersNone);
            NN_LOG("%s[%s] --> CreateRequest\n", name, nn::nifm::test::ElapsedTime());
            nn::nifm::detail::RequestClient *pRequestClient = nn::nifm::GetRequestClientPointer(request.GetHandle());
            ASSERT_NE(nullptr, pRequestClient);
            NNT_ASSERT_RESULT_SUCCESS(pRequestClient->SetRequirement(requirement));
            nn::nifm::test::SubmitRequestAndWait(&request, 5);
            NN_LOG("%s[%s] --> TimedWait\n", name, nn::nifm::test::ElapsedTime());
            result = request.GetResult();
            NN_LOG("%s[%s] --> GetResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription());

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(nn::nifm::test::Random() % 5000 + 1000));
        }
        // Temporary 設定の削除

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( nn::nifm::test::Random() % 3000 + 1000 ) );
    }
} // NOLINT(impl/function_size)

// テストスレッド
// NetworkConnection を利用した簡易接続要求
void ThreadFuncNetworkConnection(void* p)
{
    nn::os::ThreadType* thread = reinterpret_cast<nn::os::ThreadType*>(p);
    const char* name = nn::os::GetThreadNamePointer( thread );

    while( NN_STATIC_CONDITION(1) )
    {
        nn::Result result;
        NN_LOG("%s[%s] --> Scan\n", name, nn::nifm::test::ElapsedTime() );
        result = nn::nifm::Scan();
        NN_LOG("%s[%s] --> ScanResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription() );

        nn::nifm::NetworkConnection networkConnection;

        networkConnection.SubmitRequestAndWait();
        NN_LOG("%s[%s] --> RequestAndWait\n", name, nn::nifm::test::ElapsedTime());

        bool available = networkConnection.IsAvailable();
        NN_LOG("%s[%s] --> IsAvailable=%s\n", name, nn::nifm::test::ElapsedTime(), available ? "true" : "false" );
        result = networkConnection.GetResult();
        NN_LOG("%s[%s] --> GetResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription() );

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( nn::nifm::test::Random() % 5000 + 1000 ) );

        networkConnection.CancelRequest();
        NN_LOG("%s[%s] --> Cancel\n", name, nn::nifm::test::ElapsedTime());
        result = networkConnection.GetResult();
        NN_LOG("%s[%s] --> GetResult=%03u-%04u\n", name, nn::nifm::test::ElapsedTime(), result.GetModule(), result.GetDescription() );

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( nn::nifm::test::Random() % 3000 + 1000 ) );
    }
}

// 上記の三種類のスレッドを起動し，それぞれがスキャンと接続要求を繰り返す
TEST_F(StressTest, Basic)
{
    nn::nifm::test::ElapsedTime(nn::os::GetSystemTick().GetInt64Value());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // 通常の接続要求スレッド
    nn::os::ThreadType thread1;
    NNT_ASSERT_RESULT_SUCCESS( nn::os::CreateThread( &thread1, &ThreadFunc, &thread1, g_StackOfThread1, 4 * 1024, 24 ) );
    nn::os::SetThreadName( &thread1, "Thread_1" );
    nn::os::StartThread( &thread1 );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    // テンポラリ接続要求スレッド
    nn::os::ThreadType thread2;
    NNT_ASSERT_RESULT_SUCCESS( nn::os::CreateThread( &thread2, &ThreadFuncTemporarySetting, &thread2, g_StackOfThread2, 4 * 1024, 24 ) );
    nn::os::SetThreadName( &thread2, "Thread_2" );
    nn::os::StartThread( &thread2 );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    // 簡易接続要求スレッド
    nn::os::ThreadType thread3;
    NNT_ASSERT_RESULT_SUCCESS( nn::os::CreateThread( &thread3, &ThreadFuncNetworkConnection, &thread3, g_StackOfThread3, 4 * 1024, 24 ) );
    nn::os::SetThreadName( &thread3, "Thread_3" );
    nn::os::StartThread( &thread3 );

    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );
    nn::os::WaitThread( &thread3 );
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
