﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ResultPrivate.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_CharacterEncoding.h>

#include <algorithm>
#include <cstring>

#include "../Common/nifm_TestUtility.h"

#define NN_NIFM_CONFIG_TEST_PRINT_PROFILE_INFORMATION 1

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    static const nn::util::Uuid NetworkProfileIdInternet       = { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x60 } };
    static const nn::util::Uuid NetworkProfileIdLocalAreaOnly  = { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x61 } };
}

class InternetConncetionConfirmationCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(true);
    }

    static void TearDownTestCase()
    {
        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }
};


TEST_F(InternetConncetionConfirmationCaseTest, InternetSimple)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, LocalAreaOnlySimple)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_SUCCESS(request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultConnectionTestFailed, request1a.GetResult());
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

        // 接続
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultConnectionTestFailed, request1a.GetResult());
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, ConnectionSharing_Prohibited)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, ConnectionSharing_Prohibited_Simultaneously)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::InternetConnectionStatus status;

        {
            nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
            nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestInstant(request1a.GetHandle(), false);
            nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            request1a.Submit();

            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        ASSERT_TRUE(nn::nifm::test::Disconnect());

        {
            nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
            nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestInstant(request1a.GetHandle(), false);
            nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            request1a.Submit();

            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        ASSERT_TRUE(nn::nifm::test::Disconnect());

        {
            nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
            nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestInstant(request1a.GetHandle(), false);
            nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            request1a.Submit();

            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        ASSERT_TRUE(nn::nifm::test::Disconnect());

        {
            nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
            nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestInstant(request1a.GetHandle(), false);
            nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            request1a.Submit();

            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        ASSERT_TRUE(nn::nifm::test::Disconnect());

        {
            nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
            nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestInstant(request1a.GetHandle(), false);
            nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            request1a.Submit();

            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, ConnectionSharing_NotConfirmed_Internet)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, ConnectionSharing_NotConfirmed_LocalAreaOnly)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultConnectionTestHttpCouldntResolveHost, request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultConnectionTestHttpCouldntResolveHost, request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, ConnectionSharing_Confirmed)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        // Prohibited で確立された接続は LocalAreaOnly になって、追加で疎通確認は行われない

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }

    {
        // Prohibited で確立された接続は LocalAreaOnly になって、追加で疎通確認は行われない

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 144);   // 先に出したものより低優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, TakeOver_Prohibited)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

            NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult()); // プロファイル ID が同一な利用要求には相乗り
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

            NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult()); // プロファイル ID が同一な利用要求には相乗り
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);

            NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult()); // プロファイル ID が同一な利用要求には相乗り
                                                              // 疎通確認が行われても Prohibited は却下されない
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);

            NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult()); // プロファイル ID が同一な利用要求には相乗り
                                                              // 疎通確認が行われても Prohibited は却下されない
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_ASSERT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);

            NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult()); // プロファイル ID が同一な利用要求には相乗り
                                                              // 疎通確認が行われても Prohibited は却下されない
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, TakeOver_NotConfirmed_Internet)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, TakeOver_NotConfirmed_LocalAreaOnly)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdLocalAreaOnly);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultConnectionTestFailed, request2a.GetResult());

            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetInternetConnectionStatus(&status));
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdLocalAreaOnly);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultConnectionTestFailed, request2a.GetResult());

            NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetInternetConnectionStatus(&status));
        }
    }
}

TEST_F(InternetConncetionConfirmationCaseTest, TakeOver_Confirmed)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // Dispatch Loop 開始時のスキャンでつながるはず
    //NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    {
        // Prohibited で確立された接続は LocalAreaOnly になって、追加で疎通確認は行われない

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }
    }

    ASSERT_TRUE(nn::nifm::test::Disconnect());

    {
        // Prohibited で確立された接続は LocalAreaOnly になって、追加で疎通確認は行われない

        nn::nifm::Request request1a(nn::nifm::test::RequestParametersInternetGeneric);
        nn::nifm::SetRequestRawPriority(request1a.GetHandle(), 128);
        nn::nifm::SetRequestNetworkProfileId(request1a.GetHandle(), NetworkProfileIdInternet);
        nn::nifm::SetRequestConnectionConfirmationOption(request1a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        // 接続
        ASSERT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request1a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(request1a.GetResult());

        nn::nifm::InternetConnectionStatus status;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
        EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_NotConfirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Preferred);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Required);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }

        {
            nn::nifm::Request request2a(nn::nifm::test::RequestParametersInternetGeneric);
            nn::nifm::SetRequestRawPriority(request2a.GetHandle(), 112);   // 先に出したものより高優先度
            nn::nifm::SetRequestNetworkProfileId(request2a.GetHandle(), NetworkProfileIdInternet);
            nn::nifm::SetRequestConnectionConfirmationOption(request2a.GetHandle(), nn::nifm::ConnectionConfirmationOption_Forced);

            // 接続
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&request2a, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
            NNT_EXPECT_RESULT_SUCCESS(request2a.GetResult());

            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetInternetConnectionStatus(&status));
            EXPECT_EQ(nn::nifm::InternetAvailability_Confirmed, status.internetAvailability);
        }
    }
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
