﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/nifm/nifm_ApiScan.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>

#include <nn/os.h>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;
}

class CompatibilityCaseTest : public ::testing::Test
{
protected:
    virtual void TearDown() { }
};


TEST_F(CompatibilityCaseTest, BasicSuccess)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());

    nn::nifm::SubmitNetworkRequestAndWait();

    EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
    EXPECT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::HandleNetworkRequestResult());

    nn::nifm::CancelNetworkRequest();

    nn::nifm::SetLocalNetworkMode(true);

    nn::nifm::SubmitNetworkRequest();

    while (nn::nifm::IsNetworkRequestOnHold())
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    EXPECT_FALSE(nn::nifm::IsNetworkRequestOnHold());
    EXPECT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_EXPECT_RESULT_SUCCESS(nn::nifm::HandleNetworkRequestResult());

    nn::nifm::SetLocalNetworkMode(false);
}

TEST_F(CompatibilityCaseTest, TemporaryConnectWifi)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

        const int bufferSize = nn::nifm::AccessPointCountMax;
        nn::nifm::AccessPointData accessPointData[bufferSize];
        int32_t outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetScanData(accessPointData, &outCount, bufferSize));
        int count = std::min(outCount, bufferSize);
        for (int i = 0; i < count; ++i)
        {
            nn::nifm::test::PrintAccessPoint(accessPointData[i]);
        }
    }

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        // テンポラリ接続要求
        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnection.GetRequestHandle(), 0));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Required));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        in_addr inAddr;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentPrimaryIpAddress(&inAddr));
        EXPECT_TRUE(inAddr.s_addr != 0);
        NN_LOG("IpAddress: %u.%u.%u.%u\n", (inAddr.s_addr >> 0) & 0xff, (inAddr.s_addr >> 8) & 0xff, (inAddr.s_addr >> 16) & 0xff, (inAddr.s_addr >> 24) & 0xff);

        nn::nifm::AccessPointData accessPointData;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentAccessPoint(&accessPointData));
        EXPECT_EQ(accessPointData.ssid.length, wirelessSetting0.ssidConfig.ssid.length);
        EXPECT_EQ(0, memcmp(accessPointData.ssid.hex, wirelessSetting0.ssidConfig.ssid.hex, accessPointData.ssid.length));
        EXPECT_EQ(accessPointData.authentication, wirelessSetting0.security.authEncryption.authentication);
        EXPECT_EQ(accessPointData.encryption, wirelessSetting0.security.authEncryption.encryption);
        nn::nifm::test::PrintAccessPoint(accessPointData);

        nn::nifm::NetworkProfileData networkProfileData;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData));

        EXPECT_TRUE(wirelessSetting0.ssidConfig.ssid.length == networkProfileData.wirelessSetting.ssidConfig.ssid.length);
        EXPECT_TRUE(memcmp(wirelessSetting0.ssidConfig.ssid.hex, networkProfileData.wirelessSetting.ssidConfig.ssid.hex, wirelessSetting0.ssidConfig.ssid.length) == 0);
        EXPECT_TRUE(wirelessSetting0.ssidConfig.nonBroadcast == networkProfileData.wirelessSetting.ssidConfig.nonBroadcast);
        EXPECT_TRUE(wirelessSetting0.security.authEncryption.authentication == networkProfileData.wirelessSetting.security.authEncryption.authentication);
        EXPECT_TRUE(wirelessSetting0.security.authEncryption.encryption == networkProfileData.wirelessSetting.security.authEncryption.encryption);
        EXPECT_TRUE(memcmp(wirelessSetting0.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.keyMaterial, networkProfileData.wirelessSetting.security.sharedKey.length) == 0);

        EXPECT_TRUE(ipSetting0.ip.isAuto == networkProfileData.ipSetting.ip.isAuto);
        EXPECT_TRUE(memcmp(ipSetting0.ip.ipAddress.data, networkProfileData.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size) == 0);
        EXPECT_TRUE(memcmp(ipSetting0.ip.subnetMask.data, networkProfileData.ipSetting.ip.subnetMask.data, nn::nifm::IpV4Address::Size) == 0);
        EXPECT_TRUE(memcmp(ipSetting0.ip.defaultGateway.data, networkProfileData.ipSetting.ip.defaultGateway.data, nn::nifm::IpV4Address::Size) == 0);
        EXPECT_TRUE(ipSetting0.dns.isAuto == networkProfileData.ipSetting.dns.isAuto);
        EXPECT_TRUE(memcmp(ipSetting0.dns.preferredDns.data, networkProfileData.ipSetting.dns.preferredDns.data, nn::nifm::IpV4Address::Size) == 0);
        EXPECT_TRUE(memcmp(ipSetting0.dns.alternateDns.data, networkProfileData.ipSetting.dns.alternateDns.data, nn::nifm::IpV4Address::Size) == 0);
        EXPECT_TRUE(ipSetting0.proxy.isEnabled == networkProfileData.ipSetting.proxy.isEnabled);
        EXPECT_TRUE(ipSetting0.proxy.port == networkProfileData.ipSetting.proxy.port);
        EXPECT_TRUE(strncmp(ipSetting0.proxy.proxy, networkProfileData.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize) == 0);
        EXPECT_TRUE(ipSetting0.proxy.authentication.isEnabled == networkProfileData.ipSetting.proxy.authentication.isEnabled);
        EXPECT_TRUE(strncmp(ipSetting0.proxy.authentication.username, networkProfileData.ipSetting.proxy.authentication.username, nn::nifm::AuthenticationSetting::UsernameSize) == 0);
        EXPECT_TRUE(strncmp(ipSetting0.proxy.authentication.password, networkProfileData.ipSetting.proxy.authentication.password, nn::nifm::AuthenticationSetting::PasswordSize) == 0);
        EXPECT_TRUE(ipSetting0.mtu == networkProfileData.ipSetting.mtu);
    }
}// NOLINT(impl/function_size)


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
