﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>

#include <nn/util/util_BitUtil.h>

#include <algorithm>
#include <cstring>

#include "../Common/nifm_TestUtility.h"

namespace
{
    class ScanRunner
    {
    private:
        static const size_t StackSize = 32 * 1024;

        nn::os::ThreadType m_Thread;
        char* m_pStackBuffer;
        bool m_IsThreadInitialized;

        nn::Result m_Result;

        static void ThreadFunc(void* p) NN_NOEXCEPT
        {
            reinterpret_cast<ScanRunner*>(p)->ThreadFunc();
        }

        void ThreadFunc() NN_NOEXCEPT
        {
            m_Result = nn::nifm::Scan();
        }

    public:
        ScanRunner() NN_NOEXCEPT
            : m_pStackBuffer(nullptr),
              m_IsThreadInitialized(false)
        {
        }

        ~ScanRunner() NN_NOEXCEPT
        {
            if (m_IsThreadInitialized)
            {
                nn::os::DestroyThread(&m_Thread);
            }

            if (m_pStackBuffer != nullptr)
            {
                delete[] m_pStackBuffer;
            }
        }

        nn::Result Start() NN_NOEXCEPT
        {
            m_pStackBuffer = new char[StackSize + nn::os::ThreadStackAlignment - 1];
            NN_ASSERT_NOT_NULL(m_pStackBuffer);
            void* pStack = reinterpret_cast<void*>(nn::util::align_up(reinterpret_cast<uintptr_t>(m_pStackBuffer), nn::os::ThreadStackAlignment));
            NN_RESULT_DO(nn::os::CreateThread(&m_Thread, ThreadFunc, this, pStack, StackSize, nn::os::DefaultThreadPriority));
            m_IsThreadInitialized = true;
            nn::os::StartThread(&m_Thread);

            NN_RESULT_SUCCESS;
        }

        nn::Result Wait() NN_NOEXCEPT
        {
            nn::os::WaitThread(&m_Thread);

            NN_RESULT_THROW(m_Result);
        }
    };
}

class CancelCaseTest : public ::testing::Test
{
protected:
    virtual void SetUp()
    {
    }
    virtual void TearDown()
    {
        nn::nifm::FinalizeAdminForTest();
    }

    static void TearDownTestCase()
    {
        nn::nifm::Initialize();
        NN_UTIL_SCOPE_EXIT
        {
            nn::nifm::FinalizeForTest();
        };

        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }
};


TEST_F(CancelCaseTest, Scan)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;

    {
        nn::nifm::NetworkConnection networkConnectionCanceller;
        nn::nifm::SetRequestRequirementPreset(networkConnectionCanceller.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet);

        ScanRunner scanRunner;

        // 自律スキャンの末尾に相乗りしないために，一度スキャンを完了させる
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());
        NNT_ASSERT_RESULT_SUCCESS(scanRunner.Start());

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(600));
        networkConnectionCanceller.SubmitRequest();

        nn::os::Tick start = nn::os::GetSystemTick();

        auto result = scanRunner.Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return1: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultLowPriority, result);

        networkConnectionCanceller.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    {
        ScanRunner scanRunner;

        // 自律スキャンの末尾に相乗りしないために，一度スキャンを完了させる
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());
        NNT_ASSERT_RESULT_SUCCESS(scanRunner.Start());

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        nn::nifm::SetWirelessCommunicationEnabled(false);

        nn::os::Tick start = nn::os::GetSystemTick();

        auto result = scanRunner.Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return2: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, result);

        nn::nifm::SetWirelessCommunicationEnabled(true);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

} // NOLINT(impl/function_size)

TEST_F(CancelCaseTest, Connect)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // imatake-wpa2aes
    const nn::nifm::WirelessSettingData wirelessSetting1 = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting1 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile1 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting1
        },
        ipSetting1
    };

    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);

        nn::nifm::NetworkConnection networkConnectionCancellee;
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCancellee.GetRequestHandle(), temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnectionCanceller;
        nn::util::Uuid anotherId = { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x60 } };
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCanceller.GetRequestHandle(), anotherId);
        nn::nifm::SetRequestPriority(networkConnectionCanceller.GetRequestHandle(), nn::nifm::UserRequestPriorityTop);

        networkConnectionCancellee.SubmitRequest();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));   // AP への接続が始まっているが、まだ終わっていないくらい待つ
        networkConnectionCanceller.SubmitRequest();

        nn::os::Tick start = nn::os::GetSystemTick();

        networkConnectionCancellee.GetSystemEvent().Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return1: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultAggregatedRequestModified, networkConnectionCancellee.GetResult());

        networkConnectionCanceller.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);

        nn::nifm::NetworkConnection networkConnectionCancellee;
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCancellee.GetRequestHandle(), temporaryNetworkProfile.GetId());

        networkConnectionCancellee.SubmitRequest();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));   // AP への接続が始まっているが、まだ終わっていないくらい待つ
        nn::nifm::SetWirelessCommunicationEnabled(false);

        nn::os::Tick start = nn::os::GetSystemTick();

        networkConnectionCancellee.GetSystemEvent().Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return2: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, networkConnectionCancellee.GetResult());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

} // NOLINT(impl/function_size)

TEST_F(CancelCaseTest, IpSetting)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // "imatake-openwep104"
    const nn::nifm::WirelessSettingData wirelessSetting3 = {
        {  // ssidConfig
            {  // ssid
                18,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x31,0x30,0x34 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Open,  // authentication
                nn::nifm::Encryption_Wep  // encryption
            },
            {  // sharedKey
                13,  // length
                "WrongWepKeyXX"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting3 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile3 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true, // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting3
        },
        ipSetting3
    };

    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile3);

        nn::nifm::NetworkConnection networkConnectionCancellee;
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCancellee.GetRequestHandle(), temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnectionCanceller;
        nn::util::Uuid anotherId = { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x60 } };
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCanceller.GetRequestHandle(), anotherId);
        nn::nifm::SetRequestPriority(networkConnectionCanceller.GetRequestHandle(), nn::nifm::UserRequestPriorityTop);

        networkConnectionCancellee.SubmitRequest();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10000));     // AP につながって IP 設定でブロックしているくらいまで待つ
        networkConnectionCanceller.SubmitRequest();

        nn::os::Tick start = nn::os::GetSystemTick();

        networkConnectionCancellee.GetSystemEvent().Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return1: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultAggregatedRequestModified, networkConnectionCancellee.GetResult());

        networkConnectionCanceller.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile3);

        nn::nifm::NetworkConnection networkConnectionCancellee;
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCancellee.GetRequestHandle(), temporaryNetworkProfile.GetId());

        networkConnectionCancellee.SubmitRequest();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10000));     // AP につながって IP 設定でブロックしているくらいまで待つ
        nn::nifm::SetWirelessCommunicationEnabled(false);

        nn::os::Tick start = nn::os::GetSystemTick();

        networkConnectionCancellee.GetSystemEvent().Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return2: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, networkConnectionCancellee.GetResult());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

} // NOLINT(impl/function_size)

TEST_F(CancelCaseTest, ConnectionConfirmation)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // imatake-wpa2aes
    const nn::nifm::WirelessSettingData wirelessSetting1 = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting1 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile1 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting1
        },
        ipSetting1
    };

    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);

        nn::nifm::NetworkConnection networkConnectionCancellee;
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCancellee.GetRequestHandle(), temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnectionCanceller;
        nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionCanceller.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Prohibited);
        nn::nifm::SetRequestPriority(networkConnectionCanceller.GetRequestHandle(), nn::nifm::UserRequestPriorityTop);

        networkConnectionCancellee.SubmitRequest();

        // 30秒以内に疎通確認が始まったら即キャンセル
        for (int i = 0; i < 30 * 1000; ++i)
        {
            nn::nifm::InternetConnectionStatus internetConnectionStatus;
            auto result = nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus);

            if (result.IsSuccess() && internetConnectionStatus.internetAvailability == nn::nifm::InternetAvailability_InProgress)
            {
                break;
            }

            if (!(networkConnectionCancellee.GetResult() <= nn::nifm::ResultProcessing()))
            {
                NNT_ASSERT_RESULT_SUCCESS(networkConnectionCancellee.GetResult());
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        networkConnectionCanceller.SubmitRequest();

        nn::os::Tick start = nn::os::GetSystemTick();

        networkConnectionCancellee.GetSystemEvent().Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return1: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultAggregatedRequestModified, networkConnectionCancellee.GetResult());

        networkConnectionCanceller.CancelRequest();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

    {
        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);

        nn::nifm::NetworkConnection networkConnectionCancellee;
        nn::nifm::SetRequestNetworkProfileId(networkConnectionCancellee.GetRequestHandle(), temporaryNetworkProfile.GetId());

        networkConnectionCancellee.SubmitRequest();

        // 30秒以内に疎通確認が始まったら即キャンセル
        for (int i = 0; i < 30 * 1000; ++i)
        {
            nn::nifm::InternetConnectionStatus internetConnectionStatus;
            auto result = nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus);

            if (result.IsSuccess() && internetConnectionStatus.internetAvailability == nn::nifm::InternetAvailability_InProgress)
            {
                break;
            }

            if (!(networkConnectionCancellee.GetResult() <= nn::nifm::ResultProcessing()))
            {
                NNT_ASSERT_RESULT_SUCCESS(networkConnectionCancellee.GetResult());
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        nn::nifm::SetWirelessCommunicationEnabled(false);

        nn::os::Tick start = nn::os::GetSystemTick();

        networkConnectionCancellee.GetSystemEvent().Wait();

        nn::os::Tick end = nn::os::GetSystemTick();

        NN_LOG("Time to return2: %lld ms\n", (end - start).ToTimeSpan().GetMilliSeconds());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkInterfaceNoLongerAvailable, networkConnectionCancellee.GetResult());

        nn::nifm::SetWirelessCommunicationEnabled(true);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }

} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
