﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// テスト用のヘッダ
#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_ErrorResult.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

// テスト対象のヘッダ
#include <../../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_Sbv.h>

namespace {

/**
 * @brief   テストに与えるパラメータです。
 */
enum SbvTestParam
{
    SbvTestParam_UseMalloc               = 0,   //!< デフォルトのアロケータ(std::malloc)を利用する
    SbvTestParam_UseWorkBufAllocator            //!< nn::ngc::detail::WorkBufAllocator を利用する
};

// テストフィクスチャ
class SuccinctSbvTest : public ::testing::TestWithParam<int>
{
protected:
    SuccinctSbvTest() NN_NOEXCEPT : m_HeapSize(128 * 1024), m_HeapAddr(NULL),
        m_Allocator(), m_Set(), m_Sbv(), m_SparseSet() {}

    // ビットベクトルを作成します
    template<typename T>
    void CreateBitVec(T* pBitVec, uint32_t bitVecSize, size_t assertedSize, uint32_t* pAsserted )
    {
        ASSERT_TRUE(pBitVec->Init(bitVecSize));
        ASSERT_EQ(pBitVec->GetBitVectorSize(), bitVecSize);

        for(size_t i = 0; i < assertedSize; ++i)
        {
            ASSERT_TRUE(pBitVec->TurnOn(pAsserted[i]));
            ASSERT_TRUE(pBitVec->Has(pAsserted[i]));
            // TurnOff の確認のため一旦ビットを落とす
            ASSERT_TRUE(pBitVec->TurnOff(pAsserted[i]));
            ASSERT_FALSE(pBitVec->Has(pAsserted[i]));
            ASSERT_TRUE(pBitVec->TurnOn(pAsserted[i]));
            ASSERT_TRUE(pBitVec->Has(pAsserted[i]));
        }
        ASSERT_TRUE(pBitVec->Build());
    }

    // ビットベクトルを作成します (SparseSet)
    void CreateBitVec(nn::ngc::detail::SparseSet* pBitVec, uint64_t bitVecSize, size_t assertedSize, uint64_t* pAsserted)
    {
        ASSERT_TRUE(pBitVec->Init(bitVecSize));
        ASSERT_EQ(pBitVec->GetBitVectorSize(), bitVecSize);

        for (size_t i = 0; i < assertedSize; ++i)
        {
            ASSERT_TRUE(pBitVec->TurnOn(pAsserted[i]));
        }
        ASSERT_TRUE(pBitVec->Build());
    }

    // ビルド済みのビットベクトルに対して Rank/Select のテストを行います
    template<typename T>
    void DoRankSelectTest(const T* pBitVec, uint32_t bitVecSize, size_t assertedSize, uint32_t* pAsserted)
    {
        int zeroCount = 0;      // ビットベクトル内の 0 の数
        for (uint32_t i = 0; i < bitVecSize; ++i)
        {
            int assertedCount = 0;
            bool isFind = false;
            for (uint32_t j = 0; j < assertedSize; ++j)
            {
                if (i >= pAsserted[j])
                {
                    assertedCount++;
                }
                if (i == pAsserted[j])
                {
                    isFind = true;
                }
            }
            EXPECT_EQ(pBitVec->Rank0(i), (i + 1) - assertedCount);
            EXPECT_EQ(pBitVec->Rank1(i), assertedCount);
            if (!isFind)
            {
                EXPECT_EQ(pBitVec->Select0(zeroCount), i);
                zeroCount++;
            }
        }
        // Select1 の確認
        for (size_t i = 0; i < assertedSize; ++i)
        {
            EXPECT_TRUE(pBitVec->Has(pAsserted[i]));
            EXPECT_EQ(pBitVec->Select1(static_cast<uint32_t>(i)), pAsserted[i]);
        }
    }

    // ビルド済みのビットベクトルに対して Rank/Select のテストを行います (SparseSet)
    void DoRankSelectTest(const nn::ngc::detail::SparseSet* pBitVec, uint64_t bitVecSize, size_t assertedSize, uint64_t* pAsserted)
    {
        for (uint64_t i = 0; i < bitVecSize; ++i)
        {
            int assertedCount = 0;
            for (uint64_t j = 0; j < assertedSize; ++j)
            {
                if (i >= pAsserted[j])
                {
                    assertedCount++;
                }
            }
            EXPECT_EQ(pBitVec->Rank0(i), (i + 1) - assertedCount);
            EXPECT_EQ(pBitVec->Rank1(i), assertedCount);
        }
        // Select1 の確認
        for (size_t i = 0; i < assertedSize; ++i)
        {
            EXPECT_TRUE(pBitVec->Has(pAsserted[i]));
            EXPECT_EQ(pBitVec->Select1(static_cast<uint32_t>(i)), pAsserted[i]);
            EXPECT_EQ(pBitVec->Select1Ex(static_cast<uint32_t>(i)), pAsserted[i]);
        }
    }

    // ムーブに対するチェックをします
    template<typename T>
    void DoMoveTest(T& bitVec, uint32_t bitVecSize, size_t assertedSize, uint32_t* pAsserted)
    {

        // ムーブ(C++11) の確認
        T bitVec2 = std::move(bitVec);
        DoRankSelectTest(&bitVec2, bitVecSize, assertedSize, pAsserted);
        T bitVec3(std::move(bitVec2));
        DoRankSelectTest(&bitVec3, bitVecSize, assertedSize, pAsserted);

        bitVec = std::move(bitVec3);    // 元に戻す
    }

    // ムーブに対するチェックをします (SparseSet)
    void DoMoveTest(nn::ngc::detail::SparseSet& bitVec, uint64_t bitVecSize, size_t assertedSize, uint64_t* pAsserted)
    {
        // ムーブ(C++11) の確認
        nn::ngc::detail::SparseSet bitVec2(std::move(bitVec));
        DoRankSelectTest(&bitVec2, bitVecSize, assertedSize, pAsserted);
        bitVec = std::move(bitVec2);
        DoRankSelectTest(&bitVec, bitVecSize, assertedSize, pAsserted);
    }

    // Reset() に対するチェックをします
    template<typename T, typename U>
    void DoResetTest(T& outBitVec, U bitVecSize, size_t assertedSize, U* pAsserted)
    {
        outBitVec.Reset();
        ASSERT_TRUE(outBitVec.Init(bitVecSize));
        ASSERT_TRUE(outBitVec.Build());
        EXPECT_EQ(outBitVec.Select1(0), -1);
        for (size_t i = 0; i < assertedSize; ++i)
        {
            EXPECT_FALSE(outBitVec.Has(pAsserted[i]));
        }
    }

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        if (GetParam() == SbvTestParam_UseWorkBufAllocator)
        {
            m_HeapAddr = std::malloc(m_HeapSize);
            NN_ABORT_UNLESS(m_HeapAddr);
            m_Allocator.Initialize(m_HeapAddr, m_HeapSize);
            m_Set.SetAllocator(&m_Allocator);
            m_Sbv.SetAllocator(&m_Allocator);
            m_SparseSet.SetAllocator(&m_Allocator);
        }
    }
    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        if (GetParam() == SbvTestParam_UseWorkBufAllocator)
        {
            m_Set.ReleaseAllocator();
            m_Sbv.ReleaseAllocator();
            m_SparseSet.ReleaseAllocator();
            m_Allocator.Finalize();
            std::free(m_HeapAddr);
        }
    }

protected:
    const size_t m_HeapSize;
    void* m_HeapAddr;                               // ワークバッファ用メモリ
    nn::ngc::detail::WorkBufAllocator m_Allocator;  // ワークバッファ用アロケータ
    nn::ngc::detail::Set m_Set;
    nn::ngc::detail::Sbv m_Sbv;
    nn::ngc::detail::SparseSet m_SparseSet;
};

}   // namespace

/**
 * @brief   Set の API 機能を確認します
 */
TEST_P(SuccinctSbvTest, SetSimple)
{
    const uint32_t bitVecSize = 64;                 // 作成するビットベクトルの長さ
    const size_t assertedSize = 2;                  // 立てるビットの数
    uint32_t pAsserted[assertedSize] = { 1, 22 };   // 立てるビットの位置

    // ビットベクトルの作成
    CreateBitVec(&m_Set, bitVecSize, assertedSize, pAsserted);

    // 作成したビットベクトルに対する Rank/Select の確認
    DoRankSelectTest(&m_Set, bitVecSize, assertedSize, pAsserted);

    // ムーブの確認
    DoMoveTest(m_Set, bitVecSize, assertedSize, pAsserted);

    // Reset の確認
    DoResetTest(m_Set, bitVecSize, assertedSize, pAsserted);
}

/**
 * @brief   Sbv の API 機能を確認します
 */
TEST_P( SuccinctSbvTest, SbvSimple )
{
    const uint32_t bitVecSize = 64;                 // 作成するビットベクトルの長さ
    const size_t assertedSize = 2;                  // 立てるビットの数
    uint32_t pAsserted[assertedSize] = { 1, 22 };   // 立てるビットの位置

    // ビットベクトルの作成
    CreateBitVec(&m_Sbv, bitVecSize, assertedSize, pAsserted);

    // 作成したビットベクトルに対する Rank/Select の確認
    DoRankSelectTest(&m_Sbv, bitVecSize, assertedSize, pAsserted);

    // ムーブの確認
    DoMoveTest(m_Sbv, bitVecSize, assertedSize, pAsserted);

    // Reset の確認
    DoResetTest(m_Sbv, bitVecSize, assertedSize, pAsserted);
}

/**
 * @brief   SparseSet の API 機能を確認します
 */
TEST_P( SuccinctSbvTest, SparseSetSimple )
{
    const uint64_t bitVecSize = 64;                 // 作成するビットベクトルの長さ
    const size_t assertedSize = 2;                  // 立てるビットの数
    uint64_t pAsserted[assertedSize] = { 1, 22 };   // 立てるビットの位置

    // ビットベクトルの作成
    CreateBitVec(&m_SparseSet, bitVecSize, assertedSize, pAsserted);

    // 作成したビットベクトルに対する Rank/Select の確認
    DoRankSelectTest(&m_SparseSet, bitVecSize, assertedSize, pAsserted);

    // ムーブの確認
    DoMoveTest(m_SparseSet, bitVecSize, assertedSize, pAsserted);

    // Reset の確認
    DoResetTest(m_SparseSet, bitVecSize, assertedSize, pAsserted);
}

/**
 * @brief   SparseSet で入力できる最大値を使って API 機能を確認します
 */
TEST_P(SuccinctSbvTest, SparseSetMaxValue)
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    if (GetParam() == SbvTestParam_UseMalloc)   // WorkBufAllocator の場合メモリが不足する(17 GB くらい消費)
    {
        const uint64_t bitVecSize = 0x2000000000ULL - 1;                // 作成するビットベクトルの長さ
        const size_t assertedSize = 2;                                  // 立てるビットの数
        uint64_t pAsserted[assertedSize] = { 32, 0x200000000ULL - 2 };  // 立てるビットの位置

        // ビットベクトルの作成
        bool result = m_SparseSet.Init(bitVecSize);
        if (result)
        {
            // Init 成功時のみテストする
            // Init 失敗時は、マシンのメモリが不足している可能性が高い
            ASSERT_EQ(m_SparseSet.GetBitVectorSize(), bitVecSize);

            for (size_t i = 0; i < assertedSize; ++i)
            {
                ASSERT_TRUE(m_SparseSet.TurnOn(pAsserted[i]));
            }
            ASSERT_TRUE(m_SparseSet.Build());

            // Select1Ex の確認
            for (size_t i = 0; i < assertedSize; ++i)
            {
                EXPECT_TRUE(m_SparseSet.Has(pAsserted[i]));
                EXPECT_EQ(pAsserted[i], static_cast<uint64_t>(m_SparseSet.Select1Ex(static_cast<uint32_t>(i))));
            }
        }
    }
#endif
}

INSTANTIATE_TEST_CASE_P(SwitchAllocator,
                        SuccinctSbvTest,
                        testing::Values(SbvTestParam_UseMalloc,
                                        SbvTestParam_UseWorkBufAllocator));
