﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/hid.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNpt_Common.h>

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NptAuto : public nnt::npt::TestFramework
{
protected:

    NptAuto() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::npt::InitializeHidController();
        // API呼び出しログ出力フラグ
        //nnt::npt::wrapper::SetApiCallLoggingMode(nnt::npt::wrapper::LogMode_Aging);
    }

    ~NptAuto() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::npt::wrapper::GetState() == nn::nfc::State_Init)
        {
            nnt::npt::FinalizeSystem();
        }
    }
};

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

//================================================================================
// テストケースの実装です。
//================================================================================
TEST_F(NptAuto, TestCaseGoodOperationNone)
{
    // 状態がNoneの時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationNone start\n");

    // ----------------------------------------
    // テスト対象 : nn::nfc::GetState()
    //              nn::nfc::Initialize()
    //              nn::nfc::GetDeviceState()
    // ----------------------------------------
    EXPECT_EQ(nn::nfc::State_None, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Unexpected, nnt::npt::wrapper::GetDeviceState());
    nnt::npt::InitializeSystem();
    // Initialize()は戻り値がないので、状態がInitに変更していることを確認する
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
}

TEST_F(NptAuto, TestCaseGoodOperationInit)
{
    // 状態がInitの時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationInit start\n");

    // 状態をInitに遷移させる
    nnt::npt::wrapper::Initialize();

    // ----------------------------------------
    // テスト対象 : nn::nfc::GetState()
    //              nn::nfc::Finalize()
    //              nn::nfc::GetDeviceState()
    //              nn::nfc::AttachAvailabilityChangeEvent()
    // ----------------------------------------
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    nn::nfc::DeviceState deviceState;
    deviceState = nnt::npt::wrapper::GetDeviceState();
    if (!(deviceState == nn::nfc::DeviceState_Unexpected ||
            deviceState == nn::nfc::DeviceState_Init))
    {
        EXPECT_TRUE(false);
        NN_LOG("TestCaseGoodOperationDeviceUnexpected: State Error %d", deviceState);
    }
    nn::os::SystemEventType availabilityChangeEvent;
    nnt::npt::wrapper::AttachAvailabilityChangeEvent(&availabilityChangeEvent);
    nnt::npt::wrapper::Finalize();
    // Finalizeは戻り値がないので、状態がNoneに変更していることを確認する
    EXPECT_EQ(nn::nfc::State_None, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Unexpected, nnt::npt::wrapper::GetDeviceState());
}

TEST_F(NptAuto, TestCaseGoodOperationDeviceUnexpected)
{
    // デバイス状態がUnexpected時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceUnexpected start\n");

    // 状態をInitに遷移させる
    nnt::npt::InitializeSystem();

    // ----------------------------------------
    // テスト対象 : nn::nfc::GetDeviceState()
    //              nn::nfc::ListDevices()
    // ----------------------------------------
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    nn::nfc::DeviceState deviceState;
    deviceState = nnt::npt::wrapper::GetDeviceState();
    if (!(deviceState == nn::nfc::DeviceState_Unexpected ||
            deviceState == nn::nfc::DeviceState_Init))
    {
        EXPECT_TRUE(false);
        NN_LOG("TestCaseGoodOperationDeviceUnexpected: State Error %d", deviceState);
    }
    int outCount;
    nn::nfc::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::npt::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
    nnt::npt::SetCurrentDeviceHandle(currentDeviceHandle);
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::npt::wrapper::GetDeviceState());
}

TEST_F(NptAuto, TestCaseGoodOperationDeviceInit)
{
    // デバイス状態がInit時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceUnexpected start\n");

    // 状態をInitに遷移させる
    nnt::npt::DoSearchDeviceHandle();

    // ----------------------------------------
    // テスト対象 : nn::nfc::GetNpadId()
    //              nn::nfc::GetDeviceState()
    //              nn::nfc::StartDetection()
    //              nn::nfc::AttachActivateEvent()
    //              nn::nfc::AttachDeactivateEvent()
    //              nn::nfc::ListDevices()
    // ----------------------------------------
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::npt::wrapper::GetDeviceState());
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::GetNpadId(&npadId));
    nnt::npt::CheckNPadId(npadId);
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StartDetection());
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::npt::wrapper::GetDeviceState());
    int outCount;
    nn::nfc::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::npt::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
}

TEST_F(NptAuto, TestCaseGoodOperationDeviceActive)
{
    // デバイス状態がActive時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceActive start\n");

    // 状態をDeviceActiveに遷移させる
    nnt::npt::DoActivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::StopDetection()
    //              nn::nfc::GetDeviceState()
    //              nn::nfc::GetNpadId()
    //              nn::nfc::GetTagInfo()
    //              nn::nfc::SendCommandByPassThrough()
    //              nn::nfc::KeepPassThroughSession()
    //              nn::nfc::ListDevices()
    //              nn::nfc::AttachActivateEvent()
    //              nn::nfc::AttachDeactivateEvent()
    // ----------------------------------------
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::npt::wrapper::GetDeviceState());
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::GetNpadId(&npadId));
    nnt::npt::CheckNPadId(npadId);
    nn::nfc::TagInfo tagInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::GetTagInfo(&tagInfo));
    // タグタイプがType2となっているか確認
    EXPECT_TRUE(tagInfo.type == nn::nfc::TagType::TagType_Type2);
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);
    // 送信データ(READ コマンド)
    nn::Bit8 sendReadCommandData[2];
    // READ コマンドに対するレスポンスデータ
    size_t sendReadCommandResponseDataSize;
    nn::Bit8 sendReadCommandResponseData[16];
    // 送信データ(WRITE コマンド)
    nn::Bit8 sendWriteCommandData[6];
    // WRITE コマンドに対するレスポンスデータ
    size_t sendWriteCommandResponseDataSize;
    nn::Bit8 sendWriteCommandResponseData[1];
    nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
    nnt::npt::PrepareSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    nnt::npt::PrintSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::SendCommandByPassThrough(sendWriteCommandResponseData, &sendWriteCommandResponseDataSize,
                                                             sendWriteCommandData, sizeof(sendWriteCommandData),
                                                             sizeof(sendWriteCommandResponseData), timeout));
    nnt::npt::PrepareSendReadCommandData(sendReadCommandData, sizeof(sendReadCommandData));
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::SendCommandByPassThrough(sendReadCommandResponseData, &sendReadCommandResponseDataSize,
                                                             sendReadCommandData, sizeof(sendReadCommandData),
                                                             sizeof(sendReadCommandResponseData), timeout));
    nnt::npt::PrintSendReadCommandResponseData(sendReadCommandResponseData, sendReadCommandResponseDataSize);
    // 書き込んだデータを読み込めたか確認
    nnt::npt::CheckReadData(sendReadCommandResponseData, sendReadCommandResponseDataSize,
                            sendWriteCommandData, sizeof(sendWriteCommandData));
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::KeepPassThroughSession());
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::ReleasePassThroughSession());
    NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StopDetection());
    EXPECT_EQ(nn::nfc::State_Init, nnt::npt::wrapper::GetState());
    EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::npt::wrapper::GetDeviceState());
    int outCount;
    nn::nfc::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::npt::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
}

TEST_F(NptAuto, TestCaseInvalidOperationNoneDeathTest)
{
    // 状態がNoneの時に呼び出すことができないAPIのテストをします。
    // WinのDeathTestで実施します。
    NN_LOG("TestCaseInvalidOperationNoneDeathTest start\n");
#if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    // ----------------------------------------
    // テスト対象 : nn::nfc::Finalize()
    //            : nn::nfc::AttachActivateEvent()
    //            : nn::nfc::AttachDeactivateEvent()
    //            : nn::nfc::GetNpadId()
    //            : nn::nfc::ListDevices()
    //            : nn::nfc::StartDetection()
    //            : nn::nfc::StopDetection()
    //            : nn::nfc::SendCommandByPassThrough()
    //            : nn::nfc::KeepPassThroughSession()
    //            : nn::nfc::ReleasePassThroughSession()
    //            : nn::nfc::GetTagInfo()
    // ----------------------------------------
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::Finalize());
    int outCount;
    nn::nfc::DeviceHandle currentDeviceHandle;
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::AttachActivateEvent(&activateEvent));
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::hid::NpadIdType npadId;
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::GetNpadId(&npadId));
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::StartDetection());
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::StopDetection());
    // 送信データ(WRITE コマンド)
    nn::Bit8 sendWriteCommandData[6];
    // WRITE コマンドに対するレスポンスデータ
    size_t sendWriteCommandResponseDataSize;
    nn::Bit8 sendWriteCommandResponseData[1];
    nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
    nnt::npt::PrepareSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    nnt::npt::PrintSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::SendCommandByPassThrough(sendWriteCommandResponseData, &sendWriteCommandResponseDataSize,
                                                        sendWriteCommandData, sizeof(sendWriteCommandData),
                                                        sizeof(sendWriteCommandResponseData), timeout));
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::KeepPassThroughSession());
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::ReleasePassThroughSession());
    nn::nfc::TagInfo tagInfo;
    NNT_NPT_ASSERT_DEATH(nnt::npt::wrapper::GetTagInfo(&tagInfo));
#endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
}

TEST_F(NptAuto, TestCaseInvalidOperationInitDeathTest)
{
    // 状態がInitの時に呼び出すことができないAPIのテストをします。
    // WinのDeathTestで実施します。
    NN_LOG("TestCaseInvalidOperationNoneDeathTest start\n");
#if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    // Init状態に遷移させる
    nnt::npt::InitializeSystem();

    // ----------------------------------------
    // テスト対象 : nn::nfc::Initialize()
    // ----------------------------------------
    NNT_NPT_ASSERT_DEATH(nnt::npt::InitializeSystem());
#endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
}

TEST_F(NptAuto, TestCaseInvalidOperationDeviceUnexpected)
{
    // デバイス状態がUnexpected時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceUnexpected start\n");

    // 状態をInitに遷移させる、デバイス状態はUnexpectedのまま
    nnt::npt::InitializeSystem();

    // 連続して試験を行うとListdevicesで取得しなくても
    // 有効なデバイスハンドルを設定すると成功してしまう仕様なので(SIGLO-29745)
    // ここでは無効なデバイスハンドルを使用するようにします。
    nn::nfc::DeviceHandle invalidDeviceHandle = nnt::npt::GetCurrentDeviceHandle();
    invalidDeviceHandle._id = 0xFFFFFFFF;
    // ----------------------------------------
    // テスト対象 : nn::nfc::GetNpadId()
    //              nn::nfc::StartDetection()
    //              nn::nfc::StopDetection()
    //              nn::nfc::SendCommandByPassThrough()
    //              nn::nfc::KeepPassThroughSession()
    //              nn::nfc::ReleasePassThroughSession()
    //              nn::nfc::GetTagInfo()
    //              nn::nfc::AttachActivateEvent()
    //              nn::nfc::AttachDeactivateEvent()
    // ----------------------------------------
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
            nnt::npt::wrapper::GetNpadId(&npadId, invalidDeviceHandle));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
            nnt::npt::wrapper::StartDetection(invalidDeviceHandle));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
            nnt::npt::wrapper::StopDetection(invalidDeviceHandle));
    // 送信データ(WRITE コマンド)
    nn::Bit8 sendWriteCommandData[6];
    // WRITE コマンドに対するレスポンスデータ
    size_t sendWriteCommandResponseDataSize;
    nn::Bit8 sendWriteCommandResponseData[1];
    nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
    nnt::npt::PrepareSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    nnt::npt::PrintSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
                              nnt::npt::wrapper::SendCommandByPassThrough(sendWriteCommandResponseData, &sendWriteCommandResponseDataSize,
                                                             sendWriteCommandData, sizeof(sendWriteCommandData),
                                                             sizeof(sendWriteCommandResponseData), timeout));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
                              nnt::npt::wrapper::KeepPassThroughSession());
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
                              nnt::npt::wrapper::ReleasePassThroughSession());
    nn::nfc::TagInfo tagInfo;
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
            nnt::npt::wrapper::GetTagInfo(&tagInfo));
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
            nnt::npt::wrapper::AttachActivateEvent(&activateEvent, invalidDeviceHandle));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultNfcDeviceNotFound,
            nnt::npt::wrapper::AttachDeactivateEvent(&deactivateEvent, invalidDeviceHandle));
}

TEST_F(NptAuto, TestCaseInvalidOperationDeviceInit)
{
    // デバイス状態がInit時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceInit start\n");

    // 状態をInitに遷移させる
    nnt::npt::DoSearchDeviceHandle();

    // ----------------------------------------
    // テスト対象 : nn::nfc::StopDetection()
    //              nn::nfc::SendCommandByPassThrough()
    //              nn::nfc::KeepPassThroughSession()
    //              nn::nfc::ReleasePassThroughSession()
    //              nn::nfc::GetTagInfo()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
            nnt::npt::wrapper::StopDetection());
    // 送信データ(WRITE コマンド)
    nn::Bit8 sendWriteCommandData[6];
    // WRITE コマンドに対するレスポンスデータ
    size_t sendWriteCommandResponseDataSize;
    nn::Bit8 sendWriteCommandResponseData[1];
    nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
    nnt::npt::PrepareSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    nnt::npt::PrintSendWriteCommandData(sendWriteCommandData, sizeof(sendWriteCommandData));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
                              nnt::npt::wrapper::SendCommandByPassThrough(sendWriteCommandResponseData, &sendWriteCommandResponseDataSize,
                                                             sendWriteCommandData, sizeof(sendWriteCommandData),
                                                             sizeof(sendWriteCommandResponseData), timeout));
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
                              nnt::npt::wrapper::KeepPassThroughSession());
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
                              nnt::npt::wrapper::ReleasePassThroughSession());
    nn::nfc::TagInfo tagInfo;
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
            nnt::npt::wrapper::GetTagInfo(&tagInfo));
}

TEST_F(NptAuto, TestCaseInvalidOperationDeviceActive)
{
    // デバイス状態がActive時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceActive start\n");

    // デバイス状態をActiveに遷移させる
    nnt::npt::DoActivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::StartDetection()
    //              nn::nfc::ReleasePassThroughSession()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
            nnt::npt::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
                              nnt::npt::wrapper::ReleasePassThroughSession());
}

TEST_F(NptAuto, TestCaseInvalidOperationDeviceKeep)
{
    // デバイス状態がKeep時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceKeep start\n");

    // デバイス状態をKeepに遷移させる
    nnt::npt::DoKeep();

    // ----------------------------------------
    // テスト対象 : nn::nfc::StartDetection()
    //              nn::nfc::ReleasePassThroughSession()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
            nnt::npt::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfc::ResultInvalidDeviceState,
                              nnt::npt::wrapper::KeepPassThroughSession());
}
