﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>
#include <nnt/nfp/testMifare_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    enum WaitTime
    {
        WaitTime_PowerOnOff             = 10000,
        WaitTime_TagSwitch              = 3000,
        WaitTime_Connect2Devices        = 10000,
    };

    enum InvalidHandleType
    {
        InvalidHandleType_MissHandle,
        InvalidHandleType_PowerOff,
    };

    const uint32_t AccessId = nnt::nfp::NfpManagerUsedTagId;
} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpManualNx : public nnt::nfp::TestFramework
{
protected:
    InvalidHandleType m_InvalidHandleType;

    NfpManualNx() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
        m_InvalidHandleType = InvalidHandleType_MissHandle;
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init)
        {
            nnt::nfp::FinalizeSystem();
        }
    }

    void SetInvalidDeviceHandleType(InvalidHandleType type) NN_NOEXCEPT
    {
        m_InvalidHandleType = type;
    }

    void WaitNextTest() NN_NOEXCEPT
    {
        if(m_InvalidHandleType == InvalidHandleType_PowerOff)
        {
            //次のテストの為にNFCデバイスの電源をONするメッセージを表示して数秒待つ
            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE NFC DEVICE POWER ON (Within %dms) ...     \n",WaitTime_PowerOnOff);
            NN_LOG("==================================================\n");
            nnt::nfp::Sleep(WaitTime_PowerOnOff);
        }
    }

    nn::nfp::DeviceHandle GetInvalidDeviceHandle() NN_NOEXCEPT
    {
        nn::nfp::DeviceHandle invalidDeviceHandle;

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SearchDeviceHandle());
        invalidDeviceHandle = nnt::nfp::GetCurrentDeviceHandle();
        if(m_InvalidHandleType == InvalidHandleType_MissHandle)
        {
            invalidDeviceHandle._id =0xFFFFFFFF;
        }
        else
        {
            //NFCデバイスの電源をOFFするメッセージを表示して数秒待つ
            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE NFC DEVICE POWER OFF (Within %dms) ...     \n",WaitTime_PowerOnOff);
            NN_LOG("==================================================\n");
            nnt::nfp::Sleep(WaitTime_PowerOnOff);
        }

        return invalidDeviceHandle;
    }

    void DoTestInvalidDeviceHandleWithStateInit() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        nnt::nfp::DoSearchDeviceHandle();
        EXPECT_EQ(nn::nfp::DeviceState_Init,nnt::nfp::wrapper::GetDeviceState());
        EXPECT_EQ(nn::nfp::DeviceState_Unexpected,
                  nnt::nfp::wrapper::GetDeviceState(GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoSearchDeviceHandle();
        EXPECT_EQ(nn::nfp::DeviceState_Init,nnt::nfp::wrapper::GetDeviceState());
        nn::os::SystemEventType event;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::AttachActivateEvent(&event,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoSearchDeviceHandle();
        EXPECT_EQ(nn::nfp::DeviceState_Init,nnt::nfp::wrapper::GetDeviceState());
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::AttachDeactivateEvent(&event,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoSearchDeviceHandle();
        EXPECT_EQ(nn::nfp::DeviceState_Init,nnt::nfp::wrapper::GetDeviceState());
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::StartDetection(GetInvalidDeviceHandle()));

        WaitNextTest();

        nn::hid::NpadIdType npadId;
        nnt::nfp::DoSearchDeviceHandle();
        EXPECT_EQ(nn::nfp::DeviceState_Init,nnt::nfp::wrapper::GetDeviceState());
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetNpadId(&npadId, GetInvalidDeviceHandle()));

        WaitNextTest();
    }

    void DoTestInvalidDeviceHandleWithStateActive() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::StopDetection(GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::Format(GetInvalidDeviceHandle()));

#if defined(NNT_NFP_LIB_VERSION_ALPHA)
        WaitNextTest();

        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::Restore(GetInvalidDeviceHandle()));
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)

        WaitNextTest();

        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::Mount(GetInvalidDeviceHandle(),
                                         nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_All));

        WaitNextTest();

        nnt::nfp::DoActivate();
        nn::hid::NpadIdType npadId;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetNpadId(&npadId, GetInvalidDeviceHandle()));

        WaitNextTest();
    }

    void DoTestInvalidDeviceHandleWithStateMount() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
        std::memcpy(tagData,nnt::nfp::NormalTagData,sizeof(nnt::nfp::NormalTagData));
        nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { AccessId,
                                                           sizeof(nnt::nfp::NormalTagData),
                                                           tagData };

        nnt::nfp::DoMount();
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::Unmount(GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoMount();
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::CreateApplicationArea(GetInvalidDeviceHandle(),appAreaInfo));

        WaitNextTest();

        nnt::nfp::DoMount();
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::OpenApplicationArea(GetInvalidDeviceHandle(),AccessId));

        WaitNextTest();

        nnt::nfp::DoMount();
        nn::nfp::ModelInfo modelInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetModelInfo(&modelInfo,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoMount();
        nn::hid::NpadIdType npadId;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetNpadId(&npadId, GetInvalidDeviceHandle()));

        WaitNextTest();
    }

    void DoTestInvalidDeviceHandleWithStateMountOpened() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};

        nnt::nfp::DoOpenApplicationArea(AccessId);
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetApplicationArea(tagData,GetInvalidDeviceHandle(),sizeof(tagData)));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::SetApplicationArea(GetInvalidDeviceHandle(),tagData,sizeof(tagData)));

        WaitNextTest();

        nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { AccessId,
                                                           sizeof(tagData),
                                                           tagData };
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::RecreateApplicationArea(GetInvalidDeviceHandle(),appAreaInfo));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::Flush(GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        nn::nfp::TagInfo tagInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetTagInfo(&tagInfo,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        nn::nfp::RegisterInfo registerInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetRegisterInfo(&registerInfo,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        nn::nfp::RegisterInfoPrivate registerInfoPrivate;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetRegisterInfo(&registerInfoPrivate,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        nn::nfp::CommonInfo commonInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetCommonInfo(&commonInfo,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        nn::nfp::AdminInfo adminInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetAdminInfo(&adminInfo,GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        nn::nfp::RegisterInfoPrivate registerInfoSet;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::SetRegisterInfo(GetInvalidDeviceHandle(),registerInfoSet));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::DeleteRegisterInfo(GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::DeleteApplicationArea(GetInvalidDeviceHandle()));

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        bool outValue = false;
        NNT_EXPECT_RESULT_FAILURE(
            nn::nfp::ResultNfcDeviceNotFound,
            nnt::nfp::wrapper::ExistsApplicationArea(&outValue, GetInvalidDeviceHandle()));
        EXPECT_EQ(false, outValue);

        WaitNextTest();

        nnt::nfp::DoOpenApplicationArea(AccessId);
        nn::hid::NpadIdType npadId;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::ResultNfcDeviceNotFound,
                nnt::nfp::wrapper::GetNpadId(&npadId, GetInvalidDeviceHandle()));

        WaitNextTest();
    }

    void DoTestInvalidDeviceHandle() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        // INIT 状態に移行します。
        NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());

        NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());

        DoTestInvalidDeviceHandleWithStateInit();

        DoTestInvalidDeviceHandleWithStateActive();

        DoTestInvalidDeviceHandleWithStateMount();

        DoTestInvalidDeviceHandleWithStateMountOpened();
    }

    void DoTestAccessibleRamArea() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        //GetTagInfo()はRam指定でマウントしてもアクセス可能であることを確認
        nn::nfp::TagInfo tagInfo;
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetTagInfo(&tagInfo,nnt::nfp::GetCurrentDeviceHandle()));

        //Ram領域のデータにアクセスできることを確認

        nn::nfp::RegisterInfo registerInfo;
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetRegisterInfo(&registerInfo,nnt::nfp::GetCurrentDeviceHandle()));

        nn::nfp::RegisterInfoPrivate registerInfoPrivate;
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetRegisterInfo(&registerInfoPrivate,nnt::nfp::GetCurrentDeviceHandle()));

        nn::nfp::CommonInfo commonInfo;
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetCommonInfo(&commonInfo,nnt::nfp::GetCurrentDeviceHandle()));

        nn::nfp::AdminInfo adminInfo;
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetAdminInfo(&adminInfo,nnt::nfp::GetCurrentDeviceHandle()));

        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::GetCurrentDeviceHandle(),
                                                       AccessId));

        nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetApplicationArea(tagData,nnt::nfp::GetCurrentDeviceHandle(),
                                                      sizeof(tagData)));
    }

    void DoTestAccessibleRomArea() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        //GetTagInfo()はRom指定でマウントしてもアクセス可能であることを確認
        nn::nfp::TagInfo tagInfo;
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetTagInfo(&tagInfo,nnt::nfp::GetCurrentDeviceHandle()));

        //Rom領域のデータにアクセスできることを確認
        nn::nfp::ModelInfo modelInfo;
        NNT_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetModelInfo(&modelInfo,nnt::nfp::GetCurrentDeviceHandle()));
    }

    void DoTestInaccessibleRamArea() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        //Ram領域のデータにアクセスできないことを確認

        nn::nfp::RegisterInfo registerInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                nnt::nfp::wrapper::GetRegisterInfo(&registerInfo,nnt::nfp::GetCurrentDeviceHandle()));

        nn::nfp::RegisterInfoPrivate registerInfoPrivate;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                nnt::nfp::wrapper::GetRegisterInfo(&registerInfoPrivate,nnt::nfp::GetCurrentDeviceHandle()));

        nn::nfp::CommonInfo commonInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                nnt::nfp::wrapper::GetCommonInfo(&commonInfo,nnt::nfp::GetCurrentDeviceHandle()));

        nn::nfp::AdminInfo adminInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                nnt::nfp::wrapper::GetAdminInfo(&adminInfo,nnt::nfp::GetCurrentDeviceHandle()));

        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::GetCurrentDeviceHandle(),
                                                       AccessId));
    }

    void DoTestInaccessibleRomArea() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        //Rom領域のデータにアクセスできないことを確認

        nn::nfp::ModelInfo modelInfo;
        NNT_EXPECT_RESULT_FAILURE(
                nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                nnt::nfp::wrapper::GetModelInfo(&modelInfo,nnt::nfp::GetCurrentDeviceHandle()));
    }

    void DoTestAvailabilityChangeEventInInit() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        //NFC デバイス OFF を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE NFC DEVICE POWER OFF(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
        {
            NN_LOG("WifiOff\n");
            nnt::nfp::WifiOff();
            EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

            NN_LOG("WifiOn\n");
            nnt::nfp::WifiOn();
            EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

            //スリープへの遷移を促すメッセージを表示して数秒待つ
            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE SLEEP -> WAKE UP -> RESUME(Within %dms) ...     \n",10000);
            NN_LOG("==================================================\n");
            EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
        }

        //NFC デバイス ON を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE NFC DEVICE POWER ON(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
        {
            NN_LOG("WifiOff\n");
            nnt::nfp::WifiOff();
            EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

            NN_LOG("WifiOn\n");
            nnt::nfp::WifiOn();
            EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

            //スリープへの遷移を促すメッセージを表示して数秒待つ
            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE SLEEP -> WAKE UP -> RESUME(Within %dms) ...     \n",10000);
            NN_LOG("==================================================\n");
            EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
        }

        //NFC デバイス ON を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE NFC DEVICE POWER ON(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
    }

    void DoTestAvailabilityChangeEventInSearch() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        nnt::nfp::DoSearch();

        NN_LOG("WifiOff\n");
        nnt::nfp::WifiOff();
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        NN_LOG("WifiOn\n");
        nnt::nfp::WifiOn();
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        nnt::nfp::DoSearch();

        //スリープへの遷移を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE SLEEP -> WAKE UP -> RESUME(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        //NFC デバイス ON を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE NFC DEVICE POWER ON(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
    }

    void DoTestAvailabilityChangeEventInActive() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH AMIIBO TAG (Within %dms) ...    \n",
               WaitTime_TagSwitch);
        NN_LOG("========================================================================\n");
        nnt::nfp::Sleep(WaitTime_TagSwitch);

        nnt::nfp::DoActivate();

        NN_LOG("WifiOff\n");
        nnt::nfp::WifiOff();
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        NN_LOG("WifiOn\n");
        nnt::nfp::WifiOn();
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        nnt::nfp::DoActivate();

        //スリープへの遷移を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE SLEEP -> WAKE UP -> RESUME(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        //NFC デバイス ON を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE NFC DEVICE POWER ON(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
    }

    void DoTestAvailabilityChangeEventInMount() NN_NOEXCEPT
    {
        NN_LOG("%s start\n",NN_CURRENT_FUNCTION_NAME);

        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH AMIIBO TAG (Within %dms) ...    \n",
               WaitTime_TagSwitch);
        NN_LOG("========================================================================\n");
        nnt::nfp::Sleep(WaitTime_TagSwitch);

        nnt::nfp::DoMount();

        NN_LOG("WifiOff\n");
        nnt::nfp::WifiOff();
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        NN_LOG("WifiOn\n");
        nnt::nfp::WifiOn();
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        nnt::nfp::DoMount();

        //スリープへの遷移を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE SLEEP -> WAKE UP -> RESUME(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));

        //NFC デバイス ON を促すメッセージを表示して数秒待つ
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE NFC DEVICE POWER ON(Within %dms) ...     \n",10000);
        NN_LOG("==================================================\n");
        EXPECT_TRUE(nnt::nfp::WaitForAvailabilityChange(10000));
    }
};

//================================================================================
// テストケースの実装です。
//================================================================================

TEST_F(NfpManualNx,TestCaseInvalidDeviceHandle)
{
    //Amiiboタグをかざすメッセージを表示して数秒待つ
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH AMIIBO TAG(Created by NfpManagerEx) (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTime_TagSwitch);

    //APIで指定するDeviceHandleに不正な値を使用する
    SetInvalidDeviceHandleType(InvalidHandleType_MissHandle);

    DoTestInvalidDeviceHandle();
}

TEST_F(NfpManualNx,TestCaseInvalidDeviceHandleOfPowerDown)
{
    //Amiiboタグをかざすメッセージを表示して数秒待つ
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH AMIIBO TAG(Created by NfpManagerEx) (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTime_TagSwitch);

    //API実行直前にデバイスの電源をOFFにさせる
    SetInvalidDeviceHandleType(InvalidHandleType_PowerOff);

    DoTestInvalidDeviceHandle();
}

TEST_F(NfpManualNx,TestCaseMountAccessible)
{
    //Amiiboタグをかざすメッセージを表示して数秒待つ
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH AMIIBO TAG(Created by NfpManagerEx) (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTime_TagSwitch);

    // INIT 状態に移行します。
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());

    NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());

    nnt::nfp::DoActivate();

    //Ram領域をマウント
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
                                     nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_Ram));

    DoTestAccessibleRamArea();

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(nnt::nfp::GetCurrentDeviceHandle()));

    //Rom領域をマウント
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
                                     nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_Rom));

    DoTestAccessibleRomArea();

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(nnt::nfp::GetCurrentDeviceHandle()));

    //全ての領域をマウント
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
                                     nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_All));

    DoTestAccessibleRamArea();
    DoTestAccessibleRomArea();

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(nnt::nfp::GetCurrentDeviceHandle()));
}

TEST_F(NfpManualNx,TestCaseMountInaccessible)
{
    //Amiiboタグをかざすメッセージを表示して数秒待つ
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH AMIIBO TAG(Created by NfpManagerEx) (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTime_TagSwitch);

    // INIT 状態に移行します。
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());

    NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());

    nnt::nfp::DoActivate();

    //Ram領域をマウント
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
                                     nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_Ram));

    DoTestInaccessibleRomArea();

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(nnt::nfp::GetCurrentDeviceHandle()));

    //Rom領域をマウント
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
                                     nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_Rom));

    DoTestInaccessibleRamArea();

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(nnt::nfp::GetCurrentDeviceHandle()));

}

TEST_F(NfpManualNx,TestCaseMountValidModel)
{
    //Amiiboタグをかざすメッセージを表示して数秒待つ
    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH AMIIBO TAG(Created by NfpManagerEx) (Within %dms) ...    \n",
           WaitTime_TagSwitch);
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTime_TagSwitch);

    // INIT 状態に移行します。
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());

    NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());

    nnt::nfp::DoActivate();

    //Amiiboを指定してマウント
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
                                     nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_All));

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(nnt::nfp::GetCurrentDeviceHandle()));
}

TEST_F(NfpManualNx,TestCaseMountInvalidModel)
{
    //サポートしていないタグをかざすメッセージを表示して数秒待つ
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE ATTACH UNSUPPORT TAG (Within %dms) ...    \n",WaitTime_TagSwitch);
    NN_LOG("==================================================\n");
    nnt::nfp::Sleep(WaitTime_TagSwitch);

    // INIT 状態に移行します。
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());

    NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());

    nnt::nfp::DoActivate();

    //Amiiboを指定してマウント
    NNT_EXPECT_RESULT_FAILURE(
            nn::nfp::ResultNotSupported,
            nnt::nfp::wrapper::Mount(nnt::nfp::GetCurrentDeviceHandle(),
                                     nn::nfp::ModelType_Amiibo,nn::nfp::MountTarget_All));
}

TEST_F(NfpManualNx,TestCaseAvailabilityChangeEvent)
{
    nnt::nfp::WifiOn();

    //NFC デバイス ON を促すメッセージを表示して数秒待つ
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE NFC DEVICE POWER ON(Within %dms) ...     \n",10000);
    NN_LOG("==================================================\n");
    nnt::nfp::Sleep(10000);

    // INIT 状態に移行します。
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());

    NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultAvailabilityChangeEvent());

    DoTestAvailabilityChangeEventInInit();
    DoTestAvailabilityChangeEventInSearch();
    DoTestAvailabilityChangeEventInActive();
    DoTestAvailabilityChangeEventInActive();
}

TEST_F(NfpManualNx, TestCaseNfpNfcMultiDevice)
{
    nn::nfp::TagInfo             tagNfpInfo;
    nn::nfc::TagInfo             tagNfcInfo;
    nn::nfp::DeviceHandle nfpDeviceHandle[2];
    nn::nfc::DeviceHandle nfcDeviceHandle[2];
    nn::os::SystemEventType nfpActivateEvent;
    nn::os::SystemEventType nfcActivateEvent;
    nn::os::SystemEventType nfpDeactivateEvent;
    nn::os::SystemEventType nfcDeactivateEvent;
    nn::hid::NpadIdType nfpNpadId;
    nn::hid::NpadIdType nfcNpadId;
    nn::hid::NpadIdType npadId;
    int outCount = 0;

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE CONNECT 2 NFC DEVICES WITHOUT TAG\n");
    NN_LOG("==================================================\n");

    // -----------------------------------------------------
    // テスト対象 : Nfp Init -> Nfc Init -> Nfc Fin -> Nfp Fin
    // -----------------------------------------------------

    // Search 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::nfp::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfpNpadId, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfcNpadId, nfpDeviceHandle[1]));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[0]));
        EXPECT_EQ(npadId, nfpNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[1]));
        EXPECT_EQ(npadId, nfcNpadId);

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    // Active 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::nfp::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfpNpadId, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfcNpadId, nfpDeviceHandle[1]));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[0]));
        EXPECT_EQ(npadId, nfpNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[1]));
        EXPECT_EQ(npadId, nfcNpadId);

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    // Deactive 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::nfp::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfpNpadId, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfcNpadId, nfpDeviceHandle[1]));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&nfpDeactivateEvent, nfpDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[0]));
        EXPECT_EQ(npadId, nfpNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[1]));
        EXPECT_EQ(npadId, nfcNpadId);

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&nfcDeactivateEvent, nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }


    // -----------------------------------------------------
    // テスト対象 : Nfp Init -> Nfc Init -> Nfp Fin -> Nfc Fin
    // -----------------------------------------------------

    // Search 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::nfp::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfpNpadId, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfcNpadId, nfpDeviceHandle[1]));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[0]));
        EXPECT_EQ(npadId, nfpNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[1]));
        EXPECT_EQ(npadId, nfcNpadId);

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    // Active 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::nfp::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfpNpadId, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfcNpadId, nfpDeviceHandle[1]));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[0]));
        EXPECT_EQ(npadId, nfpNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[1]));
        EXPECT_EQ(npadId, nfcNpadId);

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    // Deactive 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::nfp::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfpNpadId, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&nfcNpadId, nfpDeviceHandle[1]));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&nfpDeactivateEvent, nfpDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[0]));
        EXPECT_EQ(npadId, nfpNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId, nfcDeviceHandle[1]));
        EXPECT_EQ(npadId, nfcNpadId);

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&nfcDeactivateEvent, nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
    }

    //@@@
    // -----------------------------------------------------
    // テスト対象 : Nfc Init -> Nfp Init -> Nfp Fin -> Nfc Fin
    // -----------------------------------------------------

    // Search 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::mifare::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfcNpadId, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfpNpadId, nfcDeviceHandle[1]));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[0]));
        EXPECT_EQ(npadId, nfcNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[1]));
        EXPECT_EQ(npadId, nfpNpadId);

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

    // Active 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::mifare::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfcNpadId, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfpNpadId, nfcDeviceHandle[1]));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[0]));
        EXPECT_EQ(npadId, nfcNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[1]));
        EXPECT_EQ(npadId, nfpNpadId);

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

    // Deactive 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::mifare::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfcNpadId, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfpNpadId, nfcDeviceHandle[1]));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&nfcDeactivateEvent, nfcDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[0]));
        EXPECT_EQ(npadId, nfcNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[1]));
        EXPECT_EQ(npadId, nfpNpadId);

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&nfpDeactivateEvent, nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }


    // -----------------------------------------------------
    // テスト対象 : Nfc Init -> Nfp Init -> Nfc Fin -> Nfp Fin
    // -----------------------------------------------------

    // Search 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::mifare::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfcNpadId, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfpNpadId, nfcDeviceHandle[1]));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[0]));
        EXPECT_EQ(npadId, nfcNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[1]));
        EXPECT_EQ(npadId, nfpNpadId);

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

    // Active 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::mifare::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfcNpadId, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfpNpadId, nfcDeviceHandle[1]));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[0]));
        EXPECT_EQ(npadId, nfcNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[1]));
        EXPECT_EQ(npadId, nfpNpadId);

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagNfpInfo));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagNfcInfo));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

    // Deactive 状態からの Finalize
    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        int outCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        if(outCount != 2)
        {
            nnt::mifare::Sleep(WaitTime_Connect2Devices);
            NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 2));
        }
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfcNpadId, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&nfpNpadId, nfcDeviceHandle[1]));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[0]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&nfcDeactivateEvent, nfcDeviceHandle[0]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection(nfcDeviceHandle[0]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfcDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::InitializeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 2));
        EXPECT_EQ(outCount, 2);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[0]));
        EXPECT_EQ(npadId, nfcNpadId);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId, nfpDeviceHandle[1]));
        EXPECT_EQ(npadId, nfpNpadId);

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[1]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&nfpDeactivateEvent, nfpDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[1]));
    //////////////////////////////////////////////////////////////////////////////////
    {
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE ATTACH TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
        NN_LOG("========================================================================\n");
        NN_LOG(" PLEASE REMOVE TAG\n");
        NN_LOG("========================================================================\n");
        nn::os::TimedWaitSystemEvent(&nfpDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));

        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_Init, nnt::mifare::wrapper::GetState());
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[0]));
        EXPECT_EQ(nn::nfc::DeviceState_Deactive, nnt::mifare::wrapper::GetDeviceState(nfcDeviceHandle[1]));
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::mifare::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[0]));
        EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState(nfpDeviceHandle[1]));
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

    //////////////////////////////////////////////////////////////////////////////////
    nnt::nfp::FinalizeSystem();
    //////////////////////////////////////////////////////////////////////////////////
    {
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
        EXPECT_EQ(nn::nfc::State_None, nnt::mifare::wrapper::GetState());
    }

} // NOLINT(impl/function_size)

TEST_F(NfpManualNx,TestCaseErrorVisible)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please ready for NFC device with invalid FW\n");
    NN_LOG("========================================================================\n");
    nnt::nfp::wrapper::Initialize();
    EXPECT_EQ(true, nnt::nfp::wrapper::IsErrorVisible());
    nnt::nfp::wrapper::SetErrorVisible(false);
    EXPECT_EQ(false, nnt::nfp::wrapper::IsErrorVisible());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SearchDeviceHandle());
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED NOT TO DISPLAY ERROR\n");
    NN_LOG("========================================================================\n");
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::StartDetection());
    nnt::nfp::wrapper::SetErrorVisible(true);
    EXPECT_EQ(true, nnt::nfp::wrapper::IsErrorVisible());
    int waitTime = 60;
    NN_LOG("========================================================================\n");
    NN_LOG(" Wait %dsec\n", waitTime);
    NN_LOG("========================================================================\n");
    for(int i = 0; i < waitTime; ++i)
    {
        nnt::nfp::Sleep(1000);
        NN_LOG(" %d ", waitTime - 1 - i);
        if((waitTime - 1 - i) % 10 == 0)
        {
            NN_LOG("\n");
        }
    }
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED TO DISPLAY ERROR\n");
    NN_LOG("========================================================================\n");
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::StartDetection());
    nnt::nfp::wrapper::Finalize();
}

TEST_F(NfpManualNx,TestCaseMultiLibraryEvent)
{
    nn::nfp::DeviceHandle nfpDeviceHandle[1];
    nn::nfc::DeviceHandle nfcDeviceHandle[1];
    nn::os::SystemEventType nfpActivateEvent;
    nn::os::SystemEventType nfcActivateEvent;
    nn::os::SystemEventType nfpDeactivateEvent;
    nn::os::SystemEventType nfcDeactivateEvent;
    int outCount = 0;
    bool isReceived;

    // 同時利用中に双方のライブラリに Activate/Deactivate イベントが通知されているか
    nnt::nfp::InitializeSystem();
    nnt::mifare::InitializeSystem();

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(nfpDeviceHandle, &outCount, 1));
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::ListDevices(nfcDeviceHandle, &outCount, 1));

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&nfpActivateEvent, nfpDeviceHandle[0]));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&nfpDeactivateEvent, nfpDeviceHandle[0]));
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&nfcActivateEvent, nfcDeviceHandle[0]));
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&nfcDeactivateEvent, nfcDeviceHandle[0]));

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(nfpDeviceHandle[0]));

    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE ATTACH TAG\n");
    NN_LOG("========================================================================\n");
    isReceived = nn::os::TimedWaitSystemEvent(&nfpActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
    EXPECT_EQ(true, isReceived);
    isReceived = nn::os::TimedWaitSystemEvent(&nfcActivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
    EXPECT_EQ(true, isReceived);

    NN_LOG("========================================================================\n");
    NN_LOG(" PLEASE REMOVE TAG\n");
    NN_LOG("========================================================================\n");
    isReceived = nn::os::TimedWaitSystemEvent(&nfpDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
    EXPECT_EQ(true, isReceived);
    isReceived = nn::os::TimedWaitSystemEvent(&nfcDeactivateEvent, nn::TimeSpan::FromMilliSeconds(10000));
    EXPECT_EQ(true, isReceived);

    nnt::mifare::FinalizeSystem();
    nnt::nfp::FinalizeSystem();
}
