﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/mii/mii_PrivateCommon.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // バックアップデータを読み込むためのバッファです。
    nn::Bit8  g_BackupBuffer[4 * 1024 * 1024];
    nn::Bit8* g_pBackupData;
    const size_t BackupBufferSize = sizeof(g_BackupBuffer) - nnt::nfp::IpcAlignment;
    const uint16_t CounterMaxValue = 65535;
#if defined(NNT_NFP_LIB_MII_ENABLE)
    const int MiiCheckflags =
            nn::mii::CheckFlag_Nickname |
            nn::mii::CheckFlag_Gender |
            nn::mii::CheckFlag_Height |
            nn::mii::CheckFlag_Build |
            nn::mii::CheckFlag_FavoriteColor;
    const int ExceptDataSize = 20;
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpAuto : public nnt::nfp::TestFramework
{
protected:

    NfpAuto() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
        // cfg ライブラリを初期化します。
        nn::cfg::init::Initialize();
#endif //!defined(NNT_NFP_PLATFORM_NX)
        // アライメントを調整します。
        g_pBackupData = static_cast<nn::Bit8*>(nnt::nfp::AlignBuffer(g_BackupBuffer,
                                                                     nnt::nfp::IpcAlignment));
        return;
    }

    ~NfpAuto() NN_NOEXCEPT
    {
#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
        // cfg ライブラリを解放します。
        nn::cfg::init::Finalize();
#endif //!defined(NNT_NFP_PLATFORM_NX)
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }


    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init)
        {
            nnt::nfp::FinalizeSystem();
        }
    }
};

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

//================================================================================
// テストケースの実装です。
//================================================================================

TEST_F(NfpAuto, TestCaseReadBackupSaveData)
{
    nn::nfp::TagInfo          tagInfo;
    nn::nfp::CommonInfo       commonInfo;
    nn::nfp::RegisterInfo     regInfo;
    nn::nfp::AdminInfo        adminInfo;
    nn::nfp::Date             date;
#if !defined(NNT_NFP_PLATFORM_NX)
    nn::nfp::BackupEntryInfo  entryInfo;
    nn::nfp::BackupHeaderInfo headerInfo;
#else
    nn::nfp::TagInfo          tagInfo2;
    nn::nfp::CommonInfo       commonInfo2;
    nn::nfp::RegisterInfo     regInfo2;
    nn::nfp::AdminInfo        adminInfo2;
    nn::nfp::BackupDataHeader headerInfo;
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // テストに使用するタグをつくっておきます。
    nnt::nfp::DoCreateNormalTagAndEnd();

    // システムセーブデータの生成直後にバックアップデータ取得 API の挙動を確認します。
    {
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
        //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
        nnt::nfp::DeleteSystemSaveData();
#endif //!defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());
        EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
        EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
        EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
        EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
        size_t readSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)

        std::memset(&headerInfo, 0xFF, sizeof(headerInfo));
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, g_pBackupData, BackupBufferSize));
#else
        std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
#endif //!defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, headerInfo.entryNum);
        EXPECT_EQ(0, headerInfo.nextEntryIndex);
        EXPECT_EQ(0, headerInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo.reserved, sizeof(headerInfo.reserved)));
#else
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo._reserved1, sizeof(headerInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo._reserved2, sizeof(headerInfo._reserved2)));
#endif //!defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

    // 現在のタグを Mount してからバックアップデータ取得 API の挙動を確認します。
    {
        nnt::nfp::DoMount();

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        date = nnt::nfp::GetCurrentDate();

#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
        EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
        EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
        EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
        size_t dataSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)

        std::memset(&headerInfo, 0xFF, sizeof(headerInfo));
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, g_pBackupData, BackupBufferSize));
#else
        std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
#endif //!defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(1, headerInfo.entryNum);
        EXPECT_EQ(1, headerInfo.nextEntryIndex);
        EXPECT_EQ(0, headerInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo.reserved, sizeof(headerInfo.reserved)));
#else
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo._reserved1, sizeof(headerInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo._reserved2, sizeof(headerInfo._reserved2)));

        nn::nfp::BackupDataToc* toc = &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc[0]);
        EXPECT_TRUE(tagInfo.tagId.length == toc->uidLength);
        EXPECT_TRUE(std::memcmp(tagInfo.tagId.uid, toc->uid, toc->uidLength) == 0);
        EXPECT_TRUE(date.year == toc->entryRegisterDate.GetYear());
        EXPECT_TRUE(date.month == toc->entryRegisterDate.GetMonth());
        EXPECT_TRUE(date.day == toc->entryRegisterDate.GetDay());
#endif //!defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
        std::memset(&entryInfo, 0xFF, sizeof(entryInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, g_pBackupData, BackupBufferSize));
        EXPECT_TRUE(std::memcmp(&tagInfo.tagId, &entryInfo.tagId, sizeof(nn::nfp::TagId)) == 0);
        EXPECT_TRUE(std::memcmp(&commonInfo, &entryInfo.commonInfo, sizeof(nn::nfp::CommonInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo, &entryInfo.registerInfo, sizeof(nn::nfp::RegisterInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&adminInfo, &entryInfo.adminInfo, sizeof(nn::nfp::AdminInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&date, &entryInfo.entryRegisterDate, sizeof(nn::nfp::Date)) == 0);
        EXPECT_EQ(true, entryInfo.isEnableRegisterInfo);
        EXPECT_EQ(false, entryInfo.isMoveReserved);
        EXPECT_TRUE(nnt::nfp::IsAllZero(entryInfo.reserved, sizeof(entryInfo.reserved)));
#else
        // 読み出したデータをそのままセーブデータに書き込んでタグ復旧でタグに反映させて確認する
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_Hmac));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DeleteSystemSaveData();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo2));
        EXPECT_TRUE(std::memcmp(&tagInfo.tagId, &tagInfo2.tagId, sizeof(nn::nfp::TagId)) == 0);
        EXPECT_TRUE(std::memcmp(&commonInfo, &commonInfo2, sizeof(nn::nfp::CommonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                regInfo.miiData, regInfo2.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&regInfo.registerDate, &regInfo2.registerDate, sizeof(regInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo.nickname[0], &regInfo2.nickname[0], sizeof(regInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo.fontRegion, &regInfo2.fontRegion, sizeof(regInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo._reserved, &regInfo2._reserved, sizeof(regInfo._reserved)) == 0);
        EXPECT_TRUE(std::memcmp(&adminInfo, &adminInfo2, sizeof(nn::nfp::AdminInfo)) == 0);
#endif //!defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

    // バックアップデータを 1001 個登録してからバックアップデータ取得 API の挙動を確認します。
    {
#if !defined(NNT_NFP_PLATFORM_NX)
        nnt::nfp::CreateDummyBackupData(nn::nfp::BACKUP_ENTRY_MAX, g_pBackupData, BackupBufferSize);
#else
        nnt::nfp::CreateDummyBackupData(nn::nfp::BackupEntryMax, g_pBackupData, BackupBufferSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)
        nnt::nfp::DoMount();

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        date = nnt::nfp::GetCurrentDate();

#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
        EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
        EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
        EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
        size_t dataSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)

        std::memset(&headerInfo, 0xFF, sizeof(headerInfo));
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, g_pBackupData, BackupBufferSize));
#else
        std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
#endif //!defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(1000, headerInfo.entryNum);
        EXPECT_EQ(1, headerInfo.nextEntryIndex);
        EXPECT_EQ(0, headerInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo.reserved, sizeof(headerInfo.reserved)));
#else
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo._reserved1, sizeof(headerInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(headerInfo._reserved2, sizeof(headerInfo._reserved2)));

        nn::nfp::BackupDataToc* toc = &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc[0]);
        EXPECT_TRUE(tagInfo.tagId.length == toc->uidLength);
        EXPECT_TRUE(std::memcmp(tagInfo.tagId.uid, toc->uid, toc->uidLength) == 0);
        EXPECT_TRUE(date.year == toc->entryRegisterDate.GetYear());
        EXPECT_TRUE(date.month == toc->entryRegisterDate.GetMonth());
        EXPECT_TRUE(date.day == toc->entryRegisterDate.GetDay());
#endif //!defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
        std::memset(&entryInfo, 0xFF, sizeof(entryInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, g_pBackupData, BackupBufferSize));
        EXPECT_TRUE(std::memcmp(&tagInfo.tagId, &entryInfo.tagId, sizeof(nn::nfp::TagId)) == 0);
        EXPECT_TRUE(std::memcmp(&commonInfo, &entryInfo.commonInfo, sizeof(nn::nfp::CommonInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo, &entryInfo.registerInfo, sizeof(nn::nfp::RegisterInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&adminInfo, &entryInfo.adminInfo, sizeof(nn::nfp::AdminInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&date, &entryInfo.entryRegisterDate, sizeof(nn::nfp::Date)) == 0);
        EXPECT_EQ(true, entryInfo.isEnableRegisterInfo);
        EXPECT_EQ(false, entryInfo.isMoveReserved);
        EXPECT_TRUE(nnt::nfp::IsAllZero(entryInfo.reserved, sizeof(entryInfo.reserved)));
#else
        // 読み出したデータをそのままセーブデータに書き込んでタグ復旧でタグに反映させて確認する
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_Hmac));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DeleteSystemSaveData();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo2));
        EXPECT_TRUE(std::memcmp(&tagInfo.tagId, &tagInfo2.tagId, sizeof(nn::nfp::TagId)) == 0);
        EXPECT_TRUE(std::memcmp(&commonInfo, &commonInfo2, sizeof(nn::nfp::CommonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                regInfo.miiData, regInfo2.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&regInfo.registerDate, &regInfo2.registerDate, sizeof(regInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo.nickname[0], &regInfo2.nickname[0], sizeof(regInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo.fontRegion, &regInfo2.fontRegion, sizeof(regInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfo._reserved, &regInfo2._reserved, sizeof(regInfo._reserved)) == 0);
        EXPECT_TRUE(std::memcmp(&adminInfo, &adminInfo2, sizeof(nn::nfp::AdminInfo)) == 0);
#endif //!defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }
} // NOLINT(impl/function_size)

#if !defined(NNT_NFP_PLATFORM_NX)
TEST_F(NfpAuto, TestCaseInvalidArgument)
#else
TEST_F(NfpAuto, TestCaseInvalidArgumentDeathTest) //NXではアボートを期待
#endif // !defined(NNT_NFP_PLATFORM_NX)
{
    nn::nfp::TagInfo                   tagInfo;
#if !defined(NNT_NFP_PLATFORM_NX) || (!defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32))
    //NXでDEATHテストをしない場合に使用しない変数
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo;
#endif // !defined(NNT_NFP_PLATFORM_NX) || (!defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32))

    // INIT 前に実行できる API のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) // NX には別の amiibo 設定起動 I/F が用意されました。
    nn::nfp::Parameter                 parameter;
    nnt::nfp::wrapper::InitializeParameter(&parameter);
    parameter.input.mode = -1;
    EXPECT_EQ(false, nnt::nfp::wrapper::StartAmiiboSettings(&parameter));
    EXPECT_EQ(false, nnt::nfp::wrapper::StartAmiiboSettings(nullptr));
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // INIT 状態で実行できる API のテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());

#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::AttachActivateEvent(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::AttachDeactivateEvent(nullptr));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::AttachActivateEvent(nullptr));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::AttachDeactivateEvent(nullptr));

    #endif  // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定

    nn::nfp::DeviceHandle deviceHandle;
    int outCount;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ListDevices(nullptr, &outCount, 1));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ListDevices(&deviceHandle, nullptr, 1));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 0));

    #endif  // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) && defined(NNT_NFP_LIB_VERSION_BETA)//NXではアボートを期待
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    nn::nfp::AmiiboSettingsStartParam startParam;
    nn::nfp::RegisterInfo regInfo                 = {};
    bool isRegistered;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            nullptr, &isRegistered, &regInfo, startParam, tagInfo, regInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, nullptr, &regInfo, startParam, tagInfo, regInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, nullptr, startParam, tagInfo, regInfo));

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            nullptr, &isRegistered, &regInfo, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, nullptr, &regInfo, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, nullptr, startParam, tagInfo));

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartGameDataEraser(
            nullptr, startParam, tagInfo));

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartRestorer(nullptr, startParam, tagInfo));

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            nullptr, &deviceHandle, &isRegistered, &regInfo, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &tagInfo, nullptr, &isRegistered, &regInfo, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &tagInfo, &deviceHandle, nullptr, &regInfo, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &tagInfo, &deviceHandle, &isRegistered, nullptr, startParam));

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartGameDataEraser(
            nullptr, &deviceHandle, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartGameDataEraser(
            &tagInfo, nullptr, startParam));

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartRestorer(nullptr, &deviceHandle, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartRestorer(&tagInfo, nullptr, startParam));

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartFormatter(nullptr, &deviceHandle, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartFormatter(&tagInfo, nullptr, startParam));
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif //defined(NNT_NFP_PLATFORM_NX) && defined(NNT_NFP_LIB_VERSION_BETA)


#if !defined(NNT_NFP_PLATFORM_NX) //NXでは機能が削除されているので除外
    uint32_t                                totalSize;
    uint32_t                                sentSize;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetConnectionStatus(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::InitializeCreateInfo(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::nnt::nfp::wrapper::InitializeRegisterInfoSet(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::UpdateTargetFirmware(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetUpdateResult(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetUpdateProgress(nullptr, &totalSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, nullptr));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16  dst[10] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE };
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultBufferIsSmall, nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::ReplaceOtherRegionName(nullptr, nickName, sizeof(dst), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nullptr, sizeof(dst), nn::nfp::FontRegion_JpUsEu, 0x3F00));
    }
#endif //!defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXでは機能が削除されているので除外
    // FANGATE の接続に関するテストです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetTargetConnectionStatus(nullptr));
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // テストに使用するタグをつくり、バックアップデータが 1 つだけ存在する状態にします。
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
#else
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DeleteSystemSaveData();
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは返り値がないため除外
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::FinalizeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
    nnt::nfp::DoCreateNormalTag();

#if !defined(NNT_NFP_PLATFORM_NX) //NXでは機能が削除されているので除外
    nn::nfp::BackupHeaderInfo          headerInfo;
    nn::nfp::BackupEntryInfo           entryInfo;
    // バックアップ関連 API のテストです。
    size_t backupDataSize = nnt::nfp::wrapper::GetBackupSaveDataSize();
    size_t smallSize = backupDataSize - nnt::nfp::IpcAlignment;
    size_t invalidSize = backupDataSize + nnt::nfp::IpcAlignment / 2;
    void*  invalidBuffer = g_pBackupData + nnt::nfp::IpcAlignment / 2;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetAllBackupSaveData(nullptr, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultBufferIsSmall, nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, smallSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidAlignment, nnt::nfp::wrapper::GetAllBackupSaveData(invalidBuffer, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidAlignment, nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, invalidSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetBackupHeaderFromMemory(nullptr, g_pBackupData, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, nullptr, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, invalidBuffer, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, g_pBackupData, invalidSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, g_pBackupData, smallSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetBackupEntryFromMemory(nullptr, 0, g_pBackupData, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, nullptr, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultEntryNotFound, nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 1, g_pBackupData, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultEntryNotFound, nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 999, g_pBackupData, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 1000, g_pBackupData, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, invalidBuffer, backupDataSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, g_pBackupData, invalidSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, g_pBackupData, smallSize));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    size_t readBackupDataSize;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ReadBackupData(nullptr, &readBackupDataSize, BackupBufferSize));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, nullptr, BackupBufferSize));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readBackupDataSize, 0));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::WriteBackupData(nullptr, sizeof(nn::nfp::BackupData)));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, 0));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData) + 1));
    #endif  // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // アプリケーション領域が存在する状態で実施できるテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは機能が削除されているので除外
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetConnectResult(nullptr));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetTagInfo(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetAdminInfo(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetCommonInfo(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetRegisterInfo(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetModelInfo(nullptr));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch, nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId + 1));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch, nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId - 1));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::SetApplicationArea(nullptr, nn::nfp::ApplicationAreaSizeV2, tagInfo.tagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData, 217, tagInfo.tagId));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32) //Releaseの場合は不定
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetTagInfo(nullptr));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetAdminInfo(nullptr));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetCommonInfo(nullptr));
    nn::nfp::RegisterInfo* pRegInfo               = nullptr;
    nn::nfp::RegisterInfoPrivate* pRegInfoPrivate = nullptr;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetRegisterInfo(pRegInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetRegisterInfo(pRegInfoPrivate));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetModelInfo(nullptr));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ExistsApplicationArea(nullptr));
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)

    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId + 1));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId - 1));

    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::SetApplicationArea(nullptr,
                                                           nn::nfp::ApplicationAreaSizeV2,
                                                           tagInfo.tagId));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                           0,
                                                           tagInfo.tagId));
    appAreaInfo.pInitialData = nullptr;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    appAreaInfo.accessId = nnt::nfp::NormalTagId;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));

    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2 + 1] = {};
    std::memcpy(tagData,nnt::nfp::NormalTagData,nn::nfp::ApplicationAreaSizeV2);
    appAreaInfo.pInitialData = tagData;
    appAreaInfo.initialDataSize = 0;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではtagInfo.tagIdが削除されているので除外
    ++tagInfo.tagId.uid[0];
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultUidMisMatch, nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData, nn::nfp::ApplicationAreaSizeV2, tagInfo.tagId));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::GetApplicationArea(nullptr, nn::nfp::ApplicationAreaSizeV2));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定

    nn::Bit8 readBuffer[nn::nfp::ApplicationAreaSizeV2];
    size_t   readSize;

    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetApplicationArea(nullptr,
                                                               &readSize,
                                                               nn::nfp::ApplicationAreaSizeV2));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetApplicationArea(readBuffer,
                                                               nullptr,
                                                               nn::nfp::ApplicationAreaSizeV2));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetApplicationArea(readBuffer,
                                                               &readSize,
                                                               0));

    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // アプリケーション領域の作成に関するテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&appAreaInfo));
    appAreaInfo.pInitialData = nullptr;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    appAreaInfo.accessId = nnt::nfp::NormalTagId;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    appAreaInfo.pInitialData = nnt::nfp::NormalTagData;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2 + 1;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
#else
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());
    #if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&appAreaInfo));
    #endif // !defined(NNT_NFP_PLATFORM_NX)
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定

    appAreaInfo.pInitialData = nullptr;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    appAreaInfo.accessId = nnt::nfp::NormalTagId;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));

    std::memcpy(tagData,nnt::nfp::NormalTagData,nn::nfp::ApplicationAreaSizeV2);
    appAreaInfo.pInitialData = tagData;
    appAreaInfo.initialDataSize = 0;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されているので除外
    appAreaInfo.reserved[47] = 1;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは_reservedを適切に設定するAPIが削除されたので不正な値というものがない
    // 初期登録情報の登録に関するテストです。
    nn::nfp::RegisterInfoPrivate regInfoSet;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));

    std::memcpy(&regInfoSet.miiData, nnt::nfp::NormalTagMiiData, sizeof(regInfoSet.miiData));

    std::memcpy(regInfoSet.nickname,
                nnt::nfp::NormalTagNickName,
                sizeof(nnt::nfp::NormalTagNickName));
    regInfoSet.fontRegion = nn::nfp::FontRegion_JpUsEu;
    regInfoSet._reserved[0] ^= 1;

    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではFormatが引数を持たないので除外
    // RW_ACTIVATE 状態で実施できるテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidPointer, nnt::nfp::wrapper::Format(nullptr, nn::nfp::ApplicationAreaSizeV2));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultInvalidArgument, nnt::nfp::wrapper::Format(nnt::nfp::ZeroTagData, nn::nfp::ApplicationAreaSizeV2 + 1));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
} // NOLINT(impl/function_size)

TEST_F(NfpAuto, TestCaseBoundary)
{
    nn::nfp::TagInfo                   tagInfo;
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo;
    nn::Bit8                           readBuffer[nn::nfp::ApplicationAreaSizeV2];
#if defined(NNT_NFP_PLATFORM_NX)
    size_t                             readSize = 0;
#endif // defined(NNT_NFP_PLATFORM_NX)

    // テストに使用するタグをつくります。
    nnt::nfp::DoCreateNormalTag();

    // タグのアプリケーション専用領域の読み書きは 1 ～ 216 バイトまで自由に指定できます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));

#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(readBuffer, 1));
#else
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(readBuffer, &readSize, 1));
    EXPECT_EQ(readSize, 1);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(readBuffer,
                                                                nn::nfp::ApplicationAreaSizeV2));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                                1,
                                                                tagInfo.tagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                                nn::nfp::ApplicationAreaSizeV2,
                                                                tagInfo.tagId));
#if defined(NNT_NFP_PLATFORM_NX)
    appAreaInfo.pInitialData = readBuffer;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    appAreaInfo.accessId = nnt::nfp::NormalTagId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
    appAreaInfo.initialDataSize = 1;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
#endif // defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではFormatが引数を持たないので除外
    // フォーマットのサイズも 0 ～ 216 バイトまで自由に設定できます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(readBuffer, 0));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(readBuffer,
                                                        nn::nfp::ApplicationAreaSizeV2));
#else
    //次のテスト用にFormatを実行しておく
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry());
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // アプリケーション領域の作成時も 1 ～ 216 バイトまで自由に設定できます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&appAreaInfo));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    appAreaInfo.pInitialData = readBuffer;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    appAreaInfo.accessId = nnt::nfp::NormalTagId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::CreateApplicationAreaWithRetry(appAreaInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());
    appAreaInfo.initialDataSize = 1;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::CreateApplicationAreaWithRetry(appAreaInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
}

TEST_F(NfpAuto, TestCaseInvalidOperationActive)
{
    bool                               outValue = false;
    nn::Bit8                           buffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::Bit8                           tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::TagInfo                   tagInfo = {};
    nn::nfp::RegisterInfo              regInfo = {};
    nn::nfp::CommonInfo                commonInfo = {};
    nn::nfp::ModelInfo                 modelInfo = {};
    nn::nfp::AdminInfo                 adminInfo = {};
    nn::nfp::RegisterInfoPrivate       regInfoSet = {};
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { nnt::nfp::ZeroTagId,
                                                       nn::nfp::ApplicationAreaSizeV2,
                                                       tagData };

    std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);

    // RW_ACTIVE 状態まで遷移させます。
    nnt::nfp::DoActivate();

    // RW_ACTIVE 状態のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは返り値を持たない
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Initialize());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::InitializeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Unmount());

#if defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
    EXPECT_EQ(false, outValue);
#endif // (NNT_NFP_PLATFORM_NX)

    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));

    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetApplicationArea(buffer,
                                                                sizeof(buffer),
                                                                tagInfo.tagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Flush());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetModelInfo(&modelInfo));
#if !defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventは成功する
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteApplicationArea());

#if !defined(NNT_NFP_PLATFORM_NX) //NXでは高速タグ検出は実装されないので除外
    // 高速タグ検出モードのテストです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(false));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(true));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_TypeA));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_All));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではFWアップデート機能は実装されないので除外
    // CTR 専用のテストです。
    if (!nn::os::IsRunOnSnake())
    {
        uint32_t                           sentSize;
        uint32_t                           totalSize;
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

}

TEST_F(NfpAuto, TestCaseInvalidOperationMountRom)
{
    nn::Bit8                           buffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::Bit8                           tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::TagInfo                   tagInfo = {};
    nn::nfp::RegisterInfo              regInfo = {};
    nn::nfp::CommonInfo                commonInfo = {};
    nn::nfp::AdminInfo                 adminInfo = {};
    nn::nfp::RegisterInfoPrivate       regInfoSet = {};
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { nnt::nfp::ZeroTagId,
                                                       nn::nfp::ApplicationAreaSizeV2,
                                                       tagData };

    std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);

    // RW_MOUNT_ROM 状態に遷移します。
    nnt::nfp::DoMountRom();

    // RW_MOUNT_ROM 状態のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは返り値を持たない
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Initialize());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::InitializeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::MountRom());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetApplicationArea(buffer,
                                                                sizeof(buffer),
                                                                tagInfo.tagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Flush());
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Restore());
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
#if !defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventは成功する
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteApplicationArea());

#if defined(NNT_NFP_PLATFORM_NX)
    bool outValue = false;
    NNT_EXPECT_RESULT_FAILURE(
        nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
        nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
    EXPECT_EQ(false, outValue);
#endif //(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXでは高速タグ検出は実装されないので除外
    // 高速タグ検出モードのテストです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(false));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(true));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_TypeA));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_All));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではFWアップデート機能は実装されないので除外
    // CTR 専用のテストです。
    if (!nn::os::IsRunOnSnake())
    {
        uint32_t                           sentSize;
        uint32_t                           totalSize;
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)
}

TEST_F(NfpAuto, TestCaseInvalidOperationMount)
{
    nn::Bit8                           buffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::TagInfo                   tagInfo = {};
#if !defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventは成功する
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // RW_MOUNT (NOT OPENED) 状態まで遷移させます。
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));

    // RW_MOUNT (NOT OPENED) 状態のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは返り値を持たない
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Initialize());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::InitializeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::MountRom());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetApplicationArea(buffer,
                                                                sizeof(buffer),
                                                                tagInfo.tagId));
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Restore());
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventは成功する
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));

#if !defined(NNT_NFP_PLATFORM_NX) //NXでは高速タグ検出は実装されないので除外
    // 高速タグ検出モードのテストです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(false));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(true));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_TypeA));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_All));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではFWアップデート機能は実装されないので除外
    // CTR 専用のテストです。
    if (!nn::os::IsRunOnSnake())
    {
        uint32_t                                sentSize;
        uint32_t                                totalSize;
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // RW_MOUNT (OPENED) 状態まで遷移させます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));

    // RW_MOUNT (OPENED) 状態のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは返り値を持たない
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Initialize());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::InitializeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::MountRom());
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Restore());
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventは成功する
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));

#if !defined(NNT_NFP_PLATFORM_NX) //NXでは高速タグ検出は実装されないので除外
    // 高速タグ検出モードのテストです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(false));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(true));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_TypeA));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_All));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではFWアップデート機能は実装されないので除外
    // CTR 専用のテストです。
    if (!nn::os::IsRunOnSnake())
    {
        uint32_t                                sentSize;
        uint32_t                                totalSize;
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)
}

TEST_F(NfpAuto, TestCaseGoodOperationActive)
{
    // テスト用のタグデータを生成した上で RW_ACTIVE 状態に遷移します。
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());

    // バックアップ関係の API のテストです。この時点で必ず 1 つ以上バックアップが存在します。
#if !defined(NNT_NFP_PLATFORM_NX)
    nn::nfp::BackupHeaderInfo  backupHeader = {};
    nn::nfp::BackupEntryInfo   backupEntry = {};
    EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
    EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&backupHeader, g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&backupEntry, 0, g_pBackupData, BackupBufferSize));
#else
    nn::nfp::BackupDataHeader headerInfo;
    EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
    EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
    size_t dataSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
    std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
    EXPECT_TRUE(0 < headerInfo.entryNum);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
    // デバイスハンドル取得のテストです。
    nn::nfp::DeviceHandle deviceHandle;
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
#endif // defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventが成功する
    // active・deactiveイベント登録のテストです。
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
    // イベント登録の試験は終了したので、イベントを破棄する
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);

    // activeイベント待ちのフラグをWaitForActivate()で待つイベントに設定
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultActivateEvent());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultDeactivateEvent());
#endif // defined(NNT_NFP_PLATFORM_NX)

    // タグの検出停止に関するテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
    EXPECT_TRUE(nnt::nfp::WaitForDeactivate());

    // StopDetection()の実行でdeactive状態に遷移している為、active状態に遷移させます
    nnt::nfp::DoActivate();

    // タグのマウント、復旧に関するテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountRomWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_Hmac));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではRestore()実行後にマウントされない
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
#endif // defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData,
                                                        sizeof(nnt::nfp::ZeroTagData)));

    // 状態取得系の API です。
    nn::nfp::TagInfo tagInfo;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
    nn::nfp::ConnectionStatus connectionStatus;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo.tagId._reserved, sizeof(tagInfo.tagId._reserved)));
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectionStatus(&connectionStatus));
    EXPECT_EQ(nn::nfp::TAG_IS_ACTIVE, connectionStatus.deactivateReason);
    EXPECT_TRUE(nnt::nfp::IsAllZero(connectionStatus.reserved, sizeof(connectionStatus.reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_ACTIVE, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(false, nnt::nfp::DoExistsApplicationArea());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
    // reserved が 0 初期化されていることを確認します。
    nn::nfp::RegisterInfoPrivate       regInfoSet;
    nn::nfp::ApplicationAreaCreateInfo createInfo;
    std::memset(&regInfoSet, 0xFF, sizeof(regInfoSet));
    std::memset(&createInfo, 0xFF, sizeof(createInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&regInfoSet, sizeof(regInfoSet)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&createInfo, sizeof(createInfo)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
    // ニックネーム変換のテストです。
    nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
        0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu));
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // デバイスハンドルからNpad IDを取得するテストです。
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));
    EXPECT_TRUE(nnt::nfp::CheckNpadStyle(npadId));

    // ライブラリの解放に関するテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    // SNAKE では FANGATE 関連の API が全て成功します。
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectとFWアップデート機能が削除されているので除外
    nnt::nfp::DoActivate();

    nn::Result                      connectResult;
    nn::Result                      updateResult;
    nn::nfp::TargetConnectionStatus targetStatus;
    nn::os::EventType                   connectEvent;
    nn::os::EventType                   disconnectEvent;
    nn::os::EventType                   updateEvent;
    uint32_t                             sentSize;
    uint32_t                             totalSize;

    if (nn::os::IsRunOnSnake())
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(connectResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(updateResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
    else
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX)
    // 試験終了時にイベントを破棄する
    nnt::nfp::DestroyDefaultActivateEvent();
    nnt::nfp::DestroyDefaultDeactivateEvent();
#endif // defined(NNT_NFP_PLATFORM_NX)
} // NOLINT(impl/function_size)

TEST_F(NfpAuto, TestCaseGoodOperationMountRom)
{
    // RW_MOUNT_ROM 状態に遷移します。バックアップデータを生成するためにタグデータを初期化します。
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountRomWithRetry());

    // バックアップ関係の API のテストです。この時点で必ず 1 つ以上バックアップが存在します。
#if !defined(NNT_NFP_PLATFORM_NX)
    nn::nfp::BackupHeaderInfo  backupHeader = {};
    nn::nfp::BackupEntryInfo   backupEntry = {};
    EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
    EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&backupHeader, g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&backupEntry, 0, g_pBackupData, BackupBufferSize));
#else
    nn::nfp::BackupDataHeader headerInfo;
    EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
    EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
    size_t dataSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
    std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
    EXPECT_TRUE(0 < headerInfo.entryNum);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
    // デバイスハンドル取得のテストです。
    nn::nfp::DeviceHandle deviceHandle;
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
#endif //  defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventが成功する
    // アタッチ・デタッチイベント登録のテストです。
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
    // イベント登録の試験は終了したので、イベントを破棄する
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);

    // activeイベント待ちのフラグをWaitForActivate()で待つイベントに設定
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultActivateEvent());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultDeactivateEvent());
#endif // defined(NNT_NFP_PLATFORM_NX)

    // タグの検出停止に関するテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
    EXPECT_TRUE(nnt::nfp::WaitForDeactivate());

    // StopDetection()の実行でdeactive状態に遷移している為、Mount(Rom)状態に遷移させます
    nnt::nfp::DoMountRom();

    // タグのアンマウントに関するテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountRomWithRetry());

    // 状態取得系の API です。
    nn::nfp::TagInfo tagInfo;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
    nnt::nfp::wrapper::ConnectionStatus connectionStatus;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo.tagId._reserved, sizeof(tagInfo.tagId._reserved)));
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectionStatus(&connectionStatus));
    EXPECT_EQ(nn::nfp::TAG_IS_ACTIVE, connectionStatus.deactivateReason);
    EXPECT_TRUE(nnt::nfp::IsAllZero(connectionStatus.reserved, sizeof(connectionStatus.reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_MOUNT_ROM, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(false, nnt::nfp::DoExistsApplicationArea());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // タグ読み込み系の API です (TODO: ROM 領域の期待値をどうするか)
    nn::nfp::ModelInfo modelInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
    // reserved が 0 初期化されていることを確認します。
    nn::nfp::RegisterInfoPrivate       regInfoSet;
    nn::nfp::ApplicationAreaCreateInfo createInfo;
    std::memset(&regInfoSet, 0xFF, sizeof(regInfoSet));
    std::memset(&createInfo, 0xFF, sizeof(createInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&regInfoSet, sizeof(regInfoSet)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&createInfo, sizeof(createInfo)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
    // ニックネーム変換のテストです。
    nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
        0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu));
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // デバイスハンドルからNpad IDを取得するテストです。
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));
    EXPECT_TRUE(nnt::nfp::CheckNpadStyle(npadId));

    // ライブラリの解放に関するテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectとFWアップデート機能が削除されているので除外
    nnt::nfp::DoMountRom();

    // SNAKE では FANGATE 関連の API が全て成功します。
    nn::Result                      connectResult;
    nn::Result                      updateResult;
    nn::nfp::TargetConnectionStatus targetStatus;;
    nn::os::EventType                   connectEvent;
    nn::os::EventType                   disconnectEvent;
    nn::os::EventType                   updateEvent;
    uint32_t                             sentSize;
    uint32_t                             totalSize;

    if (nn::os::IsRunOnSnake())
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(connectResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(updateResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
    else
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX)
    // 試験終了時にイベントを破棄する
    nnt::nfp::DestroyDefaultActivateEvent();
    nnt::nfp::DestroyDefaultDeactivateEvent();
#endif // defined(NNT_NFP_PLATFORM_NX)
}

TEST_F(NfpAuto, TestCaseGoodOperationMount)
{
    {
        // RW_MOUNT 状態に遷移します。タグが壊れている可能性があるので初期化しておきます。
        nnt::nfp::DoCreateNormalTag();

        // バックアップ関係の API のテストです。この時点で必ず 1 つ以上バックアップが存在します。
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::nfp::BackupHeaderInfo  backupHeader = {};
        nn::nfp::BackupEntryInfo   backupEntry = {};
        EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
        EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&backupHeader, g_pBackupData, BackupBufferSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&backupEntry, 0, g_pBackupData, BackupBufferSize));
#else
        nn::nfp::BackupDataHeader headerInfo;
        EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
        EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
        size_t dataSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
        std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
        EXPECT_TRUE(0 < headerInfo.entryNum);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
        // デバイスハンドル取得のテストです。
        nn::nfp::DeviceHandle deviceHandle;
        int outCount;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
#endif // defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventが成功する
        // アタッチ・デタッチイベント登録のテストです。
        nn::os::SystemEventType            activateEvent;
        nn::os::SystemEventType            deactivateEvent;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
        // イベント登録の試験は終了したので、イベントを破棄する
        nn::os::DestroySystemEvent(&activateEvent);
        nn::os::DestroySystemEvent(&deactivateEvent);

        // activeイベント待ちのフラグをWaitForActivate()で待つイベントに設定
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultActivateEvent());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultDeactivateEvent());
#endif // defined(NNT_NFP_PLATFORM_NX)

        // タグの検出停止、アンマウント、ライブラリ解放に関するテストです。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        EXPECT_TRUE(nnt::nfp::WaitForDeactivate());

        // StopDetection()の実行でdeactive状態に遷移している為、Mount状態に遷移させます
        nnt::nfp::DoMount();

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoMount();

        // 状態取得系の API です。
        nn::nfp::TagInfo tagInfo;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
        nnt::nfp::wrapper::ConnectionStatus connectionStatus;
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo.tagId._reserved, sizeof(tagInfo.tagId._reserved)));
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectionStatus(&connectionStatus));
        EXPECT_EQ(nn::nfp::TAG_IS_ACTIVE, connectionStatus.deactivateReason);
        EXPECT_TRUE(nnt::nfp::IsAllZero(connectionStatus.reserved, sizeof(connectionStatus.reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(nn::nfp::RW_MOUNT, nnt::nfp::wrapper::GetDeviceState());
#else
        EXPECT_EQ(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(true, nnt::nfp::DoExistsApplicationArea());

        // タグ読み込み系の API です。
        nn::nfp::CommonInfo   commonInfo;
        nn::nfp::RegisterInfo regInfo;
        nn::nfp::AdminInfo    adminInfo;
        nn::nfp::ModelInfo      modelInfo;
        std::memset(&regInfo, 0xFF, sizeof(regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

        // タグ書き込み系の API です。
        nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
        std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);
        nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { nnt::nfp::ZeroTagId,
                                                           nn::nfp::ApplicationAreaSizeV2,
                                                           tagData };

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteNfpRegisterInfoWithRetry());
        EXPECT_EQ(false, nnt::nfp::DoExistsApplicationArea());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::CreateApplicationAreaWithRetry(appAreaInfo));

        // デバイスハンドルからNpad IDを取得するテストです。
        nn::hid::NpadIdType npadId;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));
        EXPECT_TRUE(nnt::nfp::CheckNpadStyle(npadId));

#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
        // reserved が 0 初期化されていることを確認します。
        nn::nfp::RegisterInfoPrivate       regInfoSet;
        nn::nfp::ApplicationAreaCreateInfo createInfo;
        std::memset(&regInfoSet, 0xFF, sizeof(regInfoSet));
        std::memset(&createInfo, 0xFF, sizeof(createInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(&regInfoSet, sizeof(regInfoSet)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(&createInfo, sizeof(createInfo)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
        // ニックネーム変換のテストです。
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu));
#endif //!defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectとFWアップデート機能が削除されているので除外
        // SNAKE では FANGATE 関連の API が全て成功します。
        nn::Result                      connectResult;
        nn::Result                      updateResult;
        nn::nfp::TargetConnectionStatus targetStatus;
        nn::os::EventType                   connectEvent;
        nn::os::EventType                   disconnectEvent;
        nn::os::EventType                   updateEvent;
        uint32_t                             sentSize;
        uint32_t                             totalSize;

        if (nn::os::IsRunOnSnake())
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
            NNT_EXPECT_RESULT_SUCCESS(connectResult);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvent));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
            NNT_EXPECT_RESULT_SUCCESS(updateResult);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
            EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
            EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX)
        // 試験終了時にイベントを破棄する
        nnt::nfp::DestroyDefaultActivateEvent();
        nnt::nfp::DestroyDefaultDeactivateEvent();
#endif // defined(NNT_NFP_PLATFORM_NX)
    }
    {
        // RW_MOUNT (OPEN) 状態に遷移します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoCreateNormalTag();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));

        // バックアップ関係の API のテストです。この時点で必ず 1 つ以上バックアップが存在します。
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::nfp::BackupHeaderInfo  backupHeader = {};
        nn::nfp::BackupEntryInfo   backupEntry = {};
        EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
        EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&backupHeader, g_pBackupData, BackupBufferSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&backupEntry, 0, g_pBackupData, BackupBufferSize));
#else
        nn::nfp::BackupDataHeader headerInfo;
        EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
        EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
        size_t dataSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
        std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
        EXPECT_TRUE(0 < headerInfo.entryNum);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
        // デバイスハンドル取得のテストです。
        nn::nfp::DeviceHandle deviceHandle;
        int outCount;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
#endif // defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventが成功する
        // アタッチ・デタッチイベント登録のテストです。
        nn::os::SystemEventType            activateEvent;
        nn::os::SystemEventType            deactivateEvent;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
        // イベント登録の試験は終了したので、イベントを破棄する
        nn::os::DestroySystemEvent(&activateEvent);
        nn::os::DestroySystemEvent(&deactivateEvent);

        // activeイベント待ちのフラグをWaitForActivate()で待つイベントに設定
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultActivateEvent());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultDeactivateEvent());
#endif // defined(NNT_NFP_PLATFORM_NX)

        // タグの検出停止、アンマウント、ライブラリ解放に関するテストです。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        EXPECT_TRUE(nnt::nfp::WaitForDeactivate());

        // StopDetection()の実行でdeactive状態に遷移している為、Mount状態に遷移させます
        nnt::nfp::DoMount();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoMount();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));

        // 状態取得系の API です。
        nn::nfp::TagInfo tagInfo;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
        nnt::nfp::wrapper::ConnectionStatus connectionStatus;
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo.tagId._reserved, sizeof(tagInfo.tagId._reserved)));
#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectionStatus(&connectionStatus));
        EXPECT_EQ(nn::nfp::TAG_IS_ACTIVE, connectionStatus.deactivateReason);
        EXPECT_TRUE(nnt::nfp::IsAllZero(connectionStatus.reserved, sizeof(connectionStatus.reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(nn::nfp::RW_MOUNT, nnt::nfp::wrapper::GetDeviceState());
#else
        EXPECT_EQ(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(true, nnt::nfp::DoExistsApplicationArea());

        // タグ読み込み系の API です。 (TODO: ROM 領域の期待値をどうするか)
        nn::Bit8              buffer[nn::nfp::ApplicationAreaSizeV2];
        nn::nfp::CommonInfo   commonInfo;
        nn::nfp::RegisterInfo regInfo;
        nn::nfp::AdminInfo    adminInfo;
        nn::nfp::ModelInfo    modelInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));
        EXPECT_TRUE(std::memcmp(buffer, nnt::nfp::NormalTagData, sizeof(buffer)) == 0);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
        // reserved が 0 初期化されていることを確認します。
        nn::nfp::RegisterInfoPrivate       regInfoSet;
        nn::nfp::ApplicationAreaCreateInfo createInfo;
        std::memset(&regInfoSet, 0xFF, sizeof(regInfoSet));
        std::memset(&createInfo, 0xFF, sizeof(createInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(&regInfoSet, sizeof(regInfoSet)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(&createInfo, sizeof(createInfo)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
        // タグ書き込み系の API です。
        nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
        std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);
        nn::nfp::ApplicationAreaCreateInfo appAreaInfo = { nnt::nfp::ZeroTagId,
                                                           nn::nfp::ApplicationAreaSizeV2,
                                                           tagData };

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                                    sizeof(nnt::nfp::ZeroTagData),
                                                                    tagInfo.tagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteNfpRegisterInfoWithRetry());
        EXPECT_EQ(false, nnt::nfp::DoExistsApplicationArea());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::CreateApplicationAreaWithRetry(appAreaInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));

        // デバイスハンドルからNpad IDを取得するテストです。
        nn::hid::NpadIdType npadId;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));
        EXPECT_TRUE(nnt::nfp::CheckNpadStyle(npadId));

#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectとFWアップデート機能が削除されているので除外
        // SNAKE では FANGATE 関連の API が全て成功します。
        nn::Result                      connectResult;
        nn::Result                      updateResult;
        nn::nfp::TargetConnectionStatus targetStatus;
        nn::os::EventType                   connectEvent;
        nn::os::EventType                   disconnectEvent;
        nn::os::EventType                   updateEvent;
        uint32_t                             sentSize;
        uint32_t                             totalSize;

        if (nn::os::IsRunOnSnake())
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
            NNT_EXPECT_RESULT_SUCCESS(connectResult);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvent));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
            NNT_EXPECT_RESULT_SUCCESS(updateResult);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
            EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
            EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX)
        // 試験終了時にイベントを破棄する
        nnt::nfp::DestroyDefaultActivateEvent();
        nnt::nfp::DestroyDefaultDeactivateEvent();
#endif // defined(NNT_NFP_PLATFORM_NX)
    }
} // NOLINT(impl/function_size)

#if defined(NNT_NFP_LIB_VERSION_ALPHA)
TEST_F(NfpAuto, TestCaseRestore)
{
    // テスト用のタグをつくり、状態を記憶します。
    nn::nfp::CommonInfo   beforeCommonInfo;
    nn::nfp::RegisterInfo beforeRegInfo;
    nn::nfp::AdminInfo    beforeAdminInfo;
    nn::nfp::ModelInfo    beforeModelInfo;
    nn::Bit8              beforeAppArea[nn::nfp::ApplicationAreaSizeV2];
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&beforeCommonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&beforeRegInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&beforeAdminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&beforeModelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(beforeAppArea,
                                                                    sizeof(beforeAppArea)));

    // タグの書き込み中情報を破壊して Restore を試行します。
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Activation));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
#if defined(NNT_NFP_PLATFORM_NX) //NXではRestore()実行後にマウントされないのでマウントする
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
#endif // defined(NNT_NFP_PLATFORM_NX)

        nn::nfp::CommonInfo   afterCommonInfo;
        nn::nfp::RegisterInfo afterRegInfo;
        nn::nfp::AdminInfo    afterAdminInfo;
        nn::nfp::ModelInfo    afterModelInfo;
        nn::Bit8              afterAppArea[nn::nfp::ApplicationAreaSizeV2];

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&afterCommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&afterRegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&afterAdminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&afterModelInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(afterAppArea,
                                                                        sizeof(afterAppArea)));
        EXPECT_TRUE(std::memcmp(&beforeCommonInfo, &afterCommonInfo, sizeof(afterCommonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                beforeRegInfo.miiData, afterRegInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&beforeRegInfo.registerDate, &afterRegInfo.registerDate, sizeof(afterRegInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(beforeRegInfo.nickname, afterRegInfo.nickname, sizeof(afterRegInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&beforeRegInfo.fontRegion, &afterRegInfo.fontRegion, sizeof(afterRegInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(beforeRegInfo._reserved, afterRegInfo._reserved, sizeof(afterRegInfo._reserved)) == 0);
        EXPECT_TRUE(std::memcmp(&beforeAdminInfo, &afterAdminInfo, sizeof(afterAdminInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&beforeModelInfo, &afterModelInfo, sizeof(afterModelInfo)) == 0);
        EXPECT_TRUE(std::memcmp(beforeAppArea, afterAppArea, sizeof(afterAppArea)) == 0);
    }

    // タグの HMAC を破壊して Restore を試行します。
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Hmac));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
#if defined(NNT_NFP_PLATFORM_NX) //NXではRestore()実行後にマウントされないのでマウントする
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
#endif // defined(NNT_NFP_PLATFORM_NX)

        nn::nfp::CommonInfo   afterCommonInfo;
        nn::nfp::RegisterInfo afterRegInfo;
        nn::nfp::AdminInfo    afterAdminInfo;
        nn::nfp::ModelInfo      afterModelInfo;
        nn::Bit8                  afterAppArea[nn::nfp::ApplicationAreaSizeV2];

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&afterCommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&afterRegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&afterAdminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&afterModelInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(afterAppArea,
                                                                        sizeof(afterAppArea)));
        EXPECT_TRUE(std::memcmp(&beforeCommonInfo, &afterCommonInfo, sizeof(afterCommonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                beforeRegInfo.miiData, afterRegInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&beforeRegInfo.registerDate, &afterRegInfo.registerDate, sizeof(afterRegInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(beforeRegInfo.nickname, afterRegInfo.nickname, sizeof(afterRegInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&beforeRegInfo.fontRegion, &afterRegInfo.fontRegion, sizeof(afterRegInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(beforeRegInfo._reserved, afterRegInfo._reserved, sizeof(afterRegInfo._reserved)) == 0);
        EXPECT_TRUE(std::memcmp(&beforeAdminInfo, &afterAdminInfo, sizeof(afterAdminInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&beforeModelInfo, &afterModelInfo, sizeof(afterModelInfo)) == 0);
        EXPECT_TRUE(std::memcmp(beforeAppArea, afterAppArea, sizeof(afterAppArea)) == 0);
    }
}
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)

TEST_F(NfpAuto, TestCaseBrokenBackupHeader)
{
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
    //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
    nnt::nfp::DeleteSystemSaveData();
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // タグをフォーマットしてテストに使用する壊れたタグをつくります。
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_Activation));

    // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    nn::nfp::BackupDataHeader& header = *reinterpret_cast<nn::nfp::BackupDataHeader*>(g_pBackupData);
#else
    size_t readSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
    nn::nfp::BackupDataHeader& header = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header;
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // ヘッダを破壊して保存します。
    header.hash ^= 1;
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    // バックアップデータが壊れていてもプロセス内部で再生成することで Initialize は成功します。
    nnt::nfp::DoActivate();

    // バックアップデータは Initialize でフォーマットされているので Restore できないはずです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedFormat, nnt::nfp::MountWithRetry());

    // バックアップデータが存在しなくても Format はうまくいきます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData)));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
}

TEST_F(NfpAuto, TestCaseBrokenBackupTocItem)
{
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
    //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
    nnt::nfp::DeleteSystemSaveData();
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // 壊れた索引が Mount, Restore で無視されることを確認します。
    {
        // タグをフォーマットしてテストに使用する壊れたタグをつくります。
        nnt::nfp::DoCreateZeroTag();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Activation));

        // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
        size_t BackupBufferSize = nnt::nfp::wrapper::GetBackupSaveDataSize();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
        nn::nfp::BackupDataHeader& header = *reinterpret_cast<nn::nfp::BackupDataHeader*>(g_pBackupData);
        nn::nfp::BackupTocItem*    items = reinterpret_cast<nn::nfp::BackupTocItem*>(g_pBackupData + sizeof(header));
#else
        size_t readSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
        nn::nfp::BackupDataHeader& header = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header;
        nn::nfp::BackupDataToc* items = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc;
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NN_UNUSED(header);

        // 索引の CRC を破壊して保存します。
        items[0].hash ^= 1;
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

        // Initialize で索引のベリファイは行っていないので成功します。
        nnt::nfp::DoActivate();

        // 索引が壊れている場合、バックアップデータは存在しないものとして扱われます。
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedFormat, nnt::nfp::MountWithRetry());
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultBackupReadFailed, nnt::nfp::RestoreWithRetry());
#else
        //NX では Mount で ResultNeedRestore になっているかの確認を促すため ResultNeedRestart を返す
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestart, nnt::nfp::RestoreWithRetry());
#endif // !defined(NNT_NFP_PLATFORM_NX)

        // Format はうまくいきます。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

    // 壊れた索引が Flush で無視されることを確認します。
    {
        // タグをフォーマットしてテストに使用するタグをつくります。
        nn::nfp::AdminInfo    expectedAdminInfo;
        nn::nfp::CommonInfo   expectedCommonInfo;
        nn::nfp::RegisterInfo expectedRegInfo;
        nn::nfp::ModelInfo      expectedModelInfo;
        nnt::nfp::DoCreateNormalTag();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&expectedAdminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&expectedCommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&expectedRegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&expectedModelInfo));

        // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
        size_t BackupBufferSize = nnt::nfp::wrapper::GetBackupSaveDataSize();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
        nn::nfp::BackupDataHeader& header = *reinterpret_cast<nn::nfp::BackupDataHeader*>(g_pBackupData);
        nn::nfp::BackupTocItem*    items = reinterpret_cast<nn::nfp::BackupTocItem*>(g_pBackupData + sizeof(header));
#else
        size_t readSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
        EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
        nn::nfp::BackupDataHeader& header = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header;
        nn::nfp::BackupDataToc* items = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc;
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NN_UNUSED(header);

        // 索引を破壊して保存し、その状態でタグにデータを書き込みます。
        items[0].hash ^= 1;
#if !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Hmac));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

        // Initialize で索引のベリファイは行っていないので成功します。
        nnt::nfp::DoActivate();

        // 先頭の索引は壊れていますが、 Flush 時にはその索引を無視して、
        // 2 番目のエントリとしてバックアップデータが保存されており Restore が実行できます。
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
#if defined(NNT_NFP_PLATFORM_NX) //NXではRestore()実行後にマウントされないのでマウントする
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
#endif // defined(NNT_NFP_PLATFORM_NX)

        // 復元された情報が意図したデータであることを検証します。
        nn::nfp::AdminInfo    adminInfo;
        nn::nfp::CommonInfo   commonInfo;
        nn::nfp::RegisterInfo regInfo;
        nn::nfp::ModelInfo      modelInfo;
        nn::Bit8 appArea[nn::nfp::ApplicationAreaSizeV2];
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
        EXPECT_TRUE(std::memcmp(&expectedAdminInfo, &adminInfo, sizeof(adminInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedCommonInfo, &commonInfo, sizeof(commonInfo)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(std::memcmp(&expectedRegInfo, &regInfo, sizeof(regInfo)) == 0);
#else
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                expectedRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&expectedRegInfo.registerDate, &regInfo.registerDate, sizeof(expectedRegInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedRegInfo.nickname[0], &regInfo.nickname[0], sizeof(expectedRegInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedRegInfo.fontRegion, &regInfo.fontRegion, sizeof(expectedRegInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedRegInfo._reserved, &regInfo._reserved, sizeof(expectedRegInfo._reserved)) == 0);
#endif //!defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(std::memcmp(&expectedModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
        EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::NormalTagData, sizeof(appArea)) == 0);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }
}

TEST_F(NfpAuto, TestCaseBrokenBackupEntry)
{
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
    //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
    nnt::nfp::DeleteSystemSaveData();
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // タグをフォーマットしてテストに使用する壊れたタグをつくります。
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Hmac));

    // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    nn::nfp::BackupDataHeader& header = *reinterpret_cast<nn::nfp::BackupDataHeader*>(g_pBackupData);
    nn::nfp::BackupTocItem*    items = reinterpret_cast<nn::nfp::BackupTocItem*>(g_pBackupData + sizeof(header));
    nn::Bit8* entry = g_pBackupData + sizeof(header) + sizeof(*items) * nn::nfp::BACKUP_ENTRY_MAX;
#else
    size_t readSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
    nn::nfp::BackupDataHeader& header = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header;
    nn::nfp::BackupDataToc* items = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc;
    nn::Bit8* entry = reinterpret_cast<nn::Bit8*>(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->data);
#endif
    NN_UNUSED(header);
    NN_UNUSED(items);

    // エントリ (HMAC RAM) を破壊して保存します。
    entry[128] ^= 1;
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    // Initialize でエントリのベリファイは行っていないので成功します。
    nnt::nfp::DoActivate();

    // エントリが壊れている場合、バックアップデータは存在しないものとして扱われます。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedFormat, nnt::nfp::MountWithRetry());
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultBackupCrcBroken, nnt::nfp::RestoreWithRetry());
#else
    //NX では Mount で ResultNeedRestore になっているかの確認を促すため ResultNeedRestart を返す
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestart, nnt::nfp::RestoreWithRetry());
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // Format はうまくいきます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData)));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
}

TEST_F(NfpAuto, TestCaseMaxBackupData)
{
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
    //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
    nnt::nfp::DeleteSystemSaveData();
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // 最大数を超えると先頭にバックアップデータがつくられることを検証します。
    {
        // 999 個のバックアップデータがあるかのように偽造し、タグはフォーマットしておきます。
        nnt::nfp::DoCreateZeroTagAndEnd();
#if !defined(NNT_NFP_PLATFORM_NX)
        nnt::nfp::CreateDummyBackupData(nn::nfp::BACKUP_ENTRY_MAX - 1, g_pBackupData, BackupBufferSize);
#else
        nnt::nfp::CreateDummyBackupData(nn::nfp::BackupEntryMax - 1, g_pBackupData, BackupBufferSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)

        // 1000 個目のタグをマウントして 1000 個目のバックアップデータを生成します。
        nn::nfp::TagInfo tagInfo;
        nnt::nfp::DoMount();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));

        {
            // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
            nn::nfp::BackupDataHeader& header = *reinterpret_cast<nn::nfp::BackupDataHeader*>(g_pBackupData);
            nn::nfp::BackupTocItem*    items = reinterpret_cast<nn::nfp::BackupTocItem*>(g_pBackupData + sizeof(header));
            nn::nfp::BackupTocItem&    item = items[nn::nfp::BACKUP_ENTRY_MAX - 1];
#else
            size_t readSize;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
            EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
            nn::nfp::BackupDataHeader& header = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header;
            nn::nfp::BackupDataToc* items = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc;
            nn::nfp::BackupDataToc& item = items[nn::nfp::BackupEntryMax - 1];
#endif //!defined(NNT_NFP_PLATFORM_NX)

            // バックアップデータのヘッダと索引の内容を検証します。
#if !defined(NNT_NFP_PLATFORM_NX)
            EXPECT_EQ(nn::nfp::BACKUP_ENTRY_MAX, header.entryNum);
#else
            EXPECT_EQ(nn::nfp::BackupEntryMax, header.entryNum);
#endif //!defined(NNT_NFP_PLATFORM_NX)
            EXPECT_EQ(0, header.nextEntryIndex);
            EXPECT_EQ(tagInfo.tagId.length, item.uidLength);
            EXPECT_TRUE(std::memcmp(tagInfo.tagId.uid, item.uid, item.uidLength) == 0);

            // 1001 個目のタグとしてバックアップデータを生成するため先ほど作った索引を偽造します。
            item.uid[item.uidLength - 1] ^= 1;
#if !defined(NNT_NFP_PLATFORM_NX)
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetAllBackupSaveData(g_pBackupData, BackupBufferSize));
#else
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
        }

        // もう一度タグをマウントして 1001 個目のバックアップデータを生成します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoMount();

        {
            // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
            nn::nfp::BackupDataHeader& header = *reinterpret_cast<nn::nfp::BackupDataHeader*>(g_pBackupData);
            nn::nfp::BackupTocItem*    items = reinterpret_cast<nn::nfp::BackupTocItem*>(g_pBackupData + sizeof(header));
            nn::nfp::BackupTocItem&    item = items[0];
#else
            size_t readSize;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
            EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
            nn::nfp::BackupDataHeader& header = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header;
            nn::nfp::BackupDataToc* items = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc;
            nn::nfp::BackupDataToc& item = items[0];
#endif //!defined(NNT_NFP_PLATFORM_NX)

            // バックアップデータのヘッダと索引の内容を検証します。
#if !defined(NNT_NFP_PLATFORM_NX)
            EXPECT_EQ(nn::nfp::BACKUP_ENTRY_MAX, header.entryNum);
#else
            EXPECT_EQ(nn::nfp::BackupEntryMax, header.entryNum);
#endif //!defined(NNT_NFP_PLATFORM_NX)
            EXPECT_EQ(1, header.nextEntryIndex);
            EXPECT_EQ(tagInfo.tagId.length, item.uidLength);
            EXPECT_TRUE(std::memcmp(tagInfo.tagId.uid, item.uid, item.uidLength) == 0);
        }

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

    // 最後尾のバックアップデータでタグを復元できることを検証します。
    {
        // 999 個のバックアップデータがあるかのように偽造し、壊れたタグをつくります。
        nn::nfp::AdminInfo    expectedAdminInfo;
        nn::nfp::CommonInfo   expectedCommonInfo;
        nn::nfp::RegisterInfo expectedRegInfo;
        nn::nfp::ModelInfo      expectedModelInfo;
        nnt::nfp::DoCreateNormalTag();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&expectedAdminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&expectedCommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&expectedRegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&expectedModelInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#if !defined(NNT_NFP_PLATFORM_NX)
        nnt::nfp::CreateDummyBackupData(nn::nfp::BACKUP_ENTRY_MAX - 1, g_pBackupData, BackupBufferSize);
#else
        nnt::nfp::CreateDummyBackupData(nn::nfp::BackupEntryMax - 1, g_pBackupData, BackupBufferSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)

        // 1000 個目のタグをマウントして 1000 個目のバックアップデータを生成します。
        nnt::nfp::DoMount();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Hmac));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

        // タグのマウントと復元を試みます。
        nnt::nfp::DoActivate();
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::RestoreWithRetry());
#if defined(NNT_NFP_PLATFORM_NX) //NXではRestore()実行後にマウントされないのでマウントする
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
#endif // defined(NNT_NFP_PLATFORM_NX)

        // 復元された情報が意図したデータであることを検証します。
        nn::nfp::AdminInfo    adminInfo;
        nn::nfp::CommonInfo   commonInfo;
        nn::nfp::RegisterInfo regInfo;
        nn::nfp::ModelInfo      modelInfo;
        nn::Bit8 appArea[nn::nfp::ApplicationAreaSizeV2];
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
        EXPECT_TRUE(std::memcmp(&expectedAdminInfo, &adminInfo, sizeof(adminInfo)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedCommonInfo, &commonInfo, sizeof(commonInfo)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(std::memcmp(&expectedRegInfo, &regInfo, sizeof(regInfo)) == 0);
#else
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                expectedRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&expectedRegInfo.registerDate, &regInfo.registerDate, sizeof(expectedRegInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedRegInfo.nickname[0], &regInfo.nickname[0], sizeof(expectedRegInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedRegInfo.fontRegion, &regInfo.fontRegion, sizeof(expectedRegInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(&expectedRegInfo._reserved, &regInfo._reserved, sizeof(expectedRegInfo._reserved)) == 0);
#endif //!defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(std::memcmp(&expectedModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
        EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::NormalTagData, sizeof(appArea)) == 0);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }
}

TEST_F(NfpAuto, TestCaseBackupEntryRegisterDate)
{
    // バックアップデータを一旦削除し、タグはフォーマットしておきます。
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
    //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
    nnt::nfp::DeleteSystemSaveData();
#endif //!defined(NNT_NFP_PLATFORM_NX)
    nnt::nfp::DoCreateZeroTagAndEnd();

    // 現在の日付を記憶した上で本体の日付を進めます。
    nn::nfp::Date registerDate = nnt::nfp::GetCurrentDate();
    nnt::nfp::SetNextDate();

    // もう一度タグをマウントし、適当にタグデータを変更してから書き込みを行います。
    nnt::nfp::DoMount();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());

    // バックアップデータを読み込みます。
#if !defined(NNT_NFP_PLATFORM_NX)
    nn::nfp::BackupEntryInfo entry;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&entry, 0, g_pBackupData, BackupBufferSize));

    // バックアップデータの日付が元の値のまま変化していないことを確認します。
    EXPECT_EQ(registerDate.year, entry.entryRegisterDate.year);
    EXPECT_EQ(registerDate.month, entry.entryRegisterDate.month);
    EXPECT_EQ(registerDate.day, entry.entryRegisterDate.day);
#else
    size_t readSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), readSize);
    nn::nfp::BackupDataToc& toc = reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->toc[0];

    // バックアップデータの日付が元の値のまま変化していないことを確認します。
    EXPECT_EQ(registerDate.year, toc.entryRegisterDate.GetYear());
    EXPECT_EQ(registerDate.month, toc.entryRegisterDate.GetMonth());
    EXPECT_EQ(registerDate.day, toc.entryRegisterDate.GetDay());
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // 本体の日付を元に戻します。
    nnt::nfp::SetPreviousDate();
}

TEST_F(NfpAuto, TestCaseCounterOverflow)
{
    // テストで使用するタグをつくります。
    nnt::nfp::DoCreateZeroTag();

    // カウンタが最大値となるように書き換えます。
    nn::nfp::TagInfo tagInfo;
    nn::nfp::NfpData before;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&before));
    before.systemInfo.systemWriteCounter = CounterMaxValue;
    before.commonInfo.writeCounter = CounterMaxValue;
    before.adminInfo.moveCounter = CounterMaxValue;
    before.systemInfo.hardwareId ^= 1;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetAll(before));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushDebugWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    // アプリケーション専用領域を書き換えます。
    nn::nfp::NfpData after;
    nnt::nfp::DoMount();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::NormalTagData,
                                                                sizeof(nnt::nfp::NormalTagData),
                                                                tagInfo.tagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&after));

    // カウンタの値が意図した増え方になっていることを検証します。
    EXPECT_EQ(0, after.systemInfo.systemWriteCounter);
    EXPECT_EQ(CounterMaxValue, after.commonInfo.writeCounter);
    EXPECT_EQ(CounterMaxValue, after.adminInfo.moveCounter);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    // 念のため、キャッシュを解放してからもう一度検証します。
    nnt::nfp::DoMount();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&after));
    EXPECT_EQ(0, after.systemInfo.systemWriteCounter);
    EXPECT_EQ(CounterMaxValue, after.commonInfo.writeCounter);
    EXPECT_EQ(CounterMaxValue, after.adminInfo.moveCounter);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
}

TEST_F(NfpAuto, TestCaseReadWriteAppAreaParamCheck)
{
    // テストに使用するタグをつくります。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo    prevModelInfo;
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&prevRegInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

    // テストのため日付を進めます。
    nnt::nfp::SetNextDate();
    nn::nfp::Date today = nnt::nfp::GetCurrentDate();

    // アプリケーション専用領域のキャッシュだけを書き換えます。
    const int size = sizeof(nnt::nfp::ZeroTagData) / 2;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                                size,
                                                                tagInfo.tagId));

    // この時点では共有領域に変化はなく、アプリケーション専用領域だけが書き換わっています。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo    modelInfo;
    nn::Bit8              appArea[nn::nfp::ApplicationAreaSizeV2];

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
    EXPECT_TRUE(std::memcmp(&prevAdminInfo, &adminInfo, sizeof(adminInfo)) == 0);
    EXPECT_TRUE(std::memcmp(&prevCommonInfo, &commonInfo, sizeof(commonInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    EXPECT_TRUE(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    EXPECT_TRUE(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    EXPECT_TRUE(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    EXPECT_TRUE(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::ZeroTagData, size) == 0);
    EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::ZeroTagData, sizeof(nnt::nfp::NormalTagData)) != 0);
    EXPECT_TRUE(std::memcmp(appArea + size, nnt::nfp::NormalTagData, sizeof(appArea) - size) != 0);

#if defined(NNT_NFP_PLATFORM_NX)
    {
        nn::Bit8    appArea2[nn::nfp::ApplicationAreaSizeV2 * 2];
        nn::Bit8    randNumSeq[nn::nfp::ApplicationAreaSizeV2 * 2];
        size_t      readSize = 0;

        // アプリケーション専用領域サイズより大きなデータが読み込まれないことを確認します。
        nnt::nfp::CreateRandomNumberSequence(randNumSeq, sizeof(randNumSeq));
        std::memcpy(appArea2, randNumSeq, sizeof(appArea2));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea2, &readSize, sizeof(appArea2)));
        EXPECT_EQ(readSize, nn::nfp::ApplicationAreaSizeV2);
        EXPECT_TRUE(std::memcmp(appArea, appArea2, sizeof(appArea)) == 0);
        EXPECT_TRUE(std::memcmp(
            &appArea2[nn::nfp::ApplicationAreaSizeV2],
            &randNumSeq[nn::nfp::ApplicationAreaSizeV2],
            (sizeof(appArea2) - nn::nfp::ApplicationAreaSizeV2)) == 0);
    }
#endif  // defined(NNT_NFP_PLATFORM_NX)

    // タグに書き込むことの無いまま検出を停止し、タグをマウントしなおします。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DoMount();

    // Flush していないのでアプリケーション専用領域は元の状態に戻っています。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
    EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::NormalTagData, sizeof(nnt::nfp::NormalTagData)) == 0);

    // 今度はアプリケーション専用領域の内容を実際にタグに書き込みます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                                size,
                                                                tagInfo.tagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
    EXPECT_TRUE(std::memcmp(&prevAdminInfo, &adminInfo, sizeof(adminInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    EXPECT_TRUE(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    EXPECT_TRUE(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    EXPECT_TRUE(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    EXPECT_TRUE(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::ZeroTagData, size) == 0);
    EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData)) != 0);
    EXPECT_TRUE(std::memcmp(appArea + size, nnt::nfp::NormalTagData, sizeof(appArea) - size) != 0);
    EXPECT_TRUE(std::memcmp(&today, &commonInfo.lastWriteDate, sizeof(today)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではメンバが削除された
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo) - 6) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);

    // タグの検出をやり直します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DoMount();

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    EXPECT_TRUE(std::memcmp(&prevAdminInfo, &adminInfo, sizeof(adminInfo)) == 0);
#if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
            prevRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
    EXPECT_TRUE(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
    EXPECT_TRUE(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
    EXPECT_TRUE(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
    EXPECT_TRUE(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::ZeroTagData, size) == 0);
    EXPECT_TRUE(std::memcmp(appArea, nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData)) != 0);
    EXPECT_TRUE(std::memcmp(appArea + size, nnt::nfp::NormalTagData, sizeof(appArea) - size) != 0);
    EXPECT_TRUE(std::memcmp(&today, &commonInfo.lastWriteDate, sizeof(today)) == 0);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではメンバが削除された
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo) - 6) == 0);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);

    // 日付を元に戻します。
    nnt::nfp::SetPreviousDate();
} // NOLINT(impl/function_size)

TEST_F(NfpAuto, TestCaseReadWriteRegInfoParamCheck)
{
    const int fontRegion = nn::nfp::FontRegion_Count - 1;

    // テストに使用するタグをつくります。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo      prevModelInfo;
    nnt::nfp::DoCreateZeroTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister,
                              nnt::nfp::GetRegisterInfo(&prevRegInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

    // テストのため日付を進めます。
    nn::nfp::Date defaultDay = { 2000, 1, 1 };
    nnt::nfp::SetNextDate();
    nn::nfp::Date tomorrow = nnt::nfp::GetCurrentDate();

    // 登録情報のキャッシュだけを書き換えます。
    nn::nfp::RegisterInfoPrivate regInfoSet;
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではAPIが削除された
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if defined(NNT_NFP_LIB_VERSION_BETA)
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    nnt::nfp::BuildMiiData(regInfoSet.miiData);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // !defined(NNT_NFP_LIB_VERSION_BETA)
    std::memcpy(regInfoSet.nickname,
                nnt::nfp::NormalTagNickName,
                sizeof(nnt::nfp::NormalTagNickName));
    regInfoSet.fontRegion = fontRegion;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));

    // 登録情報を取得して仕様通りの値が設定されていることを確認します。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo    modelInfo;
    nn::Bit8              appArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevAdminInfo.titleId, adminInfo.titleId);
#else
    EXPECT_EQ(prevAdminInfo.applicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::ZeroTagId, adminInfo.accessId);
    EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    EXPECT_EQ(nn::nfp::RegisterInfoFlags_CommonArea | nn::nfp::RegisterInfoFlags_ApplicationArea,
              adminInfo.registerInfo);
    EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
    EXPECT_EQ(prevAdminInfo.platform, adminInfo.platform);
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&defaultDay, &commonInfo.lastWriteDate, sizeof(defaultDay)) == 0);
    EXPECT_EQ(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo.characterID)) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)

    EXPECT_EQ(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    EXPECT_EQ(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    EXPECT_TRUE(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    EXPECT_TRUE(std::memcmp(nnt::nfp::ZeroTagData, appArea, sizeof(nnt::nfp::ZeroTagData)) == 0);
#if defined(NNT_NFP_LIB_VERSION_BETA)
    nn::nfp::RegisterInfoPrivate regInfoGet;
    nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet);
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&regInfoGet.miiData, &regInfoSet.miiData,
            regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    EXPECT_TRUE(std::memcmp(nnt::nfp::NormalTagNickName,
                            regInfo.nickname,
                            sizeof(nnt::nfp::NormalTagNickName)) == 0);
    EXPECT_TRUE(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow)) == 0);
    EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    EXPECT_EQ(fontRegion, regInfo.fontRegion);
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    EXPECT_EQ(nnt::nfp::GetCountry(), regInfo.country);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));

    // タグに書き込むことの無いまま検出を停止し、タグをマウントしなおします。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DoMount();

    // Flush していないので登録情報が存在しない状態に戻っています。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister,
                              nnt::nfp::GetRegisterInfo(&prevRegInfo));

    // 今度は登録情報を実際にタグに書き込みます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevAdminInfo.titleId, adminInfo.titleId);
#else
    EXPECT_EQ(prevAdminInfo.applicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)

    EXPECT_EQ(nnt::nfp::ZeroTagId, adminInfo.accessId);
    EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    EXPECT_EQ(nn::nfp::RegisterInfoFlags_CommonArea | nn::nfp::RegisterInfoFlags_ApplicationArea,
              adminInfo.registerInfo);
    EXPECT_EQ(prevAdminInfo.platform, adminInfo.platform);
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
    EXPECT_EQ(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo.characterID)) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    EXPECT_EQ(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_VERSION_BETA)
    std::memset(&regInfoGet, 0x00, sizeof(regInfoGet));
    nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet);
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&regInfoGet.miiData, &regInfoSet.miiData,
            regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    EXPECT_TRUE(std::memcmp(nnt::nfp::NormalTagNickName,
                            regInfo.nickname,
                            sizeof(nnt::nfp::NormalTagNickName)) == 0);
    EXPECT_TRUE(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow)) == 0);
    EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    EXPECT_EQ(fontRegion, regInfo.fontRegion);
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    EXPECT_EQ(nnt::nfp::GetCountry(), regInfo.country);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    EXPECT_TRUE(std::memcmp(nnt::nfp::ZeroTagData, appArea, sizeof(nnt::nfp::ZeroTagData)) == 0);

    // タグの検出をやり直します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DoMount();

    // 共有領域やアプリケーション専用領域が仕様通り変化していることを確認します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevAdminInfo.titleId, adminInfo.titleId);
#else
    EXPECT_EQ(prevAdminInfo.applicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::ZeroTagId, adminInfo.accessId);
    EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    EXPECT_EQ(nn::nfp::RegisterInfoFlags_CommonArea | nn::nfp::RegisterInfoFlags_ApplicationArea,
              adminInfo.registerInfo);
    EXPECT_EQ(prevAdminInfo.platform, adminInfo.platform);
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&tomorrow, &commonInfo.lastWriteDate, sizeof(tomorrow)) == 0);
    EXPECT_EQ(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, sizeof(commonInfo.characterID)) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    EXPECT_EQ(nn::nfp::ApplicationAreaSizeV2, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
#if defined(NNT_NFP_LIB_VERSION_BETA)
    std::memset(&regInfoGet, 0x00, sizeof(regInfoGet));
    nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet);
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    EXPECT_TRUE(std::memcmp(&regInfoGet.miiData, &regInfoSet.miiData,
            regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    EXPECT_TRUE(std::memcmp(nnt::nfp::NormalTagNickName,
                            regInfo.nickname,
                            sizeof(nnt::nfp::NormalTagNickName)) == 0);
    EXPECT_TRUE(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow)) == 0);
    EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    EXPECT_EQ(fontRegion, regInfo.fontRegion);
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
    EXPECT_EQ(nnt::nfp::GetCountry(), regInfo.country);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    EXPECT_TRUE(nnt::nfp::IsAllZero(regInfo._reserved, sizeof(regInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&prevModelInfo, &modelInfo, sizeof(modelInfo)) == 0);
    EXPECT_TRUE(std::memcmp(nnt::nfp::ZeroTagData, appArea, sizeof(nnt::nfp::ZeroTagData)) == 0);

    // 日付を元に戻します。
    nnt::nfp::SetPreviousDate();

    // もう一度登録情報をタグに書き込みます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());

    // 今度は新規登録では無いので writeCounter や registerDate が変動しません。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    EXPECT_EQ(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    EXPECT_TRUE(std::memcmp(&tomorrow, &regInfo.registerDate, sizeof(tomorrow)) == 0);
} // NOLINT(impl/function_size)

TEST_F(NfpAuto, TestCaseCreateAppAreaParamCheck)
{
    // テストに使用するタグをつくります。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo    prevModelInfo;
    nnt::nfp::DoFormat(nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister,
                              nnt::nfp::GetRegisterInfo(&prevRegInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

    // テストのため日付を進めます。
    nnt::nfp::SetNextDate();
    nn::nfp::Date today = nnt::nfp::GetCurrentDate();

    // アプリケーション専用領域を作成します。
    nn::nfp::ApplicationAreaCreateInfo createInfo;
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではAPIが削除されている
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(createInfo.reserved, sizeof(createInfo.reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    std::memcpy(tagData,nnt::nfp::NormalTagData,nn::nfp::ApplicationAreaSizeV2);
    createInfo.accessId = nnt::nfp::NormalTagId;
    createInfo.pInitialData = tagData;
    createInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::CreateApplicationAreaWithRetry(createInfo));

    // タグのフィールドが仕様通り変化していることを検証します。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo    modelInfo;
    nn::Bit8              appArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister, nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::TestTitleId, adminInfo.titleId);
#else
    EXPECT_EQ(nnt::nfp::TestApplicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::NormalTagId, adminInfo.accessId);
    EXPECT_EQ(prevAdminInfo.moveCounter + 1, adminInfo.moveCounter);
    EXPECT_EQ(nn::nfp::RegisterInfoFlags_ApplicationArea, adminInfo.registerInfo);
    EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
    EXPECT_EQ(nnt::nfp::Platform, adminInfo.platform);
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&today, &commonInfo.lastWriteDate, sizeof(today)) == 0);
    EXPECT_EQ(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                            modelInfo.characterId,
                            sizeof(modelInfo.characterId)) == 0);
    EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
    EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
    EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));
    EXPECT_TRUE(std::memcmp(nnt::nfp::NormalTagData, appArea, sizeof(appArea)) == 0);

    // アプリケーション専用領域が存在する状態でもう一度つくろうとすると失敗します。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAlreadyCreated,
                              nnt::nfp::CreateApplicationAreaWithRetry(createInfo));

    // タグの検出をやり直します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DoMount();

    // タグのフィールドが仕様通り変化していることをもう一度検証します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister, nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::TestTitleId, adminInfo.titleId);
#else
    EXPECT_EQ(nnt::nfp::TestApplicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::NormalTagId, adminInfo.accessId);
    EXPECT_EQ(prevAdminInfo.moveCounter + 1, adminInfo.moveCounter);
    EXPECT_EQ(nn::nfp::RegisterInfoFlags_ApplicationArea, adminInfo.registerInfo);
    EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
    EXPECT_EQ(nnt::nfp::Platform, adminInfo.platform);
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&today, &commonInfo.lastWriteDate, sizeof(today)) == 0);
    EXPECT_EQ(prevCommonInfo.writeCounter + 1, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(prevCommonInfo.nfpVersion, commonInfo.nfpVersion);
    EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                            modelInfo.characterId,
                            sizeof(modelInfo.characterId)) == 0);
    EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
    EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
    EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));
    EXPECT_TRUE(std::memcmp(nnt::nfp::NormalTagData, appArea, sizeof(appArea)) == 0);

    // アプリケーション専用領域が存在する状態でもう一度つくろうとすると失敗します。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAlreadyCreated, nnt::nfp::CreateApplicationAreaWithRetry(createInfo));

    // 日付を元に戻します。
    nnt::nfp::SetPreviousDate();
} // NOLINT(impl/function_size)

TEST_F(NfpAuto, TestCaseInvalidAccessId)
{
    // テストに使用するタグをつくります。
    const nn::Bit32 id = nnt::nfp::ZeroTagId;
    nnt::nfp::DoCreateZeroTag();

    // 誤った AccessID でアプリケーション専用領域のオープンを試行します。1
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFFFFFF00));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFFFF00FF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFF00FFFF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0x00FFFFFF));

    // アプリケーション専用領域をオープンします。 2 回実行しても成功を返します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(id));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(id));

    // 誤った AccessID でアプリケーション専用領域のオープンを試行します。1
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFFFFFF00));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFFFF00FF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFF00FFFF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0x00FFFFFF));

#if defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(id));
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo;
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    std::memcpy(tagData,nnt::nfp::NormalTagData,nn::nfp::ApplicationAreaSizeV2);
    appAreaInfo.pInitialData = tagData;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
    appAreaInfo.accessId = (id & 0xFFFFFF00);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFFFF00FF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFF00FFFF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0x00FFFFFF));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(appAreaInfo.accessId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFFFF00FF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0xFF00FFFF));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultAccessIdMisMatch,
                              nnt::nfp::wrapper::OpenApplicationArea(id & 0x00FFFFFF));
#endif
}

TEST_F(NfpAuto, TestCaseFormatParamCheck)
{
    // テストに使用するタグをつくります。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo      prevModelInfo;
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&prevRegInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

    // タグをフォーマットして Mount します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData,
                                                        sizeof(nnt::nfp::ZeroTagData)));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());

    // 想定通りに値が初期化されていることを確認します。
    nn::nfp::Date         defaultDate = { 2000, 1, 1 };
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo    modelInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister, nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, adminInfo.titleId);
#else
    EXPECT_EQ(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, adminInfo.accessId);
    EXPECT_EQ(0, adminInfo.moveCounter);
    EXPECT_EQ(0, adminInfo.registerInfo);
    EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/SNAKE/NX以外が設定される
    EXPECT_EQ(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
    EXPECT_NE(nn::nfp::PlatformType_Cafe, adminInfo.platform);
    EXPECT_NE(nn::nfp::PlatformType_Ctr, adminInfo.platform);
    EXPECT_NE(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&defaultDate, &commonInfo.lastWriteDate, sizeof(defaultDate)) == 0);
    EXPECT_EQ(0, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, commonInfo.nfpVersion);
    EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                            modelInfo.characterId,
                            sizeof(modelInfo.characterId)) == 0);
    EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
    EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
    EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    // タグの検出をやり直します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DoMount();

    // 想定通りに値が初期化されていることをもう一度確認します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister, nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, adminInfo.titleId);
#else
    EXPECT_EQ(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, adminInfo.accessId);
    EXPECT_EQ(0, adminInfo.moveCounter);
    EXPECT_EQ(0, adminInfo.registerInfo);
    EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/SNAKE/NX以外が設定される
    EXPECT_EQ(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
    EXPECT_NE(nn::nfp::PlatformType_Cafe, adminInfo.platform);
    EXPECT_NE(nn::nfp::PlatformType_Ctr, adminInfo.platform);
    EXPECT_NE(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&defaultDate, &commonInfo.lastWriteDate, sizeof(defaultDate)) == 0);
    EXPECT_EQ(0, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, commonInfo.nfpVersion);
    EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                            modelInfo.characterId,
                            sizeof(modelInfo.characterId)) == 0);
    EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
    EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
    EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    // HMAC 壊れのタグに対して ResultNeedFormat が返ることを確認します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Hmac));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
#else
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DeleteSystemSaveData();
    nnt::nfp::DoActivate();
#endif //!defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedFormat, nnt::nfp::MountWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData,
                                                        sizeof(nnt::nfp::ZeroTagData)));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());

    // 書き込み情報壊れたのタグに対して ResultNeedFormat が返ることを確認します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Activation));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::MountWithRetry());
#if !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::DeleteSystemSaveData());
#else
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DeleteSystemSaveData();
    nnt::nfp::DoActivate();
#endif //!defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedFormat, nnt::nfp::MountWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry(nnt::nfp::ZeroTagData,
                                                        sizeof(nnt::nfp::ZeroTagData)));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
} // NOLINT(impl/function_size)

TEST_F(NfpAuto, TestCaseDeleteAppAreaParamCheck)
{
    // アプリケーション領域をオープンしていない状態で削除を試行します。
    {
        // テストに使用するタグをつくります。
        nn::nfp::TagInfo      tagInfo;
        nn::nfp::AdminInfo    prevAdminInfo;
        nn::nfp::CommonInfo   prevCommonInfo;
        nn::nfp::RegisterInfo prevRegInfo;
        nn::nfp::ModelInfo      prevModelInfo;
        nnt::nfp::DoCreateNormalTag();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&prevRegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

        // アプリケーション専用領域を削除します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());

        // タグのフィールドが仕様通り変化していることを検証します。
        nn::nfp::AdminInfo    adminInfo;
        nn::nfp::CommonInfo   commonInfo;
        nn::nfp::RegisterInfo regInfo;
        nn::nfp::ModelInfo    modelInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                                  nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.titleId);
#else
        EXPECT_EQ(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.accessId);
        EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
        EXPECT_EQ(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
        EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/SNAKE/NX以外が設定される
        EXPECT_EQ(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
        EXPECT_NE(nn::nfp::PlatformType_Cafe, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Ctr, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
        EXPECT_TRUE(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
        EXPECT_EQ(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
        EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
        EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
        EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
        EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, commonInfo.nfpVersion);
        EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
        EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                                modelInfo.characterId,
                                sizeof(modelInfo.characterId)) == 0);
        EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
        EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
        EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                prevRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
        EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

        // アプリケーション専用領域が存在しない状態で削除を試行します。
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                                  nnt::nfp::DeleteApplicationAreaWithRetry());

        // タグの検出をやり直します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoMount();

        // タグのフィールドが仕様通り変化していることを検証します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                                  nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.titleId);
#else
        EXPECT_EQ(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.accessId);
        EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
        EXPECT_EQ(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
        EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/SNAKE/NX以外が設定される
        EXPECT_EQ(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
        EXPECT_NE(nn::nfp::PlatformType_Cafe, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Ctr, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
        EXPECT_TRUE(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
        EXPECT_EQ(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
        EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
        EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
        EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
        EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, commonInfo.nfpVersion);
        EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
        EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                                modelInfo.characterId,
                                sizeof(modelInfo.characterId)) == 0);
        EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
        EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
        EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                prevRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
        EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

        // アプリケーション専用領域が存在しない状態で削除を試行します。
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                                  nnt::nfp::DeleteApplicationAreaWithRetry());

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }

    // アプリケーション領域をオープンしている状態で削除を試行します。
    {
        // テストに使用するタグをつくります。
        nn::nfp::TagInfo      tagInfo;
        nn::nfp::AdminInfo    prevAdminInfo;
        nn::nfp::CommonInfo   prevCommonInfo;
        nn::nfp::RegisterInfo prevRegInfo;
        nn::nfp::ModelInfo      prevModelInfo;
        nnt::nfp::DoCreateNormalTag();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&prevRegInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
        EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

        // アプリケーション専用領域をオープンしてから削除します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());

        // タグのフィールドが仕様通り変化していることを検証します。
        nn::nfp::AdminInfo    adminInfo;
        nn::nfp::CommonInfo   commonInfo;
        nn::nfp::RegisterInfo regInfo;
        nn::nfp::ModelInfo      modelInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                                  nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.titleId);
#else
        EXPECT_EQ(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.accessId);
        EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
        EXPECT_EQ(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
        EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/SNAKE/NX以外が設定される
        EXPECT_EQ(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
        EXPECT_NE(nn::nfp::PlatformType_Cafe, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Ctr, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
        EXPECT_TRUE(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
        EXPECT_EQ(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
        EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
        EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
        EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
        EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, commonInfo.nfpVersion);
        EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
        EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                                modelInfo.characterId,
                                sizeof(modelInfo.characterId)) == 0);
        EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
        EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
        EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                prevRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
        EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

        // アプリケーション専用領域が存在しない状態で削除を試行します。
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                                  nnt::nfp::DeleteApplicationAreaWithRetry());

        // タグの検出をやり直します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
        nnt::nfp::DoMount();

        // タグのフィールドが仕様通り変化していることを検証します。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&regInfo));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate,
                                  nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.titleId);
#else
        EXPECT_EQ(nn::ncm::ApplicationId{0}, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, adminInfo.accessId);
        EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
        EXPECT_EQ(nn::nfp::RegisterInfoFlags_CommonArea, adminInfo.registerInfo);
        EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
#if !defined(NNT_NFP_PLATFORM_NX)// NX ではCTR/SNAKE/NX以外が設定される
        EXPECT_EQ(nn::nfp::PLATFORM_OTHER, adminInfo.platform);
#else
        EXPECT_NE(nn::nfp::PlatformType_Cafe, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Ctr, adminInfo.platform);
        EXPECT_NE(nn::nfp::PlatformType_Nx, adminInfo.platform);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
        EXPECT_TRUE(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
        EXPECT_EQ(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
        EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
        EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
        EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
        EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ(0, commonInfo.nfpVersion);
        EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
        EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
        EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                                modelInfo.characterId,
                                sizeof(modelInfo.characterId)) == 0);
        EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
        EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
        EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(nn::mii::CompareCharInfo(nullptr, MiiCheckflags,
                prevRegInfo.miiData, regInfo.miiData) == nn::mii::CheckResult_Same);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&prevRegInfo.registerDate, &regInfo.registerDate, sizeof(regInfo.registerDate)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo.nickname, regInfo.nickname, sizeof(regInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&prevRegInfo.fontRegion, &regInfo.fontRegion, sizeof(regInfo.fontRegion)) == 0);
        EXPECT_TRUE(std::memcmp(prevRegInfo._reserved, regInfo._reserved, sizeof(regInfo._reserved)) == 0);
        EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

        // アプリケーション専用領域が存在しない状態で削除を試行します。
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedCreate, nnt::nfp::DeleteApplicationAreaWithRetry());

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    }
} // NOLINT(impl/function_size)

TEST_F(NfpAuto, TestCaseDeleteRegInfoParamCheck)
{
    // テストに使用するタグをつくります。
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::AdminInfo    prevAdminInfo;
    nn::nfp::CommonInfo   prevCommonInfo;
    nn::nfp::RegisterInfo prevRegInfo;
    nn::nfp::ModelInfo      prevModelInfo;
    nnt::nfp::DoCreateNormalTag();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&prevAdminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&prevCommonInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&prevRegInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&prevModelInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(tagInfo._reserved, sizeof(tagInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevAdminInfo._reserved, sizeof(prevAdminInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved1, sizeof(prevCommonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevCommonInfo._reserved2, sizeof(prevCommonInfo._reserved2)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevRegInfo._reserved, sizeof(prevRegInfo._reserved)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(prevModelInfo._reserved, sizeof(prevModelInfo._reserved)));

    // 登録情報を削除します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteNfpRegisterInfoWithRetry());

    // タグのフィールドが仕様通り変化していることを検証します。
    nn::nfp::AdminInfo    adminInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfo regInfo;
    nn::nfp::ModelInfo      modelInfo;
    nn::Bit8                  appArea[nn::nfp::ApplicationAreaSizeV2];
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister, nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::TestTitleId, adminInfo.titleId);
#else
    EXPECT_EQ(nnt::nfp::TestApplicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::NormalTagId, adminInfo.accessId);
    EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    EXPECT_EQ(nn::nfp::RegisterInfoFlags_ApplicationArea, adminInfo.registerInfo);
    EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
    EXPECT_EQ(nnt::nfp::Platform, adminInfo.platform);
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
    EXPECT_EQ(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, commonInfo.nfpVersion);
    EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                            modelInfo.characterId,
                            sizeof(modelInfo.characterId)) == 0);
    EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
    EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
    EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));
    EXPECT_TRUE(std::memcmp(nnt::nfp::NormalTagData, appArea, sizeof(appArea)) == 0);

    // タグの検出をやり直します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    nnt::nfp::DoMount();

    // タグのフィールドが仕様通り変化していることを検証します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRegister, nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(appArea, sizeof(appArea)));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::TestTitleId, adminInfo.titleId);
#else
    EXPECT_EQ(nnt::nfp::TestApplicationId, adminInfo.applicationId);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nnt::nfp::NormalTagId, adminInfo.accessId);
    EXPECT_EQ(prevAdminInfo.moveCounter, adminInfo.moveCounter);
    EXPECT_EQ(nn::nfp::RegisterInfoFlags_ApplicationArea, adminInfo.registerInfo);
    EXPECT_EQ(nnt::nfp::TestNoftVersion, adminInfo.formatVersion);
    EXPECT_EQ(nnt::nfp::Platform, adminInfo.platform);
    EXPECT_TRUE(nnt::nfp::IsAllZero(adminInfo._reserved, sizeof(adminInfo._reserved)));
    EXPECT_TRUE(std::memcmp(&prevCommonInfo.lastWriteDate, &commonInfo.lastWriteDate, sizeof(commonInfo.lastWriteDate)) == 0);
    EXPECT_EQ(prevCommonInfo.writeCounter, commonInfo.writeCounter);
#if !defined(NNT_NFP_PLATFORM_NX) //NXではメンバが削除されている
    EXPECT_TRUE(std::memcmp(prevCommonInfo.characterID, commonInfo.characterID, 3) == 0);
    EXPECT_EQ(prevCommonInfo.seriesId, commonInfo.seriesId);
    EXPECT_EQ(prevCommonInfo.numberingId, commonInfo.numberingId);
    EXPECT_EQ(prevCommonInfo.nfpType, commonInfo.nfpType);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, commonInfo.nfpVersion);
    EXPECT_EQ(prevCommonInfo.applicationAreaSize, commonInfo.applicationAreaSize);
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved1, sizeof(commonInfo._reserved1)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(commonInfo._reserved2, sizeof(commonInfo._reserved2)));
    EXPECT_TRUE(std::memcmp(prevModelInfo.characterId,
                            modelInfo.characterId,
                            sizeof(modelInfo.characterId)) == 0);
    EXPECT_EQ(prevModelInfo.seriesId, modelInfo.seriesId);
    EXPECT_EQ(prevModelInfo.numberingId, modelInfo.numberingId);
    EXPECT_EQ(prevModelInfo.nfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));
    EXPECT_TRUE(std::memcmp(nnt::nfp::NormalTagData, appArea, sizeof(appArea)) == 0);
}

TEST_F(NfpAuto, TestCaseSystemWriteCounter)
{
    nn::nfp::NfpData         prevNfpData;
    nn::nfp::NfpData         nfpData;

    // テストに使用するタグをつくります。
    nnt::nfp::DoFormat(nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&prevNfpData));
    uint16_t expectedCounter = prevNfpData.systemInfo.systemWriteCounter;

    // CreateApplicationArea で SystemWriteCounter がインクリメントされることを検証します。
    expectedCounter = (expectedCounter + 1) & 0xFFFF;
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    std::memcpy(tagData,nnt::nfp::NormalTagData,nn::nfp::ApplicationAreaSizeV2);
    nn::nfp::ApplicationAreaCreateInfo createInfo = { nnt::nfp::NormalTagId,
                                                      nn::nfp::ApplicationAreaSizeV2,
                                                      tagData };

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::CreateApplicationAreaWithRetry(createInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&nfpData));
    EXPECT_EQ(expectedCounter, nfpData.systemInfo.systemWriteCounter);

    // アプリケーション専用領域を書き換えます。
    nn::nfp::TagInfo tagInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                                sizeof(nnt::nfp::ZeroTagData),
                                                                tagInfo.tagId));

    // アプリケーション専用領域書き換え後の Flush で SystemWriteCounter がインクリメントされることを検証します。
    expectedCounter = (expectedCounter + 1) & 0xFFFF;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&nfpData));
    EXPECT_EQ(expectedCounter, nfpData.systemInfo.systemWriteCounter);

#if defined(NNT_NFP_PLATFORM_NX)
    // RecreateApplicationArea にてアプリケーション専用領域を書き換えます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
    createInfo.accessId = nnt::nfp::NormalTagId + 1;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::RecreateApplicationArea(createInfo));

    // アプリケーション専用領域書き換え後の Flush で SystemWriteCounter がインクリメントされることを検証します。
    expectedCounter = (expectedCounter + 1) & 0xFFFF;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&nfpData));
    EXPECT_EQ(expectedCounter, nfpData.systemInfo.systemWriteCounter);
#endif // defined(NNT_NFP_PLATFORM_NX)

    // 初期登録情報を登録します。
    nn::nfp::RegisterInfoPrivate regInfoSet;
#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されている
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if defined(NNT_NFP_LIB_VERSION_BETA)
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    nnt::nfp::BuildMiiData(regInfoSet.miiData);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
    std::memcpy(regInfoSet.nickname,
                nnt::nfp::NormalTagNickName,
                sizeof(nnt::nfp::NormalTagNickName));
    regInfoSet.fontRegion = nn::nfp::FontRegion_JpUsEu;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));

    // 初期登録情報登録後の Flush で SystemWriteCounter がインクリメントされることを検証します。
    expectedCounter = (expectedCounter + 1) & 0xFFFF;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&nfpData));
    EXPECT_EQ(expectedCounter, nfpData.systemInfo.systemWriteCounter);

    // 何も書き換えていない状態の Flush で SystemWriteCounter がインクリメントされることを検証します。
    expectedCounter = (expectedCounter + 1) & 0xFFFF;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FlushWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&nfpData));
    EXPECT_EQ(expectedCounter, nfpData.systemInfo.systemWriteCounter);

    // アプリケーション専用領域の削除で SystemWriteCounter がインクリメントされることを検証します。
    expectedCounter = (expectedCounter + 1) & 0xFFFF;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteApplicationAreaWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&nfpData));
    EXPECT_EQ(expectedCounter, nfpData.systemInfo.systemWriteCounter);

    // 初期登録情報の削除で SystemWriteCounter がインクリメントされることを検証します。
    expectedCounter = (expectedCounter + 1) & 0xFFFF;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::DeleteNfpRegisterInfoWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAll(&nfpData));
    EXPECT_EQ(expectedCounter, nfpData.systemInfo.systemWriteCounter);
}

TEST_F(NfpAuto, TestCaseGetModelInfo)
{
    nn::nfp::ModelInfo modelInfo;

    // テストに使用するタグをつくります。
    nnt::nfp::DoCreateNormalTag();

    // MOUNT 状態で ROM を読み込んでみます。
    std::memset(&modelInfo, 0xFF, sizeof(modelInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    EXPECT_TRUE(std::memcmp(nnt::nfp::TestCharacterId,
                            modelInfo.characterId,
                            nn::nfp::CharacterIdSize) == 0);
    EXPECT_EQ(nnt::nfp::TestNumberingId, modelInfo.numberingId);
    EXPECT_EQ(nnt::nfp::TestSeriesId, modelInfo.seriesId);
    EXPECT_EQ(nnt::nfp::TestNfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    // タグの ROM 領域だけをマウントします。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountRomWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    EXPECT_TRUE(std::memcmp(nnt::nfp::TestCharacterId,
                            modelInfo.characterId,
                            nn::nfp::CharacterIdSize) == 0);
    EXPECT_EQ(nnt::nfp::TestNumberingId, modelInfo.numberingId);
    EXPECT_EQ(nnt::nfp::TestSeriesId, modelInfo.seriesId);
    EXPECT_EQ(nnt::nfp::TestNfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));

    // タグデータを壊します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::MountWithRetry());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::BreakTagWithRetry(nn::nfp::BreakType_Hmac));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());

    // 壊れていても ROM 領域だけならマウントできます。
    nnt::nfp::DoMountRom();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    EXPECT_TRUE(std::memcmp(nnt::nfp::TestCharacterId,
                            modelInfo.characterId,
                            nn::nfp::CharacterIdSize) == 0);
    EXPECT_EQ(nnt::nfp::TestNumberingId, modelInfo.numberingId);
    EXPECT_EQ(nnt::nfp::TestSeriesId, modelInfo.seriesId);
    EXPECT_EQ(nnt::nfp::TestNfpType, modelInfo.nfpType);
    EXPECT_TRUE(nnt::nfp::IsAllZero(modelInfo._reserved, sizeof(modelInfo._reserved)));
}

TEST_F(NfpAuto, TestCaseRepeatStartStopDetection)
{
    nnt::nfp::DoCreateNormalTag();
    const uint32_t AgingRepeat = 1000;

    for (uint32_t i = 0; i < AgingRepeat; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは状態取得のAPIが違う
        EXPECT_EQ(nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState());
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは状態取得のAPIが違う
        EXPECT_TRUE(
            (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::RW_SEARCH)
            || (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::RW_ACTIVE)
            );
#else
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
        EXPECT_TRUE(
            (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::DeviceState_Search)
            || (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::DeviceState_Active)
            );
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }

    const uint32_t WaitRange = 1000;
    for (uint32_t i = 0; i < WaitRange; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(i));
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは状態取得のAPIが違う
        EXPECT_EQ(nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState());
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(i));
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは状態取得のAPIが違う
        EXPECT_TRUE(
            (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::RW_SEARCH)
            || (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::RW_ACTIVE)
            );
#else
        EXPECT_TRUE(
            (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::DeviceState_Search)
            || (nnt::nfp::wrapper::GetDeviceState() == nn::nfp::DeviceState_Active)
            );
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
}

#if !defined(NNT_NFP_PLATFORM_NX)
TEST_F(NfpAuto, TestCaseFastDetection)
{
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeSystemWithRetry());

    // 高速タグ検出モードに切り替えます。
    NNT_EXPECT_RESULT_SUCCESS(nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_TypeA));
    NNT_EXPECT_RESULT_SUCCESS(nn::nfp::SetPowerSavingModeEnabled(false));

    // タグを検出できることを確認します。
    nnt::nfp::DoActivate();

    // タグの情報を取得します。
    nn::nfp::TagInfo tagInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    EXPECT_EQ(nn::nfp::TagType_Type2, tagInfo.type);
    EXPECT_EQ(nn::nfp::NfcProtocol_TypeA, tagInfo.protocol);

    // 高速タグ検出モードではタグをマウントできません。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Format(nnt::nfp::ZeroTagData, sizeof(nnt::nfp::ZeroTagData)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Restore());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::MountRom());

    // タグの検出を停止します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Finalize());
}
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
TEST_F(NfpAuto, TestCaseReplaceOtherRegionName)
{
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeWithRetry());

    // 現在のリージョンを取得します。
    nn::cfg::CfgRegionCode region;
    NNT_EXPECT_RESULT_SUCCESS(nn::cfg::init::GetRegion(&region));

    // 最大文字数を指定し、かつ入力元と出力先が同じ場合のテストです。
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_JAPAN:
        case nn::cfg::CFG_REGION_AMERICA:
        case nn::cfg::CFG_REGION_EUROPE:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_China, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_CHINA:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Korea, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_KOREA:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Taiwan, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_TAIWAN:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_JAPAN:
        case nn::cfg::CFG_REGION_AMERICA:
        case nn::cfg::CFG_REGION_EUROPE:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_China, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_CHINA:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Korea, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_KOREA:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Taiwan, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_TAIWAN:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }

    // 最大文字数を指定し、かつ入力元と出力先が異なる場合のテストです。
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_JAPAN:
        case nn::cfg::CFG_REGION_AMERICA:
        case nn::cfg::CFG_REGION_EUROPE:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_China, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_CHINA:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_Korea, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_KOREA:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000 };
        nn::Bit16 expected[] = { 0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
            0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_Taiwan, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_TAIWAN:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_JAPAN:
        case nn::cfg::CFG_REGION_AMERICA:
        case nn::cfg::CFG_REGION_EUROPE:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_China, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_CHINA:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_Korea, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_KOREA:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x3F00, 0x3F00, 0x3F00, 0x3F00,
            0x0100, 0x2000, 0x3000, 0x4000, 0x7F00, 0x0000 };
        nn::Bit16  dst[nn::nfp::NicknameLengthMax + 1] = { 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD,
            0xC0DE, 0xDEAD, 0xC0DE, 0xDEAD, 0xC0DE, 0xFFFF };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_Taiwan, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_TAIWAN:
            EXPECT_TRUE(std::memcmp(dst, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(dst, expected, sizeof(nickName)) == 0);
            break;
        }
    }

    // 最小文字数を指定した場合のテストです。
    {
        nn::Bit16 nickName[] = { 0x0100, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_China, 0x3F00));
        EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Korea, 0x3F00));
        EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
    }
    {
        nn::Bit16 nickName[] = { 0x0100, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Taiwan, 0x3F00));
        EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
    }
    {
        nn::Bit16 nickName[] = { 0x0001, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_JAPAN:
        case nn::cfg::CFG_REGION_AMERICA:
        case nn::cfg::CFG_REGION_EUROPE:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0001, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_China, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_CHINA:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0001, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Korea, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_KOREA:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }
    {
        nn::Bit16 nickName[] = { 0x0001, 0x0000 };
        nn::Bit16 expected[] = { 0x3F00, 0x0000 };
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_Taiwan, 0x3F00));
        switch (region)
        {
        case nn::cfg::CFG_REGION_TAIWAN:
            EXPECT_TRUE(std::memcmp(nickName, nickName, sizeof(nickName)) == 0);
            break;
        default:
            EXPECT_TRUE(std::memcmp(nickName, expected, sizeof(nickName)) == 0);
            break;
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Finalize());
} // NOLINT(impl/function_size)
#endif //!defined(NNT_NFP_PLATFORM_NX)
