﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/hid.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testMifare_Common.h>

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class MifareAuto : public nnt::mifare::TestFramework
{
protected:

    MifareAuto() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::mifare::InitializeHidController();
        // API呼び出しログ出力フラグ
        //nnt::mifare::wrapper::SetApiCallLoggingMode(nnt::mifare::wrapper::LogMode_Aging);
    }

    ~MifareAuto() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::mifare::wrapper::GetState() == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init)
        {
            nnt::mifare::FinalizeSystem();
        }
    }
};

//================================================================================
// テストスイート全体で共通の処理です。
//================================================================================

//================================================================================
// テストケースの実装です。
//================================================================================
TEST_F(MifareAuto, TestCaseGoodOperationNone)
{
    // 状態がNoneの時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationNone start\n");

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::GetState()
    //              nn::nfc::mifare::Initialize()
    //              nn::nfc::mifare::GetDeviceState()
    // ----------------------------------------
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_None, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected, nnt::mifare::wrapper::GetDeviceState());
    nnt::mifare::InitializeSystem();
    // Initialize()は戻り値がないので、状態がInitに変更していることを確認する
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
}

TEST_F(MifareAuto, TestCaseGoodOperationInit)
{
    // 状態がInitの時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationInit start\n");

    // 状態をInitに遷移させる
    nnt::mifare::wrapper::Initialize();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::GetState()
    //              nn::nfc::mifare::Finalize()
    //              nn::nfc::mifare::GetDeviceState()
    //              nn::nfc::mifare::AttachAvailabilityChangeEvent()
    // ----------------------------------------
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState deviceState;
    deviceState = nnt::mifare::wrapper::GetDeviceState();
    if (!(deviceState == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected ||
            deviceState == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init))
    {
        EXPECT_TRUE(false);
        NN_LOG("TestCaseGoodOperationDeviceUnexpected: State Error %d", deviceState);
    }
    nn::os::SystemEventType availabilityChangeEvent;
    nnt::mifare::wrapper::AttachAvailabilityChangeEvent(&availabilityChangeEvent);
    nnt::mifare::wrapper::Finalize();
    // Finalizeは戻り値がないので、状態がNoneに変更していることを確認する
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_None, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected, nnt::mifare::wrapper::GetDeviceState());
}

TEST_F(MifareAuto, TestCaseGoodOperationDeviceUnexpected)
{
    // デバイス状態がUnexpected時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceUnexpected start\n");

    // 状態をInitに遷移させる
    nnt::mifare::InitializeSystem();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::GetDeviceState()
    //              nn::nfc::mifare::ListDevices()
    // ----------------------------------------
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState deviceState;
    deviceState = nnt::mifare::wrapper::GetDeviceState();
    if (!(deviceState == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected ||
            deviceState == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init))
    {
        EXPECT_TRUE(false);
        NN_LOG("TestCaseGoodOperationDeviceUnexpected: State Error %d", deviceState);
    }
    int outCount;
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::mifare::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
    nnt::mifare::SetCurrentDeviceHandle(currentDeviceHandle);
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState());
}

TEST_F(MifareAuto, TestCaseGoodOperationDeviceInit)
{
    // デバイス状態がInit時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceUnexpected start\n");

    // 状態をInitに遷移させる
    nnt::mifare::DoSearchDeviceHandle();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::GetNpadId()
    //              nn::nfc::mifare::GetDeviceState()
    //              nn::nfc::mifare::StartDetection()
    //              nn::nfc::mifare::AttachActivateEvent()
    //              nn::nfc::mifare::AttachDeactivateEvent()
    //              nn::nfc::mifare::ListDevices()
    // ----------------------------------------
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState());
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId));
    nnt::mifare::CheckNPadId(npadId);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StartDetection());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Search, nnt::mifare::wrapper::GetDeviceState());
    int outCount;
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::mifare::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
}

TEST_F(MifareAuto, TestCaseGoodOperationDeviceActive)
{
    // デバイス状態がActive時に呼び出すことができるAPIのテストをします。
    NN_LOG("TestCaseGoodOperationDeviceActive start\n");

    // 状態をDeviceActiveに遷移させる
    nnt::mifare::DoActivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::StopDetection()
    //              nn::nfc::mifare::GetDeviceState()
    //              nn::nfc::mifare::GetNpadId()
    //              nn::nfc::mifare::GetTagInfo()
    //              nn::nfc::mifare::Read()
    //              nn::nfc::mifare::Write()
    //              nn::nfc::mifare::ListDevices()
    //              nn::nfc::mifare::AttachActivateEvent()
    //              nn::nfc::mifare::AttachDeactivateEvent()
    // ----------------------------------------
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active, nnt::mifare::wrapper::GetDeviceState());
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetNpadId(&npadId));
    nnt::mifare::CheckNPadId(npadId);
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagInfo tagInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::GetTagInfo(&tagInfo));
    // タグタイプがMifareとなっているか確認
    EXPECT_TRUE(tagInfo.type == NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagType::TagType_Mifare);
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::os::DestroySystemEvent(&activateEvent);
    nn::os::DestroySystemEvent(&deactivateEvent);
    // 読み込みブロックカウント
    const int BlockCount = 2;
    // 読み込みデータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[BlockCount] = {};
    // 読み込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[BlockCount] = {};
    // 書き込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[BlockCount] = {};
    nnt::mifare::PrepareWriteParam(writeBlockParameter, BlockCount);
    nnt::mifare::PrintWriteBlockParameter(writeBlockParameter, BlockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Write(writeBlockParameter, BlockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, BlockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Read(
            readBlockData, readBlockParameter, BlockCount));
    nnt::mifare::PrintReadBlockData(readBlockData, BlockCount);
    // 書き込んだデータを読み込めたか確認
    nnt::mifare::CheckReadData(readBlockData, writeBlockParameter, BlockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::StopDetection());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init, nnt::mifare::wrapper::GetState());
    EXPECT_EQ(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init, nnt::mifare::wrapper::GetDeviceState());
    int outCount;
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle currentDeviceHandle;
    NNT_EXPECT_RESULT_SUCCESS(
            nnt::mifare::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
}

TEST_F(MifareAuto, TestCaseGoodOperationOneBlocklWriteRead)
{
    // 1ブロックにデータを書き込んで読み込むテストをします。
    NN_LOG("TestCaseGoodOperationOneBlocklWriteRead start\n");

    // デバイス状態をActiveに遷移させる
    nnt::mifare::DoActivate();

    // 1ブロックにランダムのデータを書き込む
    // データを書き込む
    const int OneBlockCount = 1;
    // 読み込みデータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[OneBlockCount] = {};
    // 読み込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[OneBlockCount] = {};
    // 書き込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[OneBlockCount] = {};

    nnt::mifare::PrepareWriteParam(writeBlockParameter, OneBlockCount);
    nnt::mifare::PrintWriteBlockParameter(writeBlockParameter, OneBlockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Write(writeBlockParameter, OneBlockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, OneBlockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Read(
            readBlockData, readBlockParameter, OneBlockCount));
    nnt::mifare::PrintReadBlockData(readBlockData, OneBlockCount);
    // 書き込んだデータを読み込めたか確認
    nnt::mifare::CheckReadData(readBlockData, writeBlockParameter, OneBlockCount);
}

TEST_F(MifareAuto, TestCaseGoodOperationFullBlocklWriteRead)
{
    // APIで1度に書き込むことができる最大のデータを書き込むテストをします。
    NN_LOG("TestCaseGoodOperationFullBlocklWriteRead start\n");

    // デバイス状態をActiveに遷移させる
    nnt::mifare::DoActivate();

    // 16ブロックにランダムのデータを書き込む
    // データを書き込む
    const int FullBlockCount = 16;
    // 読み込みデータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[FullBlockCount] = {};
    // 読み込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[FullBlockCount] = {};
    // 書き込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[FullBlockCount] = {};

    nnt::mifare::PrepareWriteParam(writeBlockParameter, FullBlockCount);
    nnt::mifare::PrintWriteBlockParameter(writeBlockParameter, FullBlockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Write(writeBlockParameter, FullBlockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, FullBlockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Read(
            readBlockData, readBlockParameter, FullBlockCount));
    nnt::mifare::PrintReadBlockData(readBlockData, FullBlockCount);
    // 書き込んだデータを読み込めたか確認
    nnt::mifare::CheckReadData(readBlockData, writeBlockParameter, FullBlockCount);
}

TEST_F(MifareAuto, TestCaseInvalidOperationNoneDeathTest)
{
    // 状態がNoneの時に呼び出すことができないAPIのテストをします。
    // WinのDeathTestで実施します。
    NN_LOG("TestCaseInvalidOperationNoneDeathTest start\n");
#if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::Finalize()
    //            : nn::nfc::mifare::AttachActivateEvent()
    //            : nn::nfc::mifare::AttachDeactivateEvent()
    //            : nn::nfc::mifare::GetNpadId()
    //            : nn::nfc::mifare::ListDevices()
    //            : nn::nfc::mifare::StartDetection()
    //            : nn::nfc::mifare::StopDetection()
    //            : nn::nfc::mifare::Read()
    //            : nn::nfc::mifare::Write()
    //            : nn::nfc::mifare::GetTagInfo()
    // ----------------------------------------
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::Finalize());
    int outCount;
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle currentDeviceHandle;
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::ListDevices(&currentDeviceHandle, &outCount, 1));
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::AttachActivateEvent(&activateEvent));
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::AttachDeactivateEvent(&deactivateEvent));
    nn::hid::NpadIdType npadId;
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::GetNpadId(&npadId));
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::StartDetection());
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::StopDetection());
    const int blockCount = 2;
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[blockCount] = {};
    nnt::mifare::PrepareReadParam(readBlockParameter, blockCount);
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::Read(readBlockData, readBlockParameter, blockCount));
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::Write(writeBlockParameter, blockCount));
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagInfo tagInfo;
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::wrapper::GetTagInfo(&tagInfo));
#endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
}

TEST_F(MifareAuto, TestCaseInvalidOperationInitDeathTest)
{
    // 状態がInitの時に呼び出すことができないAPIのテストをします。
    // WinのDeathTestで実施します。
    NN_LOG("TestCaseInvalidOperationNoneDeathTest start\n");
#if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    // Init状態に遷移させる
    nnt::mifare::InitializeSystem();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::Initialize()
    // ----------------------------------------
    NNT_MIFARE_ASSERT_DEATH(nnt::mifare::InitializeSystem());
#endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
}

TEST_F(MifareAuto, TestCaseInvalidOperationDeviceUnexpected)
{
    // デバイス状態がUnexpected時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceUnexpected start\n");

    // 状態をInitに遷移させる、デバイス状態はUnexpectedのまま
    nnt::mifare::InitializeSystem();

    // 連続して試験を行うとListdevicesで取得しなくても
    // 有効なデバイスハンドルを設定すると成功してしまう仕様なので(SIGLO-29745)
    // ここでは無効なデバイスハンドルを使用するようにします。
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceHandle invalidDeviceHandle = nnt::mifare::GetCurrentDeviceHandle();
    invalidDeviceHandle._id = 0xFFFFFFFF;
    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::GetNpadId()
    //              nn::nfc::mifare::StartDetection()
    //              nn::nfc::mifare::StopDetection()
    //              nn::nfc::mifare::Read()
    //              nn::nfc::mifare::Write()
    //              nn::nfc::mifare::GetTagInfo()
    //              nn::nfc::mifare::AttachActivateEvent()
    //              nn::nfc::mifare::AttachDeactivateEvent()
    // ----------------------------------------
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::GetNpadId(&npadId, invalidDeviceHandle));
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::StartDetection(invalidDeviceHandle));
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::StopDetection(invalidDeviceHandle));
    const int blockCount = 2;
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[blockCount] = {};
    nnt::mifare::PrepareWriteParam(writeBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::Write(writeBlockParameter, blockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::Read(readBlockData, readBlockParameter, blockCount));
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagInfo tagInfo;
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::GetTagInfo(&tagInfo));
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::AttachActivateEvent(&activateEvent, invalidDeviceHandle));
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound,
            nnt::mifare::wrapper::AttachDeactivateEvent(&deactivateEvent, invalidDeviceHandle));
}

TEST_F(MifareAuto, TestCaseInvalidOperationDeviceInit)
{
    // デバイス状態がInit時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceInit start\n");

    // 状態をInitに遷移させる
    nnt::mifare::DoSearchDeviceHandle();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::StopDetection()
    //              nn::nfc::mifare::Read()
    //              nn::nfc::mifare::Write()
    //              nn::nfc::mifare::GetTagInfo()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultInvalidDeviceState,
            nnt::mifare::wrapper::StopDetection());
    const int blockCount = 2;
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[blockCount] = {};
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[blockCount] = {};
    nnt::mifare::PrepareWriteParam(writeBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultInvalidDeviceState,
            nnt::mifare::wrapper::Write(writeBlockParameter, blockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, blockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultInvalidDeviceState,
            nnt::mifare::wrapper::Read(readBlockData, readBlockParameter, blockCount));
    NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::TagInfo tagInfo;
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultInvalidDeviceState,
            nnt::mifare::wrapper::GetTagInfo(&tagInfo));
}

TEST_F(MifareAuto, TestCaseInvalidOperationDeviceActive)
{
    // デバイス状態がActive時に呼び出すことができないAPIのテストをします。
    NN_LOG("TestCaseInvalidOperationDeviceActive start\n");

    // デバイス状態をActiveに遷移させる
    nnt::mifare::DoActivate();

    // ----------------------------------------
    // テスト対象 : nn::nfc::mifare::StartDetection()
    // ----------------------------------------
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultInvalidDeviceState,
            nnt::mifare::wrapper::StartDetection());
}

TEST_F(MifareAuto, TestCaseInvalidOperationOverBlocklWriteRead)
{
    // APIで1度に書き込むことができる最大を超えるデータを書き込むテストをします。
    NN_LOG("TestCaseInvalidOperationOverBlocklWriteRead start\n");

    // デバイス状態をActiveに遷移させる
    nnt::mifare::DoActivate();

    // 47ブロック以上にランダムのデータを書き込むみ読み込む
    // データを書き込む
    const int WriteFullBlockCount = 48;
    const int ReadFullBlockCount = 48;
    // 読み込みデータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) readBlockData[ReadFullBlockCount] = {};
    // 読み込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) readBlockParameter[ReadFullBlockCount] = {};
    // 書き込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) writeBlockParameter[WriteFullBlockCount] = {};

    nnt::mifare::PrepareWriteParam(writeBlockParameter, WriteFullBlockCount);
    nnt::mifare::PrintWriteBlockParameter(writeBlockParameter, WriteFullBlockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultAccessError,
            nnt::mifare::wrapper::Write(writeBlockParameter, WriteFullBlockCount));
    nnt::mifare::PrepareReadParam(readBlockParameter, ReadFullBlockCount);
    NNT_EXPECT_RESULT_FAILURE(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultAccessError,
            nnt::mifare::wrapper::Read(
            readBlockData, readBlockParameter, ReadFullBlockCount));
    nnt::mifare::PrintReadBlockData(readBlockData, ReadFullBlockCount);
}
