﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testMifare_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // エイジングの回数を指定します。タグの書き込み回数には最大 110,000 回の制限があります。
    const int AgingCountMax = 10000;
    // ブロックカウント最小
    const int MinBlockCount = 1;
    // ブロックカウント最大
    const int MaxBlockCount = 16;
    // 読み込みデータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) g_readBlockData[MaxBlockCount] = {};
    // 読み込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) g_readBlockParameter[MaxBlockCount] = {};
    // 書き込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) g_writeBlockParameter[MaxBlockCount] = {};
} // end of anonymous namespace

//================================================================================
// タグの共有領域やアプリケーション専用領域の読み書きを繰り返すエイジングです。
// 必ず新しいタグを設置してからテストを開始してください。
// 使い込んだタグでは耐性を超えてしまい、書き込みの失敗が頻発する可能性があります。
//================================================================================

//================================================================================
// テスト用のユーティリティです。
//================================================================================

//================================================================================
// エイジングテスト用のフレームワークです。
//================================================================================

class MifareAgingStableTag : public nnt::mifare::TestFramework
{
protected:

    MifareAgingStableTag() NN_NOEXCEPT
    {
        //amiibo設定実行用にoeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::mifare::InitializeHidController();
        nnt::mifare::wrapper::SetApiCallLoggingMode(nnt::mifare::wrapper::LogMode_Aging);
    }

    void TestReadWrite(int blockCount) NN_NOEXCEPT;
};

void MifareAgingStableTag::TestReadWrite(int blockCount)
{
    NN_LOG("Mifare AGING: - TestReadWrite Start\n");
    nnt::mifare::PrepareWriteParam(g_writeBlockParameter, blockCount);
    nnt::mifare::PrintWriteBlockParameter(g_writeBlockParameter, blockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Write(g_writeBlockParameter, blockCount));
    nnt::mifare::PrepareReadParam(g_readBlockParameter, blockCount);
    NNT_EXPECT_RESULT_SUCCESS(nnt::mifare::wrapper::Read(
            g_readBlockData, g_readBlockParameter, blockCount));
    nnt::mifare::PrintReadBlockData(g_readBlockData, blockCount);
    // 書き込んだデータを読み込めたか確認
    nnt::mifare::CheckReadData(g_readBlockData, g_writeBlockParameter, blockCount);
    return;
}

//================================================================================
// エイジングの実装です。
//================================================================================

TEST_F(MifareAgingStableTag,TestCaseAgingStableTag)
{
    // 書き込みブロックカウント
    int blockCount = MinBlockCount;

    NNT_MIFARE_ASSERT_RESULT_SUCCESS_FATAL(nnt::mifare::InitializeWithRetry());
    nnt::mifare::DoSearchDeviceHandle();

    // タグの耐用回数までエイジングを行います。
    for (int i = 0; i < AgingCountMax; ++i)
    {
        NN_LOG("MIFARE AGING: @COUNT = %d/%d\n", i + 1, AgingCountMax);
        //StopDetection()が実行可能な状態の時のみStopDetection()を実行する
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState deviceState = nnt::mifare::wrapper::GetDeviceState();
        if((deviceState != NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init) &&
           (deviceState != NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected))
        {
            nn::Result stopResult = nnt::mifare::wrapper::StopDetection();
            if (stopResult.IsFailure())
            {
                NNT_MIFARE_AGING_ASSERT_RESULT_SUCCESS(stopResult);
                continue;
            }
        }

        // タグの検出を開始します。
        nn::Result startResult = nnt::mifare::wrapper::StartDetection();
        if (startResult.IsFailure())
        {
            NNT_MIFARE_AGING_ASSERT_RESULT_SUCCESS(startResult);
            continue;
        }

        // タグが見つかるまで待機します。
        bool isActivated = nnt::mifare::WaitForActivate();
        if (!isActivated)
        {
            NNT_MIFARE_AGING_ASSERT_EQUAL_N(true, isActivated);
            continue;
        }

        // データ領域の読み書きに関するテストです。
        TestReadWrite(blockCount);
        // ブロックは最大16ブロック
        blockCount++;
        if (blockCount > MaxBlockCount)
        {
            blockCount = MinBlockCount;
        }
    }

    NNT_MIFARE_ASSERT_RESULT_SUCCESS_FATAL(nnt::mifare::FinalizeSystem());
}
