﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNews_Util.h"

#include <nn/news/detail/service/core/news_FileSystem.h>
#include <nn/news/detail/service/core/news_FileSystemConfig.h>

namespace nnt { namespace news {

namespace
{
#if NN_DETAIL_NEWS_SYSTEM_STORAGE_MOUNT_ALWAYS == 1

    nn::Result MountSystemSaveData(const char* name, nn::fs::SystemSaveDataId id, int64_t size, int64_t journalSize) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nn::fs::MountSystemSaveData(name, id))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, size, journalSize, 0));
                NN_RESULT_DO(nn::fs::MountSystemSaveData(name, id));
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

#endif
}

nn::Result MountSystemStorage(nn::fs::SystemSaveDataId id) NN_NOEXCEPT
{
#if NN_DETAIL_NEWS_SYSTEM_STORAGE_MOUNT_ALWAYS == 1

    NN_RESULT_DO(MountSystemSaveData(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME, id,
        nn::news::detail::service::core::SystemStorageSize, nn::news::detail::service::core::SystemStorageJournalSize));

#else

    NN_UNUSED(id);

#endif

    NN_RESULT_SUCCESS;
}

void UnmountSystemStorage() NN_NOEXCEPT
{
#if NN_DETAIL_NEWS_SYSTEM_STORAGE_MOUNT_ALWAYS == 1

    nn::fs::Unmount(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME);

#endif
}

nn::Result PostLocalNews(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    static nn::Bit8 g_MountRomCacheBuffer[4 * 1024];

    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
    NN_ABORT_UNLESS(mountRomCacheUseSize <= sizeof (g_MountRomCacheBuffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", g_MountRomCacheBuffer, sizeof (g_MountRomCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    nn::fs::FileHandle handle = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    static char buffer[2 * 1024 * 1024];
    size_t read = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&read, handle, 0, buffer, sizeof (buffer)));
    NN_RESULT_DO(nn::news::PostLocalNews(buffer, read));

    NN_RESULT_SUCCESS;
}

}}
