﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "GenTestlist_Win.h"
#include "NetTest/Horizon/NetTest_Types.h"
#include "NetTest/Horizon/NetTest_OS.h"

#include <cstdio>

#include <windows.h>

using namespace ParserHelpers;

TestlistGenerator::TestlistGenerator() NN_NOEXCEPT
    : m_platformCount(0),
      m_pRootDefine(nullptr),
      m_pRootCommandParser(nullptr),
      m_buildTypeCount(0),
      m_pReadFile(nullptr),
      m_pWriteFile(nullptr),
      m_curPlatform(0),
      m_curBuildType(0)
{
    memset(m_pBaseTestlistFileIn, '\0', sizeof(m_pBaseTestlistFileIn));
    memset(m_pOutDir,             '\0', sizeof(m_pOutDir));
    memset(m_pBuildTypesParam,    '\0', sizeof(m_pBuildTypesParam));
    memset(m_pBuildTypes,         '\0', sizeof(m_pBuildTypes));
}

bool TestlistGenerator::Init() NN_NOEXCEPT
{
    int argC = 0;
    const char * const * pArgV = nullptr;

    NETTEST_GET_ARGS(argC, pArgV);

    m_parser.AddParser(NATF::Utils::StringParser ("--BaseTestlistFileIn", nullptr, m_pBaseTestlistFileIn, sizeof(m_pBaseTestlistFileIn)));
    m_parser.AddParser(NATF::Utils::StringParser ("--OutDir", nullptr, m_pOutDir, sizeof(m_pOutDir)));
    m_parser.AddParser(NATF::Utils::StringParser ("--BuildTypes", nullptr, m_pBuildTypesParam, sizeof(m_pBuildTypesParam)));

    if (!m_parser.Parse(argC, pArgV))
    {
        NN_LOG(" * Failed to parse command line arguements!\n\n");
        return false;
    }

    if(strlen(m_pBuildTypesParam) == 0)
    {
        NN_LOG(" * ERROR: --BuildTypes must not be blank!\n\n");
        return false;
    }

    uint32_t buildTypeCount = 1;
    for(char* pCur = m_pBuildTypesParam; *pCur != '\0'; ++pCur)
    {
        if(',' == *pCur)
        {
            ++buildTypeCount;
        }
    }

    if(buildTypeCount > MaxBuildTypes)
    {
        NN_LOG(" * ERROR: --BuildTypes only supports up to %d platforms at once.\n\n", MaxBuildTypes);
        return false;
    }

    char* pBeginBuildType = m_pBuildTypesParam;
    char* pCur = m_pBuildTypesParam;
    do
    {
        if(*pCur == ',' || *pCur == '\0')
        {
            if(pCur - pBeginBuildType > MaxBuildTypeStrBufLen - 1) // -1 for null terminator
            {
                NN_LOG(" * ERROR: Each platform in --BuildTypes must be less than %d in length.\n\n", MaxBuildTypeStrBufLen);
                return false;
            }

            // Buffer will already be null terminated in the constructor
            memcpy(m_pBuildTypes[m_buildTypeCount], pBeginBuildType, pCur - pBeginBuildType);
            pBeginBuildType = pCur + 1;
            ++m_buildTypeCount;

            if(*pCur == '\0')
            {
                break;
            }
        }

        ++pCur;
    } while(NN_STATIC_CONDITION(TRUE));

    RETURN_FALSE_ON_FALSE(RegisterAllCommandParsers());
    RETURN_FALSE_ON_FALSE(FindPlatformDefinitions(m_pBaseTestlistFileIn));

    if(m_platformCount == 0)
    {
        NN_LOG("ERROR: Could not find %s\n", CommandBeginDefinePlatform);
        return false;
    }

    NN_LOG("Platform Count: %d\n", m_platformCount);
    for(uint32_t iPlatform = 0; iPlatform < m_platformCount; ++iPlatform)
    {
        NN_LOG("Line: %-4d, Name: %s\n", m_pPlatforms[iPlatform].startLine, m_pPlatforms[iPlatform].pName);
        NN_LOG("Data:\n%s\n\n", m_pPlatforms[iPlatform].pConstants);
    }

    return true;
}

bool TestlistGenerator::RegisterCommandParser(CommandParser::ParserFn parserFn) NN_NOEXCEPT
{
    CommandParser* pNewParser = new CommandParser(parserFn);
    if(nullptr == pNewParser)
    {
        NN_LOG("Allocation error!\n\n");
        return false;
    }

    if(nullptr == m_pRootCommandParser)
    {
        m_pRootCommandParser.reset(pNewParser);
    }
    else
    {
        CommandParser* pLast = m_pRootCommandParser.get();
        while(nullptr != pLast->pNext)
        {
            pLast = pLast->pNext;
        }

        pLast->pNext = pNewParser;
    }

    return true;
}

bool TestlistGenerator::RegisterAllCommandParsers() NN_NOEXCEPT
{
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::IgnorePlatformDefinition));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForDisablePlatform));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForIgnoreEmptyLine));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForInclude));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForComment));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForBuild));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForPlatform));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForConstants));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForParamLoop));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForLoopParam));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckOnlyPlatforms));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForDefinitions));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForDefineReference));
    RETURN_FALSE_ON_FALSE(RegisterCommandParser(&TestlistGenerator::CheckForUnexpectedCommand));

    return true;
}

bool TestlistGenerator::ParseAndWriteLine(ParseInfo& parseInfo) NN_NOEXCEPT
{
    const char* pPrev = parseInfo.pLine;
    const char* pFind = nullptr;

    do
    {
        pFind = strchr(pPrev, CommandChar);
        if(nullptr != pFind)
        {
            size_t restOfLineLen = LineLen(pFind);
            ParseInfo newParseInfo(pFind, restOfLineLen, parseInfo);

            // Write the first chunk with no commands.
            size_t writeLen = pFind - pPrev;

            bool isContainsWhiteNonWhiteSpace = ContainsNonWhiteSpace(pPrev, writeLen);

            if(isContainsWhiteNonWhiteSpace || false == parseInfo.isIgnoreBeginningWhiteSpaceEnabled)
            {
                size_t bytesWritten = fwrite(pPrev, 1, writeLen, m_pWriteFile);
                if( bytesWritten != writeLen )
                {
                    NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
                    return false;
                }
            }

            // Loop through all registerd command parsers and try to parse the line.
            for(CommandParser* pCurParser = m_pRootCommandParser.get(); nullptr != pCurParser; pCurParser = pCurParser->pNext)
            {
                CommandParser::ParserFn pParserFn = pCurParser->pParserFn;
                int32_t bytesParsed = (this->*pParserFn)(newParseInfo);
                if(bytesParsed != 0)
                {
                    return (bytesParsed > 0);
                }
            }

            // No command was found, write one byte and search again.
            size_t bytesWritten = fwrite(pFind, 1, 1, m_pWriteFile);
            if( bytesWritten != 1 )
            {
                NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
                return false;
            }

            pPrev = pFind + 1;
        }
        else
        {
            size_t writeLen = LineLen(pPrev);

            // Should we ignore an empty line?
            if(parseInfo.isIgnoreEmptyLinesEnabled && writeLen == 1 && *pPrev == '\n')
            {
                pFind = nullptr; // We are done searching this line
                continue;
            }

            // Write the whole line
            size_t bytesWritten = fwrite(pPrev, 1, writeLen, m_pWriteFile);
            if( bytesWritten != writeLen )
            {
                NN_LOG("File error while writing! err: %d\n\n", ferror(m_pWriteFile));
                return false;
            }

            pFind = nullptr; // We are done searching this line
        }
    } while(nullptr != pFind);

    return true;
}

bool TestlistGenerator::FindPlatformDefinitions(const char* pFileName) NN_NOEXCEPT
{
    bool isSuccess = true;
    uint32_t lineNumber = 0;

    std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
    if(nullptr == pReadBuf)
    {
        NN_LOG("Allocation error!\n\n");
        return false;
    }

    FILE* pReadFile = fopen(pFileName, "rt");
    if( !pReadFile )
    {
        NN_LOG("Failed to open file: %s\n\n", pFileName);
        isSuccess = false;
        goto out;
    }

    // Read through whole file and look for platform definitions
    while(!feof(pReadFile))
    {
        char* pRet = fgets(pReadBuf.get(), ReadWriteBufLen, pReadFile);
        if( pRet )
        {
            ++lineNumber;
            if(pRet[0] == CommandChar)
            {
                size_t lineLen = LineLen(pRet);
                if(lineLen >= CommandIncludeLen && memcmp(pRet, CommandInclude, CommandIncludeLen) == 0)
                {
                    if(lineLen <= 2)
                    {
                        NN_LOG("Line: %-4d - ERROR: A file name must be specified after %s\n\n", lineNumber, CommandInclude);
                        isSuccess = false;
                        goto out;
                    }

                    char* pIncludeFileName = pRet + CommandIncludeLen + 1; // +1 for space

                    // Remove newline char from the end.
                    size_t fileNameLen = strlen(pIncludeFileName);
                    if(pIncludeFileName[fileNameLen - 1] == '\n')
                    {
                        pIncludeFileName[fileNameLen - 1] = '\0';
                    }

                    isSuccess = FindPlatformDefinitions(pIncludeFileName);
                    if(false == isSuccess)
                    {
                        goto out;
                    }
                }
                else if(!CheckForPlatformDefinition(pReadFile, pReadBuf.get(), ReadWriteBufLen, lineNumber))
                {
                    isSuccess = false;
                    goto out;
                }
            }
        }
    }

out:
    if(nullptr != pReadFile)
    {
        fclose(pReadFile);
        pReadFile = nullptr;
    }

    return isSuccess;
}

bool TestlistGenerator::CheckForPlatformDefinition(FILE* pFile, const char* pCheck, size_t lineLen, uint32_t& lineNumber) NN_NOEXCEPT
{
    if(lineLen >= CommandBeginDefinePlatformLen && memcmp(pCheck, CommandBeginDefinePlatform, CommandBeginDefinePlatformLen) == 0)
    {
        uint32_t startLine = lineNumber;
        const char* pPlatformName = pCheck + CommandBeginDefinePlatformLen + 1; // +1 for space
        size_t restOfLine = LineLen(pPlatformName);

        NN_LOG("Line: %-4d - Found: %s\n", lineNumber, CommandBeginDefinePlatform);

        if(m_platformCount >= MaxPlatforms)
        {
            NN_LOG("Line: %-4d - Error after %s, Maximum platforms reached: %d\n", lineNumber, CommandBeginDefinePlatform, MaxPlatforms);
            return false;
        }

        if(restOfLine <= 1)
        {
            NN_LOG("Line: %-4d - Failed to find name of platform after %s\n", lineNumber, CommandBeginDefinePlatform);
            return false;
        }

        // -1 for restOfLine is to remove endline, -1 for MaxPlatformNameBufLen is to leave room for null terminator
        if((restOfLine - 1) > (MaxPlatformNameBufLen - 1))
        {
            NN_LOG("Line: %-4d - Error after %s. Platform name is too long! Max platform name length: %d\n", lineNumber, CommandBeginDefinePlatform, MaxPlatformNameBufLen - 1);
            return false;
        }

        m_pPlatforms[m_platformCount].startLine = lineNumber;

        // Copy platform name into struct
        memcpy(m_pPlatforms[m_platformCount].pName, pPlatformName, restOfLine - 1); // -1 to remove endline
        m_pPlatforms[m_platformCount].pName[restOfLine - 1] = '\0'; // null terminate name
        size_t platformDataLen = 0;

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return false;
        }

        // Read platform definition into buffers
        while(!feof(pFile))
        {
            char* pRet = fgets(pReadBuf.get(), ReadWriteBufLen, pFile);
            if( pRet )
            {
                ++lineNumber;
                lineLen = strlen(pRet);

                // Check for end command
                if(lineLen >= CommandEndDefinePlatformLen && memcmp(pRet, CommandEndDefinePlatform, CommandEndDefinePlatformLen) == 0)
                {
                    NN_LOG("Line: %-4d - Found: %s\n", lineNumber, CommandEndDefinePlatform);

                    // Strip newline char off the end
                    if(m_pPlatforms[m_platformCount].pConstants[platformDataLen - 1] == '\n')
                    {
                        m_pPlatforms[m_platformCount].pConstants[platformDataLen - 1] = '\0';
                    }
                    else
                    {
                        m_pPlatforms[m_platformCount].pConstants[platformDataLen] = '\0';
                    }

                    ++m_platformCount;
                    return true;
                }
                // Check for define platform command
                else if(lineLen >= CommandBeginDefinePlatformLen && memcmp(pRet, CommandBeginDefinePlatform, CommandBeginDefinePlatformLen) == 0)
                {
                    NN_LOG("Line: %-4d - Error: Cannot have nested %s\n", lineNumber, CommandBeginDefinePlatform);
                    return false;
                }
                else
                {
                    char* pFindCommand = strchr(pRet, CommandChar);
                    size_t restOfLineLen = LineLen(pFindCommand);
                    if(restOfLineLen >= CommandPlatformConstantsLen && memcmp(pFindCommand, CommandPlatformConstants, CommandPlatformConstantsLen) == 0)
                    {
                        NN_LOG("Line: %-4d - ERROR %s can't be inside of a platform definition!\n", lineNumber, CommandPlatformConstants);
                        return false;
                    }
                }

                // Check buffer size
                if((lineLen + platformDataLen) > (MaxPlatformConstantsBufLen - 1))
                {
                    NN_LOG("Error after %s. Platform data is too large. Max platform data length: %d. Platform started at line %d.\n", CommandBeginDefinePlatform, MaxPlatformConstantsBufLen - 1, startLine);
                    return false;
                }

                // Add data to the end of the buffer
                memcpy(&m_pPlatforms[m_platformCount].pConstants[platformDataLen], pRet, lineLen);
                platformDataLen += lineLen;
            }
        }

        NN_LOG("Error: Found end of file before '%s'. Platform definition started at line %d\n", CommandEndDefinePlatform, startLine);
        return false;
    }

    return true;
}

bool TestlistGenerator::GenerateTestlist(uint32_t platform, uint32_t buildType) NN_NOEXCEPT
{
    bool isSuccess = true;
    std::string outFile;
    m_curPlatform  = platform;
    m_curBuildType = buildType;
    size_t foundIndex = std::string::npos;
    char* pBeginName  = nullptr;
    char* pLastSlash1 = nullptr;
    char* pLastSlash2 = nullptr;

    NN_LOG("\n **** Generating testlist for %s for %s ****\n\n", m_pPlatforms[m_curPlatform].pName, m_pBuildTypes[m_curBuildType]);

    m_pReadFile = fopen(m_pBaseTestlistFileIn, "rt");
    if( !m_pReadFile )
    {
        NN_LOG("Failed to open file!\n\n");
        isSuccess = false;
        goto out;
    }


    pLastSlash1 = strrchr(m_pBaseTestlistFileIn, '\\');
    pLastSlash2 = strrchr(m_pBaseTestlistFileIn, '/');

    if(nullptr == pLastSlash1 && nullptr == pLastSlash2)
    {
        pBeginName = m_pBaseTestlistFileIn;
    }
    else if(pLastSlash1 > pLastSlash2)
    {
        pBeginName = pLastSlash1 + 1;
    }
    else
    {
        pBeginName = pLastSlash2 + 1;
    }

    outFile = m_pOutDir;
    outFile += pBeginName;
    foundIndex = outFile.find_last_of(StripExt);
    if(foundIndex != std::string::npos)
    {
        size_t extLen = strlen(StripExt);
        size_t eraseIndex = foundIndex - (extLen - 1);
        outFile.erase(eraseIndex, extLen);
    }

    outFile += m_pPlatforms[m_curPlatform].pName;
    outFile += '.';
    outFile += m_pBuildTypes[m_curBuildType];
    outFile += AppendExt;

    m_pWriteFile = fopen(outFile.c_str(), "wt");
    if( !m_pWriteFile )
    {
        NN_LOG("Failed to open file!\n\n");
        isSuccess = false;
        goto out;
    }

    fwrite(AutoGeneratedMessage_1,            1, strlen(AutoGeneratedMessage_1),            m_pWriteFile);
    fwrite(m_pBaseTestlistFileIn,             1, strlen(m_pBaseTestlistFileIn),             m_pWriteFile);
    fwrite(AutoGeneratedMessage_2,            1, strlen(AutoGeneratedMessage_2),            m_pWriteFile);
    fwrite(m_pPlatforms[m_curPlatform].pName, 1, strlen(m_pPlatforms[m_curPlatform].pName), m_pWriteFile);
    fwrite(SpaceStr,                          1, strlen(SpaceStr),                          m_pWriteFile);
    fwrite(m_pBuildTypes[m_curBuildType],     1, strlen(m_pBuildTypes[m_curBuildType]),     m_pWriteFile);

    {
        FileLineReader fileReader(m_pReadFile);
        uint32_t lineNumber = 0;

        std::unique_ptr<char[]> pReadBuf(new char[ReadWriteBufLen]);
        if(nullptr == pReadBuf)
        {
            NN_LOG("Allocation error!\n\n");
            return false;
        }

        do
        {
            char* pRet = fileReader.ReadLine(pReadBuf.get(), ReadWriteBufLen);
            if( pRet )
            {
                ++lineNumber;

                ParseInfo parseInfo(fileReader, pRet, LineLen(pRet), nullptr, nullptr, lineNumber, false, true, m_pBaseTestlistFileIn);
                if(false == ParseAndWriteLine(parseInfo))
                {
                    isSuccess = false;
                    goto out;
                }
            }
        } while(false == fileReader.IsEndOfStream());
    }

out:
    if(nullptr != m_pReadFile)
    {
        fclose(m_pReadFile);
        m_pReadFile = nullptr;
    }

    if(nullptr != m_pWriteFile)
    {
        fclose(m_pWriteFile);
        m_pWriteFile = nullptr;
    }

    if(nullptr != m_pRootDefine)
    {
        delete m_pRootDefine; // Destructor recursively deletes the whole list
        m_pRootDefine = nullptr;
    }

    m_curPlatform  = 0;
    m_curBuildType = 0;

    return isSuccess;
}

bool TestlistGenerator::GenerateAllLists() NN_NOEXCEPT
{
    for(uint32_t iBuildType = 0; iBuildType < m_buildTypeCount; ++iBuildType)
    {
        for(uint32_t iPlatform = 0; iPlatform < m_platformCount; ++iPlatform)
        {
            if(m_pPlatforms[iPlatform].isEnabled)
            {
                if(!GenerateTestlist(iPlatform, iBuildType))
                {
                    NN_LOG("Failed to generate testlist for %s for %s\n", m_pPlatforms[iPlatform].pName, m_pBuildTypes[iBuildType]);
                    return false;
                }
            }
        }
    }

    return true;
}

extern "C"
{
int nnMain()
{
    TestlistGenerator testlistGenerator;

    if(!testlistGenerator.Init())
    {
        NN_LOG("\n ******************** FAIL ********************\n\n");
        return -1;
    }

    if(!testlistGenerator.GenerateAllLists())
    {
        NN_LOG("\n ******************** FAIL ********************\n\n");
        return -1;
    }

    NN_LOG("\n ******************** PASS ********************\n\n");
    return 0;
}
}
