﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include "NetTest_Port.h"

#include "Tests/CurlHttpDownloadMultiTest.h"
#include "Utils/CommandLineParser.h"

#include <nn/ssl/ssl_SessionCache.h>

#include <nnt/nntest.h>
#include <curl/curl.h>

#include <memory> // std::unique_ptr

static const uint32_t SessionCacheCount = 8;
static const uint32_t SessionTypeBufLen = 8;

// CurlSslSessionCache
TEST(natf, CurlSslSessionCache)
{
    bool isSuccess = true;
    NATF::Utils::ParserGroup parser;
    NATF::Modules::LibCurl::Params params;
    int argc;
    uint32_t timeoutMs = 0;
    nn::util::Uuid netProfile;
    const char * const * pArgv;
    static const uint32_t SdkPathBufLen = 1024 * 2;
    char pSdkPath[SdkPathBufLen] = {0};
    char pSessionTypeStr[SessionTypeBufLen];
    nn::ssl::Connection::SessionCacheMode sessionCacheMode = nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId;

    NN_NETTEST_LOG("\nEnter NATF Process\n\n");

    NETTEST_GET_ARGS(argc, pArgv);

    parser.AddParser(NATF::Utils::UuidParser    ("--NetProfile", &nn::util::InvalidUuid, netProfile));
    parser.AddParser(NATF::Utils::UInt32Parser  ("--TimeoutMs", nullptr, timeoutMs));
    parser.AddParser(NATF::Utils::StringParser  ("--SdkRoot", nullptr, pSdkPath, sizeof(pSdkPath)));
    parser.AddParser(NATF::Utils::StringParser  ("--SessionCacheMode", nullptr, pSessionTypeStr, sizeof(pSessionTypeStr)));

    if (!parser.Parse(argc, pArgv))
    {
        NN_NETTEST_LOG(" * Failed to parse command line arguements!\n\n");
        EXPECT_EQ(false, true);
        return;
    }

    NetTest::StrUpr(pSessionTypeStr);
    if(strcmp(pSessionTypeStr, "ID") == 0)
    {
        sessionCacheMode = nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId;
    }
    else if(strcmp(pSessionTypeStr, "TICKET") == 0)
    {
        sessionCacheMode = nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionTicket;
    }
    else
    {
        NN_NETTEST_LOG(" * ERROR: Unknown value for --SessionType! Valid options: 'ID', 'TICKET'\n\n");
        FAIL();
        return;
    }

    NATF::Utils::InitApi initApi(NATF::Tests::CurlHttpDownloadMulti::InitFlags);
    if(!initApi.Init(netProfile))
    {
        NN_NETTEST_LOG(" * Failed to initialize an api!\n\n");
        FAIL();
        return;
    }

    params.sessionCacheMode = sessionCacheMode;
    params.checkHash = false;

    // Params for create session cache 1
    NATF::Modules::LibCurl::Params paramsCreateSessionCache1(params);
    paramsCreateSessionCache1.expectedHttpResponse = 200;
    paramsCreateSessionCache1.doVerifyPeer         = true;
    paramsCreateSessionCache1.doVerifyHostname     = true;
    paramsCreateSessionCache1.doVerifyDate         = true;

    snprintf(paramsCreateSessionCache1.pServerCertPath, sizeof(paramsCreateSessionCache1.pServerCertPath), "%s\\Tests\\Net\\Resources\\Certs\\serverCert.pem", pSdkPath);
    strncpy(paramsCreateSessionCache1.pUrl, "https://natf.com:441/kb32", sizeof(paramsCreateSessionCache1.pUrl));

    // Params for create session cache 2
    NATF::Modules::LibCurl::Params paramsCreateSessionCache2(paramsCreateSessionCache1);
    paramsCreateSessionCache2.doVerifyHostname = false;
    snprintf(paramsCreateSessionCache2.pServerCertPath, sizeof(paramsCreateSessionCache2.pServerCertPath), "%s\\Tests\\Net\\Resources\\Certs\\Abuse\\Chains\\Chain_2\\Intermediate_1\\Leaf_1\\leaf.pem.crt", pSdkPath);
    strncpy(paramsCreateSessionCache2.pUrl, "https://natf_2.com:441/kb32", sizeof(paramsCreateSessionCache2.pUrl));

    // Params for create session cache 3
    NATF::Modules::LibCurl::Params paramsCreateSessionCache3(paramsCreateSessionCache2);
    paramsCreateSessionCache3.doVerifyDate = false; // Different from paramsCreateSessionCache2

    // Params for create session cache 4
    NATF::Modules::LibCurl::Params paramsCreateSessionCache4(paramsCreateSessionCache2);
    strncpy(paramsCreateSessionCache4.pUrl, "https://natf_4.com:441/kb32", sizeof(paramsCreateSessionCache4.pUrl));
    snprintf(paramsCreateSessionCache4.pServerCertPath, sizeof(paramsCreateSessionCache4.pServerCertPath), "%s\\Tests\\Net\\Resources\\Certs\\Abuse\\Chains\\Chain_4\\Intermediate_1\\Leaf_1\\leaf.pem.crt", pSdkPath);

    // Params for create session cache 5
    NATF::Modules::LibCurl::Params paramsCreateSessionCache5(paramsCreateSessionCache2);
    strncpy(paramsCreateSessionCache5.pUrl, "https://natf_5.com:441/kb32", sizeof(paramsCreateSessionCache5.pUrl));
    snprintf(paramsCreateSessionCache5.pServerCertPath, sizeof(paramsCreateSessionCache5.pServerCertPath), "%s\\Tests\\Net\\Resources\\Certs\\Abuse\\Chains\\Chain_5\\Intermediate_1\\Leaf_1\\leaf.pem.crt", pSdkPath);

    // Params for no session cache 1
    NATF::Modules::LibCurl::Params paramsFailSessionCache1(params);
    paramsFailSessionCache1.expectedHttpResponse = 0;
    paramsFailSessionCache1.expectedCurlReturn   = 60;
    paramsFailSessionCache1.expectedAuthReturn   = 157819;
    paramsFailSessionCache1.doVerifyPeer         = true;
    paramsFailSessionCache1.doVerifyHostname     = true;
    paramsFailSessionCache1.doVerifyDate         = true;
    paramsFailSessionCache1.pServerCertPath[0] = '\0';

    strncpy(paramsFailSessionCache1.pUrl, "https://natf.com:441/kb32", sizeof(paramsFailSessionCache1.pUrl));

    // Params for no session cache 2
    NATF::Modules::LibCurl::Params paramsFailSessionCache2(paramsFailSessionCache1);
    paramsFailSessionCache2.doVerifyHostname = false;
    strncpy(paramsFailSessionCache2.pUrl, "https://natf_2.com:441/kb32", sizeof(paramsFailSessionCache2.pUrl));

    // Params for no session cache 3
    NATF::Modules::LibCurl::Params paramsFailSessionCache3(paramsFailSessionCache2);
    paramsFailSessionCache3.doVerifyDate = false; // Different from paramsFailSessionCache2

    // Params for no session cache 4
    NATF::Modules::LibCurl::Params paramsFailSessionCache4(paramsFailSessionCache2);
    strncpy(paramsFailSessionCache4.pUrl, "https://natf_4.com:441/kb32", sizeof(paramsFailSessionCache4.pUrl));

    // Params for no session cache 5
    NATF::Modules::LibCurl::Params paramsFailSessionCache5(paramsFailSessionCache2);
    strncpy(paramsFailSessionCache5.pUrl, "https://natf_5.com:441/kb32", sizeof(paramsFailSessionCache5.pUrl));

    // Params for verifying session cache 1
    NATF::Modules::LibCurl::Params paramsPassSessionCache1(params);
    paramsPassSessionCache1.expectedHttpResponse = 200;
    paramsPassSessionCache1.doVerifyPeer         = true;
    paramsPassSessionCache1.doVerifyHostname     = true;
    paramsPassSessionCache1.doVerifyDate         = true;

    paramsPassSessionCache1.pServerCertPath[0] = '\0';
    strncpy(paramsPassSessionCache1.pUrl, "https://natf.com:441/kb32", sizeof(paramsPassSessionCache1.pUrl));

    // Params for verifying session cache 2
    NATF::Modules::LibCurl::Params paramsPassSessionCache2(paramsPassSessionCache1);
    paramsPassSessionCache2.doVerifyHostname = false;
    strncpy(paramsPassSessionCache2.pUrl, "https://natf_2.com:441/kb32", sizeof(paramsPassSessionCache2.pUrl));

    // Params for verifying session cache 3
    NATF::Modules::LibCurl::Params paramsPassSessionCache3(paramsPassSessionCache2);
    paramsPassSessionCache3.doVerifyDate = false; // Different from paramsPassSessionCache2

    // Params for verifying session cache 4
    NATF::Modules::LibCurl::Params paramsPassSessionCache4(paramsPassSessionCache2);
    strncpy(paramsPassSessionCache4.pUrl, "https://natf_4.com:441/kb32", sizeof(paramsPassSessionCache4.pUrl));

    // Params for verifying session cache 5
    NATF::Modules::LibCurl::Params paramsPassSessionCache5(paramsPassSessionCache2);
    strncpy(paramsPassSessionCache5.pUrl, "https://natf_5.com:441/kb32", sizeof(paramsPassSessionCache5.pUrl));

    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> createSessionCache1(new NATF::Tests::CurlHttpDownloadMulti("CreateSessionCache1", netProfile, timeoutMs, 1, paramsCreateSessionCache1));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> createSessionCache2(new NATF::Tests::CurlHttpDownloadMulti("CreateSessionCache2", netProfile, timeoutMs, 1, paramsCreateSessionCache2));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> createSessionCache3(new NATF::Tests::CurlHttpDownloadMulti("CreateSessionCache3", netProfile, timeoutMs, 1, paramsCreateSessionCache3));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> createSessionCache4(new NATF::Tests::CurlHttpDownloadMulti("CreateSessionCache4", netProfile, timeoutMs, 1, paramsCreateSessionCache4));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> createSessionCache5(new NATF::Tests::CurlHttpDownloadMulti("CreateSessionCache5", netProfile, timeoutMs, 1, paramsCreateSessionCache5));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> failSessionCache1(new NATF::Tests::CurlHttpDownloadMulti("FailSessionCache1", netProfile, timeoutMs, 1, paramsFailSessionCache1));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> failSessionCache2(new NATF::Tests::CurlHttpDownloadMulti("FailSessionCache2", netProfile, timeoutMs, 1, paramsFailSessionCache2));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> failSessionCache3(new NATF::Tests::CurlHttpDownloadMulti("FailSessionCache3", netProfile, timeoutMs, 1, paramsFailSessionCache3));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> failSessionCache4(new NATF::Tests::CurlHttpDownloadMulti("FailSessionCache4", netProfile, timeoutMs, 1, paramsFailSessionCache4));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> failSessionCache5(new NATF::Tests::CurlHttpDownloadMulti("FailSessionCache5", netProfile, timeoutMs, 1, paramsFailSessionCache5));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> passSessionCache1(new NATF::Tests::CurlHttpDownloadMulti("PassSessionCache1", netProfile, timeoutMs, 1, paramsPassSessionCache1));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> passSessionCache2(new NATF::Tests::CurlHttpDownloadMulti("PassSessionCache2", netProfile, timeoutMs, 1, paramsPassSessionCache2));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> passSessionCache3(new NATF::Tests::CurlHttpDownloadMulti("PassSessionCache3", netProfile, timeoutMs, 1, paramsPassSessionCache3));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> passSessionCache4(new NATF::Tests::CurlHttpDownloadMulti("PassSessionCache4", netProfile, timeoutMs, 1, paramsPassSessionCache4));
    std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> passSessionCache5(new NATF::Tests::CurlHttpDownloadMulti("PassSessionCache5", netProfile, timeoutMs, 1, paramsPassSessionCache5));

    // Make sure there is no session cache 1
    isSuccess = failSessionCache1->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Create session cache 1
    isSuccess = createSessionCache1->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache 1 was created
    isSuccess = passSessionCache1->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure there is no session cache 2
    isSuccess = failSessionCache2->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Create session cache 2
    isSuccess = createSessionCache2->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache 2 was created
    isSuccess = passSessionCache2->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure there is no session cache 3
    isSuccess = failSessionCache3->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Create session cache 3
    isSuccess = createSessionCache3->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache 3 was created
    isSuccess = passSessionCache3->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure there is no session cache 4
    isSuccess = failSessionCache4->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Create session cache 4
    isSuccess = createSessionCache4->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache 4 was created
    isSuccess = passSessionCache4->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure there is no session cache 5
    isSuccess = failSessionCache5->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Create session cache 5
    isSuccess = createSessionCache5->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache 5 was created
    isSuccess = passSessionCache5->Run(false);
    EXPECT_EQ(isSuccess, true);

    uint32_t deletedEntries = 0;

    // Delete session cache 1
    nn::Result result = nn::ssl::FlushSessionCache(&deletedEntries, "natf.com", static_cast<uint32_t>(strlen("natf.com") + 1), nn::ssl::FlushSessionCacheOptionType_SingleHost);
    if(result.IsFailure())
    {
        NN_NETTEST_LOG("\n\n\nFailed to flush session cache for host. Mod: %d, Desc: %d\n\n\n", result.GetModule(), result.GetDescription());
        FAIL();
    }

    // Verify only one session cache was deleted
    EXPECT_EQ(deletedEntries, 1);

    // Make sure session cache 1 is deleted.
    isSuccess = failSessionCache1->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache for 2 still exsists
    isSuccess = passSessionCache2->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache for 3 still exsists
    isSuccess = passSessionCache3->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache for 4 still exsists
    isSuccess = passSessionCache4->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache for 5 still exsists
    isSuccess = passSessionCache5->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Delete session cache 2 and 3
    result = nn::ssl::FlushSessionCache(&deletedEntries, "natf_2.com", static_cast<uint32_t>(strlen("natf_2.com") + 1), nn::ssl::FlushSessionCacheOptionType_SingleHost);
    if(result.IsFailure())
    {
        NN_NETTEST_LOG("\n\n\nFailed to flush session cache for host. Mod: %d, Desc: %d\n\n\n", result.GetModule(), result.GetDescription());
        FAIL();
    }

    // Verify that two session caches were deleted
    EXPECT_EQ(deletedEntries, 2);

    // Make sure session cache 2 is deleted.
    isSuccess = failSessionCache2->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache 3 is deleted.
    isSuccess = failSessionCache3->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache for 4 still exsists
    isSuccess = passSessionCache4->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache for 5 still exsists
    isSuccess = passSessionCache5->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Delete the rest of the session cache
    result = nn::ssl::FlushSessionCache(&deletedEntries, nullptr, 0, nn::ssl::FlushSessionCacheOptionType_AllHosts);
    if(result.IsFailure())
    {
        NN_NETTEST_LOG("\n\n\nFailed to flush session cache for all host. Mod: %d, Desc: %d\n\n\n", result.GetModule(), result.GetDescription());
        FAIL();
    }

    // Make sure both session cache were deleted
    EXPECT_EQ(deletedEntries, 2);

    // Make sure session cache 4 is deleted.
    isSuccess = failSessionCache4->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Make sure session cache 5 is deleted.
    isSuccess = failSessionCache5->Run(false);
    EXPECT_EQ(isSuccess, true);

    // Create 'SessionCacheCount' different session cache entries
    for(uint32_t iServer = 0; iServer < SessionCacheCount; ++iServer)
    {
        NATF::Modules::LibCurl::Params paramsCreateSessionCache(params);
        paramsCreateSessionCache.expectedHttpResponse = 200;
        paramsCreateSessionCache.doVerifyPeer         = true;
        paramsCreateSessionCache.doVerifyHostname     = true;
        paramsCreateSessionCache.doVerifyDate         = false;
        paramsCreateSessionCache.sessionCacheMode     = sessionCacheMode;

        snprintf(paramsCreateSessionCache.pServerCertPath, sizeof(paramsCreateSessionCache.pServerCertPath), "%s\\Tests\\Net\\Resources\\Certs\\Abuse\\Chains\\Chain_%d\\Intermediate_1\\Leaf_1\\leaf.pem.crt", pSdkPath, iServer + 1);

        snprintf(paramsCreateSessionCache.pUrl, sizeof(paramsCreateSessionCache.pUrl), "https://natf_%d.com:441/kb32", iServer + 1);

        std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> createSessionCache(new NATF::Tests::CurlHttpDownloadMulti("createSessionCache",netProfile, timeoutMs, 1, paramsCreateSessionCache));

        isSuccess = createSessionCache->Run(false);
        EXPECT_EQ(isSuccess, true);
    }

    // Clear all session cache entries
    result = nn::ssl::FlushSessionCache(&deletedEntries, nullptr, 0, nn::ssl::FlushSessionCacheOptionType_AllHosts);
    if(result.IsFailure())
    {
        NN_NETTEST_LOG("\n\n\nFailed to flush session cache for all host. Mod: %d, Desc: %d\n\n\n", result.GetModule(), result.GetDescription());
        FAIL();
    }

    // Verify all expected session caches were deleted
    EXPECT_EQ(deletedEntries, SessionCacheCount);

    // Make sure all expected session cache entries are actually deleted
    for(uint32_t iServer = 0; iServer < SessionCacheCount; ++iServer)
    {
        NATF::Modules::LibCurl::Params paramsFailWithVerify(params);
        paramsFailWithVerify.expectedHttpResponse = 0;
        paramsFailWithVerify.doVerifyPeer         = true;
        paramsFailWithVerify.doVerifyHostname     = true;
        paramsFailWithVerify.doVerifyDate         = false;
        paramsFailWithVerify.expectedHttpResponse = 0;
        paramsFailWithVerify.expectedCurlReturn   = 60;
        paramsFailWithVerify.expectedAuthReturn   = 157819;
        paramsFailWithVerify.pServerCertPath[0]   = '\0';

        snprintf(paramsFailWithVerify.pUrl, sizeof(paramsFailWithVerify.pUrl), "https://natf_%d.com:441/kb32", iServer + 1);

        std::unique_ptr<NATF::Tests::CurlHttpDownloadMulti> failWithVerify(new NATF::Tests::CurlHttpDownloadMulti("failWithVerify", netProfile, timeoutMs, 1, paramsFailWithVerify));

        isSuccess = failWithVerify->Run(false);
        EXPECT_EQ(isSuccess, true);
    }

    initApi.Finalize();

    NN_NETTEST_LOG("\nExit NATF Process\n\n");
} // NOLINT(impl/function_size)
