﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

nn::os::ThreadType             g_Mntcp_tid;
extern int mntcp_main(int argc, char* argv[]);

namespace
{
    const int TcpSocketLimit    = 16;
    const int UdpSocketLimit    = 16;
    const int ConcurrencyLimit  = 8;

    nn::socket::SystemConfigDefaultWithMemory
        <
        TcpSocketLimit, // Tcp Sockets
        UdpSocketLimit   // UDP Sockets
        > g_SocketConfigWithMemory(ConcurrencyLimit);
}


namespace nns {
namespace socket {

// Constants
const int                            TimeoutOnNicConfigurationInSec = 15;        //<! How long to wait for NIFM
const size_t                         kMaxThreads = 32;
const size_t                         kThreadStackSize = 65536;
NN_OS_ALIGNAS_THREAD_STACK uint8_t   kThreadStack[kMaxThreads + 1][kThreadStackSize];

// Socket Library
//nn::socket::ConfigDefaultWithMemory  g_SocketConfigWithMemory;

// Nifm values
struct in_addr                       g_NifmIpv4Addr = { 0 };
struct in_addr                       g_NifmIpv4Mask = { 0 };
struct in_addr                       g_NifmIpv4Gateway = { 0 };
struct in_addr                       g_NifmIpv4Dns1 = { 0 };
struct in_addr                       g_NifmIpv4Dns2 = { 0 };
bool                                 g_IsNetworkInitialized = false;

// ------------------------------------------------------------------------------------------------
// Setup network interface
// ------------------------------------------------------------------------------------------------
int SetupNetwork() NN_NOEXCEPT
{
    int            ret = 0;
    nn::Result     result = nn::ResultSuccess();
    uint32_t       timeoutSec = 0;
    bool           isSocketLibInitialized = false;
    bool           isNifmLibInitialized = false;

    do
    {
        // If already initialized
        if (g_IsNetworkInitialized == true)
        {
            NN_LOG("Network is already initialized!\n");
            break;
        }

        // Initialize NIFM
        result = nn::nifm::Initialize();
        if(result.IsFailure())
        {
            NN_LOG("Failed calling: 'nn::nifm::Initialize()'\n");
            ret = -1;
            break;
        }

        // Submit the NIFM Network Request
        nn::nifm::SubmitNetworkRequest();

        // Wait for NIFM to go online
        while(nn::nifm::IsNetworkRequestOnHold())
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            if(timeoutSec++ > TimeoutOnNicConfigurationInSec)
            {
                NN_LOG("Failed to setup NIFM network interface.\n");
                ret = -1;
                break;
            }
        }

        // Did NIFM request timeout?
        if (ret < 0)
        {
            break;  // Yes
        }

        // Is the NIFM Network Available?
        if(nn::nifm::IsNetworkAvailable() == 0)
        {
            NN_LOG("Network is not available.\n");
            ret = -1;
            break;
        }
        isNifmLibInitialized = true;

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &g_NifmIpv4Addr, &g_NifmIpv4Mask, &g_NifmIpv4Gateway, &g_NifmIpv4Dns1, &g_NifmIpv4Dns2 );

        NN_LOG( "Network Interface is UP!\n" );
        NN_LOG( "NIFM Address Ipv4      : %s\n", nn::socket::InetNtoa(g_NifmIpv4Addr));
        NN_LOG( "NIFM Network Mask Ipv4 : %s\n", nn::socket::InetNtoa(g_NifmIpv4Mask));
        NN_LOG( "NIFM Gateway Ipv4      : %s\n", nn::socket::InetNtoa(g_NifmIpv4Gateway));
        NN_LOG( "NIFM DNS1 Ipv4         : %s\n", nn::socket::InetNtoa(g_NifmIpv4Dns1));
        NN_LOG( "NIFM DNS2 Ipv4         : %s\n", nn::socket::InetNtoa(g_NifmIpv4Dns2));

        // Initialize the socket library
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        if(result.IsFailure())
        {
            NN_LOG("Failed to initialize socket library.\n");
            ret = -1;
            break;
        }
        isSocketLibInitialized = true;

        // Successfully initialized the network
        g_IsNetworkInitialized = true;

        // All done
        break;

    } while (NN_STATIC_CONDITION(true));

    // If failed - disable network
    if (ret != 0)
    {
        if (isSocketLibInitialized == true)
        {
            nn::socket::Finalize();
            isSocketLibInitialized  = false;
            ret = -1;
        }

        if (isNifmLibInitialized == true)
        {
            nn::nifm::CancelNetworkRequest();
            isNifmLibInitialized = false;
            ret = -1;
        }
        g_IsNetworkInitialized = false;

    }

    return(ret);

}  // NOLINT(impl/function_size)

// ------------------------------------------------------------------------------------------------
// Finalize Network
// ------------------------------------------------------------------------------------------------
int FinalizeNetwork() NN_NOEXCEPT
{
    nn::Result     result = nn::ResultSuccess();
    int            ret = 0;

    if (g_IsNetworkInitialized == true)
    {
        result = nn::socket::Finalize();
        if (result.IsFailure())
        {
            NN_LOG("Failed calling 'nn::socket::Finalize()'!\n");
        }

        nn::nifm::CancelNetworkRequest();
        g_IsNetworkInitialized = false;
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Start Thread
// ------------------------------------------------------------------------------------------------
void startThread(void *)
{
    char   sz_arg0[] = "mntcpswi";
    char*  v_arg[2] = {sz_arg0, NULL};
    int    c_arg = 1;
    int    ret = 0;

    ret = mntcp_main(c_arg, v_arg);
    if (ret != 0)
    {
        NN_LOG("mntcp_main returned: %d\n", ret);
    }

    return;
}


// ------------------------------------------------------------------------------------------------
// Main Loop
// ------------------------------------------------------------------------------------------------
void mainLoop() NN_NOEXCEPT
{
    int         ret = 0;

    NN_LOG("Start Up!\n");

    do
    {
        // Setup network interface
        ret = SetupNetwork();
        if (ret != 0)
        {
            NN_LOG("Failed to enable network interface\n");
            break;
        }

        /* one server, two client threads */
        nn::os::CreateThread(&g_Mntcp_tid, &startThread, NULL, kThreadStack, sizeof(kThreadStack), 30);

        nn::os::StartThread(&g_Mntcp_tid);

        nn::os::WaitThread(&g_Mntcp_tid);

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    // Finalize NIFM and Socket Library
    if (g_IsNetworkInitialized == true)
    {
        ret = FinalizeNetwork();
        if (ret != 0)
        {
            NN_LOG("Failing calling 'FinalizeNetwork()'\n");
        }
    }

    NN_LOG("Shut Down!\n");
}

}}   // Namespace nns / socket

// ------------------------------------------------------------------------------------------------
// Startup function
// ------------------------------------------------------------------------------------------------
extern "C" void nninitStartup() NN_NOEXCEPT
{
    // Setup the size of total memory heap
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    NN_ASSERT(result.IsSuccess());

    // Allcate memory block from the heap for malloc
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock(&address, MemoryHeapSize);
    NN_ASSERT( result.IsSuccess() );

    // Set memory chunk for malloc
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MemoryHeapSize);
}

// ------------------------------------------------------------------------------------------------
// nnMain
// ------------------------------------------------------------------------------------------------
extern "C" void nnMain() NN_NOEXCEPT
{
    nns::socket::mainLoop();
}
