﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>

#include <Common/testCommonUtil.h>

#include <Utils/CommandLineParser.h>

// ------------------------------------------------------------------------------------------------
// Grobal parameters
// ------------------------------------------------------------------------------------------------
namespace
{
const char TestHostList[][64] = {
    "www.google.com",
    "facebook.com",
    "twitter.com",
    "youtube.com",
    "www.netflix.com",
    "www.amazon.com",
    "www.google.co.jp",
};
const int StandardSslPortNumber = 443;

SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    nn::util::Uuid netProfile = nn::util::InvalidUuid;
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser ("--NetProfile", &nn::util::InvalidUuid, netProfile));

    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    ASSERT_TRUE(g_CommonUtil.SetupNetwork(netProfile));
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());

    ASSERT_TRUE(nn::ssl::Initialize().IsSuccess());
}

TEST(Internet, Success)
{
    SimpleHttpsClient* pHttpsClient;
    nn::ssl::Connection::VerifyOption verifyOption =
        nn::ssl::Connection::VerifyOption::VerifyOption_None;

    // Create session cache
    for (int i = 0; i < (sizeof(TestHostList) / sizeof(TestHostList[0])); i++)
    {
        pHttpsClient = new SimpleHttpsClient(
            false, // Blocking
            TestHostList[i],
            StandardSslPortNumber);

        EXPECT_TRUE(pHttpsClient->Initialize(
            nn::ssl::Context::SslVersion::SslVersion_Auto,
            verifyOption));
        EXPECT_TRUE(pHttpsClient->PerformSslHandshake(false));
        pHttpsClient->Finalize();
        delete pHttpsClient;
    }

    // Performs SSL handshake again with the session cache
    for (int i = 0; i < (sizeof(TestHostList) / sizeof(TestHostList[0])); i++)
    {
        pHttpsClient = new SimpleHttpsClient(
            false, // Blocking
            TestHostList[i],
            StandardSslPortNumber);

        EXPECT_TRUE(pHttpsClient->Initialize(
            nn::ssl::Context::SslVersion::SslVersion_Auto,
            verifyOption));
        EXPECT_TRUE(pHttpsClient->PerformSslHandshake(false));

        //EXPECT_TRUE(pHttpsClient->GetSslConnection()->FlushSessionCache().IsSuccess());
        pHttpsClient->Finalize();
        delete pHttpsClient;
    }
}

TEST(FinalizeTest, Success)
{
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
