﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ssl.h>
#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/TargetConfigs/build_Platform.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>

#include <Utils/CommandLineParser.h>


// ------------------------------------------------------------------------------------------------
// Build flags for tests to run
// ------------------------------------------------------------------------------------------------
#define PERFORM_NON_BLOCKING_IO
//#define IMPORT_SERVER_PKI
#define IMPORT_CLIENT_PKI

#ifdef IMPORT_SERVER_PKI
#define VERIFY_PEER
#define VERIFY_HOST
#define VERIFY_DATE
#endif    // IMPORT_SERVER_PKI

// ------------------------------------------------------------------------------------------------
// Configuration parameters
// ------------------------------------------------------------------------------------------------
namespace
{
const uint32_t NumberOfThreads = 8;
const size_t   ThreadStackSize = 1024 * 32;

SslTestCommonUtil                  g_CommonUtil;
NN_ALIGNAS(4096)           uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
NN_OS_ALIGNAS_THREAD_STACK uint8_t g_ThreadStack[NumberOfThreads][ThreadStackSize];
nn::os::ThreadType                 g_ThreadTid[NumberOfThreads];
}

extern "C" void nninitStartup()
{
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MemoryHeapSize );
}

void WorkerFunction(void* arg)
{
    bool isPrintServerCert    = false;
#if defined(PERFORM_NON_BLOCKING_IO)
    bool isPerformNonBlocking = true;
#else
    bool isPerformNonBlocking = false;
#endif

    nn::ssl::Connection::VerifyOption verifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
#if defined(VERIFY_PEER)
    verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa;
#endif
#if defined(VERIFY_HOST)
    verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_HostName;
#endif
#if defined(VERIFY_DATE)
    verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_DateCheck;
#endif

    SimpleHttpsClient httpsClient(isPerformNonBlocking,
        ServerName,
#ifdef IMPORT_CLIENT_PKI
        ServerPort_ClientAuth
#else
        ServerPort_Normal
#endif
);


    EXPECT_TRUE(httpsClient.Initialize(
        nn::ssl::Context::SslVersion::SslVersion_Auto,
        verifyOption));
#ifdef IMPORT_SERVER_PKI
    EXPECT_TRUE(httpsClient.ImportServerPki(
        g_pTestCaCert,
        sizeof(g_pTestCaCert),
        nn::ssl::CertificateFormat_Pem));
#endif
#ifdef IMPORT_CLIENT_PKI
    EXPECT_TRUE(httpsClient.ImportClientPki(
            (const char *)g_pTestClientPki,
            (const char *)g_pTestClientPkiPassword,
            g_pTestClientPkiSize,
            g_TestClientPkiPasswordLength));
#endif
    EXPECT_TRUE(httpsClient.PerformSslHandshake(isPrintServerCert));
    EXPECT_TRUE(httpsClient.SendHttpGetOverSsl());
    EXPECT_TRUE(httpsClient.ReceiveAllHttpData());

    httpsClient.Finalize();
}

// ------------------------------------------------------------------------------------------------
// Tests
// ------------------------------------------------------------------------------------------------
TEST(Setup, Success)
{
    nn::util::Uuid netProfile = nn::util::InvalidUuid;
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser ("--NetProfile", &nn::util::InvalidUuid, netProfile));

    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    // Setup network interface
    ASSERT_TRUE(g_CommonUtil.SetupNetwork(netProfile));

    nn::Result result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);
    EXPECT_TRUE(result.IsSuccess());

    result = nn::ssl::Initialize();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(QuickSingleThread, Success)
{
    int threadCount = 1;
    int i = 0;
    nn::Result result;
    for(i =0; i < threadCount; i++)
    {
        result = nn::os::CreateThread(
            &g_ThreadTid[i],
            WorkerFunction,
            nullptr,
            g_ThreadStack[i],
            ThreadStackSize,
            nn::os::LowestThreadPriority);
        EXPECT_TRUE(result.IsSuccess());
    }

    for(i =0; i < threadCount; i++)
    {
        nn::os::StartThread(&g_ThreadTid[i]);
    }

    for(i =0; i < threadCount; i++)
    {
        NN_LOG("Waiting thread %d\n", i);
        nn::os::WaitThread(&g_ThreadTid[i]);
        NN_LOG("DONE:Waiting thread %d\n", i);
    }

    for(i =0; i < threadCount; i++)
    {
        NN_LOG("cleaning thread %d\n", i);
        nn::os::DestroyThread(&g_ThreadTid[i]);
        NN_LOG("DONE:cleaning thread %d\n", i);
    }

    NN_LOG("Finished...\n");
}


TEST(MultithreadThDestroyBeforeFinalize, Success)
{
    int i = 0;
    nn::Result result;
    for(i =0; i < NumberOfThreads; i++)
    {
        result = nn::os::CreateThread(
            &g_ThreadTid[i],
            WorkerFunction,
            nullptr,
            g_ThreadStack[i],
            ThreadStackSize,
            nn::os::LowestThreadPriority);
        EXPECT_TRUE(result.IsSuccess());
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        nn::os::StartThread(&g_ThreadTid[i]);
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        NN_LOG("Waiting thread %d\n", i);
        nn::os::WaitThread(&g_ThreadTid[i]);
        NN_LOG("DONE:Waiting thread %d\n", i);
    }

    NN_LOG("Wait for 6 seconds to run cleanup thread of NSPR for testing ");
    for(i=0; i<6; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG(".");
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        NN_LOG("cleaning thread %d\n", i);
        nn::os::DestroyThread(&g_ThreadTid[i]);
        NN_LOG("DONE:cleaning thread %d\n", i);
    }

    NN_LOG("Finished...\n");
}


TEST(MultithreadThDestroyAfterFinalize, Success)
{
    int i = 0;
    nn::Result result;
    for(i =0; i < NumberOfThreads; i++)
    {
        result = nn::os::CreateThread(
            &g_ThreadTid[i],
            WorkerFunction,
            nullptr,
            g_ThreadStack[i],
            ThreadStackSize,
            nn::os::LowestThreadPriority);
        EXPECT_TRUE(result.IsSuccess());
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        nn::os::StartThread(&g_ThreadTid[i]);
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        NN_LOG("Waiting thread %d\n", i);
        nn::os::WaitThread(&g_ThreadTid[i]);
        NN_LOG("DONE:Waiting thread %d\n", i);
    }

    NN_LOG("Wait for 5 seconds to run cleanup thread of NSPR for testing ");
    for(i=0; i<5; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        NN_LOG(".");
    }
    NN_LOG("\n");

    result = nn::ssl::Finalize();
    EXPECT_TRUE(result.IsSuccess());

    for(i =0; i < NumberOfThreads; i++)
    {
        NN_LOG("cleaning thread %d\n", i);
        nn::os::DestroyThread(&g_ThreadTid[i]);
        NN_LOG("DONE:cleaning thread %d\n", i);
    }

    NN_LOG("Finished...\n");
}

TEST(Cleanup, Success)
{
    // nn::ssl::Finalize is called in the test above.
    g_CommonUtil.FinalizeNetwork();
    nn::socket::Finalize();
}
