﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/ssl.h>
#include <nn/socket.h>
#include <nn/nifm.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>

#include <Utils/CommandLineParser.h>

// ------------------------------------------------------------------------------------------------
// Build flags for tests to run
// ------------------------------------------------------------------------------------------------

// ------------------------------------------------------------------------------------------------
// Configuration parameters
// ------------------------------------------------------------------------------------------------
namespace
{
const uint32_t NumberOfThreads = 4;
const size_t   ThreadStackSize = 1024 * 32;
const char     UrlString[]     = "https://natf.com:444";
}

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
#define RESULT_EXPECT_SUCCESS_OR_BREAK(func) \
do { \
    nn::Result result = func; \
    if(result.IsFailure()) \
    { \
        NN_LOG("Result is not ResultSuccess: (desc:%d)\n", result.GetDescription()); \
        break; \
    } \
    EXPECT_TRUE(result.IsSuccess()); \
}while(NN_STATIC_CONDITION(false))

#define RESULT_EXPECT_SUCCESS(result) \
do { \
    if(result.IsFailure()) \
    { \
        NN_LOG("Result is not ResultSuccess: (desc:%d)\n", result.GetDescription()); \
    } \
    EXPECT_TRUE(result.IsSuccess()); \
}while(NN_STATIC_CONDITION(false))


#define PRINT_HEX_TABLE(pData,len)                                  \
do                                                                  \
{                                                                   \
    unsigned char* pChar = reinterpret_cast<unsigned char*>(pData); \
    for(uint32_t i = 0; i < len; i++) {                             \
        NN_LOG("0x%02x", pChar[i]);                                 \
        if((i + 1) % 16 == 0) {                                     \
            NN_LOG("\n");                                           \
        } else {                                                    \
            if(((i + 1) % 8 == 0) && (i % 16 != 0)) {               \
                NN_LOG(" | ");                                      \
            } else {                                                \
                NN_LOG(" ");                                        \
            }                                                       \
        }                                                           \
    }                                                               \
} while(NN_STATIC_CONDITION(0))

namespace
{
SslTestCommonUtil                  g_CommonUtil;
NN_ALIGNAS(4096)           uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
NN_OS_ALIGNAS_THREAD_STACK uint8_t g_ThreadStack[NumberOfThreads][ThreadStackSize];
nn::os::ThreadType                 g_ThreadTid[NumberOfThreads];

size_t CurlSslContextCallback(CURL* pCurl, void* pSslContext, void* pUserData)
{
    // Obtain pointer to the SSL context passed by CURLOPT_SSL_CTX_FUNCTION
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);

    // Create SSL context
    nn::Result result = pContext->Create(nn::ssl::Context::SslVersion_Auto);
    if( result.IsFailure() )
    {
        return (size_t) - 1;
    }

    nn::ssl::CertStoreId id;

    result = pContext->ImportClientPki(
        &id,
        reinterpret_cast<const char*>(g_pTestClientPki),
        g_pTestClientPkiPassword,
        g_pTestClientPkiSize,
        g_TestClientPkiPasswordLength
    );

    if( result.IsFailure() )
    {
        return (size_t) - 1;
    }

    return 0;
}

}

extern "C" void nninitStartup()
{
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MemoryHeapSize );
}

void WorkerFunction(void* arg)
{
    CURL*    curl = nullptr;
    CURLcode res = CURLE_OK;

    curl = curl_easy_init();
    ASSERT_TRUE(curl != nullptr);

    curl_easy_setopt(curl, CURLOPT_URL, UrlString);
    //curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L)
    curl_easy_setopt(curl, CURLOPT_SSL_SKIP_DEFAULT_VERIFY, 1L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
    curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
    curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextCallback);

    res = curl_easy_perform(curl);
    /* Check for errors */
    if( res != CURLE_OK )
    {
        NN_LOG("\nCURL_EASY_PERFORM() FAILED! curl error: %d\n", res);
        g_CommonUtil.DumpCurlResults(curl);
        ASSERT_TRUE(NN_STATIC_CONDITION(false));
    }
    else
    {
        NN_LOG("\nCURL_EASY_PERFORM() SUCCEEDED!\n");
    }

    /* always cleanup */
    curl_easy_cleanup(curl);
}

// ------------------------------------------------------------------------------------------------
// Tests
// ------------------------------------------------------------------------------------------------
TEST(Setup, Success)
{
    nn::util::Uuid netProfile = nn::util::InvalidUuid;
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser ("--NetProfile", &nn::util::InvalidUuid, netProfile));

    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    ASSERT_TRUE(g_CommonUtil.SetupNetwork(netProfile));

    nn::Result result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);
    EXPECT_TRUE(result.IsSuccess());

    ASSERT_TRUE(curl_global_init(CURL_GLOBAL_DEFAULT) == CURLE_OK);
}

TEST(MultiThreadWithClientImport, Success)
{
    int i = 0;
    nn::Result result;

    for(i =0; i < NumberOfThreads; i++)
    {
        result = nn::os::CreateThread(
            &g_ThreadTid[i],
            WorkerFunction,
            nullptr,
            g_ThreadStack[i],
            ThreadStackSize,
            nn::os::LowestThreadPriority);
        EXPECT_TRUE(result.IsSuccess());
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        nn::os::StartThread(&g_ThreadTid[i]);
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        NN_LOG("Waiting thread %d\n", i);
        nn::os::WaitThread(&g_ThreadTid[i]);
        NN_LOG("DONE:Waiting thread %d\n", i);
    }

    for(i =0; i < NumberOfThreads; i++)
    {
        NN_LOG("cleaning thread %d\n", i);
        nn::os::DestroyThread(&g_ThreadTid[i]);
        NN_LOG("DONE:cleaning thread %d\n", i);
    }

    NN_LOG("Finished...\n");
}

TEST(Cleanup, Success)
{
    curl_global_cleanup();
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
