﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*
  Test code for Performance logger
 *---------------------------------------------------------------------------*/


#include <nn/perflog/Daemon.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer
#include <nn/perflog/sfdl/perflog.sfdl.h>
#include <nn/os/os_Result.public.h>
#include <nnt/nntest.h>




namespace nn {  namespace perflog {

// peek into semi-private function for test purposes
nn::sf::SharedPointer<sf::IDaemon> CreateDaemonByHipc() NN_NOEXCEPT;

} } // namespace nn::perflog




namespace
{
    nn::sf::SharedPointer<nn::perflog::sf::IDaemon> g_TestDaemon;
    const size_t THREAD_STACK_SIZE  = ( 64 * 1024 );
    char selftriggerstack[ THREAD_STACK_SIZE + ( nn::os::StackRegionAlignment * 2 ) ];
}

using namespace nn::perflog;




static void SelfTriggerThread( void *pvEvent )
{
    nn::Result nnRes;
    nn::os::SystemEventType *e = static_cast<nn::os::SystemEventType *>(pvEvent);


    NN_LOG("v2Self-signal sleeping 5s before signaling remote event\n" );
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(4000));
    nnRes = g_TestDaemon->TestTriggerEvent();
    EXPECT_EQ( true, nnRes.IsSuccess() );

    NN_LOG("v2Self-signal sleeping 3s b4 attaching event locally\n" );
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));


    NN_LOG("v2Self-signal atttaching eventlocally\n" );
    nn::os::SystemEventType evt;
    nn::os::AttachWritableHandleToSystemEvent(&evt,
                                              nn::os::GetWritableHandleOfSystemEvent(e), false,
                                              nn::os::EventClearMode_AutoClear  );


    NN_LOG("v2Self-signal sleeping 4s before signaling local eventg\n" );
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(4000));

    nn::os::SignalSystemEvent(&evt);
    nn::os::DetachWritableHandleOfSystemEvent(&evt);


    NN_LOG("Self-signal done signaling, will return\n" );
}



TEST(SFEventTest, IPC_Wait)
{
    nn::Result nnRes;
    void *pAlignedStack = nullptr;
    nn::os::ThreadType hthread;
    nn::os::SystemEventType hevent;

    // get hipc ref
    g_TestDaemon = CreateDaemonByHipc();


    nnRes = nn::os::CreateSystemEvent(&hevent, nn::os::EventClearMode_AutoClear , true );
    EXPECT_EQ( true, nnRes.IsSuccess() );


    nnRes = g_TestDaemon->TestRegisterEvent( nn::sf::NativeHandle( nn::os::GetWritableHandleOfSystemEvent(&hevent),false) );
    EXPECT_EQ( true, nnRes.IsSuccess() );

    NN_LOG("v1.Registered event\n" );

    pAlignedStack = reinterpret_cast<void*>(((reinterpret_cast<uintptr_t>(selftriggerstack) + (nn::os::StackRegionAlignment - 1)) &
                                      ~(nn::os::StackRegionAlignment - 1)));
    nnRes = CreateThread( &hthread, SelfTriggerThread, &hevent, pAlignedStack, THREAD_STACK_SIZE, nn::os::DefaultThreadPriority);
    EXPECT_EQ( true, nnRes.IsSuccess() );

    nn::os::StartThread( &hthread );

    NN_LOG("Main Thread Waiting for first signal\n" );
    for(;;)
    {
        if( nn::os::TimedWaitSystemEvent( &hevent,  nn::TimeSpan::FromMilliSeconds(5000) ) )
        {
            NN_LOG("SUCCESS: Got first signal!\n" );
            break;
        }
        else
        {
            NN_LOG("Timeout, still waiting(1)...\n" );
        }
    }

    NN_LOG("Main Thread Waiting for second signal\n" );
    for(;;)
    {
        if( nn::os::TimedWaitSystemEvent( &hevent,  nn::TimeSpan::FromMilliSeconds(5000) ) )
        {
            NN_LOG("SUCCESS: Got second signal!\n" );
            break;
        }
        else
        {
            NN_LOG("Timeout, still waiting(2)...\n" );
        }
    }


    NN_LOG("Main Thread Waiting for third signal\n" );
    for(;;)
    {
        if( nn::os::TimedWaitSystemEvent( &hevent,  nn::TimeSpan::FromMilliSeconds(5000) ) )
        {
            NN_LOG("SUCCESS: Got third signal!\n" );
            break;
        }
        else
        {
            NN_LOG("Timeout, still waiting(3)...\n" );
        }
    }


    nn::os::WaitThread( &hthread ); // no way to tell success

    nn::os::DestroyThread( &hthread );

    g_TestDaemon->TestUnregisterEvent();

    nn::os::DestroySystemEvent(&hevent);


    g_TestDaemon = nullptr;
    NN_LOG("Done finalize\n");
}

TEST(SFEventTest, IPC_Signal)
{
    nn::Result nnRes;

    // get hipc ref
    g_TestDaemon = CreateDaemonByHipc();
    NN_LOG("Done init\n");

    nnRes = g_TestDaemon->TestTriggerEvent( );
    EXPECT_EQ( true, nnRes.IsSuccess() );

    NN_LOG("Done trigger event\n");

    g_TestDaemon = nullptr;
    NN_LOG("Done finalize\n");
}
