﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_SocketApi.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt.h>


#include <nn/dhcps.h>

namespace NATF {
namespace API {

namespace {
#define STATIC_IP_ADDR      "192.168.0.2"
#define STATIC_GW_ADDR      "192.168.0.1"
#define STATIC_SUBNET_MASK  "255.255.255.0"
#define DHCP_RANGE_BEGIN    "192.168.0.3"
#define DHCP_RANGE_END      "192.168.0.13"
#define STATIC_DNS_1        "8.8.8.8"
#define STATIC_DNS_2        "8.8.4.4"
#define WLAN_INTERFACE_NAME "wl0"
#define WLAN_PRIORITY_MODE  true

const unsigned int g_PidLocalExperimental2 = 0x88B6;

const nn::wlan::ReceivedDataMatchInfo DummyMatchInfo = {
    { 0x00, 0x00, 0x00, 0x00, 0x00 }, 5
};

uint32_t g_LocalRxId = 0;

nn::socket::SystemConfigDefaultWithMemory<5, 5> g_SocketConfig;

nn::Result ConfigureInterface(const char* interfaceName, bool enable)
{
    nn::Result result;

    if (enable)
    {
        nn::bsdsocket::cfg::IfSettings ifcfg;
        memset(&ifcfg, 0, sizeof(ifcfg));

        ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
        ifcfg.mtu  = 1500;
        nn::socket::InetAton(STATIC_IP_ADDR, &ifcfg.u.modeStatic.addr);
        nn::socket::InetAton(STATIC_GW_ADDR, &ifcfg.u.modeStatic.gatewayAddr);
        nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
        ifcfg.u.modeStatic.broadcastAddr.S_addr =
            (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
            ~ifcfg.u.modeStatic.subnetMask.S_addr;
        nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
        nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);

        result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
        if (result.IsFailure())
        {
            NN_LOG("failed to configure interface %s - %d:%d\n",
                   interfaceName,
                   result.GetModule(),
                   result.GetDescription());
        };
    }
    else
    {
        result = nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
    };

    return result;
};

const char* GetRandomSsid()
{
    const int MaximumSsidSize = 33;
    static char ssid[MaximumSsidSize];
    memset(ssid, 0, sizeof(ssid));
    int idx = 0;

    static const char alphanum[] =
        "0123456789"
        "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
        "abcdefghijklmnopqrstuvwxyz";

    for (idx = 0; idx < sizeof(ssid) - 1; ++idx)
    {
        ssid[idx] = alphanum[rand() % (sizeof(alphanum) - 1)];
    };

    ssid[idx] = 0;
    return ssid;
};

enum Step
{
    Step_1_WlanInitializeLocalManager  = 1,
    Step_2_WlanInitializeSocketManager = 2,
    Step_3_SocketInitialize            = 3,
    Step_4_ConfigureInterface          = 4,
    Step_5_WlanOpenMasterMode          = 5,
    Step_6_WlanCreateBss               = 6,
    Step_7_WlanCreateRxEntry           = 7,
    Step_8_WlanAddMatchingData         = 8,
    Step_All                           = 9
};

nn::Result InitializeAccessPoint(unsigned int step)
{
    nn::Result result;
    uint16_t pidOuiExp2 = g_PidLocalExperimental2;
    nn::wlan::MasterBssParameters param;
    const char* ssid = nullptr;

    if (step >= Step_1_WlanInitializeLocalManager)
    {
        result = nn::wlan::InitializeLocalManager();
        NN_ASSERT(result.IsSuccess());
    };

    if (step >= Step_2_WlanInitializeSocketManager)
    {
        result = nn::wlan::InitializeSocketManager();
        NN_ASSERT(result.IsSuccess());
    };

    if (step >= Step_3_SocketInitialize)
    {
        result = nn::socket::Initialize(g_SocketConfig);
        NN_ASSERT(result.IsSuccess());
    };

    if (step >= Step_4_ConfigureInterface)
    {
        result = ConfigureInterface(WLAN_INTERFACE_NAME, true);
        NN_ASSERT(result.IsSuccess());
    };

    if (step >= Step_5_WlanOpenMasterMode)
    {
        result = nn::wlan::Local::OpenMasterMode();
        NN_ASSERT(result.IsSuccess());
    };

    if (step >= Step_6_WlanCreateBss)
    {
        param.channel = -1;
        param.autoKeepAlive = false;
        param.basicRates = nn::wlan::RateSetLegacy_11gMask;
        param.supportedRates = nn::wlan::RateSetLegacy_11gMask;
        param.beaconInterval = 100;
        param.hiddenSsid = false;
        param.inactivePeriod = 10;
        param.security.privacyMode = nn::wlan::SecurityMode_Open;
        param.security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
        ssid  = GetRandomSsid();
        NN_LOG("Attempting to create an SSID with the name: %s\n", ssid);
        param.ssid.Set(ssid);
        nn::wlan::Local::CreateBss(param);
    };

    if (step >= Step_7_WlanCreateRxEntry)
    {
        result = nn::wlan::Local::CreateRxEntry(&g_LocalRxId,
                                                &pidOuiExp2,
                                                1,
                                                30);
        NN_ASSERT(result.IsSuccess());
    }

    if (step >= Step_8_WlanAddMatchingData)
    {
        result = nn::wlan::Local::AddMatchingDataToRxEntry(g_LocalRxId, DummyMatchInfo);
        NN_ASSERT(result.IsSuccess());
    }

    return result;
};

void FinalizeAccessPoint(unsigned int step)
{
    if (step >= Step_8_WlanAddMatchingData)
    {
        nn::wlan::Local::RemoveMatchingDataFromRxEntry(g_LocalRxId, DummyMatchInfo).IsSuccess();
    };

    if (step >= Step_7_WlanCreateRxEntry)
    {
        nn::wlan::Local::DeleteRxEntry(g_LocalRxId).IsSuccess();
    };

    if (step >= Step_6_WlanCreateBss)
    {
        nn::wlan::Local::DestroyBss().IsSuccess();
    };

    if (step >= Step_5_WlanOpenMasterMode)
    {
        nn::wlan::Local::CloseMasterMode().IsSuccess();
    };

    if (step >= Step_4_ConfigureInterface)
    {
        ConfigureInterface(WLAN_INTERFACE_NAME, false);
    };

    if (step >= Step_3_SocketInitialize)
    {
        nn::socket::Finalize();
    }

    if (step >= Step_2_WlanInitializeSocketManager)
    {
        nn::wlan::FinalizeSocketManager();
    };

    if (step >= Step_1_WlanInitializeLocalManager)
    {
        nn::wlan::FinalizeLocalManager();
    };
};

const int LeaseTableMemorySize = 4096;
uint8_t pLeaseTableMemory[LeaseTableMemorySize];

void IgnoreEverythingCallback(void* context, nn::dhcps::Event event, const void* pData)
{
};

class Context
{
public:
    size_t memorySize = 0;
    nn::dhcps::UserConfiguration userConfiguration;
    void* pLeaseTableMemory = nullptr;

    Context()
    {
        userConfiguration.pInterfaceName = WLAN_INTERFACE_NAME;
        userConfiguration.pCallback = IgnoreEverythingCallback;
        userConfiguration.pContext = nullptr;
        userConfiguration.offerSeconds = 10;
        userConfiguration.leaseSeconds = 600;
        userConfiguration.t1Ratio = nn::dhcps::DefaultT1Ratio;
        userConfiguration.t2Ratio = nn::dhcps::DefaultT2Ratio;
        userConfiguration.hasGateway = false;
        userConfiguration.dnsServerCount = 0;
        userConfiguration.pDnsServers = nullptr;
        nn::socket::InetPton(nn::socket::Family::Af_Inet, DHCP_RANGE_BEGIN, &userConfiguration.begin);
        nn::socket::InetPton(nn::socket::Family::Af_Inet, DHCP_RANGE_END, &userConfiguration.end);
    };
};

}; // end anonymous namespace

/*

// The DHCP server depends on the socket layer to be initialized
// the wireless network does not need to be set up therefore
// these tests produce unexpected results and are commented out
// until the access point functionality is hardened and the DHCP
// server can do the right thing or we end up deleting them

// @brief attempt to use the dhcp server with a partially initialized
// access point. This test should fail.
TEST(DhcpServer, ConfigurationStep1)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_1_WlanInitializeLocalManager);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_1_WlanInitializeLocalManager);

    if (true == success)
    {
        NN_LOG("Able to initialize the DHCP server with a partially "
               "enabled access point. Initialize should have failed\n");
        ADD_FAILURE();
    };
};

// @brief attempt to use the dhcp server with a partially initialized
// access point. This test should fail.
TEST(DhcpServer, ConfigurationStep2)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_2_WlanInitializeSocketManager);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_2_WlanInitializeSocketManager);

    if (true == success)
    {
        NN_LOG("Able to initialize the DHCP server with a partially "
               "enabled access point. Initialize should have failed\n");
        ADD_FAILURE();
    };
};

// @brief attempt to use the dhcp server with a partially initialized
// access point. This test should fail.
TEST(DhcpServer, ConfigurationStep3)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_3_SocketInitialize);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_3_SocketInitialize);

    if (true == success)
    {
        NN_LOG("Able to initialize the DHCP server with a partially "
               "enabled access point. Initialize should have failed\n");
        ADD_FAILURE();
    };
};

// @brief attempt to use the dhcp server with a partially initialized
// access point. This test should fail.
TEST(DhcpServer, ConfigurationStep4)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_4_ConfigureInterface);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_4_ConfigureInterface);

    if (true == success)
    {
        NN_LOG("Able to initialize the DHCP server with a partially "
               "enabled access point. Initialize should have failed\n");
        ADD_FAILURE();
    };
};

// @brief attempt to use the dhcp server with a partially initialized
// access point. This test should fail.
TEST(DhcpServer, ConfigurationStep5)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_5_WlanOpenMasterMode);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_5_WlanOpenMasterMode);

    if (true == success)
    {
        NN_LOG("Able to initialize the DHCP server with a partially "
               "enabled access point. Initialize should have failed\n");
        ADD_FAILURE();
    };
};

// @brief attempt to use the dhcp server with a partially initialized
// access point. This test should fail.
TEST(DhcpServer, ConfigurationStep6)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_6_WlanCreateBss);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_6_WlanCreateBss);

    if (true == success)
    {
        NN_LOG("Able to initialize the DHCP server with a partially "
               "enabled access point. Initialize should have failed\n");
        ADD_FAILURE();
    };
};

// @brief attempt to use the dhcp server with a partially initialized
// access point. This test should fail.
TEST(DhcpServer, ConfigurationStep7)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_7_WlanCreateRxEntry);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_7_WlanCreateRxEntry);

    if (true == success)
    {
        NN_LOG("Able to initialize the DHCP server with a partially "
               "enabled access point. Initialize should have failed\n");
        ADD_FAILURE();
    };
};
*/

// @brief attempt to use the dhcp server with a fully initialized
// access point and a good configuration object. This is the happy
// golden path and should succeed.
TEST(DhcpServer, AllConfigurationSteps)
{
    bool success = false;
    Context context;

    InitializeAccessPoint(Step_8_WlanAddMatchingData);

    if (nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
         success = true;
         nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_8_WlanAddMatchingData);

    if (false == success)
    {
        NN_LOG("Able to initialize the DHCP server with a fully "
               "enabled access point. Initialize should have succeeded.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with pInterfaceName = nullptr;
// should fail with ResultInterfaceNameStringInvalid.
TEST(DhcpServer, ResultInterfaceNameStringInvalid)
{
    bool correct = false;
    Context context;
    nn::Result result;

    InitializeAccessPoint(Step_All);
    context.userConfiguration.pInterfaceName = nullptr;

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultInterfaceNameStringInvalid::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with invalid pInterfaceName."
               "Should failed with ResultInterfaceNameStringInvalid.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with pCallback = nullptr;
// should fail with ResultNullCallback
TEST(DhcpServer, InitializeNullCallback)
{
    bool correct = false;
    Context context;
    nn::Result result;

    InitializeAccessPoint(Step_All);
    context.userConfiguration.pCallback = nullptr;

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultNullCallback::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with invalid pCallback. "
               "Should failed with ResultNullCallback.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with swapped begin and end range.
// Should fail with ResultRangeIsInvalid.
TEST(DhcpServer, InitializeInvalidRange)
{
    bool correct = false;
    Context context;
    nn::Result result;

    InitializeAccessPoint(Step_All);
    std::swap(context.userConfiguration.begin, context.userConfiguration.end);
    result = nn::dhcps::Initialize(context.userConfiguration);

    if (nn::dhcps::ResultRangeIsInvalid::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with swapped begin and end range."
               "Should fail with ResultRangeIsInvalid\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize offerSeconds = 0;
// should fail with ResultOfferTimerSecondsInvalid.
TEST(DhcpServer, InitializeOfferTimerSecondsInvalid)
{
    bool correct = false;
    Context context;
    nn::Result result;

    InitializeAccessPoint(Step_All);
    context.userConfiguration.offerSeconds = 0;

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultOfferTimerSecondsInvalid::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };


    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with invalid offerSeconds."
               "Should fail with ResultOfferTimerSecondsInvalid.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize leaseSeconds = 0
// should fail with ResultLeaseTimerSecondsInvalid.
TEST(DhcpServer, InitializeLeaseTimerSecondsInvalid)
{
    bool correct = false;
    Context context;
    nn::Result result;

    InitializeAccessPoint(Step_All);
    context.userConfiguration.leaseSeconds = 0;
    result = nn::dhcps::Initialize(context.userConfiguration);

    if (nn::dhcps::ResultLeaseTimerSecondsInvalid::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        FinalizeAccessPoint(Step_All);
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with invalid leaseSeconds."
               "Should fail with ResultLeaseTimerSecondsInvalid.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize t1Ratio = 0;
// should fail with ResultT1RatioInvalid.
TEST(DhcpServer, ResultT1RatioInvalid)
{
    bool correct = false;
    Context context;
    nn::Result result;

    context.userConfiguration.t1Ratio = 0;

    InitializeAccessPoint(Step_All);

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultT1RatioInvalid::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with invalid t1Ratio."
               "Should fail with ResultT1RatioInvalid.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with t2Ratio = 0
// should fail with ResultT2RatioInvalid.
TEST(DhcpServer, ResultT2RatioInvalid)
{
    bool correct = false;
    Context context;
    nn::Result result;

    context.userConfiguration.t2Ratio = 0;

    InitializeAccessPoint(Step_All);

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultT2RatioInvalid::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with invalid t1Ratio."
               "Should fail with ResultT2RatioInvalid.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with t2Ratio = 0
// should fail with ResultT1GreaterOrEqualToT2
TEST(DhcpServer, ResultT1EqualToT2)
{
    bool correct = false;
    Context context;
    nn::Result result;

    context.userConfiguration.t1Ratio = context.userConfiguration.t2Ratio;

    InitializeAccessPoint(Step_All);

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultT1GreaterOrEqualToT2::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with t1Ratio = t2Ratio."
               "Should fail with ResultT1GreaterOrEqualToT2.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with t1Ratio and t2Ratio swapped.
// should fail with ResultT1GreaterOrEqualToT2
TEST(DhcpServer, ResultT1GreaterThanToT2)
{
    bool correct = false;
    Context context;
    nn::Result result;

    InitializeAccessPoint(Step_All);
    std::swap(context.userConfiguration.t1Ratio, context.userConfiguration.t2Ratio);

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultT1GreaterOrEqualToT2::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with t1Ratio = t2Ratio."
               "Should fail with ResultT1GreaterOrEqualToT2.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with dnsServerCount = 1 & null DNS server array.
// should fail with ResultNullDnsServerArray
TEST(DhcpServer, ResultNullDnsServerArray)
{
    bool correct = false;
    Context context;
    nn::Result result;

    context.userConfiguration.dnsServerCount = 1;

    InitializeAccessPoint(Step_All);

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultNullDnsServerArray::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with dnsServerCount = 1 and pDnsServers = nullptr."
               "Should fail with ResultNullDnsServerArray.\n");
        ADD_FAILURE();
    };
};

// @brief Attempt to initialize with a range outside of the acquired network values.
// should fail with ResultNullDnsServerArray
TEST(DhcpServer, ResultRangeNotInNetwork)
{
    bool correct = false;
    Context context;
    nn::Result result;

    const char* begin = "138.128.120.101"; //landsberger.com public range
    const char* end = "138.128.120.106";

    nn::socket::InetPton(nn::socket::Family::Af_Inet, begin, &context.userConfiguration.begin);
    nn::socket::InetPton(nn::socket::Family::Af_Inet, end, &context.userConfiguration.end);

    InitializeAccessPoint(Step_All);

    result = nn::dhcps::Initialize(context.userConfiguration);
    if (nn::dhcps::ResultRangeNotInNetwork::Includes(result))
    {
         correct = true;
    }
    else if (result.IsSuccess())
    {
        nn::dhcps::Finalize();
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize dhcpserver with begin: %s, end: %s but self: %s, and netmask: %s"
               "Should fail with ResultRangeNotInNetwork.\n", begin, end,
               STATIC_IP_ADDR, STATIC_SUBNET_MASK);
        ADD_FAILURE();
    };
};

// @brief Do not initialize, run GetMemorySize, should fail
TEST(DhcpServer, GetMemoryWithoutInitialize)
{
    bool correct = false;
    Context context;
    size_t memorySize = 0;

    InitializeAccessPoint(Step_All);

    if (nn::dhcps::GetMemorySize(&memorySize).IsFailure())
    {
        correct = true;
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("GetMemorySize without Initialize succeeded but shouldn't have.\n");
        ADD_FAILURE();
    };
};


// @brief Do not initialize, run SetMemorySize, should fail
TEST(DhcpServer, SetMemoryWithoutInitialize)
{
    bool correct = false;
    Context context;

    InitializeAccessPoint(Step_All);

    if (nn::dhcps::SetMemory(pLeaseTableMemory, LeaseTableMemorySize).IsFailure())
    {
        correct = true;
    };

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("SetMemory without Initialize succeeded but shouldn't have.\n");
        ADD_FAILURE();
    };
};

// @brief Initialize, run SetMemorySize with nullptr, should fail
TEST(DhcpServer, SetMemoryWithoutNull)
{
    bool correct = false;
    Context context;
    size_t memorySize;

    InitializeAccessPoint(Step_All);

    if (nn::dhcps::Initialize(context.userConfiguration).IsFailure())
    {
        goto bail;
    };

    if (nn::dhcps::GetMemorySize(&memorySize).IsFailure())
    {
        goto bail;
    };

    if (nn::dhcps::SetMemory(nullptr, LeaseTableMemorySize).IsFailure())
    {
        correct = true;
    };

bail:
    nn::dhcps::Finalize();

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("SetMemory with pMemory=nullptr succeeded but shouldn't have.\n");
        ADD_FAILURE();
    };
};

// @brief Initialize, run SetMemorySize with nullptr, should fail
TEST(DhcpServer, SetMemoryWithZeroSize)
{
    bool correct = false;
    Context context;
    size_t memorySize;

    InitializeAccessPoint(Step_All);

    if (nn::dhcps::Initialize(context.userConfiguration).IsFailure())
    {
        goto bail;
    };

    if (nn::dhcps::GetMemorySize(&memorySize).IsFailure())
    {
        goto bail;
    };

    if (nn::dhcps::SetMemory(pLeaseTableMemory, 0).IsFailure())
    {
        correct = true;
    };

bail:

    nn::dhcps::Finalize();

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("SetMemorySize without size = 0 succeeded but shouldn't have.\n");
        ADD_FAILURE();
    };
};


// @brief Set up the access point, initialize dhcp server library, and set up memory
// should succeed
TEST(DhcpServer, GetAndSetMemory)
{
    bool correct = false;
    Context context;
    nn::Result result;
    size_t memorySize = 0;

    InitializeAccessPoint(Step_All);

    if (!nn::dhcps::Initialize(context.userConfiguration).IsSuccess())
    {
        goto bail;
    };

    if (!nn::dhcps::GetMemorySize(&memorySize).IsSuccess())
    {
        goto bail;
    };

    if (!nn::dhcps::SetMemory(pLeaseTableMemory, LeaseTableMemorySize).IsSuccess())
    {
        goto bail;
    };

    correct = true;

bail:

    nn::dhcps::Finalize();

    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize, Get, and Set Memory should have succeeded but didn't.");
        ADD_FAILURE();
    };
};

// @brief EventTicker structure
// used to report event ordinality and count
struct EventTicker
{
    int ordinal;

    struct Report {
        int lastOrdinal;
        int count;
    };

    Report OnStart;
    Report OnRunning;
    Report OnStopping;
    Report OnStopped;
    Report OnDetectedNetworkChange;
    Report OnRecoverableError;
    Report OnLeaseOffered;
    Report OnLeaseOfferResent;
    Report OnLeaseOfferAdvancedState;
    Report OnLeaseOfferReused;
    Report OnLeaseOfferExpired;
    Report OnLeaseDecline;
    Report OnLeaseDeclineInvalid;
    Report OnLeaseInform;
    Report OnLeaseInformInvalid;
    Report OnLeaseRequest;
    Report OnLeaseRequestInvalid;
    Report OnLeaseRenewing;
    Report OnLeaseRebinding;
    Report OnLeaseExpired;
    Report OnLeaseRelease;
    Report OnLeaseReleaseInvalid;
    Report OnTableFull;
    Report OnTableAvailable;
    Report OnDiscoverWhileTableFull;
    Report OnDhcpUnhandledMessage;
};

// @brief EventTickerCallback
// used to report event ordinality and count
void EventTickerCallback(void* context, nn::dhcps::Event event, const void* pData)
{
    EventTicker& ticker = *reinterpret_cast<EventTicker*>(context);

    ++ticker.ordinal;
    switch (event)
    {
#define EVENT_TICKER_HANDLER(event) \
        case nn::dhcps::Event::event:                             \
            NN_LOG(#event"\n");                                   \
            ticker.event.lastOrdinal = ticker.ordinal;            \
            ++ticker.event.count;                                 \
            break;
        EVENT_TICKER_HANDLER(OnStart);
        EVENT_TICKER_HANDLER(OnRunning);
        EVENT_TICKER_HANDLER(OnStopping);
        EVENT_TICKER_HANDLER(OnStopped);
        EVENT_TICKER_HANDLER(OnDetectedNetworkChange);
        EVENT_TICKER_HANDLER(OnRecoverableError);
        EVENT_TICKER_HANDLER(OnLeaseOffered);
        EVENT_TICKER_HANDLER(OnLeaseOfferResent);
        EVENT_TICKER_HANDLER(OnLeaseOfferAdvancedState);
        EVENT_TICKER_HANDLER(OnLeaseOfferReused);
        EVENT_TICKER_HANDLER(OnLeaseOfferExpired);
        EVENT_TICKER_HANDLER(OnLeaseDecline);
        EVENT_TICKER_HANDLER(OnLeaseDeclineInvalid);
        EVENT_TICKER_HANDLER(OnLeaseInform);
        EVENT_TICKER_HANDLER(OnLeaseInformInvalid);
        EVENT_TICKER_HANDLER(OnLeaseRequest);
        EVENT_TICKER_HANDLER(OnLeaseRequestInvalid);
        EVENT_TICKER_HANDLER(OnLeaseRenewing);
        EVENT_TICKER_HANDLER(OnLeaseRebinding);
        EVENT_TICKER_HANDLER(OnLeaseExpired);
        EVENT_TICKER_HANDLER(OnLeaseRelease);
        EVENT_TICKER_HANDLER(OnLeaseReleaseInvalid);
        EVENT_TICKER_HANDLER(OnTableFull);
        EVENT_TICKER_HANDLER(OnTableAvailable);
        EVENT_TICKER_HANDLER(OnDiscoverWhileTableFull);
        EVENT_TICKER_HANDLER(OnDhcpUnhandledMessage);
    default:
        ;
#undef EVENT_TICKER_HANDLER
    };
};

// @brief Initialize, Get & Set Memory, Start, wait a few seconds, stop, finalize
// OnStart, OnRunning, OnStopping, and OnStopped should all have a single event
// and be delivered in that order
TEST(DhcpServer, EventOrdinalityAndCount)
{
    bool correct = false;
    Context context;
    nn::Result result;
    const int Slack = 3;
    size_t memorySize = 0;
    EventTicker ticker;
    memset(&ticker, 0, sizeof(ticker));

    InitializeAccessPoint(Step_All);
    context.userConfiguration.pCallback = EventTickerCallback;
    context.userConfiguration.pContext = &ticker;

    if (nn::dhcps::Initialize(context.userConfiguration).IsFailure())
    {
        goto bail;
    };

    if (nn::dhcps::GetMemorySize(&memorySize).IsFailure())
    {
        goto bail;
    };

    if (nn::dhcps::SetMemory(pLeaseTableMemory, LeaseTableMemorySize).IsFailure())
    {
        goto bail;
    };

    if (nn::dhcps::Start().IsFailure())
    {
        goto bail;
    };

    NN_LOG("Giving the server %d seconds for processing ...\n", Slack);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(Slack));

    if (nn::dhcps::Stop().IsFailure())
    {
        goto bail;
    };

    NN_LOG("ticker.OnStart.lastOrdinal:    %d  ticker.OnStart.count:    %d\n"
           "ticker.OnRunning.lastOrdinal:  %d  ticker.OnRunning.count:  %d\n"
           "ticker.OnStopping.lastOrdinal: %d  ticker.OnStopping.count: %d\n"
           "ticker.OnStopped.lastOrdinal:  %d  ticker.OnStopped.count:  %d\n",
           ticker.OnStart.lastOrdinal, ticker.OnStart.count,
           ticker.OnRunning.lastOrdinal, ticker.OnRunning.count,
           ticker.OnStopping.lastOrdinal, ticker.OnStopping.count,
           ticker.OnStopped.lastOrdinal, ticker.OnStopped.count);

    if (1 == ticker.OnStart.lastOrdinal    && 1 == ticker.OnStart.count    &&
        2 == ticker.OnRunning.lastOrdinal  && 1 == ticker.OnRunning.count  &&
        3 == ticker.OnStopping.lastOrdinal && 1 == ticker.OnStopping.count &&
        4 == ticker.OnStopped.lastOrdinal  && 1 == ticker.OnStopped.count)
    {
        correct = true;
    };

bail:
    nn::dhcps::Stop();
    nn::dhcps::Finalize();
    FinalizeAccessPoint(Step_All);

    if (false == correct)
    {
        NN_LOG("Initialize and start dhcpserver and then check "
               "OnStart, OnRunning, OnStopping, and OnStopped "
               "ordinality (1-4) and count (all == 1)\n");
        ADD_FAILURE();
    };
};

}} // namespace NATF::API

extern "C"
void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    NN_LOG("\n=== End Test of DhcpServerTest\n");

    nnt::Exit(result);
};
