﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"
#include "Unit/testNet_CommonFunctions.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>
#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;   // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static EchoServer *         g_echoServer = NULL;

static uint8_t              g_noiseVal = 1;

// Anonymous Namespace
namespace {

// Integer Value Results
typedef struct {

    int   value;
    bool  result;

} SockOptValueResult;

static SockOptValueResult g_SockOptValueResults[] =
{
    {-2147483648, true},
    {-1, true},
    {0, false},
    {1, true},
    {INT_MAX, true}

};

// Struct Timeval Results
typedef struct {

    int      tv_sec;
    int      tv_usec;
    bool     result;

} SockOptTimeoutResult;

static SockOptTimeoutResult g_SockOptTimeoutResults[] =
{
    {-2147483648, -2147483648, false},
    {-1, -1, false},
    {0, 0, true},
    {INT_MAX, INT_MAX}

};

} // Anonymous Namespace

static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;

    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    delete g_echoServer;
    g_echoServer = NULL;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}


static bool
SetSocketOpt( int & sockfds, nn::socket::Level socklevel, nn::socket::Option sockopt, void * opt,
              nn::socket::SockLenT & optlen, int inPort, const char * inInterface,  bool isUDP,
              int expectedRC )
{
    nn::socket::SockAddrIn   localAddr;
    bool                 isSuccess = true;
    int                  rc;

    // Create Socket
    if ( isUDP == true )
    {
        sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    }
    else
    {
        sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    }

    ERROR_IF(sockfds < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell echo server to REUSE it's TCP address port
    PRINT_AND_CALL( rc = nn::socket::SetSockOpt( sockfds, socklevel, sockopt, opt, optlen ) );
    ERROR_IF(rc < 0, "SetSockOpt failed: rc: %d, socklevel: %d, sockopt: %d, errno: %d",
                           rc, socklevel, sockopt, nn::socket::GetLastError() );

    // Client: Local Address
    memset(&localAddr, 0, sizeof( localAddr ));
    localAddr.sin_port        = nn::socket::InetHtons( inPort );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, inInterface, &localAddr.sin_addr.S_addr );
    ERROR_IF(rc != 1, "InetPton() Failed but Success was expected!" );

    // Client: Bind (self)
    rc = nn::socket::Bind( sockfds, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF(rc != expectedRC, "Bind() returned: %d but expectedRC: %d was expected!", rc, expectedRC );

    // For TCP - Add a listen queue
    if (isUDP == false)
    {
        rc = nn::socket::Listen( sockfds, 5 );
        ERROR_IF( rc < 0, "Listen() returned: %d but errno: %d!", rc, nn::socket::GetLastError() );
    }

    return( true );

out:

    return( false);

}


static bool
ConnectToBoth(int svrsock, int svrsock2, bool isUDP)
{
    nn::socket::TimeVal  mytimeout;
    nn::socket::FdSet    readSet;
    int                  sockfds, idx, rc;
    uint64_t             unused1, unused2;
    unsigned char        inBuff[128];
    bool                 connsvr1 = false, connsvr2 = false;
    bool                 isSuccess = true;

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl( svrsock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( svrsock2, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // For()ever
    sockfds = -1;
    for( idx = 0; idx < 100; idx++)
    {
        if ( connsvr1 == true && connsvr2 == true )
        {
           break;
        }

        nn::socket::FdSetZero( &readSet );

        if ( svrsock > -1  ) nn::socket::FdSetSet( svrsock,  &readSet );
        if ( svrsock2 > -1 ) nn::socket::FdSetSet( svrsock2, &readSet );

        mytimeout.tv_sec  = 1;
        mytimeout.tv_usec = 0;

        rc = nn::socket::Select( svrsock2 + 1, &readSet, NULL, NULL, &mytimeout );

        if ( rc == 0 )
        {
            NN_LOG( "Connecting to [%s:%d]\n", g_echoServer->kInterfaceIPv4Addr, 8090 );

            if ( sockfds > -1 )
            {
                nn::socket::Close( sockfds );
                sockfds = -1;
            }

            // Connect Client
            if ( isUDP == false )
            {
                sockfds = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8090 );
                ERROR_IF( sockfds < 0, "Connect failed but should have succeeded!\n" );
            }
            else  // UDP
            {
                sockfds = NATF::API::COMMON::MakeUDPConnection( g_echoServer->kInterfaceIPv4Addr, 8091,
                                             g_echoServer->kInterfaceIPv4Addr, 8090 );
                ERROR_IF( sockfds < 0, "Connect failed but should have succeeded!\n" );

                memset( inBuff, 0, sizeof( inBuff ) );
                sprintf( (char *) inBuff, "idx = %d\n", idx );
                NATF::API::COMMON::WriteData( sockfds, (unsigned char *) inBuff, strlen( (char *) inBuff ), unused1, unused2 );
            }
            continue;
        }

        if ( nn::socket::FdSetSet( svrsock, &readSet ) )
        {
            NN_LOG( "Connect to svr1 worked\n" );

            if ( isUDP == false )
            {
                int newsock;
                newsock = nn::socket::Accept( svrsock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
                if ( newsock < 0 )
                {
                    NN_LOG( "svr1: Accept failed - errno: %d\n", nn::socket::GetLastError() );
                }
                else
                {
                    NN_LOG( "svr1: Accepted new socket: %d\n", newsock );
                    nn::socket::Close( newsock );
                }
            }
            else // UDP
            {
                memset( inBuff, 0, sizeof( inBuff ) );
                rc = NATF::API::COMMON::ReadData( svrsock, inBuff, sizeof( inBuff ), unused1 );
                if ( rc < 0)
                {
                    printf( "Read failed - errno: %d\n", nn::socket::GetLastError() );
                }
                else
                {
                    printf( "Sv2: rc %d, %s\n", rc, (char *) inBuff );
                }
            }

            if ( idx > 5 )
            {
               connsvr1 = true;
               nn::socket::Close( svrsock );
               svrsock = -1;
            }
        }

        if ( nn::socket::FdSetSet( svrsock2, &readSet )  )
        {
            NN_LOG( "Connect to svr2 worked\n" );

            if ( isUDP == false )
            {
                int newsock;
                newsock = nn::socket::Accept( svrsock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
                if ( newsock < 0 )
                {
                    NN_LOG( "svr2: Accept failed - errno: %d\n", nn::socket::GetLastError() );
                }
                else
                {
                    NN_LOG( "svrs2: Accepted new socket: %d\n", newsock );
                    nn::socket::Close( newsock );
                }
            }
            else // UDP
            {
                memset( inBuff, 0, sizeof( inBuff ) );
                rc = NATF::API::COMMON::ReadData( svrsock2, inBuff, sizeof( inBuff ), unused1 );
                if ( rc < 0)
                {
                    NN_LOG( "Read failed - errno: %d\n", nn::socket::GetLastError() );
                }
                else
                {
                    NN_LOG( "Sv2: rc, %d, %s\n", rc, (char *) inBuff );
                }
            }

            if ( idx > 5 )
            {
                connsvr2 = true;
                nn::socket::Close( svrsock2 );
                svrsock2 = -1;
            }
        }
    }

    if ( idx == 100 )
    {
        return( false );
    }

    return( true );

out:

    return( false );

} // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_SO_Initialize)
{
    InitializeTesting();
}




TEST(ApiUnit,SetSockOpt_SO_REUSEADDR)
{
    int                       intoptval = 1;
    nn::socket::SockLenT      intoptlen = 0;

    uint64_t                  unused1 = 0, unused2 = 0;
    size_t                    myResultSize = 0;
    int                       tcpsock = -1, udpsock = -1;
    int                       clisock = -1, svrsock = -1, svrsock2 = -1, idx = 0;
    bool                      isSuccess = true, rval = false;

    ////////////////
    // GetSockOpt Tests for: SO_REUSEADDR
    ///////////////

    /// TCP and UDP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    myResultSize = sizeof(g_SockOptValueResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_REUSEADDR - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_REUSEADDR - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_REUSEADDR - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_ReuseAddr [TCP] - Option: ON
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr][TCP][Option: ON]\n" );

    // BIND Server 1
    intoptval = 1;   // ON
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, false, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval == 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // Connect Client
    clisock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8090 );
    ERROR_IF_AND_COUNT( clisock < 0, "Connect failed but should have succeeded!\n" );

    // Client and Server Write Data
    NATF::API::COMMON::WriteData( svrsock, (unsigned char *) "Hello!", 6, unused1, unused2 );
    NATF::API::COMMON::WriteData( clisock, (unsigned char *) "Hello!", 6, unused1, unused2 );

    // Close Server Sock
    nn::socket::Close( svrsock );
    svrsock = -1;                 // Socket in TIMEWAIT?

    // BIND Server 1
    intoptval = 1;
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &intoptval, intoptlen,
                                            8090, g_echoServer->kInterfaceIPv4Addr, false, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval == 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // Cleanup
    nn::socket::Close( svrsock );
    svrsock = -1;

    nn::socket::Close( clisock );
    clisock = -1;

    NN_LOG( "PASS\n" );

    ////////////////
    // nn::socket::Option::So_ReuseAddr [TCP] - Option: OFF
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr][TCP][Option: OFF]\n" );

    // BIND Server 1
    intoptval = 0;  // OFF
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, false, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval != 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // Connect Client
    clisock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8090 );
    ERROR_IF_AND_COUNT( clisock < 0, "Connect failed but should have succeeded!\n" );

    // Client and Server Write Data
    NATF::API::COMMON::WriteData( svrsock, (unsigned char *) "Hello!", 6, unused1, unused2 );
    NATF::API::COMMON::WriteData( clisock, (unsigned char *) "Hello!", 6, unused1, unused2 );

    // Close Server Sock
    nn::socket::Close( svrsock );
    svrsock = -1;                 // Socket in TIMEWAIT?

    // BIND Server 1
    intoptval = 0;
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, false, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval != 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // Cleanup
    nn::socket::Close( svrsock );
    svrsock = -1;

    nn::socket::Close( clisock );
    clisock = -1;

    NN_LOG( "PASS\n" );

out:
    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    if ( svrsock > -1 )
    {
        nn::socket::Close( svrsock );
        svrsock = -1;
    }

    if ( svrsock2 > -1 )
    {
        nn::socket::Close( svrsock2 );
        svrsock2 = -1;
    }

    return;

}  // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_SO_REUSEPORT)
{
    int                     intoptval = -1;
    nn::socket::SockLenT    intoptlen = -1;

    size_t         myResultSize = 0;
    int            tcpsock = -1, udpsock = -1, idx = 0;
    int            clisock = -1, svrsock = -1, svrsock2 = -1;
    bool           isSuccess = true, rval = false;

    /////////////////////////////////
    //
    // Was: SIGLONTD-9695.  It is normal for TCP listen ports to be bound to the same (address) and (port) without error.
    // This was tested on FreeBSD 10.3, Linux and NX and all allow the (same) listen port and address to be bound without
    // error.  3/7/2017
    //
    /////////////////////////////////

    ////////////////
    // GetSockOpt Tests for: SO_REUSPORT
    ///////////////

    /// TCP and UDP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    myResultSize = sizeof(g_SockOptValueResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_REUSEPORT - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_REUSEPORT - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_REUSEPORT - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_ReusePort [TCP] - Option: OFF
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort][TCP][Option: OFF]\n" );

    // BIND Server 1
    intoptval = 0;   // OFF
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, false, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval != 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReusePort set to OFF!  rval: %d, intoptval: %d\n", rval, intoptval );

    // BIND Server 2
    intoptval = 0;   // OFF
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock2, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, false, -1 );

    ERROR_IF_AND_COUNT( rval == false || intoptval != 0,
    "Server 2 failed to bind with nn::socket::Option::So_ReusePort set to OFF!  rval: %d, intoptval: %d\n", rval, intoptval );

    // Cleanup
    nn::socket::Close( svrsock );
    svrsock = -1;

    nn::socket::Close( svrsock2 );
    clisock = -1;

    NN_LOG( "PASS\n" );


    ////////////////
    // nn::socket::Option::So_ReusePort [TCP] - Option: ON
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort][TCP][Option: ON]\n" );

    // BIND Server 1
    intoptval = 1;   // ON
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, false, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval == 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // BIND Server 2
    intoptval = 1;   // ON
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock2, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, false, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval == 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // Connect to both
    rval = ConnectToBoth( svrsock, svrsock2, false );
    ERROR_IF_AND_COUNT( rval == false, "Failed to Connect to both server sockets\n" );

    // Cleanup
    nn::socket::Close( svrsock );
    svrsock = -1;

    nn::socket::Close( svrsock2 );
    clisock = -1;

    NN_LOG( "PASS\n" );


    ////////////////
    // nn::socket::Option::So_ReusePort [UDP] - Option: ON
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort][UDP][Option: ON]\n" );

    // BIND Server 1
    intoptval = 1;   // ON
    intoptlen = sizeof( intoptval );
    rval =SetSocketOpt( svrsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, true, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval == 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // BIND Server 2
    intoptval = 1;   // ON
    intoptlen = sizeof( intoptval );
    rval = SetSocketOpt( svrsock2, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReusePort, &intoptval, intoptlen,
                         8090, g_echoServer->kInterfaceIPv4Addr, true, 0 );

    ERROR_IF_AND_COUNT( rval == false || intoptval == 0,
    "Failed to bind server 1 with nn::socket::Option::So_ReuseAddr!  rval: %d, intoptval: %d\n", rval, intoptval );

    // Connect to both
    rval = ConnectToBoth( svrsock, svrsock2, true );
    ERROR_IF_AND_COUNT( rval == false, "Failed to Connect to both server sockets\n" );

    // Cleanup
    nn::socket::Close( svrsock );

    NN_LOG( "PASS\n" );

out:
    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    if ( svrsock > -1 )
    {
        nn::socket::Close( svrsock );
        svrsock = -1;
    }

    if ( svrsock2 > -1 )
    {
        nn::socket::Close( svrsock2 );
        svrsock2 = -1;
    }

    return;

}  // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_SO_SNDBUF)
{
    int                     intoptval = -1;
    nn::socket::SockLenT    intoptlen = -1;

    uint64_t                unused1 = -1, unused2 = -1;
    size_t                  workPtrLen = 0;
    int                     tcpsock = -1, udpsock = -1;
    int                     clisock = -1, svrsock = -1, svrsock2 = -1, idx = 0, idx2 = 0;
    int                     rc = -1, writeCount = 0, readCount = 0;
    bool                    isSuccess = true, rval = false;
    unsigned char *         workPtr = NULL;

    ////////////////
    // GetSockOpt Tests for: SO_SNDBUF
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    // UDP
    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    static int sizeArrayCertBad[] = {-65535, -32767, -16384, -9000, -1, 0, INT_MAX};
    static int sizeArrayCertBadCount = sizeof(sizeArrayCertBad) / sizeof(int);

    for(idx = 0; idx < sizeArrayCertBadCount; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_SNDBUF (BAD) size: %d\n", sizeArrayCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, false, sizeArrayCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_SNDBUF (BAD) size: %d\n", sizeArrayCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, false, sizeArrayCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_SNDBUF (BAD) size: %d\n", sizeArrayCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, false, sizeArrayCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    static int sizeArrayCertGood[] = {128, 256, 9000, 16384, 32767, 65535,
                                      nn::socket::ConfigDefault::DefaultTcpInitialSendBufferSize,
                                      nn::socket::ConfigDefault::DefaultTcpAutoSendBufferSizeMax,
                                      nn::socket::DefaultTcpAutoBufferSizeMax};

    static int sizeArrayCertGoodCount = sizeof(sizeArrayCertGood) / sizeof(int);

    for(idx = 0; idx < sizeArrayCertGoodCount; idx++ )
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_SNDBUF (GOOD) size: %d\n", sizeArrayCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, true, sizeArrayCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_SNDBUF (GOOD) size: %d\n", sizeArrayCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, true, sizeArrayCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_SNDBUF (GOOD) size: %d\n", sizeArrayCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, true, sizeArrayCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_SndBuf [TCP] - Option: (Various sizes)
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf][UDP][Option: ON]\n" );


    /////////////////
    // UDP - nn::socket::Option::So_SndBuf
    /////////////////

    static size_t sizeArrayUDP[] = { 128, 256, 384, 400, 416, 448, 480, 512, 640, 1024, 4096, 8192, 9000 };
    static int sizeArraySizeUDP = sizeof ( sizeArrayUDP ) / sizeof( size_t);

    for( idx = 0; idx < sizeArraySizeUDP; idx++ )
    {
        NN_LOG( "Test size is: %d\n", sizeArrayUDP[idx] );
        writeCount = 0;

        // Build Test of that size..
        workPtrLen = sizeArrayUDP[idx];
        rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
        ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen  );

        // Tell Echo Server to (NOT) echo
        g_echoServer->setFalseEcho( true );

        // Client gets smaller SNDBUF
        intoptval = workPtrLen;
        intoptlen = sizeof( intoptval );
        rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &intoptval, intoptlen,
                                       (9000 + idx), g_echoServer->kInterfaceIPv4Addr,
                                       8053, g_echoServer->kInterfaceIPv4Addr, true, 0 );   // isUDP = true

        // Set Non Blocking I/O
        rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
        ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

        /*
         * UDP does not overflow buffers - instead (inbound) packets are silently dropped.  We know if we are
         * losing data by reading back what the echo server has stored
         */
         for( idx2 = 0; idx2 < 100; idx2++ )
         {
             rc = NATF::API::COMMON::WriteData( clisock, workPtr, workPtrLen, unused1, unused2 );
             if ( rc < 1 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     NN_LOG( "Write: SNDBUF size: %d, Maxed out at: %d\n", workPtrLen, writeCount );
                     break;
                 }

                 ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n", nn::socket::GetLastError() );
             }
             writeCount = writeCount + rc;
         }

         // Tell Server to Echo
         g_echoServer->setFalseEcho( false );
         nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

         for( ; ; )
         {
             rc = NATF::API::COMMON::ReadData( clisock, workPtr, workPtrLen, unused1 );
             if ( rc < 1 )
             {
                if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                {
                   NN_LOG( "Read: SNDBUF size: %d, Maxed out at: %d\n", workPtrLen, readCount );
                   break;
                }

                ERROR_IF( rc < 0, "Read Failed for size: %d - errno: %d\n",
                                                  workPtrLen, nn::socket::GetLastError() );
             }
             readCount = readCount + rc;
        }

        NN_LOG( "Done!  Read Count: %d, Write Count: %d, percent loss: %lld%%\n", readCount, writeCount,
                    abs ( (int) ( (double) ( (double) readCount / (double) writeCount ) * 100 ) - 100 ) );

        nTestsPassing++;

        // Close Client Socket
        nn::socket::Close( clisock );
        clisock = -1;

        // Free Payload memory
        if (  workPtr != NULL )
        {
            free( workPtr );
            workPtr = NULL;
        }
    }

    /////////////////
    // TCP - nn::socket::Option::So_SndBuf
    /////////////////

    static size_t sizeArrayTCP [] = { 128, 256, 384, 400, 416, 448, 480 ,512, 640, 1024, 4096, 8192, 9000, 16384, 32767, 65535 };
    static int sizeArraySizeTCP = sizeof ( sizeArrayTCP ) / sizeof( size_t);

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf][TCP][Option: ON]\n" );

    for( idx = 0; idx < sizeArraySizeTCP; idx++ )
    {
        // Free Payload memory
        if (  workPtr != NULL )
        {
            free( workPtr );
            workPtr = NULL;
        }

        NN_LOG( "Test size is: %d\n", sizeArrayTCP[idx] );
        writeCount = 0;

        // Build Test of that size..
        workPtrLen = sizeArrayTCP[idx];
        rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
        ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen );

        // Tell Echo Server to (NOT) echo
        g_echoServer->setFalseEcho( true );

        // Client gets smaller SNDBUF
        intoptval = workPtrLen;
        intoptlen = sizeof( intoptval );
        rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &intoptval, intoptlen,
                                       (9000 + idx), g_echoServer->kInterfaceIPv4Addr,
                                       8053, g_echoServer->kInterfaceIPv4Addr, false, 0 );  // isUDP = false

        // Set Non Blocking I/O
        rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
        ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

        /*
         * TCP controls overflow buffers
         */

         writeCount = 0;
         for( idx2 = 0; idx2 < 100; idx2++ )
         {
             rc = NATF::API::COMMON::WriteData( clisock, workPtr, workPtrLen, unused1, unused2 );
             if ( rc < 1 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     NN_LOG( "Write: SNDBUF size: %d, Maxed out at: %d\n", workPtrLen, writeCount );
                     break;
                 }

                 ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n",
                                                  workPtrLen, nn::socket::GetLastError() );
             }
             writeCount = writeCount + rc;

         }

         // Tell Server to Echo
         g_echoServer->setFalseEcho( false );
         nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

         for( ; ; )
         {
             rc = NATF::API::COMMON::ReadData( clisock, workPtr, workPtrLen, unused1 );
             if ( rc < 1 )
             {
                if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                {
                   NN_LOG( "Read: SNDBUF size: %d, Maxed out at: %d\n", workPtrLen , readCount );
                   break;
                }

                ERROR_IF( rc < 0, "Read Failed for size: %d - errno: %d\n",
                                                  workPtrLen, nn::socket::GetLastError() );
             }

             readCount = readCount + rc;
        }

        NN_LOG( "Done!  Read Count: %d, Write Count: %d, percent loss: %lld%%\n", readCount, writeCount,
                    abs ( (int) ( (double) ( (double) writeCount / (double) readCount ) * 100 ) - 100 ) );
        nTestsPassing++;

        // Close Client Socket
        nn::socket::Close( clisock );
        clisock = -1;

        // Free Payload memory
        if (  workPtr != NULL )
        {
            free( workPtr );
            workPtr = NULL;
        }
    }

out:

    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    if ( svrsock > -1 )
    {
        nn::socket::Close( svrsock );
        svrsock = -1;
    }

    if ( svrsock2 > -1 )
    {
        nn::socket::Close( svrsock2 );
        svrsock2 = -1;
    }

    // Free Payload memory
    if (  workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    return;
}   // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_SO_RCVBUF)
{
    int                     intoptval = 0;
    nn::socket::SockLenT    intoptlen = 0;

    uint64_t          unused1 = 0, unused2 = 0;
    size_t            workPtrLen = 0;
    int               tcpsock = -1, udpsock = -1;
    int               clisock = -1, svrsock = -1, svrsock2 = -1, idx = 0, idx2 = 0;
    int               rc = -1, writeCount = 0, readCount = 0;
    int               againCount = 0;
    bool              isSuccess = true, rval = false;
    unsigned char *   workPtr = nullptr;

    ////////////////
    // GetSockOpt Tests for: SO_RCVBUF
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    // UDP
    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    static int sizeArrayCertBad[] = {-65535, -32767, -16384, -9000, -1, 0, INT_MAX};
    static int sizeArrayCertBadCount = sizeof(sizeArrayCertBad) / sizeof(int);

    for(idx = 0; idx < sizeArrayCertBadCount; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_RCVBUF (BAD) size: %d\n", sizeArrayCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, false, sizeArrayCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_RCVBUF (BAD) size: %d\n", sizeArrayCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, false, sizeArrayCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_RCVBUF (BAD) size: %d\n", sizeArrayCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, false, sizeArrayCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    static int sizeArrayCertGood[] = {128, 256, 9000, 16384, 32767, 65535,
                                      nn::socket::ConfigDefault::DefaultTcpInitialSendBufferSize,
                                      nn::socket::ConfigDefault::DefaultTcpAutoSendBufferSizeMax,
                                      nn::socket::DefaultTcpAutoBufferSizeMax};

    static int sizeArrayCertGoodCount = sizeof(sizeArrayCertGood) / sizeof(int);

    for(idx = 0; idx < sizeArrayCertGoodCount; idx++ )
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_RCVBUF (GOOD) size: %d\n", sizeArrayCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, true, sizeArrayCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_RCVBUF (GOOD) size: %d\n", sizeArrayCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, true, sizeArrayCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_RCVBUF (GOOD) size: %d\n", sizeArrayCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, true, sizeArrayCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    clisock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_RcvBuf [UDP] - Option: (Various sizes)
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf][UDP][Option: ON]\n" );

    static size_t sizeArrayUDP [] = { 128, 256, 384, 400, 416, 448, 480, 512, 640, 1024, 4096, 8192, 9000 };
    static int sizeArraySizeUDP = sizeof ( sizeArrayUDP ) / sizeof( size_t);

    workPtr = NULL;
    for( idx = 0; idx < sizeArraySizeUDP; idx++ )
    {
        // Free Memory
        if ( workPtr != NULL )
        {
            free( workPtr );
            workPtr = NULL;
        }

        NN_LOG( "Test size is: %d\n", sizeArrayUDP[idx] );

        // Build Test of that size..
        workPtrLen = sizeArrayUDP[idx];
        rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
        ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen  );

        // Client: Gets smaller RCVBUF
        //
        // IMPORTANT: UDP needs an additional 16 bytes /per message
        intoptval = workPtrLen;
        intoptlen = sizeof( intoptval );

        rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &intoptval, intoptlen,
                                       (9000 + idx), g_echoServer->kInterfaceIPv4Addr,
                                        8053, g_echoServer->kInterfaceIPv4Addr, true, 0 );   // isUDP = true

        // Set Non Blocking I/O
        rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
        ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

        /*
         * UDP automatically "echos" response to us
         */

         againCount = 0;
         writeCount = 0;
         for( idx2 = 0; idx2 < 100; idx2++ )
         {
             rc = NATF::API::COMMON::WriteData( clisock, workPtr, workPtrLen, unused1, unused2 );
             if ( rc < 1 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     if ( ++againCount < 2 )
                     {
                         continue;
                     }

                     NN_LOG( "Write: RCVBUF size: %d, Maxed out at: %d\n", workPtrLen, writeCount );
                     break;
                 }

                 ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n",
                                                  workPtrLen, nn::socket::GetLastError() );
             }
             writeCount = writeCount + rc;

         }

         // Server automatically ECHOs response
         nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

         againCount = 0;
         readCount = 0;
         for( ; ; )
         {
             rc = NATF::API::COMMON::ReadData( clisock, workPtr, workPtrLen, unused1 );
             if ( rc < 1 )
             {
                if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                {
                    if ( ++againCount < 2 )
                    {
                        continue;
                    }

                    NN_LOG( "Read: RCVBUF size: %d, Maxed out at: %d\n", workPtrLen, readCount );
                    break;
                }

                ERROR_IF( rc < 0, "Read Failed for size: %d - errno: %d\n",
                                                  workPtrLen, nn::socket::GetLastError() );
             }

             readCount = readCount + rc;
        }

        NN_LOG( "Done!  Read Count: %d, Write Count: %d, percent loss: %lld%%\n", readCount, writeCount,
                    abs ( (int) ( (double) ( (double) readCount / (double) writeCount ) * 100 ) - 100 ) );

        nTestsPassing++;

        // Close Client Socket
        nn::socket::Close( clisock );
    }

    // Free Memory
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }


    ////////////////
    // nn::socket::Option::So_RcvBuf [TCP] - Option: (Various sizes)
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf][TCP][Option: ON]\n" );

    static size_t sizeArrayTCP [] = { 128, 256, 384, 400, 416, 448, 480, 512, 640, 1024, 4096, 8192, 9000, 16384, 32767,
                                      65535, 131070, 262140 };

    static int sizeArraySizeTCP = sizeof ( sizeArrayTCP ) / sizeof( size_t);

    workPtr = NULL;

    for( idx = 0; idx < sizeArraySizeTCP; idx++ )
    {
        // Free Memory
        if ( workPtr != NULL )
        {
            free( workPtr );
            workPtr = NULL;
        }

        NN_LOG( "Test size is: %d\n", sizeArrayTCP[idx] );

        // Build Test of that size..
        workPtrLen = sizeArrayTCP[idx];
        rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
        ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen  );

        // Set nn::socket::Option::So_RcvBuf Option
        intoptval = workPtrLen;
        intoptlen = sizeof( intoptval );

        rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &intoptval, intoptlen,
                                       (9000 + idx), g_echoServer->kInterfaceIPv4Addr,
                                        8053, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

        // Set Non Blocking I/O
        rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
        ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

        /*
         * UDP automatically "echos" response to us
         */

         againCount = 0;
         writeCount = 0;
         for( idx2 = 0; idx2 < 100; idx2++ )
         {
             // Save Index
             rc = NATF::API::COMMON::WriteData( clisock, workPtr, workPtrLen, unused1, unused2 );
             if ( rc < 1 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     if ( ++againCount < 2 )
                     {
                         continue;
                     }

                     NN_LOG( "Write: RCVBUF size: %d, Maxed out at: %d\n", workPtrLen, writeCount );
                     break;
                 }

                 ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n",
                                                  workPtrLen, nn::socket::GetLastError() );
             }
             writeCount = writeCount + rc;

         }

         // Server automatically ECHOs response
         nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

         againCount = 0;
         readCount = 0;
         for( ; ; )
         {
             rc = NATF::API::COMMON::ReadData( clisock, workPtr, workPtrLen, unused1 );
             if ( rc < 1 )
             {
                if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                {
                    if ( ++againCount < 2 )
                    {
                        continue;
                    }

                    NN_LOG( "Read: RCVBUF size: %d, Maxed out at: %d\n", workPtrLen, readCount );
                    break;
                }

                ERROR_IF( rc < 0, "Read Failed for size: %d - errno: %d\n",
                                                  workPtrLen, nn::socket::GetLastError() );
             }

             readCount = readCount + rc;
        }

        NN_LOG( "Done!  Read Count: %d, Write Count: %d, percent loss: %lld%%\n", readCount, writeCount,
                    abs ( (int) ( (double) ( (double) readCount / (double) writeCount ) * 100 ) - 100 ) );

        nTestsPassing++;

        // Close Client Socket
        nn::socket::Close( clisock );
    }

out:

    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    if ( svrsock > -1 )
    {
        nn::socket::Close( svrsock );
        svrsock = -1;
    }

    if ( svrsock2 > -1 )
    {
        nn::socket::Close( svrsock2 );
        svrsock2 = -1;
    }

    // Free Payload memory
    if (  workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    return;

} // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_SO_SNDLOWAT)
{
    nn::socket::TimeVal    mytime = { 0 };
    nn::socket::FdSet      writeSet;
    nn::socket::SockLenT   intoptlen = 0;
    uint64_t               unused1 = 0, unused2 = 0;
    size_t                 workPtrLen = 0;
    int                    tcpsock = -1, udpsock = -1;
    int                    clisock = -1, writeCount, highWaterMark = -1;
    int                    intoptval = 0;
    int                    rc = -1, idx = 0;
    bool                   isSuccess = true, rval = false;
    unsigned char *        workPtr = nullptr;

    ////////////////
    // GetSockOpt Tests for: SO_SNDLOWAT
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    static int sizeArrayTCPCertBad[] = {-65535, -32767, -16384, -9000, -1, 0, INT_MAX};
    static int sizeArrayTCPSizeCertBad = sizeof(sizeArrayTCPCertBad) / sizeof(int);

    for(idx = 0; idx < sizeArrayTCPSizeCertBad; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_SNDLOWAT (BAD) size: %d\n", sizeArrayTCPCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat, false, sizeArrayTCPCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_SNDLOWAT (BAD) size: %d\n", sizeArrayTCPCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat, false, sizeArrayTCPCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    static int sizeArrayTCPCertGood[] = {128, 256, 9000, 16384, 32767, 48996};
    static int sizeArrayTCPSizeCertGood = sizeof(sizeArrayTCPCertGood) / sizeof(int);

    for(idx = 0; idx < sizeArrayTCPSizeCertGood; idx++ )
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_SNDLOWAT (GOOD) size: %d\n", sizeArrayTCPCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat, true, sizeArrayTCPCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_SNDLOWAT (GOOD) size: %d\n", sizeArrayTCPCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat, true, sizeArrayTCPCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // UDP
    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    static int sizeArrayUDPCertBad[] = {-65535, -32767, -16384, -9000, -1, 0, INT_MAX};
    static int sizeArrayUDPSizeCertBad = sizeof(sizeArrayUDPCertBad) / sizeof(int);

    for(idx = 0; idx < sizeArrayUDPSizeCertBad; idx++)
    {
        NN_LOG("GetSockOpt:  [UDP]: SO_SNDLOWAT (BAD) size: %d\n", sizeArrayUDPCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat, false, sizeArrayUDPCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    static int sizeArrayUDPCertGood[] = {128, 256, 9000, 9216};
    static int sizeArrayUDPSizeCertGood = sizeof(sizeArrayUDPCertGood) / sizeof(int);

    for(idx = 0; idx < sizeArrayUDPSizeCertGood; idx++ )
    {
        NN_LOG("GetSockOpt:  [UDP]: SO_SNDLOWAT (GOOD) size: %d\n", sizeArrayUDPCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat, true, sizeArrayUDPCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_SndLoWat [TCP] - Option
    //
    // NOTE: No UDP test because UDP alwasy succeeds with UDP + Write()
    ////////////////

    workPtrLen = 1024;
    rc = NATF::API::COMMON::MakeRandomNoise(&workPtr, workPtrLen, g_noiseVal);
    ERROR_IF(rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen);

    // Client: Set nn::socket::Option::So_SndLoWat
    clisock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8053);
    ERROR_IF(clisock < 0, "Connect failed but should have succeeded!\n" );

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
    ERROR_IF(rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell server to (NOT) echo
    g_echoServer->setFalseEcho(true);

    // Find High Water mark...
    writeCount = 0;
    for( ; ; )
    {
         nn::socket::FdSetZero(&writeSet);
         nn::socket::FdSetSet(clisock, &writeSet);

         mytime.tv_sec  = 0;
         mytime.tv_usec = 100;

         rc = nn::socket::Select(clisock + 1, nullptr, &writeSet, nullptr, &mytime);
         if ( rc == 0  )
         {
             NN_LOG( "Socket not ready for writing after bytes: %d sent\n", writeCount);
             break;
         }
         else if ( rc < 0 )
         {
             NN_LOG( "Select fail with errno: %d\n", nn::socket::GetLastError());
             return;
         }

         rc = NATF::API::COMMON::WriteData(clisock, workPtr, workPtrLen, unused1, unused2);
         if ( rc < 1 )
         {
             if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock)
             {
                 continue;
             }

             ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n", nn::socket::GetLastError() );
         }

         writeCount = writeCount + rc;
    }

    // Close Clinet Socket
    nn::socket::Close( clisock );

    // At High Water mark...
    highWaterMark = writeCount;
    NN_LOG( "=======> High Water Mark: %d\n", writeCount );


    // Client: Set nn::socket::Option::So_SndLoWat to A little over High Water Mark
    //
    intoptval = workPtrLen;
    intoptlen = sizeof( intoptval );
    rval = NATF::API::COMMON::SetSocketOptAndConnect(clisock,
                                   nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat,
                                   &intoptval, intoptlen,
                                   9000, g_echoServer->kInterfaceIPv4Addr,
                                   8053, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell server to (NOT) echo
    g_echoServer->setFalseEcho( true );

    // For()ever
    writeCount = 0;
    for( ; ; )
    {
         nn::socket::FdSetZero(&writeSet);
         nn::socket::FdSetSet(clisock, &writeSet);

         mytime.tv_sec  = 0;
         mytime.tv_usec = 100;

         rc = nn::socket::Select(clisock + 1, nullptr, &writeSet, nullptr, &mytime);
         if ( rc == 0  )
         {
             NN_LOG( "Socket not ready for writing after bytes: %d sent\n", writeCount );
             break;
         }
         else if ( rc < 0 )
         {
             NN_LOG( "Select fail with errno: %d\n", nn::socket::GetLastError() );
             return;
         }

         rc = NATF::API::COMMON::WriteData(clisock, workPtr, workPtrLen, unused1, unused2);
         if ( rc < 1 )
         {
             if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
             {
                 continue;
             }

             ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n", nn::socket::GetLastError() );
         }

         writeCount = writeCount + rc;
    }

    // At High Water mark...
    nn::socket::Close( clisock );
    NN_LOG( "=======> High Water Mark: %d\n", highWaterMark );
    NN_LOG( "=======> Low  Water Mark: %d\n", writeCount );
    NN_LOG( "=======> Difference     : %d\n", ( highWaterMark - writeCount ) );

    nTestsPassing++;

    ERROR_IF_AND_COUNT( writeCount > highWaterMark, "Low water mark is higher than High Water mark!  Failed!\n" );

out:
    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if (clisock > -1)
    {
        nn::socket::Close(clisock);
        clisock = -1;
    }

    // Free Payload memory
    if(workPtr != NULL)
    {
        free(workPtr);
        workPtr = NULL;
    }

    return;

}  // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_SO_RCVLOWAT)
{

    int                    intoptval = 0;
    nn::socket::SockLenT   intoptlen = 0;

    uint64_t             unused1 = 0, unused2 = 0;
    int                  tcpsock = -1, udpsock = -1;
    int                  clisock = -1, writeCount = 0;
    int                  rc = -1, idx;
    bool                 isSuccess = true, rval = false;
    unsigned char *      workPtr = nullptr;
    size_t               workPtrLen = 0;
    nn::socket::TimeVal  mytime = { 0 };
    nn::socket::FdSet    readSet;

    ////////////////
    // GetSockOpt Tests for: SO_SNDLOWAT
    ///////////////

    /// TCP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    static int sizeArrayTCPCertBad[] = {-65535, -32767, -16384, -9000, -1, 0, INT_MAX};
    static int sizeArrayTCPSizeCertBad = sizeof(sizeArrayTCPCertBad) / sizeof(int);

    for(idx = 0; idx < sizeArrayTCPSizeCertBad; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_RCVLOWAT (BAD) size: %d\n", sizeArrayTCPCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, false, sizeArrayTCPCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_RCVLOWAT (BAD) size: %d\n", sizeArrayTCPCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, false, sizeArrayTCPCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    static int sizeArrayTCPCertGood[] = {128, 256, 9000, 16384, 32767, 48996};
    static int sizeArrayTCPSizeCertGood = sizeof(sizeArrayTCPCertGood) / sizeof(int);

    for(idx = 0; idx < sizeArrayTCPSizeCertGood; idx++ )
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_RCVLOWAT (GOOD) size: %d\n", sizeArrayTCPCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, true, sizeArrayTCPCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_RCVLOWAT (GOOD) size: %d\n", sizeArrayTCPCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, true, sizeArrayTCPCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");

    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // UDP
    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    static int sizeArrayUDPCertBad[] = {-65535, -32767, -16384, -9000, -1, 0, INT_MAX};
    static int sizeArrayUDPSizeCertBad = sizeof(sizeArrayUDPCertBad) / sizeof(int);

    for(idx = 0; idx < sizeArrayUDPSizeCertBad; idx++)
    {
        NN_LOG("GetSockOpt:  [UDP]: SO_RCVLOWAT (BAD) size: %d\n", sizeArrayUDPCertBad[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, false, sizeArrayUDPCertBad[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    static int sizeArrayUDPCertGood[] = {128, 256, 9000, 9216};
    static int sizeArrayUDPSizeCertGood = sizeof(sizeArrayUDPCertGood) / sizeof(int);

    for(idx = 0; idx < sizeArrayUDPSizeCertGood; idx++ )
    {
        NN_LOG("GetSockOpt:  [UDP]: SO_RCVLOWAT (GOOD) size: %d\n", sizeArrayUDPCertGood[idx]);
        rval = NATF::API::COMMON::CheckNumericSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, true, sizeArrayUDPCertGood[idx]);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckNumericSetSockOpt()\n");
    }

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_RcvLoWat [TCP] - Option
    ////////////////

    workPtrLen = 30;
    rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
    ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen  );

    // Client: Set nn::socket::Option::So_RcvLoWat
    //
    intoptval = 20;      // LESS THAN workPtrLen
    intoptlen = sizeof( intoptval );
    rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, &intoptval, intoptlen,
                                   9000, g_echoServer->kInterfaceIPv4Addr,
                                   8053, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell server (to Echo) responses
    g_echoServer->setFalseEcho( false );


    ////////////////
    // TCP: For()ever
    //
    // Select (SHOULD NOT) unblock until 20 bytes are available on the socket (receive)
    ////////////////

    writeCount = 0;
    for( idx = 0; idx < workPtrLen; idx++ )
    {
         nn::socket::FdSetZero( &readSet );

         nn::socket::FdSetSet( clisock, &readSet );

         mytime.tv_sec  = 0;
         mytime.tv_usec = 100;

         rc = nn::socket::Select( clisock + 1, &readSet, nullptr, nullptr, &mytime );
         if ( rc > 0 )
         {
             ERROR_IF_AND_COUNT( idx < 20, "[TCP] Select unblocked with return code: %d, at index: %d - exiting\n", rc, idx );
             NN_LOG( "PASS - Recieve Water mark honored at index: %d\n", idx );
             break;
         }


         if ( rc == 0  )
         {
             rc = NATF::API::COMMON::WriteData( clisock, &workPtr[idx], 1, unused1, unused2 );
             if ( rc < 1 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     continue;
                 }

                 NN_LOG( "Write failed with errno: %d\n", nn::socket::GetLastError() );
                 break;
             }
        }

        ERROR_IF( rc < 0, "Select unblocked with FAILED return code: %d, errno: %d - exiting\n",
                           rc, nn::socket::GetLastError() );
    }

    nTestsPassing++;


    // TCP: Close Socket
    nn::socket::Close( clisock );
    clisock = -1;

#if 0

    /////////////////////
    // I found that on UDP, the Select call unblocks after 2 bytes are available on the inbound UDP socket.
    // I then duplicated this test on Freebsd and foud the (same result).  For the moment will assume this
    // works as designed.  JLT - 9/21/2016
    /////////////////////

    ////////////////
    // UDP:  RCVLOWAT
    ////////////////

    // Client: Set nn::socket::Option::So_RcvLoWat
    //
    intoptval = 20;      // LESS THAN workPtrLen
    intoptlen = sizeof( intoptval );
    rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, &intoptval, intoptlen,
                                   9000, g_echoServer->kInterfaceIPv4Addr,
                                   8053, g_echoServer->kInterfaceIPv4Addr, true, 0 );   // isUDP = true

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell server (to Echo) responses
    g_echoServer->setFalseEcho( false );

    ////////////////
    // UDP: For()ever
    ////////////////

    writeCount = 0;
    for( idx = 0; idx < workPtrLen; idx++ )
    {
         nn::socket::FdSetZero( &readSet );
         nn::socket::FdSetSet( clisock, &readSet );

         mytime.tv_sec  = 0;
         mytime.tv_usec = 100;

         rc = nn::socket::Select( clisock + 1, &readSet, nullptr, nullptr, &mytime );
         if ( rc > 0 )
         {
             ERROR_IF_AND_COUNT( idx < 20, "[UDP] - Select unblocked with return code: %d, at index: %d - exiting\n", rc, idx );
             NN_LOG( "PASS - Recieve Water mark honored at index: %d\n", idx );
             break;
         }

         if ( rc == 0  )
         {
             rc = NATF::API::COMMON::WriteData( clisock, &workPtr[idx], 1, unused1, unused2 );
             if ( rc < 1 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     continue;
                 }

                 NN_LOG( "Write failed with errno: %d\n", nn::socket::GetLastError() );
                 break;
             }
        }

        ERROR_IF( rc < 0, "Select unblocked with FAILED return code: %d, errno: %d - exiting\n",
                           rc, nn::socket::GetLastError() );
    }

    // UDP: Close Socket
    nn::socket::Close( clisock );
    clisock = -1;

#endif


out:
    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    // Free Payload memory
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    // } // Wrap Failing Test

    return;
} // NOLINT(impl/function_size)



TEST(ApiUnit,SetSockOpt_SO_SNDTIMEO)
{
    nn::socket::TimeVal    tsopt = { 0 };
    nn::socket::SockLenT   optlen = 0;

    uint64_t          unused1 = 0, unused2 = 0;
    size_t            workPtrLen = 0, myResultSize = 0;
    int               clisock = -1, tcpsock = -1, udpsock = -1;
    int               rc = -1, idx = 0;
    bool              isSuccess = true, rval = false;
    unsigned char *   workPtr = nullptr;

    ////////////////
    // GetSockOpt Tests for: SO_SNDTIMEO
    ///////////////

    /// TCP and UDP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    myResultSize = sizeof(g_SockOptTimeoutResults) / sizeof(SockOptTimeoutResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        tsopt.tv_sec  = g_SockOptTimeoutResults[idx].tv_sec;
        tsopt.tv_usec = g_SockOptTimeoutResults[idx].tv_usec;

        NN_LOG("GetSockOpt:  [TCP Client]: SO_SNDTIMEO - Test [tv_sec: %d, tv_usec: %d]\n", tsopt.tv_sec, tsopt.tv_usec);
        rval = NATF::API::COMMON::CheckTimevalSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, g_SockOptTimeoutResults[idx].result, tsopt);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckTimevalSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_SNDTIMEO - Test [tv_sec: %d, tv_usec: %d]\n", tsopt.tv_sec, tsopt.tv_usec);
        rval = NATF::API::COMMON::CheckTimevalSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, g_SockOptTimeoutResults[idx].result, tsopt);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckTimevalSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_SNDTIMEO - Test [tv_sec: %d, tv_usec: %d]\n", tsopt.tv_sec, tsopt.tv_usec);
        rval = NATF::API::COMMON::CheckTimevalSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, g_SockOptTimeoutResults[idx].result, tsopt);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckTimevalSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_SndTimeo [TCP] - Option
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo][TCP][Option: ON]\n" );

    workPtrLen = 1024;
    rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
    ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen  );

    // Client: Set nn::socket::Option::So_SndTimeo
    //
    tsopt.tv_sec  = 5;
    tsopt.tv_usec = 0;
    optlen = sizeof( nn::socket::TimeVal );

    rval = NATF::API::COMMON::SetSocketOptAndConnect(clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, &tsopt, optlen,
                                   9000, g_echoServer->kInterfaceIPv4Addr,
                                   8053, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

    // Tell server (DO NOT Echo) responses
    g_echoServer->setFalseEcho( true );

    // For()Ever
    for(; ; )
    {
        // WARNING: Blocking Write !  Should unblock..
        rc = NATF::API::COMMON::WriteData(clisock, workPtr, workPtrLen, unused1, unused2);
        if ( rc < 0 )
        {
            if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain ||  nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "==================\n");
                NN_LOG( "PASSED!  Function call was unblocked - errno: %d\n", nn::socket::GetLastError() );
                NN_LOG( "==================\n");
                nTestsPassing++;

                break;
            }

            ERROR_IF( rc < 0, "Wrirtte Failed!  Errno: %d\n", nn::socket::GetLastError() );
        }
    }

    // If we got here - it passed!

out:
    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    // Free Payload memory
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    return;
}


TEST(ApiUnit,SetSockOpt_SO_RCVTIMEO)
{
    nn::socket::TimeVal    tsopt = { 0 };
    nn::socket::SockLenT   optlen = 0;

    uint64_t          unused1 = 0;
    size_t            workPtrLen = 0, myResultSize = 0;
    int               clisock = -1, tcpsock = -1, udpsock = -1;
    int               rc = -1, idx = 0;
    bool              isSuccess = true, rval = false;
    unsigned char *   workPtr = nullptr;

    ////////////////
    // GetSockOpt Tests for: SO_RCVTIMEO
    ///////////////

    /// TCP and UDP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    myResultSize = sizeof(g_SockOptTimeoutResults) / sizeof(SockOptTimeoutResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        tsopt.tv_sec  = g_SockOptTimeoutResults[idx].tv_sec;
        tsopt.tv_usec = g_SockOptTimeoutResults[idx].tv_usec;

        NN_LOG("GetSockOpt:  [TCP Client]: SO_RCVTIMEO - Test [tv_sec: %d, tv_usec: %d]\n", tsopt.tv_sec, tsopt.tv_usec);
        rval = NATF::API::COMMON::CheckTimevalSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, g_SockOptTimeoutResults[idx].result, tsopt);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckTimevalSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_RCVTIMEO - Test [tv_sec: %d, tv_usec: %d]\n", tsopt.tv_sec, tsopt.tv_usec);
        rval = NATF::API::COMMON::CheckTimevalSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, g_SockOptTimeoutResults[idx].result, tsopt);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckTimevalSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_RCVTIMEO - Test [tv_sec: %d, tv_usec: %d]\n", tsopt.tv_sec, tsopt.tv_usec);
        rval = NATF::API::COMMON::CheckTimevalSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, g_SockOptTimeoutResults[idx].result, tsopt);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckTimevalSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////
    // nn::socket::Option::So_RcvTimeo [TCP] - Option
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo][TCP][Option: ON]\n" );

    workPtrLen = 1024;
    rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
    ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen  );

    // Client: Set nn::socket::Option::So_RcvTimeo
    //
    tsopt.tv_sec  = 5;
    tsopt.tv_usec = 0;
    optlen = sizeof( nn::socket::TimeVal );

    rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, &tsopt, optlen,
                                   9000, g_echoServer->kInterfaceIPv4Addr,
                                   8053, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

    // Tell server (DO NOT Echo) responses
    g_echoServer->setFalseEcho( true );

    // WARNING: Blocking read !  Should unblock..
    rc = NATF::API::COMMON::ReadData( clisock, workPtr, workPtrLen, unused1 );
    ERROR_IF( rc > -1, "Read succeeded but should have timed out!\n" );

    // If we got here - it passed!
    NN_LOG( "==================\n");
    NN_LOG( "PASSED!  Function call was unblocked - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "==================\n");
    nTestsPassing++;

out:
    if (tcpsock > -1)
    {
        nn::socket::Close(tcpsock);
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close(udpsock);
        udpsock = -1;
    }

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    // Free Payload memory
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    return;
}


TEST(ApiUnit,SetSockOpt_SO_Get_Only_Options)
{
    int                    intoptval = 0;
    nn::socket::SockLenT   intoptlen = 0;

    int               tcpsock = -1, udpsock = -1;
    int               rc = -1;
    bool              isSuccess = true;

    tcpsock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    udpsock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram,  nn::socket::Protocol::IpProto_Udp );


    ////////////////
    // nn::socket::Option::So_AcceptConn - TCP "ON"
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_AcceptConn][TCP][Option: ON]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( g_echoServer->serverTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_AcceptConn, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "GetSockOpt on  nn::socket::Option::So_AcceptConn failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_AcceptConn = <%d>\n", intoptval );
    ERROR_IF( intoptval == 0, "GetockOpt on nn::socket::Option::So_AcceptConn failed!  Optval: %d\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_AcceptConn - TCP "OFF"
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_AcceptConn][TCP][Option: OFF]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_AcceptConn, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "GetSockOpt on  nn::socket::Option::So_AcceptConn failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_AcceptConn = <%d>\n", intoptval );
    ERROR_IF( intoptval != 0, "GetockOpt on nn::socket::Option::So_AcceptConn failed!  Optval: %d\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_Type - TCP (nn::socket::Type::Sock_Stream)
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_Type][TCP][nn::socket::Type::Sock_Stream]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Type, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "GetSockOpt on  nn::socket::Option::So_Type failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_Type = <%d>\n", intoptval );
    ERROR_IF( intoptval != static_cast<int>(nn::socket::Type::Sock_Stream), "TCP GetockOpt on nn::socket::Option::So_Type failed!  Optval: %d\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_Type - UDP (SOCK_DRGAM)
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_Type][UDP][nn::socket::Type::Sock_Dgram]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Type, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "UDP GetSockOpt on nn::socket::Option::So_Type failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_Type = <%d>\n", intoptval );
    ERROR_IF( intoptval != static_cast<int>(nn::socket::Type::Sock_Dgram), "UDP GetockOpt on nn::socket::Option::So_Type failed!  Optval: %d\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_Type - TCP (nn::socket::Protocol::IpProto_Tcp)
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_Protocol][TCP][nn::socket::Protocol::IpProto_Tcp]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Protocol, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "TCP GetSockOpt on  nn::socket::Option::So_Protocol failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_Type = <%d>\n", intoptval );
    ERROR_IF( intoptval != static_cast<int>(nn::socket::Protocol::IpProto_Tcp), "TCP GetockOpt on nn::socket::Option::So_Protocol failed!  Optval: %d\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_Type - UDP (nn::socket::Protocol::IpProto_Udp)
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_Protocol][UDP][nn::socket::Protocol::IpProto_Udp]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Protocol, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "UDP GetSockOpt on nn::socket::Option::So_Protocol failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_Type = <%d>\n", intoptval );
    ERROR_IF( intoptval != static_cast<int>(nn::socket::Protocol::IpProto_Udp), "UDP GetockOpt on nn::socket::Option::So_Protocol failed!  Optval: %d\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_Error
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_Error]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( g_echoServer->serverTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Error, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "GetSockOpt on nn::socket::Option::So_Error failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_Error = <%d>\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_ListenQLimit
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ListenQLimit][TCP]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( g_echoServer->serverTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ListenQLimit, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "GetSockOpt on nn::socket::Option::So_ListenQLimit failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_ListenQLimit = <%d>\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_ListenQLen
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ListenQLen][TCP]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( g_echoServer->serverTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ListenQLen, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "GetSockOpt on nn::socket::Option::So_ListenQLen failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_ListenQLen = <%d>\n", intoptval );
    nTestsPassing++;

    ////////////////
    // nn::socket::Option::So_ListenIncQLen
    ////////////////

    NN_LOG( "\nTest: [nn::socket::Level::Sol_Socket, nn::socket::Option::So_ListenIncQLen][TCP]\n" );

    intoptlen = sizeof( int );
    rc = nn::socket::GetSockOpt( g_echoServer->serverTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ListenIncQLen, &intoptval, &intoptlen );
    ERROR_IF( rc < 0, "GetSockOpt on nn::socket::Option::So_ListenQLen failed!  Errno: %d\n", nn::socket::GetLastError() );

    NN_LOG( "nn::socket::Option::So_ListenIncQLen = <%d>\n", intoptval );
    nTestsPassing++;

out:
    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( udpsock > -1 )
    {
        nn::socket::Close( udpsock );
        udpsock = -1;
    }

    return;
}


TEST(ApiUnit,SetSockOpt_SO_OOB_INLINE)
{

// WRAP_FAILING_TEST( "SIGLO-58243", "[NX] - Socket API: SockAtMark() does not return the OOB mark" )
// {
    nn::socket::Errno        myError = nn::socket::Errno::ESuccess;
    nn::socket::SockAddrIn   localAddr = { 0 };
    size_t                   myResultSize = 0;
    int                      tcpsock = -1, udpsock = -1;
    int                      cliSock = -1, svrSock = -1, acceptedSock = -1;
    int                      rc = -1, idx = 0, count = 0, atMark = 0;
    char                     buf1[1024] = { 0 };
    bool                     isSuccess = true, rval = false, isOOB = false;

    const char * msgs[]
    {
      "The rain in Spain...",
      "X"
    };

    const char *            pAnswer = "The rain in Spain...X";

    ////////////////
    // GetSockOpt Tests for: SO_OOB_INLINE
    ///////////////

    /// TCP and UDP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    myResultSize = sizeof(g_SockOptValueResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_OOB_INLINE - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_OOB_INLINE - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling    ()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_OOB_INLINE - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    ////////////////////////////////////////////////////////////////////////////////////////////////////////
    //  T E S T:    OOB_INLINE (ON)
    ////////////////////////////////////////////////////////////////////////////////////////////////////////

    /////////////////
    // Create Server
    /////////////////

    svrSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF(svrSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Local Address
    memset(&localAddr, 0, sizeof( localAddr ));
    localAddr.sin_port        = nn::socket::InetHtons(9001);
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr);
    ERROR_IF(rc != 1, "InetPton() Failed but Success was expected!" );

    // Set Reuse Socket
    rval = NATF::API::COMMON::CheckBooleanSetSockOpt(svrSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, true, 1);
    ERROR_IF(rval == false, "SetSockOpt() Failed to set SO_REUSEADDR to (true)!" );

    // SET SO_OOB_INLINE - ON
    rval = NATF::API::COMMON::CheckBooleanSetSockOpt(svrSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, true, 1);
    ERROR_IF(rval == false, "SetSockOpt() Failed to set SO_OOB_INLINE to (true)!" );

    // Client: Bind (self)
    rc = nn::socket::Bind(svrSock, (nn::socket::SockAddr *) &localAddr, sizeof(localAddr));
    ERROR_IF(rc < 0, "Bind() returned: %d but was unexpected!", rc);

    // For TCP - Add a listen queue
    rc = nn::socket::Listen(svrSock, 5 );
    ERROR_IF(rc < 0, "Listen() returned: %d but errno: %d!", rc, nn::socket::GetLastError());

    NN_LOG( "Server established at: loopback:9000\n");

    /////////////////
    // Create Client
    /////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, 9001);
    ERROR_IF_AND_COUNT(cliSock < 0, "[TCP] Failed to create Client socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );

    ////////////////
    // Accept Socket
    ////////////////

    acceptedSock = nn::socket::Accept( svrSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT(acceptedSock < 0, "Failed to Accept socket from server socket\n" );

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl(acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    ///////////////
    // Client Writes (regular) and (OOB) data to server..
    ///////////////

    NN_LOG( "Send(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "Send() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1] );

    ///////////////
    // OOB_INLINE:  Server reads OOB data as "regular data".  (DO NOT) specify the Msg_Oob flag during read(2) or recv(2)
    ///////////////

    // Initialize Recieve Buffer
    memset(buf1, 0, sizeof(buf1));
    count = 0;

    NN_LOG( "Nonblocking Read: [" );
    for( ; ; )
    {
        rc = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None);
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();

            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    ///////////////
    // Did we receive the right message?
    ///////////////

    if (  (count == strlen(pAnswer))                       &&
          (strncmp(buf1, pAnswer, strlen(pAnswer)) == 0)   )
    {
        NN_LOG( "Test Passed!  Received [%.*s]\n", count, buf1);
    }
    else
    {
        ERROR_IF_AND_COUNT(rc > -1, "Recv: Got [len: %d, Val: %s], but expected [len: %d, Val: %s]\n", rc, buf1, strlen(pAnswer), pAnswer);
    }

    // Close Client Socket
    nn::socket::Close(cliSock);
    cliSock = -1;

    // Close Accepted Socket
    nn::socket::Close(acceptedSock);
    acceptedSock = -1;

    // Close Server Socket
    nn::socket::Close(svrSock);
    svrSock = -1;

    ////////////////////////////////////////////////////////////////////////////////////////////////////////
    //  T E S T:    OOB_INLINE (OFF)
    ////////////////////////////////////////////////////////////////////////////////////////////////////////

    /////////////////
    // Create Server
    /////////////////

    svrSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF(svrSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Local Address
    memset(&localAddr, 0, sizeof( localAddr ));
    localAddr.sin_port        = nn::socket::InetHtons(9001);
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr);
    ERROR_IF(rc != 1, "InetPton() Failed but Success was expected!" );

    // Set Reuse Socket
    rval = NATF::API::COMMON::CheckBooleanSetSockOpt(svrSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, true, 1);
    ERROR_IF(rval == false, "SetSockOpt() Failed to set SO_REUSEADDR to (true)!" );

    // SET SO_OOB_INLINE - OFF
    rval = NATF::API::COMMON::CheckBooleanSetSockOpt(svrSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, false, 0);
    ERROR_IF(rval == false, "SetSockOpt() Failed to set SO_OOB_INLINE to (true)!" );

    // Client: Bind (self)
    rc = nn::socket::Bind(svrSock, (nn::socket::SockAddr *) &localAddr, sizeof(localAddr));
    ERROR_IF(rc < 0, "Bind() returned: %d but was unexpected!", rc);

    // For TCP - Add a listen queue
    rc = nn::socket::Listen(svrSock, 5 );
    ERROR_IF(rc < 0, "Listen() returned: %d but errno: %d!", rc, nn::socket::GetLastError());

    NN_LOG( "Server established at: loopback:9000\n");

    /////////////////
    // Create Client
    /////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, 9001);
    ERROR_IF_AND_COUNT(cliSock < 0, "[TCP] Failed to create Client socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );

    ////////////////
    // Accept Socket
    ////////////////

    acceptedSock = nn::socket::Accept( svrSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT(acceptedSock < 0, "Failed to Accept socket from server socket\n" );

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl(acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    ///////////////
    // Client Writes (regular) and (OOB) data to server..
    ///////////////

    NN_LOG( "Send(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "Send() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::Send( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1] );

    ///////////////
    // OOB_INLINE:  Server reads OOB data as "regular data".  Do not specify the Msg_Oob flag during read(2) or recv(2)
    ///////////////

    // Initialize Recieve Buffer
    memset(buf1, 0, sizeof(buf1));
    count = 0;

    // TOP: of loop
    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ;; )
    {
        atMark = nn::socket::SockAtMark(acceptedSock);
        if (atMark < 0)
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError());
        }
        if (atMark == 1)
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc = nn::socket::Recv(acceptedSock, (unsigned char *) &buf1[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None) );
        if (rc < 0)
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                // Regular (NOT-OOB) data read
                rc = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None );
                if (rc < 0)
                {
                    myError = nn::socket::GetLastError();

                    if (myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock)
                    {
                        break;
                    }

                    ERROR_IF(rc < 0, "Recv() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                }
                NN_LOG( "%c", (char) buf1[count] );
                count++;
                continue;
            }

            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
    }

    NN_LOG( "]\n" );
    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    ///////////////
    // Did we receive the right message?
    ///////////////

    if (  (count == strlen(pAnswer))                       &&
          (strncmp(buf1, pAnswer, strlen(pAnswer)) == 0)   )
    {
        NN_LOG( "Test Passed!  Received [%.*s]\n", count, buf1);
    }
    else
    {
        NN_LOG( "Recv: Got [len: %d, Val: %s], but expected [len: %d, Val: %s]\n", count, buf1, strlen(pAnswer), pAnswer);
    }

    // Close Client Socket
    nn::socket::Close(cliSock);
    cliSock = -1;

    // Close Accepted Socket
    nn::socket::Close(acceptedSock);
    acceptedSock = -1;

    // Close Server Socket
    nn::socket::Close(svrSock);
    svrSock = -1;

out:
    if ( tcpsock > -1 )
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if ( udpsock > -1 )
    {
        nn::socket::Close( udpsock );
        udpsock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( svrSock > -1 )
    {
        nn::socket::Close( svrSock );
        svrSock = -1;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

// } // END - WRAP FAILING TEST

    return;

}   // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_SO_DONTROUTE)
{

    nn::socket::SockLenT         intoptlen = 0;
    nn::socket::SockAddrIn       targetAddr = { 0 };
    size_t                       myResultSize = 0;
    int                          tcpsock = -1, udpsock = -1;
    int                          sockfds = -1, rc = -1, intoptval = 0, idx = 0;
    bool                         isSuccess = true, rval = false;

    ////////////////
    // GetSockOpt Tests for: SO_OOB_INLINE
    ///////////////

    /// TCP and UDP
    tcpsock = NATF::API::COMMON::MakeTCPConnection(g_echoServer->kInterfaceIPv4Addr, g_echoServer->kEchoServerTCPPort);
    ERROR_IF_AND_COUNT(tcpsock < 0, "Failed to call MakeTCPConnection()\n");

    udpsock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8089, "127.0.0.1", 8089);
    ERROR_IF_AND_COUNT(udpsock < 0, "Failed to call MakeUDPConnection()\n");

    myResultSize = sizeof(g_SockOptValueResults) / sizeof(SockOptValueResult);
    for(idx = 0; idx < myResultSize; idx++)
    {
        NN_LOG("GetSockOpt:  [TCP Client]: SO_DONT_ROUTE - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(tcpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [TCP Server]: SO_DONT_ROUTE - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(g_echoServer->acceptedTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");

        NN_LOG("GetSockOpt:  [UDP]: SO_DONT_ROUTE - Test [Value: %d]\n", g_SockOptValueResults[idx].value);
        rval = NATF::API::COMMON::CheckBooleanSetSockOpt(udpsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, g_SockOptValueResults[idx].result, g_SockOptValueResults[idx].value);
        ERROR_IF_AND_COUNT(rval == false, "Failed calling CheckBooleanSetSockOpt()\n");
    }

    // Close TCP Socket
    nn::socket::Close(tcpsock);
    tcpsock = -1;

    // Close UDP Socket
    nn::socket::Close(udpsock);
    udpsock = -1;

    /////////////////
    // Create Server
    /////////////////

    sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( sockfds < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Set Option
    intoptval = 1;
    intoptlen = sizeof( intoptval );
    PRINT_AND_CALL(rc = nn::socket::SetSockOpt( sockfds, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, &intoptval, intoptlen ));
    ERROR_IF( rc < 0, "SetSockOpt failed: rc: %d, socklevel: %d, sockopt: %d, errno: %d",
              rc, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, nn::socket::GetLastError() );

    // Is Option on?
    PRINT_AND_CALL( rc = nn::socket::GetSockOpt( sockfds, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, &intoptval, &intoptlen ) );
    ERROR_IF( rc < 0, "GetSockOpt failed! rc: %d, socklevel: %d, sockopt: %d, optlen: %d, errno: %d\n",
              rc, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, nn::socket::GetLastError() );

    // Client: Target Address
    memset(&targetAddr, 0, sizeof( targetAddr ));
    targetAddr.sin_port        = nn::socket::InetHtons(53); // domain = dns
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client:  Connect USC B-Root
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet,"192.228.79.201", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client: Connect should fail!  Can't directly route there..
    rc = nn::socket::Connect( sockfds, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    if (rc > -1)
    {
        NN_LOG( "nn::socket::Option::So_DontRoute should have failed to: 192.228.79.201, but it succeeded!" );
        goto out;
    }

    // If we got here - it passed!
    NN_LOG( "==================\n");
    NN_LOG( "PASSED!  Connection failed using nn::socket::Option::So_DontRoute: %d\n", nn::socket::GetLastError() );
    NN_LOG( "==================\n");
    nTestsPassing++;

out:
    if (tcpsock > -1)
    {
        nn::socket::Close( tcpsock );
        tcpsock = -1;
    }

    if (udpsock > -1)
    {
        nn::socket::Close( udpsock );
        udpsock = -1;
    }

    if (sockfds > -1)
    {
        nn::socket::Close( sockfds );
        sockfds = -1;
    }

}


TEST(ApiUnit,SetSockOpt_SO_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,SetSockOpt_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
