﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/socket_ConstantsPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions
#include "Unit/testNet_ApiUnitCommon.h"

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************
namespace
{
    const int            MyThreadPriority = 10;       // Thread Priority
    const int            MyMaxNumOfSockets = 3;       // Max test sockets
    ThreadedController   *g_pThreadedControllerTest = NULL;
    int                  g_EchoServerThreadId = -1;
    EchoServer           *g_pEchoServer = NULL;

    void
    StartEchoServer( void * inArg )
    {
        // Tell Log we are done
        NN_LOG( "[Echo Server]: Thread starting\n" );

        EchoServer * echoServer = static_cast<EchoServer *>(inArg);

        // Start the DNS Server Main Loop
        echoServer->Start( nullptr );

        // Tell Log we are done
        NN_LOG( "[Echo Server]: Thread exiting\n" );
    }


    bool
    InitializeTesting()
    {
        bool        isSuccess = true;

        NN_LOG( "In\n\n" );

        ///////////////////////////
        //// Test Counts: Initialize
        ///////////////////////////

        INITIALIZE_TEST_COUNTS;


        ///////////////////////////
        //// NIFM Library: Initialize
        ///////////////////////////

        ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

        // NIFM: Network Interface is Online
        NN_LOG( "====================================\n" );
        NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
        NN_LOG( "====================================\n" );


        ///////////////////////////
        ////  Allocate Echo Server
        ///////////////////////////

        g_pEchoServer = new EchoServer();
        if ( g_pEchoServer == NULL )
        {
            NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
            goto out;
        }

        NN_LOG( "===> Starting [Echo Server]\n" );

        // Create: ThreadedTest
        g_pThreadedControllerTest = new ThreadedController();
        if ( g_pThreadedControllerTest == NULL )
        {
            NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
            goto out;
        }

        // Allocate 1 slot
        g_pThreadedControllerTest->Initialize( MyMaxNumOfSockets );
        // Create a thread to run the Server in

        g_EchoServerThreadId = g_pThreadedControllerTest->CreateThread( &StartEchoServer, static_cast<void *>(g_pEchoServer), MyThreadPriority );
        if (  g_EchoServerThreadId < 0 )
        {
            NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
            goto out;
        }

        NN_LOG( "Successfully started Echo Server" );

        // Give Thread time to start
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        return( true );

    out:

        // If the loopback DNS server is allocated
        if ( g_pThreadedControllerTest != NULL )
        {
            delete g_pThreadedControllerTest;
            g_pThreadedControllerTest = NULL;
        }

        // If the loopback DNS server is allocated
        if ( g_pEchoServer != NULL )
        {
            delete g_pEchoServer;
            g_pEchoServer = NULL;
        }

        return( false );
    }


    bool
    TeardownTesting()
    {
         bool    isSuccess = true;

        ///////////////////
        //// Stop DNS Server
        ////////////////////

        NN_LOG( "Stopping Echo Server\n" );

        // If the echo server is allocated
        if ( g_pEchoServer != NULL )
        {
            g_pEchoServer->Stop();
        }

        // Wait for Echo Server thread to die
        g_pThreadedControllerTest->WaitThread( g_EchoServerThreadId );

        // Destroy Echo Server Thread
        g_pThreadedControllerTest->DestroyThread( g_EchoServerThreadId );

        // Delete Echo Server
        delete g_pEchoServer;
        g_pEchoServer = NULL;

        // Delete Threaded Test
        delete g_pThreadedControllerTest;
        g_pThreadedControllerTest = NULL;
        g_EchoServerThreadId = -1;

        NN_LOG( "Echo Server successfully stopped!\n" );


        ////////////////////
        ////  Stop Testing
        ////////////////////

        ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

        ////////////////////
        ////  Print Test Counts
        ////////////////////

        PRINT_TEST_COUNTS;

        EXPECT_EQ( isSuccess, true );

        NN_LOG( "Out\n\n" );

        return( true );

    out:

        return( false );
    }

    const int MaxMessages = nn::socket::MaxNumMessages;     // 32
    const int MaxMessageSize = nn::socket::MaxMessageSize;  // 1500
    const int OverloadBufDifference = 50;

}

////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,RecvMMsg_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,RecvMMsg_Various_Tests)
{
    int                     tcpSocket = -1, udpSocket = -1, udpServerSocket = -1, tcpServerSocket, acceptedSocket = -1;
    int                     rc = -1, idx = 0, msgIdx = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    nn::socket::SockAddrIn  localAddr = { 0 }, targetAddr = { 0 }, serverAddr = { 0 };
    nn::socket::SockLenT    targetAddrLen = sizeof(targetAddr);
    nn::socket::MMsgHdr     *pSendMsgs = new nn::socket::MMsgHdr[MaxMessages](),
                            *pRecvMsgs = new nn::socket::MMsgHdr[MaxMessages](),
                            *pBigMsgs = new nn::socket::MMsgHdr[MaxMessages](),
                            *pBigReceiveMsgs = new nn::socket::MMsgHdr[MaxMessages]();
    nn::socket::Iovec       *pOutIovecs = new nn::socket::Iovec[MaxMessages](),
                            *pInIovecs = new nn::socket::Iovec[2](),
                            *pBigOutIovecs = new nn::socket::Iovec[MaxMessages](),
                            *pBigIov = new nn::socket::Iovec[1]();
    nn::socket::SockAddrIn  *pNames = new nn::socket::SockAddrIn[MaxMessages]();
    nn::socket::SockLenT    optlen = 0;


    nn::TimeSpan    timeout = nn::TimeSpan::FromSeconds(5);
    nn::socket::InAddr         myIp = { 0 }, myMask = { 0 }, myGw = { 0 }, myDns1 = { 0 }, myDns2 = { 0 };
    nn::socket::TimeVal         myTime = { 0 };
    bool            isSuccess = true;

    static char     bufs[MaxMessages][1400] = { { 0 } },
                    bigReceiveBufs[MaxMessages][MaxMessageSize + OverloadBufDifference] = { { 0 } };

    char            bigBuf[MaxMessageSize + OverloadBufDifference] = { 0 };

    const char * msgs[]
    {
        "sendmmsg",
        "test",
        "sendmmsgtest"
    };

    pInIovecs[0].iov_base = const_cast<char *>( msgs[0] );
    pInIovecs[0].iov_len = strlen( static_cast<const char *>(pInIovecs[0].iov_base) );
    pInIovecs[1].iov_base = const_cast<char *>( msgs[1] );
    pInIovecs[1].iov_len = strlen( static_cast<const char *>(pInIovecs[1].iov_base) );

    // Fill Send Message Vector
    for (int msgidx = 0; msgidx < MaxMessages; msgidx++ )
    {
        pSendMsgs[msgidx].msg_hdr.msg_iov = pInIovecs;
        pSendMsgs[msgidx].msg_hdr.msg_iovlen = 2;
    }

    // Initialize recv msghdrs
    for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++)
    {
        pOutIovecs[msgIdx].iov_base          = bufs[msgIdx];
        pOutIovecs[msgIdx].iov_len           = 1400;
        pRecvMsgs[msgIdx].msg_hdr.msg_iov     = &pOutIovecs[msgIdx];
        pRecvMsgs[msgIdx].msg_hdr.msg_iovlen  = 1;
        pRecvMsgs[msgIdx].msg_hdr.msg_name    = &pNames[msgIdx];
        pRecvMsgs[msgIdx].msg_hdr.msg_namelen = sizeof(pNames[msgIdx]);
    }


    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_pEchoServer->serverTCP )  ||
              ( idx == g_pEchoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        rc = nn::socket::RecvMMsg( idx, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
        myError = nn::socket::GetLastError();
        if ( rc < 0 )
        {
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nRecvMMsg() bad fds: %d should have failed with nn::socket::Errno::EBadf. Errno=<%d>\n", idx, myError );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nRecvMMsg() succeeded but fds: %d. Expected to fail.\n" );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EBadf failed\n" );
    isSuccess = true;


    ////////////
    // Make TCP Connection
    ///////////

    tcpSocket = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSocket < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSocket = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSocket < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL Message Vector
    ///////////

    NN_LOG( "Testing: [TCP] - Null Message Vector - nn::socket::Errno::EInval\n" );

    rc = nn::socket::RecvMMsg( tcpSocket, nullptr, 1, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP]: RecvMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvMMsg() expected nn::socket::Errno::EInval. Errno:%d\n", myError );

    ////////////
    // [UDP] - Test NULL Message Vector
    ///////////

    NN_LOG( "Testing: [UDP] - Null Message Vector - nn::socket::Errno::EInval\n" );

    rc = nn::socket::RecvMMsg( udpSocket, nullptr, 1, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP]: RecvMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvMMsg() expected nn::socket::Errno::EInval. Errno:%d\n", myError );


    ////////////
    // [TCP] - Test No Message Vector Length
    ///////////

    NN_LOG( "Testing: [TCP] - No Message Vector Length - nn::socket::Errno::EInval\n" );

    rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, 0, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP]: RecvMMsg() succeded but should have failed.\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvMMsg expected nn::socket::Errno::EInval. Errno:%d\n", myError );


    ////////////
    // [UDP] - Test No Message Vector Length
    ///////////

    NN_LOG( "Testing: [UDP] - No message Vector Length - nn::socket::Errno::EInval\n" );

    rc = nn::socket::RecvMMsg( udpSocket, pRecvMsgs, 0, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP]: RecvMMsg() succeded but should have failed." );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvMMsg expected nn::socket::Errno::EInval. Errno:%d\n", myError );


    ////////////
    // [TCP] - Test Overload Message Vector Length
    ///////////

    NN_LOG( "Testing: [TCP] - Overload Message Vector Length - nn::socket::Errno::EInval\n" );

    rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, MaxMessages + 1, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP]: RecvMMsg() succeded but should have failed." );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvMMsg expected nn::socket::Errno::EInval. Errno:%d\n", myError );


    ////////////
    // [UDP] - Test Overload Message Vector Length
    ///////////

    NN_LOG( "Testing: [UDP] - Overload Message Vector Length - nn::socket::Errno::EInval\n" );

    rc = nn::socket::RecvMMsg( udpSocket, pRecvMsgs, MaxMessages + 1, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP]: RecvMMsg() succeded but should have failed." );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvMMsg expected nn::socket::Errno::EInval. Errno:%d\n", myError );


    ////////////
    // Fill mmsghdr with overloaded messsage and total bytes allowed
    ///////////

    memset( bigBuf, 78, sizeof(bigBuf) );
    pBigIov[0].iov_base = bigBuf;
    pBigIov[0].iov_len = sizeof(bigBuf);

    for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
    {
        pBigMsgs[msgIdx].msg_hdr.msg_iov = pBigIov;
        pBigMsgs[msgIdx].msg_hdr.msg_iovlen = 1;
    }

    ////////////
    // Initialize big receive mmsghdr
    ///////////
    for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++)
    {
        pBigOutIovecs[msgIdx].iov_base          = &bigReceiveBufs[msgIdx];
        pBigOutIovecs[msgIdx].iov_len           = sizeof(bigReceiveBufs[msgIdx]);
        pBigReceiveMsgs[msgIdx].msg_hdr.msg_iov     = &pBigOutIovecs[msgIdx];
        pBigReceiveMsgs[msgIdx].msg_hdr.msg_iovlen  = 1;
        pBigReceiveMsgs[msgIdx].msg_hdr.msg_name    = &pNames[msgIdx];
        pBigReceiveMsgs[msgIdx].msg_hdr.msg_namelen = sizeof(pNames[msgIdx]);
    }


    ////////////
    // [TCP] - Test Overload Message bytes - Small buffer; extra data will come in as another message
    ///////////

    NN_LOG( "Testing: [TCP] - Overload Message bytes - Small buffer\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pBigMsgs, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 1, "SendMMsg() failed to send 1 msg. Result:%d\n", rc );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    rc = nn::socket::RecvMMsg( tcpSocket, pBigReceiveMsgs, 2, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT(rc != 2, "RecvMMsg() failed to receive 2 msgs. Result:%d\n", rc );
    ERROR_IF_AND_COUNT( pBigReceiveMsgs[0].msg_len != MaxMessageSize && pBigReceiveMsgs[1].msg_len != OverloadBufDifference,
    "RecvMMsg() expected max message size of %d bytes, Actually received %d bytes", MaxMessageSize, pBigReceiveMsgs[0].msg_len );


    ////////////
    // [UDP] - Test Overload Message bytes - Truncated received data
    ///////////

    NN_LOG( "Testing: [UDP] - Overload Message bytes - Truncated received data\n" );

    rc = nn::socket::SendMMsg( udpSocket, pBigMsgs, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 1, "SendMMsg() failed to send 1 msg. Result:%d\n", rc );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    rc = nn::socket::RecvMMsg( udpSocket, pBigReceiveMsgs, 1, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT(rc != 1, "RecvMMsg() failed to receive 1 msg. Result:%d\n", rc );
    ERROR_IF_AND_COUNT( pBigReceiveMsgs[0].msg_len != MaxMessageSize,
        "RecvMMsg() expected max message size of %d bytes, Actually received %d bytes", MaxMessageSize, pBigReceiveMsgs[0].msg_len );


    ////////////
    // [TCP] - nn::socket::MsgFlag::Msg_DontWait - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock
    ///////////

    NN_LOG( "Testing: [TCP]- nn::socket::MsgFlag::Msg_DontWait - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock\n" );

    rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_DontWait, nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP] RecvMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock , "[TCP] RecvMMsg() got Errno:%d\n", myError );


    ////////////
    // [UDP] - nn::socket::MsgFlag::Msg_DontWait - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock
    ///////////

    NN_LOG( "Testing: [UDP]- nn::socket::MsgFlag::Msg_DontWait - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock\n" );

    rc = nn::socket::RecvMMsg( udpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_DontWait, nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] RecvMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock , "[UDP] RecvMMsg() got Errno:%d\n", myError );


    ////////////
    // [TCP] - Timeout - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock
    ///////////

    NN_LOG( "Testing: [TCP] - Timeout - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock\n" );

    rc = NATF::API::COMMON::SetBlocking( tcpSocket );
    ERROR_IF( rc < 0, "Fcntl() failed disabling nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP] RecvMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock, "[TCP] RecvMMsg() got Errno:%d\n", myError );


    ////////////
    // [UDP] - Timeout - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock
    ///////////

    NN_LOG( "Testing: [UDP] - Timeout - nn::socket::Errno::EAgain/nn::socket::Errno::EWouldBlock\n" );

    rc = NATF::API::COMMON::SetBlocking( udpSocket );
    ERROR_IF( rc < 0, "Fcntl() failed disabling nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    rc = nn::socket::RecvMMsg( udpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] RecvMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock, "[UDP] RecvMMsg() got Errno:%d\n", myError );


    ////////////
    // [TCP] - nn::socket::MsgFlag::Msg_Peek
    ///////////

    NN_LOG( "Testing: [TCP] - nn::socket::MsgFlag::Msg_Peek\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "[TCP]: SendMMsg() Failed. Expected to Succeed. Errno:%d\n", nn::socket::GetLastError() );

    // Wait for Echo back
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    for ( idx = 0; idx < 5; idx++ )
    {
        NN_LOG( "[TCP]: nn::socket::MsgFlag::Msg_Peek [%d]\n", idx );

        rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_Peek, &timeout );
        if ( rc != MaxMessages )
        {
            NN_LOG( "\tRecvMMsg() - nn::socket::MsgFlag::Msg_Peek sent %d/%d messages, Errno:%d\n", rc, MaxMessages, nn::socket::GetLastError() );
            isSuccess = false;
        }
        else
        {
            for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
            {
                // Strings are received combined - "sendmmsgtest"
                if ( strncmp(bufs[msgIdx], msgs[2], strlen(msgs[2])) != 0 )
                {
                    NN_LOG( "\tRecvMMsg failed to read data: Expected: [%s], Got [%.*s]\n", msgs[2], rc, bufs[msgIdx] );
                    isSuccess = false;
                }
            }

            NN_LOG( "First Message received: %s\n", bufs[0] );
        }

        // Reset Buffers
        for( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
        {
            memset( bufs[msgIdx], 0, sizeof(bufs[msgIdx]) );
        }
    }

    // Flush Socket
    rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc < 0, "RecvMMsg() Failed. Expected to Succeed. Errno:%d\n", nn::socket::GetLastError() );

    ERROR_IF_AND_COUNT( !isSuccess, "[UDP] - nn::socket::MsgFlag::Msg_Peek Failed\n" );
    isSuccess = true;


    ////////////
    // [UDP] - nn::socket::MsgFlag::Msg_Peek
    ///////////

    NN_LOG( "Testing: [UDP] - nn::socket::MsgFlag::Msg_Peek\n" );

    rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "[UDP]: SendMMsg() Failed. Expected to Succeed. Errno:%d\n", nn::socket::GetLastError() );

    // Wait for Echo back
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    for ( idx = 0; idx < 5; idx++ )
    {
        NN_LOG( "[UDP]: nn::socket::MsgFlag::Msg_Peek [%d]\n", idx );

        rc = nn::socket::RecvMMsg( udpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_Peek, &timeout );
        if ( rc != MaxMessages )
        {
            NN_LOG( "\tRecvMMsg() - nn::socket::MsgFlag::Msg_Peek sent %d/%d messages, Errno:%d\n", rc, MaxMessages, nn::socket::GetLastError() );
            isSuccess = false;
        }
        else
        {
            for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
            {
                // Strings are received combined - "sendmmsgtest"
                if ( strncmp(bufs[msgIdx], msgs[2], strlen(msgs[2])) != 0 )
                {
                    NN_LOG( "\tRecvMMsg failed to read data: Expected: [%s], Got [%.*s]\n", msgs[2], rc, bufs[msgIdx] );
                    isSuccess = false;
                }
            }

            NN_LOG( "First Message received: %s\n", bufs[0] );
        }

        // Reset Buffers
        for( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
        {
            memset( bufs[msgIdx], 0, sizeof(bufs[msgIdx]) );
        }
    }

    // Flush Socket
    rc = nn::socket::RecvMMsg( udpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc < 0, "RecvMMsg() Failed. Expected to Succeed. Errno:%d\n", nn::socket::GetLastError() );

    ERROR_IF_AND_COUNT( !isSuccess, "[UDP] - nn::socket::MsgFlag::Msg_Peek Failed\n" );
    isSuccess = true;


    ////////////
    // [UDP] - Test Simple Network Echo Recieve
    ///////////

    NN_LOG( "Testing: [UDP] - Normal Send/Recieve\n" );

    rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "[UDP]: SendMMsg() Failed to write data to the Socket!\n" );
    NN_LOG( "Send - Sent %d messages \n", rc );

    // Wait for Echo back
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    // Got right response?
    NN_LOG( "RecvMMsg..\n" );
    rc = nn::socket::RecvMMsg( udpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc < 0,
        "Failed to read data from the UDP Socket!  Packets read: %d, expected: %d\n", rc, MaxMessages );
    NN_LOG( "Compare..\n" );
    ERROR_IF_AND_COUNT( strncmp(bufs[0], static_cast<const char *>(pInIovecs[0].iov_base), pInIovecs[0].iov_len) != 0,
        "Failed to read data, Expected: [%.*s], Got [%.*s]\n",
        pInIovecs[0].iov_len, static_cast<const char *>(pInIovecs[0].iov_base), strlen(bufs[0]), bufs[0] );


    ////////////
    // [TCP] - Test Simple Network Echo Recieve - causes memory access crash
    ///////////

    NN_LOG( "Testing: [TCP] - Normal Send/Recieve\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "[TCP]: SendMMsg() Failed to write data to the Socket!\n" );
    NN_LOG( "Send - Sent %d messages \n", rc );

    // Wait for Echo back
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    // Got right response?
    NN_LOG( "RecvMMsg..\n" );
    rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc < 0,
        "Failed to read data from the TCP Socket!  Packets read: %d, expected: %d\n", rc, MaxMessages );
    NN_LOG( "Compare..\n" );
    ERROR_IF_AND_COUNT( strncmp( bufs[0], static_cast<const char *>(pInIovecs[0].iov_base), pInIovecs[0].iov_len ) != 0,
        "Failed to read data, Expected: [%.*s], Got [%.*s]\n",
        pInIovecs[0].iov_len, static_cast<const char *>(pInIovecs[0].iov_base), strlen(bufs[0]), bufs[0] );

    ////////////
    // [TCP] - nn::socket::Errno::ENotConn
    ///////////

    NN_LOG( "Testing: [TCP] - nn::socket::Errno::ENotConn \n" );

    nn::socket::Close( tcpSocket );
    tcpSocket = -1;

    // TCP Client - Create
    tcpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSocket < 0, "Socket() Failed to create TCP Client. Errno:%d", nn::socket::GetLastError() );

    rc = nn::socket::RecvMMsg( tcpSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > -1, "RecvMMsg() Succeeded with %d messages. Expected to Fail", rc );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::ENotConn, "RecvMMsg expected to fail with nn::socket::Errno::EConnReset. Errno:%d", myError );

    // Close Accpeted socket
    nn::socket::Close( tcpSocket );
    tcpSocket = -1;


    ////////////
    // [TCP] - Test RecvMMsg From Direct SendMMsg on Lan Iface
    ///////////

    NN_LOG( "Testing: [TCP] - RecvMMsg From Direct SendMMsg on Lan Iface\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // TCP server Sock
    tcpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServerSocket < 0, "Failed to create TCP Server socket. Errno:%d\n", nn::socket::GetLastError() );

    // TCP Server - Local Address
    memset(&serverAddr, 0, sizeof(serverAddr));
    serverAddr.sin_port = nn::socket::InetHtons( 8058 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;

    NN_LOG( "Server IP Address: %08x\n", myIp.S_addr );
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    // TCP Server = Don't wait forever
    myTime.tv_sec = 9;
    myTime.tv_usec = 15000;
    optlen = sizeof(myTime);

    rc = nn::socket::SetSockOpt( tcpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // TCP Server - Local Bind
    rc = nn::socket::Bind( tcpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed!\n" );

    // TCP Server - Listen
    rc = nn::socket::Listen( tcpServerSocket, 5 );
    ERROR_IF( rc < 0, "Listen() failed! Errno:%d\n", nn::socket::GetLastError() );

    // TCP Client - Close
    nn::socket::Close( tcpSocket );
    tcpSocket = -1;

    // TCP Client - Create
    tcpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );

    // TCP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port = nn::socket::InetHtons( 8058 ); // Point to UDP Server
    targetAddr.sin_family = nn::socket::Family::Af_Inet;

    // TCP - Target IP: Lan Interface
    targetAddr.sin_addr.S_addr = myIp.S_addr;
    NN_LOG( "Connected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

    // TCP Client - Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( tcpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc < 0, "Connect() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    // TCP Server - Accept client connection
    acceptedSocket = nn::socket::Accept( tcpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), &targetAddrLen );
    ERROR_IF( acceptedSocket < 0, "Accept() expected to Succeed. Failed with Errno:%d \n", nn::socket::GetLastError() );

    rc = nn::socket::SendMMsg( tcpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "SendMMsg() failed. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "SendMMsg - Sent %d messages\n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

    isSuccess = true;

    rc = nn::socket::RecvMMsg( acceptedSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    if (rc == -1)
    {
        isSuccess = false;
        NN_LOG( "RecvMMsg() returned %d. Errno:%d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "Packets received: %d\n", rc );
        bufs[rc - 1][pRecvMsgs[rc - 1].msg_len] = '\0';
        char addrbuf[16] = { '\0' };
        nn::socket::InetNtop(
            nn::socket::Family::Af_Inet,
            &(static_cast<nn::socket::SockAddrIn *>(pRecvMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
            addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]) );
        NN_LOG( "\tLast message from %s: %s\n", addrbuf, bufs[rc - 1] );
    }

    ERROR_IF_AND_COUNT( !isSuccess, "RecvMMsg() did not recieve any messages\n" );

    // TCP Client for broadcast testing - Close
    nn::socket::Close( tcpSocket );
    tcpSocket = -1;

    // TCP Server - Close
    nn::socket::Close( tcpServerSocket );
    tcpServerSocket = -1;

    // TCP Accepted Sock - Close
    nn::socket::Close( acceptedSocket );
    acceptedSocket = -1;


    ////////////
    // [UDP] - Test RecvMMsg From Direct SendMMsg on Lan Iface
    ///////////

    NN_LOG( "Testing: [UDP] - RecvMMsg From Direct SendMMsg on Lan Iface\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP server Sock
    udpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Server - Local Address
    memset( &serverAddr, 0, sizeof(serverAddr) );
    serverAddr.sin_port = nn::socket::InetHtons( 9056 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;

    NN_LOG("Server IP Address: %08x\n", myIp.S_addr);
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP Server = Don't wait forever
    myTime.tv_sec = 9;
    myTime.tv_usec = 15000;
    optlen = sizeof(myTime);

    rc = nn::socket::SetSockOpt(udpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed!\n" );

    // UDP Client - Close
    nn::socket::Close( udpSocket );

    // UDP Client - Create
    udpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port = nn::socket::InetHtons( 9057 );
    localAddr.sin_family = nn::socket::Family::Af_Inet;

    NN_LOG( "Client IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port = nn::socket::InetHtons( 9056 ); // Point to UDP Server
    targetAddr.sin_family = nn::socket::Family::Af_Inet;

    // UDP - Target IP: Lan Interface
    targetAddr.sin_addr.S_addr = myIp.S_addr;
    NN_LOG( "Connected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

    // UDP Client - Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( udpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc < 0, "Connect() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::SendMMsg(udpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None);
    ERROR_IF( rc < 0, "SendMMsg() failed. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "SendMMsg - Sent %d messages\n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

    isSuccess = true;

    rc = nn::socket::RecvMMsg( udpServerSocket, pRecvMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    if ( rc == -1 )
    {
        isSuccess = false;
        NN_LOG( "RecvMMsg() returned %d. Errno:%d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "Packets received: %d\n", rc );
        bufs[rc - 1][pRecvMsgs[rc - 1].msg_len] = '\0';
        char addrbuf[16] = { '\0' };
        nn::socket::InetNtop(
            nn::socket::Family::Af_Inet,
            &(static_cast<nn::socket::SockAddrIn *>(pRecvMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
            addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]) );
        NN_LOG( "\tLast message from %s: %s\n", addrbuf, bufs[rc - 1] );
    }

    ERROR_IF_AND_COUNT( !isSuccess, "RecvMMsg() did not recieve any messages\n" );

    // UDP Client for broadcast testing - Close
    nn::socket::Close( udpSocket );
    udpSocket = -1;

    // UDP Server - Close
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;


out:

    if ( tcpSocket > -1 )
    {
        nn::socket::Close( tcpSocket );
        tcpSocket = -1;
    }

    if ( udpSocket > -1 )
    {
        nn::socket::Close( udpSocket );
        udpSocket = -1;
    }

    if ( udpServerSocket > -1 )
    {
        nn::socket::Close( udpServerSocket );
        udpServerSocket = -1;
    }

    if ( pNames != NULL )
    {
        delete[] pNames;
        pNames = NULL;
    }

    if ( pBigIov != NULL )
    {
        delete[] pBigIov;
        pBigIov = NULL;
    }

    if ( pBigOutIovecs != NULL )
    {
        delete[] pBigOutIovecs;
        pBigOutIovecs = NULL;
    }

    if ( pInIovecs != NULL )
    {
        delete[] pInIovecs;
        pInIovecs = NULL;
    }

    if ( pOutIovecs != NULL )
    {
        delete[] pOutIovecs;
        pOutIovecs = NULL;
    }

    if ( pBigReceiveMsgs != NULL )
    {
        delete[] pBigReceiveMsgs;
        pBigReceiveMsgs = NULL;
    }

    if ( pBigMsgs != NULL )
    {
        delete[] pBigMsgs;
        pBigMsgs = NULL;
    }

    if ( pRecvMsgs != NULL )
    {
        delete[] pRecvMsgs;
        pRecvMsgs = NULL;
    }

    if ( pSendMsgs != NULL )
    {
        delete[] pSendMsgs;
        pSendMsgs = NULL;
    }

} // NOLINT(impl/function_size)


TEST(ApiUnit,RecvMMsg_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,RecvMMsg_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

} // Namespace: API
} // Namespace: NATF
