﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;

static bool                 g_workBool = false;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}


static void
PollPrintFlags( nn::socket::PollEvent event )
{
    NN_LOG( "POLL flags: [" );
    if ( (event & nn::socket::PollEvent::PollIn) != nn::socket::PollEvent::PollNone )     NN_LOG( "nn::socket::PollEvent::PollIn | ");
    if ( (event & nn::socket::PollEvent::PollRdNorm) != nn::socket::PollEvent::PollNone ) NN_LOG( "nn::socket::PollEvent::PollRdNorm | ");
    if ( (event & nn::socket::PollEvent::PollRdBand) != nn::socket::PollEvent::PollNone ) NN_LOG( "nn::socket::PollEvent::PollRdBand | ");
    if ( (event & nn::socket::PollEvent::PollPri) != nn::socket::PollEvent::PollNone )    NN_LOG( "nn::socket::PollEvent::PollPri | ");
    if ( (event & nn::socket::PollEvent::PollOut) != nn::socket::PollEvent::PollNone )    NN_LOG( "nn::socket::PollEvent::PollOut | ");
    if ( (event & nn::socket::PollEvent::PollWrNorm) != nn::socket::PollEvent::PollNone)  NN_LOG( "nn::socket::PollEvent::PollWrNorm | ");
    if ( (event & nn::socket::PollEvent::PollWrBand) != nn::socket::PollEvent::PollNone)  NN_LOG( "nn::socket::PollEvent::PollWrBand | ");
    if ( (event & nn::socket::PollEvent::PollErr) != nn::socket::PollEvent::PollNone)     NN_LOG( "nn::socket::PollEvent::PollErr | ");
    if ( (event & nn::socket::PollEvent::PollHup) != nn::socket::PollEvent::PollNone)     NN_LOG( "nn::socket::PollEvent::PollHup | ");
    if ( (event & nn::socket::PollEvent::PollNVal) != nn::socket::PollEvent::PollNone)    NN_LOG( "nn::socket::PollEvent::PollNVal | ");
    NN_LOG( "]\n" );
}


////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_Poll_Initialize)
{
    InitializeTesting();
}

static void
PollUnblockReadThread( void * inArg )
{
    int * sockfds = (int *) inArg;
    int rc = 0;

    NN_LOG( "(Thread) - Start, Sleeping...\n" );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_LOG( "(Thread) - Wakup! Writing 1 bytes\n" );

    rc = nn::socket::Write( *sockfds, "1", 1 );
    if ( rc != 1 )
    {
        NN_LOG( "##### FAILED - Return Code from Write is: %d\n", rc );
    }

    g_workBool = true;

    NN_LOG( "(Thread) - Exit\n" );
}


TEST(ApiUnit,Win_Poll_Block_For_Infinte_Time_Test)
{
    nn::socket::PollFd  myPollFds[1];
    int                 clifds = -1, svrfds = -1, rc = 0 , thisThread = -1;
    bool                isSuccess = true;

    ////////////
    // Create UDP Client / Server connection
    ///////////

    clifds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000,  "127.0.0.1", 9001 );
    ERROR_IF( clifds < 0, "MakeUDPConnection failed for client, errno: %d\n", nn::socket::GetLastError() );

    svrfds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9001,  "127.0.0.1", 9000 );
    ERROR_IF( svrfds < 0, "MakeUDPConnection failed for server, errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // Create a thread to run the "Unblock" routine
    ////////////

    g_workBool = false;
    thisThread = g_pThreadedTest->CreateThread( &PollUnblockReadThread, (void *) &svrfds, kThreadPriority );
    if ( thisThread < 0 )
    {
        NN_LOG( "Failed to create and start Unblock (thread)!\n" );
        goto out;
    }

    ////////////
    // Test (Read) Unblocking Select
    ////////////

    myPollFds[0].fd      = clifds;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );

    NN_LOG( "Calling Poll with INFTIM (Infinite Timeout) - BLOCKING..\n" );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, -1  ) );  // -1 == INFTIM (Infinite Time)
    NN_LOG( "Poll Unblocked - PASS\n" );

    if ( g_workBool == false )
    {
        NN_LOG( "===> Poll() unblocked - but Poll wasn't unblocked by thread!!  This was unexpected\n" );
        nTestsFailing++;
        goto out;
    }

    PollPrintFlags( myPollFds[0].revents );

    if ( (myPollFds[0].revents & nn::socket::PollEvent::PollIn) != nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "================\n");
        NN_LOG( "== Pass - Poll was unblocked by Write/nn::socket::PollEvent::PollIn\n" );
        NN_LOG( "================\n");
        nTestsPassing++;
    }

    // Cleanup test - FALL THRU

out:

    if ( thisThread > -1 )
    {
        // Wait DNS Server threads to die
        g_pThreadedTest->WaitThread( thisThread );

        // Destroy all Threads
        g_pThreadedTest->DestroyThread( thisThread );

        // Clear Thread
        thisThread = -1;
    }

    if ( clifds > -1 )
    {
        nn::socket::Close( clifds );
        clifds = -1;
    }

    if ( svrfds > -1 )
    {
        nn::socket::Close( svrfds );
        svrfds = -1;
    }

    return;
}


TEST(ApiUnit,Win_Poll_Various_Tests)
{
    nn::socket::PollFd  myPollFds[1];
    int            tcpSock = -1, udpSock = -1, udpServSock = -1, listenSock = -1, acceptSock =-1, rc = 0;
    //int            optval = -1;
    //nn::socket::SockLenT      optlen = static_cast<nn::socket::SockLenT>(-1);
    bool           isSuccess = true, rval;
    uint64_t       unused1 = 0, unused2 = 0;
    ssize_t        rc1 = 0;
    unsigned char  buf1[128];

    memset( buf1, 0, sizeof( buf1 ) );

    ////////////
    // Make TCP Client / Server
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9020, "127.0.0.1" );
    ERROR_IF_AND_COUNT( rval == false, "Failed calling MakeiListenSocket!  Errno:%d\n", nn::socket::GetLastError() );

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9020 );
    ERROR_IF_AND_COUNT( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptSock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpServSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 9001 );
    ERROR_IF_AND_COUNT( udpServSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9001, "127.0.0.1", 9000 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Call POLL with a NULL decriptor ARRAY
    ///////////

    NN_LOG( "Call POLL with a NULL decriptor ARRAY\n" );

    PRINT_AND_CALL(rc = nn::socket::Poll( static_cast<nn::socket::PollFd*>(nullptr), 0, 0 ) );
    ERROR_IF_AND_COUNT( rc != -1, "Poll() returned: %d for Null Array of Fds - Errno: %d\n", rc, nn::socket::GetLastError() );

    ////////////
    // TCP: POLL FDs Array has Negative Length
    ///////////

    NN_LOG( "[TCP] - Poll FDS Array Length is (-1)\n" );

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, static_cast<nn::socket::SockLenT>(-1), 0 ) );
    ERROR_IF_AND_COUNT( rc > -1 || nn::socket::GetLastError() != nn::socket::Errno::EInval,
                       "Poll() returned OK for -1 Array of FDs legnth!" );

    ////////////
    // UDP: POLL FDs Array has Negative Length
    ///////////

    NN_LOG( "[UDP] - Poll FDS Array Length is (-1)\n" );

    myPollFds[0].fd      = udpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, static_cast<nn::socket::SockLenT>(-1), 0 ) );
    ERROR_IF_AND_COUNT( rc > -1 || nn::socket::GetLastError() != nn::socket::Errno::EInval, "Poll() returned OK for -1 Array of FDs legnth!" );


    ////////////
    // Place Normal data on TCP and UDP sockets
    ///////////

    rc = NATF::API::COMMON::WriteData( acceptSock, (unsigned char *) "Hello!", 6, unused1, unused2 );
    ERROR_IF_AND_COUNT( rc != 6, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    rc = NATF::API::COMMON::WriteData( udpServSock, (unsigned char *) "Hello!", 6, unused1, unused2 );
    ERROR_IF_AND_COUNT( rc != 6, "Failed calling WriteData! rc: %d,  Errno:%d\n", rc, nn::socket::GetLastError() );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));


    ///////////
    // TCP: nn::socket::PollEvent::PollIn + nn::socket::PollEvent::PollRdNorm
    /////////

    NN_LOG( "[TCP] - Test nn::socket::PollEvent::PollIn / nn::socket::PollEvent::PollRdNorm\n" );

    // TCP nn::socket::PollEvent::PollIn
    memset( myPollFds, 0, sizeof( myPollFds) );
    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, 0 ) );
    PollPrintFlags( myPollFds[0].revents );

    ERROR_IF_AND_COUNT( rc != 1, "Poll Return code %d did not equal 1 descriptor!\n", rc );
    ERROR_IF_AND_COUNT(  ( myPollFds[0].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone && ( myPollFds[0].revents & nn::socket::PollEvent::PollRdNorm ) != nn::socket::PollEvent::PollNone,
                        "Poll failed - I expected nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPollFds[0].revents );

    NN_LOG( "PASSED!\n" );


WRAP_FAILING_TEST( "SIGLO-58435", "Socket API: [Win] POLL did not detect UDP data ready-to-Read" )
{

    ///////////
    // UDP: nn::socket::PollEvent::PollIn + nn::socket::PollEvent::PollRdNorm
    /////////

    NN_LOG( "[UDP] - Test nn::socket::PollEvent::PollIn / nn::socket::PollEvent::PollRdNorm\n" );

    // UDP nn::socket::PollEvent::PollIn
    memset( myPollFds, 0, sizeof( myPollFds) );
    myPollFds[0].fd      = udpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollNVal | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollErr;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, 0 ) );
    PollPrintFlags( myPollFds[0].revents );

    ERROR_IF_AND_COUNT( rc != 1, "Poll Return code %d did not equal 1 descriptor (ready-to-Read)!  Errno: %d\n", rc, nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT(  (myPollFds[0].revents & nn::socket::PollEvent::PollIn) != nn::socket::PollEvent::PollNone || (myPollFds[0].revents & nn::socket::PollEvent::PollRdNorm) != nn::socket::PollEvent::PollNone,
                        "Poll failed - I expected either nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPollFds[0].revents );

    NN_LOG( "PASSED!\n" );

}    // Wrap Failing test


    ///////////
    // Test UDP and TCP at same time
    /////////

    {
        nn::socket::PollFd myPoll[2];

        memset( myPoll, 0, sizeof( myPoll) );

        // TCP nn::socket::PollEvent::PollIn
        myPoll[0].fd      = tcpSock;
        myPoll[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm;
        myPoll[0].revents = nn::socket::PollEvent::PollNone;

        // UDP nn::socket::PollEvent::PollIn
        myPoll[1].fd      = udpSock;
        myPoll[1].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm;
        myPoll[1].revents = nn::socket::PollEvent::PollNone;

        NN_LOG( "[TCP]: " );
        PollPrintFlags( myPoll[0].events );
        NN_LOG( "[UDP]: " );
        PollPrintFlags( myPoll[1].events );

        PRINT_AND_CALL(rc = nn::socket::Poll( myPoll, 2, 0 ) );

        NN_LOG( "[TCP]: " );
        PollPrintFlags( myPoll[0].revents );
        NN_LOG( "[UDP]: " );
        PollPrintFlags( myPoll[1].revents );

        ERROR_IF_AND_COUNT( rc != 2, "Poll Return code %d did not equal 2 descriptors!\n", rc );
        ERROR_IF_AND_COUNT(  ( myPoll[0].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone && ( myPoll[0].revents & nn::socket::PollEvent::PollRdNorm ) != nn::socket::PollEvent::PollNone,
                        "[TCP] Poll failed - I expected nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPoll[0].revents );

        ERROR_IF_AND_COUNT(  ( myPoll[1].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone && ( myPoll[1].revents & nn::socket::PollEvent::PollRdNorm ) != nn::socket::PollEvent::PollNone,
                        "[UDP] Poll failed - I expected nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPoll[1].revents );

    }


    ////////////
    // Test: Place Out-Of-Band (OOB) data onto TCP socket
    ///////////

    NN_LOG( "[TCP] - Test: Place Out-Of-Band (OOB) data onto TCP socket\n" );

    rc1 = nn::socket::Send( acceptSock, (unsigned char *) "X", 1, nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc1 != 1, "Failed calling SendTo with nn::socket::MsgFlag::Msg_Oob!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    rc1 = nn::socket::Send( acceptSock, (unsigned char *) "Good Bye!", 9, nn::socket::MsgFlag::Msg_None);
    ERROR_IF_AND_COUNT( rc1 != 9, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));


    ///////////
    // TCP: nn::socket::PollEvent::PollIn + nn::socket::PollEvent::PollRdNorm + nn::socket::PollEvent::PollRdBand + nn::socket::PollEvent::PollPri
    /////////

    NN_LOG( "[UDP] - Test nn::socket::PollEvent::PollIn / nn::socket::PollEvent::PollRdNorm\n" );

    // UDP nn::socket::PollEvent::PollIn
    memset( myPollFds, 0, sizeof( myPollFds ) );
    myPollFds[0].fd      = tcpSock;
    // myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollRdBand | nn::socket::PollEvent::PollPri | nn::socket::PollEvent::PollNVal | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollErr;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollRdBand;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, 0 ) );
    PollPrintFlags( myPollFds[0].revents );

    ERROR_IF_AND_COUNT( rc != 1, "Poll Return code %d did not equal 1 descriptor (ready-to-Read)!\n", rc );
    ERROR_IF_AND_COUNT(  (myPollFds[0].revents & nn::socket::PollEvent::PollRdBand ) == nn::socket::PollEvent::PollNone,
                        "Poll failed - I expected nn::socket::PollEvent::PollRdBand but got %d\n", myPollFds[0].revents );

    ///////////
    // TCP: Read Hello!
    /////////

    rc1 = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    NN_LOG( "(Part 1) - Normal Data: %.*s\n", rc1, buf1 );
    if ( rc1 != 16 && rc1 != 15 )
    {
        NN_LOG( "Failed to receive 'Hello!XGood Bye!' data!  rc: %d\n", rc1 );
        ERROR_IF_AND_COUNT( rc1 != 16 && rc1 != 15, "Test Failed!\n" );
    }


#if 0

    ///////////
    // TCP: Read OOB data
    /////////

    rc1 = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc1 != 1, "Failed to receive OOB data!  rc: %d\n", rc1 );
    NN_LOG( "(Out of Band) - OOB Data: %.*s\n", rc1, buf1 );


    ///////////
    // TCP: Read Good Bye!
    /////////

    rc1 = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), 0 );
    ERROR_IF_AND_COUNT( rc1 != 9, "Failed to receive 'Good Bye!' data!  rc: %d\n", rc1 );
    NN_LOG( "(Part 2) - Normal Data: %.*s\n", rc1, buf1 );

#endif



WRAP_FAILING_TEST( "SIGLO-57670", "AT&T System 5 release 4 (svr4) port of POLL to FreeBSD/NX missing nn::socket::PollEvent::PollWrBand\n" )
{

    ////////////
    // Test: Ready-To-Write (OOB) data onto TCP socket
    ///////////

    NN_LOG( "[TCP] - Test: Ready-To-Write Out-Of-Band (OOB) data onto TCP socket\n" );

    // Set up Poll
    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollWrBand | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    // Invoke Poll
    PollPrintFlags(  myPollFds[0].events );
    rc = nn::socket::Poll( myPollFds, 1, 0 );
    PollPrintFlags(  myPollFds[0].revents );
    ERROR_IF_AND_COUNT( rc < 1, "Poll Failed!  rc: %d", rc);

    if (  ( myPollFds[0].revents & nn::socket::PollEvent::PollWrBand ) == nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "Poll did not return nn::socket::PollEvent::PollWrBand - but I requeted it!  Errno: %d\n", nn::socket::GetLastError() );
        goto out;
    }

} // Wrap Failing Test

    ////////////
    // Test: (OOB) data is INLINE onto TCP socket
    ////////////

#if 0   // TODO: this test is faulty; listenSock == -1 which causes Accept() to fail
    NN_LOG( "[TCP] - Test: Place Out-Of-Band (OOB) data INLINE (aka Normal Data) for TCP socket\n" );

    // Close TCP Socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    nn::socket::Close( acceptSock );
    acceptSock = -1;

    ////////////
    // Open New TCP Socket with nn::socket::Option::So_OobInline
    ///////////

    optval = 1;
    optlen = sizeof( optval );
    rval = NATF::API::COMMON::SetSocketOptAndConnect( tcpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, &optval, optlen,
                                                      0, g_echoServer->kInterfaceIPv4Addr,
                                                      9020, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

    ERROR_IF_AND_COUNT( rval == false, "Failed calling SetSocketOptAndConnect()!  Errno:%d\n", nn::socket::GetLastError() );

    acceptSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptSock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////
    // Set OBBINLINE Data
    ///////////

    rc1 = nn::socket::Send( acceptSock, (unsigned char *) "Hello!", 6, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 != 6, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc1, nn::socket::GetLastError() );

    rc1 = nn::socket::Send( acceptSock, (unsigned char *) "X", 1, nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc1 != 1, "Failed calling SendTo with nn::socket::MsgFlag::Msg_Oob!  rc: %d, Errno:%d\n", rc1, nn::socket::GetLastError() );

    rc1 = nn::socket::Send( acceptSock, (unsigned char *) "Good Bye!", 9, nn::socket::MsgFlag::Msg_None);
    ERROR_IF_AND_COUNT( rc1 != 9, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc1, nn::socket::GetLastError() );

    ///////////
    // TCP: Read Hello!XGood Bye!
    /////////

    rc1 = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    NN_LOG( "Normal Data: %.*s\n", rc1, buf1 );

    if ( rc1 != 16 && rc1 != 15 )
    {
        NN_LOG( "Failed to receive 'Hello!XGood Bye!' data!  rc: %d\n", rc1 );
        ERROR_IF_AND_COUNT( rc1 != 16 && rc1 != 15, "Test Failed!\n" );
    }
#endif

#if 0

    rc1 = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), 0 );
    NN_LOG( "Normal Data: %.*s\n", rc1, buf1 );
    ERROR_IF_AND_COUNT( rc != 10, "Failed to receive 'XGood Bye!' data!  rc: %d\n", rc1 );

#endif

#if 0

    ////////////
    // Test: Socket HANGHUP during I/O Operation
    ///////////

    // Shutdown client
    nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_RdWr );

    NN_LOG( "[TCP] - Test: Socket [HANGUP] return flag\n" );

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollWrBand  | nn::socket::PollEvent::PollHup;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    // Invoke Poll
    PollPrintFlags(  myPollFds[0].events );
    rc = nn::socket::Poll( &myPollFds, 1, 0 );
    PollPrintFlags(  myPollFds[0].revents );
    ERROR_IF_AND_COUNT( rc < 1, "Poll Failed!  rc: %d", rc);

    if (  ( myPollFds[0].revents & nn::socket::PollEvent::PollHup ) == 0 )
    {
        NN_LOG( "FAILED!  ==> Poll did not return nn::socket::PollEvent::PollHup - but I requeted it!  Errno: %d\n", nn::socket::GetLastError() );
        goto out;
    }

    // TCP Socket is closed
    tcpSock = -1;


    ////////////
    // Test: Operation on closed socket
    ///////////

    NN_LOG( "[TCP] - Test: Closed Socket should not be able to READ or WRITE\n" );

    // TCP Socket is closed
    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollRdBand | nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollWrBand | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    // Invoke Poll
    PollPrintFlags(  myPollFds[0].events );
    rc = nn::socket::Poll( &myPollFds, 1, 0 );
    PollPrintFlags(  myPollFds[0].revents );
    ERROR_IF_AND_COUNT( rc != 0, "Poll Success was expected, but Poll actually failed!  rc: %d", rc);

    if (  ( myPollFds[0].revents & nn::socket::PollEvent::PollIn ) != 0  || ( myPollFds[0].revents & nn::socket::PollEvent::PollOut ) != 0  )

    {
        NN_LOG( "FAILED!  ==> Poll Indicated that a closed socket can (READ or WRITE), result: 0x%x\n",
                myPollFds[0].revents );
        goto out;
    }

#endif

    // Fall Thru to cleanup

out:

    if ( udpServSock > 0 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

    if ( acceptSock > 0 )
    {
        nn::socket::Close( acceptSock );
        acceptSock = -1;
    }

    if ( listenSock > 0 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }


    if ( tcpSock > 0 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > 0 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

}  // NOLINT(impl/function_size)


TEST(ApiUnit,Win_Poll_Teardown)
{
    TeardownTesting();
}

}}  // Namespace: NATF::API
