﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////////////////////////////////////////////////////////////////
// DEFAULT: DISABLED

//#define ENABLE_FAILING_TESTS

#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nn/os/os_Thread.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <cstdio>

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"
#include <nn/socket/resolver/resolver_Client.h>

#define BUILD_NAME_ADD_STRING(nameBuffer, nameLen, str, strLen, totalCharsWritten) \
do \
{ \
    ERROR_IF((totalCharsWritten + static_cast<int>(strLen)) > static_cast<int>(nameLen), "BuildName: insufficient space in nameBuffer"); \
    memcpy(nameBuffer + totalCharsWritten, str, strLen); \
    totalCharsWritten += static_cast<int>(strLen); \
} while( NN_STATIC_CONDITION(false) )

namespace {

    const int NUM_THREADS = 16;
    const int MY_STACK_SIZE = 16 * 1024;
    const char g_DefaultHostName[] = "www.nintendo.com";

    NN_ALIGNAS(4096) unsigned char g_pStacks[NUM_THREADS][NUM_THREADS][MY_STACK_SIZE];
    char g_PrintBuffer[512];
    bool g_TestModeContinueOnFailure;
    bool g_TestModeVerbose;

    nn::socket::ResolverOption DisableDnsCache()
    {
        nn::socket::ResolverOption   myOption;

        NN_LOG( "DisableDnsCache() - Begin\n" );
        myOption.key = static_cast<nn::socket::ResolverOptionKey>( nn::socket::ResolverOptionKey::RequestEnableDnsCacheBoolean );
        myOption.type = nn::socket::ResolverOptionType::Boolean;
        myOption.size = sizeof(bool);
        myOption.data.booleanValue = false;
        NN_LOG( "DisableDnsCache() - End\n" );

        return myOption;
    }

    bool BuildName(char *nameBuffer, size_t nameLen, const char *label, size_t labelLen, int n, bool isTldUsed )
    {
        bool isSuccess = true;
        int totalCharsWritten = 0;

        ERROR_IF(nullptr == nameBuffer, "BuildName precondition failed: nameBuffer null");
        ERROR_IF(nullptr == label, "BuildName precondition failed: label null");
        ERROR_IF(1 > labelLen, "BuildName precondition failed: labelLen less than 1");

        // Build name string as N copies of label string separated by period characters
        // The name may include top level domain (TLD) string (.com) appended to the name string
        // Otherwise, names include one fewer character included in the last label string copy
        for( int i = 0; i < n; ++i )
        {
            // For gaps between label strings, add period character
            if( i != 0 )
            {
                BUILD_NAME_ADD_STRING(nameBuffer, nameLen, ".", strlen("."), totalCharsWritten);
            }

            // For the last label string when TLD is not used, add label string with one fewer character
            if( !isTldUsed && (i == (n - 1)) )
            {
                BUILD_NAME_ADD_STRING(nameBuffer, nameLen, label, labelLen - 1, totalCharsWritten);
            }
            // Otherwise, add label string with normal characters
            else
            {
                BUILD_NAME_ADD_STRING(nameBuffer, nameLen, label, labelLen, totalCharsWritten);
            }
        }

        // When TLD is used, add TLD string after name building is done
        if( isTldUsed )
        {
            BUILD_NAME_ADD_STRING(nameBuffer, nameLen, ".com", strlen(".com"), totalCharsWritten);
        }

out:
        return isSuccess;
    }

    bool TestGetHostByName_Name(const char *name, bool isSupported)
    {
        bool isSuccess = true;
        nn::socket::HostEnt* rval = nullptr;

        // Disable DNS Cache
        nn::socket::ResolverOption myOption = DisableDnsCache();

        if( g_TestModeVerbose )
        {
            NN_LOG("** TestModeVerbose is ENABLED **\n");
            NN_LOG("TestGetHostByName_Name: name: strlen=%d, str=%s\n", strlen(name), name);
            PRINT_AND_CALL(rval = nn::socket::GetHostEntByName(name, &myOption, 1));
        }
        else
        {
            rval = nn::socket::GetHostEntByName( name, &myOption, 1);
        }

        if( g_TestModeContinueOnFailure )
        {
            NN_LOG("** TestModeContinueOnFailure is ENABLED **\n");
            NN_LOG("isSupported=%d, isConfirmed=%d (confirmed: nullptr != rval)\n", isSupported, nullptr != rval);
            if( g_TestModeVerbose && (nullptr != rval) )
            {
                NN_LOG("** TestModeVerbose is ENABLED **\n");
                NATF::API::PrintHostent(rval);
            }
        }
        else
        {
            if( isSupported )
            {
                ERROR_IF(nullptr == rval, "GetHostByName failed when expected to succeed.");
                if( g_TestModeVerbose )
                {
                    NN_LOG("** TestModeVerbose is ENABLED **\n");
                    ERROR_IF(!NATF::API::PrintHostent(rval), "NATF::API::PrintHostent failed when expected to succeed.");
                }
            }
            else
            {
                ERROR_IF(nullptr != rval, "GetHostByName succeeded when expected to fail.");
            }
        }

out:
        return isSuccess;
    }

    bool TestGetHostByName_Main()
    {
        bool isSuccess = true;
        bool rval = true;

        const char label63[] = "012345678901234567890123456789012345678901234567890123456789012";
        const char label64[] = "0123456789012345678901234567890123456789012345678901234567890123";
        const char *namesByCountry[] =
        {
            "amazon.fr",
            "amazon.jp",
            "amazon.gr"
        };
        const char *namesByTld[] =
        {
            "google.com",
            "microsoft.net",
            "amazon.jobs",
            "fora.tv",
            "windows.microsoft"
        };

        const char ASCII_SPACE = 0x20;

        char nameBuffer[512];

       ////////////////////////////////////////////////////////////////
        // Function:
        // nn::socket::HostEnt *gethostbyname(const char *name);

        ////////////////////////////////////////////////////////////////
        // Argument:
        // const char *name

        PRINT_AND_CALL(rval = TestGetHostByName_Name("www.google.com", true));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        PRINT_AND_CALL(rval = TestGetHostByName_Name("google.com", true));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        // Equivalent: "名がドメイン.com", "xn--v8jxj3d1dzdz08w.com"
        PRINT_AND_CALL(rval = TestGetHostByName_Name("名がドメイン.com", false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        // Equivalent: "名がドメイン.com", "xn--v8jxj3d1dzdz08w.com"
        PRINT_AND_CALL(rval = TestGetHostByName_Name("xn--v8jxj3d1dzdz08w.com", true));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");
        EXPECT_HERRNO_NX(nn::socket::HErrno::Netdb_Success);
        EXPECT_ERRNO_NX(nn::socket::Errno::ESuccess);
        EXPECT_HERRNO_WIN(nn::socket::HErrno::Netdb_Success);
        EXPECT_ERRNO_WIN(nn::socket::Errno::ESuccess);

        PRINT_AND_CALL(rval = TestGetHostByName_Name("google", false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        PRINT_AND_CALL(rval = TestGetHostByName_Name(".com", false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        for( int i = 0; i < SIZE_OF_ARRAY(namesByCountry); ++i )
        {
            PRINT_AND_CALL(rval = TestGetHostByName_Name(namesByCountry[i], true));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x3
        }

        for( int i = 0; i < SIZE_OF_ARRAY(namesByTld); ++i )
        {
            PRINT_AND_CALL(rval = TestGetHostByName_Name(namesByTld[i], true));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x5
        }

        PRINT_AND_CALL(rval = TestGetHostByName_Name("google.idonotexist111", false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        for( int i = 1; i <= 4; ++i )
        {
            // names: X, X.X, X.X.X, X.X.X.X
            ERROR_IF(!BuildName(nameBuffer, 512, label63, strlen(label63), i, false), "BuildName() failed");
            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x4

            // names: X, X.X, X.X.X, X.X.X.X
            ERROR_IF(!BuildName(nameBuffer, 512, label64, strlen(label64), i, false), "BuildName() failed");
            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x4
        }

        for( int i = 0; i <= 3; ++i )
        {
            // names: com, X.com, X.X.com, X.X.X.com
            ERROR_IF(!BuildName(nameBuffer, 512, label63, strlen(label63), i, true), "BuildName() failed");
            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x4

            // names: com, X.com, X.X.com, X.X.X.com
            ERROR_IF(!BuildName(nameBuffer, 512, label64, strlen(label64), i, true), "BuildName() failed");
            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x4
        }

WRAP_FAILING_TEST_NX("SIGLO-58192", "[NX] GetHostByName h_errno: Expected NO_DATA, Returned NETDB_INTERNAL with errno ECANCELED (names: X.X.X.X.X, X.X.X.X.X.X, X.X.X.X.X.X.X, X.X.X.X.X.X.X.X)")
{
        for( int i = 5; i <= 8; ++i )
        {
            // names: X.X.X.X.X, X.X.X.X.X.X, X.X.X.X.X.X.X, X.X.X.X.X.X.X.X
            ERROR_IF(!BuildName(nameBuffer, 512, label63, strlen(label63), i, false), "BuildName() failed");
            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x4
            EXPECT_HERRNO_NX(nn::socket::HErrno::No_Data); // FAILED: Expected NO_DATA, Returned NETDB_INTERNAL with errno nn::socket::Errno::ECanceled
            EXPECT_HERRNO_WIN(nn::socket::HErrno::No_Data);
        }
}

WRAP_FAILING_TEST_NX("SIGLO-58192", "[NX] GetHostByName h_errno: Expected NO_DATA, Returned NETDB_INTERNAL with errno ECANCELED (names: X.X.X.X.X, X.X.X.X.X.X, X.X.X.X.X.X.X, X.X.X.X.X.X.X.X)")
{
        for( int i = 4; i <= 7; ++i )
        {
            // names: X.X.X.X.com, X.X.X.X.X.com, X.X.X.X.X.X.com, X.X.X.X.X.X.X.com
            ERROR_IF(!BuildName(nameBuffer, 512, label63, strlen(label63), i, true), "BuildName() failed");
            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // x4
            EXPECT_HERRNO_NX(nn::socket::HErrno::No_Data); // FAILED: Expected NO_DATA, Returned NETDB_INTERNAL with errno nn::socket::Errno::ECanceled
            EXPECT_HERRNO_WIN(nn::socket::HErrno::No_Data);
        }
}

        PRINT_AND_CALL(rval = TestGetHostByName_Name("", false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed"); // WIN FAILED
        EXPECT_HERRNO_NX(nn::socket::HErrno::No_Data);

        PRINT_AND_CALL(rval = TestGetHostByName_Name("idonotexist.localhost", false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");
        EXPECT_HERRNO_NX(nn::socket::HErrno::Host_Not_Found); // FAILED: Expected HOST_NOT_FOUND, Returned NETDB_INTERNAL with errno nn::socket::Errno::ECanceled
        EXPECT_HERRNO_WIN(nn::socket::HErrno::Host_Not_Found);

        memset(nameBuffer, ASCII_SPACE, 256);
        nameBuffer[256] = '\0';
        PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        PRINT_AND_CALL(rval = TestGetHostByName_Name("...com", false));
        ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

        for( char c = 0; c <= UCHAR_MAX; ++c )
        {
            int stateIndex = -1;
            char urlCharAsStr[4];

            ERROR_IF(!NATF::API::VerifyUrlCharValid(&stateIndex, c), "VerifyUrlCharValid() failed: return false");
            NN_LOG("VerifyUrlCharValid: char='%c' (\\x%02X), stateIndex=%s (%d)\n", c, c, NATF::API::namesUrlCharValidStatesEnum[stateIndex], stateIndex);

            // names: X.X.X.com
            ERROR_IF(0 > sprintf(urlCharAsStr, "%c", c), "TestGetHostByName_Main: sprintf failed");
            ERROR_IF(!BuildName(nameBuffer, 512, urlCharAsStr, 1, 3, true), "BuildName() failed");

            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

            if( 255 == c )
            {
                break;
            }
        }

        for( char c = 0; c <= UCHAR_MAX; ++c )
        {
            int stateIndex = -1;
            char urlCharAsStr[4];

            ERROR_IF(!NATF::API::VerifyUrlCharValid(&stateIndex, c), "VerifyUrlCharValid() failed: return false");
            NN_LOG("VerifyUrlCharValid: char='%c' (\\x%02X), stateIndex=%s (%d)\n", c, c, NATF::API::namesUrlCharValidStatesEnum[stateIndex], stateIndex);

            // names: %AB.%AB.%AB.com
            ERROR_IF(0 > sprintf(urlCharAsStr, "%%%02X", c), "TestGetHostByName_Main: sprintf failed");
            ERROR_IF(!BuildName(nameBuffer, 512, urlCharAsStr, 1, 3, true), "BuildName() failed");

            PRINT_AND_CALL(rval = TestGetHostByName_Name(nameBuffer, false));
            ERROR_IF_AND_COUNT(rval != true, "GetHostByName() test failed");

            if( 255 == c )
            {
                break;
            }
        }

out:
        return isSuccess;
    } // NOLINT(impl/function_size)

    void TestGetHostByName_CallRampThread(void *pParam)
    {
        bool isSuccess = true;
        bool rval = false;

        char printBuffer[512];
        char defaultHostName[] = "www.nintendo.com";

        ERROR_IF(nullptr == pParam, "TestGetHostByName_CallRampThread: failed precondition check (pParam nullptr)");

        NN_LOG("TestGetHostByName_CallRampThread: thread %03d; hostName len=%d, str=%s\n", *(reinterpret_cast<int *>(pParam)), strlen(defaultHostName), defaultHostName);

        for( int i = 0; i < 200; ++i )
        {
            rval = TestGetHostByName_Name(defaultHostName, true);
            ERROR_IF(0 > sprintf(printBuffer, "TestGetHostByName_CallRampThread: thread %03d; i %03d: GetHostByName() test failed \n", *(reinterpret_cast<int *>(pParam)), i), "TestGetHostByName_CallRampThread: sprintf failed");
            ERROR_IF(rval != true, printBuffer);
        }

out:
        if( nullptr == pParam )
        {
            return;
        }

        if( rval != true )
        {
            *(reinterpret_cast<int*>(pParam)) = -1;
        }

        return;
    }

    bool TestGetHostByName_CallRamp()
    {
        bool isSuccess = true;

        nn::Result result;
        int threadArguments[NUM_THREADS][NUM_THREADS];
        nn::os::ThreadType threads[NUM_THREADS][NUM_THREADS];
        int threadControlState[NUM_THREADS][NUM_THREADS];

        int numThreadsToUse = 3;
        bool errorCaught = false;

WRAP_FAILING_TEST_NX("SIGLO-58192", "[NX] GetHostByName h_errno: Expected NO_DATA, Returned NETDB_INTERNAL with errno ECANCELED (names: X.X.X.X.X, X.X.X.X.X.X, X.X.X.X.X.X.X, X.X.X.X.X.X.X.X)")
{
        numThreadsToUse = NUM_THREADS;
}

        for( int j = 0; j < numThreadsToUse; ++j )
        {
            NN_LOG("TestGetHostByName_CallRamp: Run %d thread(s) ...\n", j + 1);

            for( int i = 0; i <= j; ++i )
            {
                threadControlState[j][i] = 0;
                threadArguments[j][i] = i;
                PRINT_AND_CALL(result = nn::os::CreateThread(&threads[j][i], TestGetHostByName_CallRampThread, &threadArguments[j][i], &g_pStacks[j][i][0], MY_STACK_SIZE, nn::os::DefaultThreadPriority));
                if( result.IsFailure() )
                {
                    NN_LOG("TestGetHostByName_CallRamp: CreateThread failed (thread_id = %03d, thread_count = %03d)\n", i, j);
                    errorCaught = true;
                    goto out_in_loop;
                }
                else
                {
                    ++threadControlState[j][i];
                }
            }

            for( int i = 0; i <= j; ++i )
            {
                PRINT_AND_CALL(nn::os::StartThread(&threads[j][i]));
                ++threadControlState[j][i];
            }

            for( int i = 0; i <= j; ++i )
            {
                PRINT_AND_CALL(nn::os::WaitThread(&threads[j][i]));
                if( -1 == threadArguments[j][i] )
                {
                    NN_LOG("TestGetHostByName_CallRamp: TestGetHostByName_CallRampThread failed (thread_id = %03d, thread_count = %03d)", i, j);
                    errorCaught = true;
                    goto out_in_loop;
                }
                else
                {
                    ERROR_IF_AND_COUNT(NN_STATIC_CONDITION(false), "Increment number of passing tests");
                    ++threadControlState[j][i];
                }
            }

out_in_loop:
            for( int i = 0; i <= j; ++i )
            {
                NN_LOG("threadControlState[%d][%d]=%d\n", j, i, threadControlState[j][i]);
                switch(threadControlState[j][i])
                {
                case 1: // Fall through
                    PRINT_AND_CALL(nn::os::StartThread(&threads[j][i]));
                case 2: // Fall through
                    PRINT_AND_CALL(nn::os::WaitThread(&threads[j][i]));
                case 3: // Fall through
                    PRINT_AND_CALL(nn::os::DestroyThread(&threads[j][i]));
                    break;

                case 0: // Fall through
                default:
                    break;
                }
            }

            ERROR_IF(0 > sprintf(g_PrintBuffer, "TestGetHostByName_CallRamp: error caught (thread_count = %03d)", j), "TestGetHostByName_CallRamp: sprintf failed");
            ERROR_IF_AND_COUNT(errorCaught, g_PrintBuffer);
        }

out:
        return isSuccess;
    }

#ifndef NN_BUILD_CONFIG_OS_WIN32
    bool TestGetHostByName_SelectDnsServer()
    {
        bool isSuccess = true;
        int rvalApi = -1;
        bool rvalTest = false;

        typedef struct
        {
            const char *name;
            const char *addrAscii;
            const char *oldName;
        } RootNameServerInfo;

        const RootNameServerInfo servers[] =
        {
            { "A-Root", "198.41.0.4", "ns.internic.net" },
            { "B-Root", "192.228.79.201", "ns1.isi.edu" },
            { "C-Root", "192.33.4.12", "c.psi.net" },
            { "D-Root", "199.7.91.13", "terp.umd.edu" },
            { "E-Root", "192.203.230.10", "ns.nasa.gov" },
            { "F-Root", "192.5.5.241", "ns.isc.org" },
            { "G-Root", "192.112.36.4", "ns.nic.ddn.mil" },
            { "H-Root", "198.97.190.53", "aos.arl.army.mil" },
            { "I-Root", "192.36.148.17", "nic.nordu.net" },
            { "J-Root", "192.58.128.30", "Verisign" },
            { "Google", "8.8.8.8", "Google" }
        };

        nn::socket::SockAddrIn dhcpAddr;
        nn::socket::SockAddrIn rootNameServerAddr;

        nn::socket::PrivateDnsAddressArrayArgument arg;
        nn::socket::ResolverOption option;

        // Call GetDnsAddressPrivate with dhcpAddr
        option.size = sizeof(arg);
        option.data.pointerValue = reinterpret_cast<char*>(&arg);
        PRINT_AND_CALL(rvalApi = nn::socket::ResolverSetOption(option));
        dhcpAddr = arg.addresses[0];

        // Call GetDnsAddressPrivate with dhcpAddr
        ERROR_IF(0 != rvalApi, "GetDnsAddressPrivate failed");
        ERROR_IF(!NATF::API::PrintSockaddrIn(&dhcpAddr), "NATF::API::PrintSockaddrIn failed");
        g_TestModeVerbose = true;
        g_TestModeContinueOnFailure = true;

        for( int i = 0; i < SIZE_OF_ARRAY(servers); ++i )
        {
            // Setup nn::socket::SockAddrIn
            memset(&rootNameServerAddr, 0, sizeof(rootNameServerAddr));
            rootNameServerAddr.sin_family = nn::socket::Family::Af_Inet;
            rootNameServerAddr.sin_port = nn::socket::InetHtons(53);
            PRINT_AND_CALL(rvalApi = nn::socket::InetPton(nn::socket::Family::Af_Inet, servers[i].addrAscii, &rootNameServerAddr.sin_addr));
            ERROR_IF(0 == rvalApi, "TestGetHostByName_SelectDnsServer: InetAton failed (rvalApi 0).");

            // Call SetDnsAddressesPrivate with rootNameServerAddr;
            arg.count = 1;
            arg.addresses[0] = rootNameServerAddr;
            nn::socket::ResolverOption option = {
                .key = static_cast<nn::socket::ResolverOptionKey>(nn::socket::ResolverOptionPrivateKey::SetDnsServerAddressesPointer),
                .type = nn::socket::ResolverOptionType::Pointer,
                .size = sizeof(arg),
                .data.pointerValue = reinterpret_cast<char*>(&arg)
            };
            PRINT_AND_CALL(rvalApi = nn::socket::ResolverSetOption(option));
            ERROR_IF(!NATF::API::PrintSockaddrIn(&rootNameServerAddr), "NATF::API::PrintSockaddrIn failed");
            ERROR_IF(0 > sprintf(g_PrintBuffer, "TestGetHostByName_SelectDnsServer: SetDnsAddressesPrivate failed (rvalApi %d)", rvalApi), "TestGetHostByName_SelectDnsServer: sprintf failed");
            ERROR_IF(0 != rvalApi, g_PrintBuffer);

            // Call TestGetHostByName_Name
            PRINT_AND_CALL(rvalTest = TestGetHostByName_Name(g_DefaultHostName, true));
            ERROR_IF(0 > sprintf(g_PrintBuffer, "TestGetHostByName_SelectDnsServer: server name %s, rootNameServerAddr %s, oldName %s: GetHostByName() test failed", servers[i].name, servers[i].addrAscii, servers[i].oldName), "TestGetHostByName_SelectDnsServer: sprintf failed");
            ERROR_IF_AND_COUNT(true != rvalTest, g_PrintBuffer);
        }

        // Call SetDnsAddressesPrivate with dhcpAddr
        arg.count = 1;
        arg.addresses[0] = dhcpAddr;

        option.key = static_cast<nn::socket::ResolverOptionKey>(nn::socket::ResolverOptionPrivateKey::SetDnsServerAddressesPointer);
        option.type = nn::socket::ResolverOptionType::Pointer;
        option.size = sizeof(arg);
        option.data.pointerValue = reinterpret_cast<char*>(&arg);

        PRINT_AND_CALL(rvalApi = nn::socket::ResolverSetOption(option));
        ERROR_IF(!NATF::API::PrintSockaddrIn(&dhcpAddr), "NATF::API::PrintSockaddrIn failed");
        ERROR_IF(0 > sprintf(g_PrintBuffer, "TestGetHostByName_SelectDnsServer: SetDnsAddressesPrivate failed (rvalApi %d)", rvalApi), "TestGetHostByName_SelectDnsServer: sprintf failed");
        ERROR_IF(0 != rvalApi, g_PrintBuffer);

out:
        g_TestModeVerbose = false;
        g_TestModeContinueOnFailure = false;
        return isSuccess;
    }
#endif

} // namespace unnamed

namespace NATF {
namespace API {

TEST(ApiUnit, TestGetHostByName_Setup)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    ::g_TestModeContinueOnFailure = false;
    ::g_TestModeVerbose = false;

out:
    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(ApiUnit, TestGetHostByName_Main)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    INITIALIZE_TEST_COUNTS;

    NN_LOG("Calling TestGetHostByName_Main...\n");
    isSuccess = ::TestGetHostByName_Main();
    ERROR_IF(isSuccess != true, "TestGetHostByName_Main() test failed");

out:
    PRINT_TEST_COUNTS;

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(ApiUnit, TestGetHostByName_CallRamp)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    INITIALIZE_TEST_COUNTS;

    NN_LOG("Calling TestGetHostByName_CallRamp...\n");
    isSuccess = ::TestGetHostByName_CallRamp();
    ERROR_IF(isSuccess != true, "TestGetHostByName_CallRamp() test failed");

out:
    PRINT_TEST_COUNTS;

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(ApiUnit, TestGetHostByName_SelectDnsServer)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

#ifndef NN_BUILD_CONFIG_OS_WIN32
    INITIALIZE_TEST_COUNTS;

// NOTE: Test is passing in degraded mode using g_TestModeContinueOnFailure
// This mode is not suitable for TeamCity normal execution.
// It continues on failures and only prints warnings to build log, instead of aborting
WRAP_FAILING_TEST("SIGLO-61200", "Not available: Ability to access custom DNS server through NTD firewall")
{
    NN_LOG("Calling TestGetHostByName_SelectDnsServer...\n");
    isSuccess = ::TestGetHostByName_SelectDnsServer();
    ERROR_IF(isSuccess != true, "TestGetHostByName_SelectDnsServer() test failed");
}

out:
    PRINT_TEST_COUNTS;
#else
    NN_LOG("Skipping TestGetHostByName_SelectDnsServer...\n");
#endif

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(ApiUnit, TestGetHostByName_Teardown)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:
    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

}} // namespace NATF::API
