﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"

#include <nn/socket/socket_ApiPrivate.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/socket/resolver/resolver_Client.h>
#endif

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static ThreadedController * g_pThreadedTest = NULL;
static EchoServer *         g_echoServer = NULL;

static int                  kThreadPriority = 10;       // Thread Priority
static int                  kMaxUDPPacketSize = 4096;   // Size of SysV4 Pipe
static uint16_t             kMaxTCPPacketSize = 4096;   // Size of SysV4 Pipe
static int                  kMaxNumOfSockets = 3;       // Max test sockets

static bool                 g_workBool;
static uint8_t              g_noiseVal = 1;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}



static bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    // g_pThreadedTest->Initialize( kMaxNumberOfThreads );
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    delete g_echoServer;
    g_echoServer = NULL;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}



//////////////////////////
//
// Begin Testing
//
//////////////////////////

typedef struct
{

    int         sockfds;
    uint64_t    bytesSent;
    uint64_t    bytesRecv;
    uint64_t    writeBlocked;
    uint64_t    writeNotFull;
    uint64_t    readBlocked;

} SocketTestPayloadType;


TEST(ApiUnit,EchoClient_Initialize)
{
    InitializeTesting();
}


static void
ReaderThread( void *   arg)
{
    unsigned char              inBuff[kMaxTCPPacketSize];
    int                        rc;
    SocketTestPayloadType *    payload;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, 0);

    // Restore Payload Pointer
    payload = (SocketTestPayloadType *) arg;

    while ( g_workBool == true)
    {
         rc = NATF::API::COMMON::ReadData( payload->sockfds, inBuff, sizeof( inBuff ), payload->readBlocked );
         if ( rc < 0 )
         {
             if ( rc == 0 )
             {
                 NN_LOG( "Client Read: End-of-File\n" );
                 nn::socket::Close( payload->sockfds );
             }

             if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
             {
                 continue; // Try again
             }

             NN_LOG( "socket: %d, Read failed - exiting!\n", payload->sockfds );
             goto out;
         }

         payload->bytesRecv += rc;
    }

out:

    NN_LOG( "Read Thread - exiting\n" );

    return;
}


static void
WriterThread( void *   arg)
{
    unsigned char *            outBuff;
    size_t                     outBuffLen;
    int                        rc;
    SocketTestPayloadType *    payload;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, 2);

    // Restore Payload Pointer
    payload = (SocketTestPayloadType *) arg;


    outBuffLen = kMaxTCPPacketSize;
    if ( NATF::API::COMMON::MakeRandomNoise( &outBuff, outBuffLen, g_noiseVal ) < 0 )
    {
        NN_LOG( "MakeRandomNoise failed!\n" );
        goto out;
    }

    while ( g_workBool == true)
    {
         rc = NATF::API::COMMON::WriteData( payload->sockfds, outBuff, outBuffLen, payload->writeBlocked,
                                            payload->writeNotFull );
         if ( rc < 0 )
         {
             if ( rc == 0 )
             {
                 NN_LOG( "Client Write: End-of-File\n" );
                 nn::socket::Close( payload->sockfds );
             }

             NN_LOG( "socket: %d, Client Write failed - exiting!\n", payload->sockfds );
             goto out;
         }
         payload->bytesSent += rc;
    }

out:

    NN_LOG( "Write Thread - exiting\n" );

    // Free Out Buffer
    if ( outBuff != NULL )
    {
        free( outBuff );
        outBuff = 0;
    }

    return;
}


TEST(ApiUnit,EchoClient_ReadWrite_10sec)
{
    bool                      isSuccess = true;
    int                       idx, rc;
    SocketTestPayloadType     payload[15];

    // Everyone should wait..
    g_workBool = false;

    // Initialize Payload Space
    memset( (char *) &payload, 0, sizeof( payload ) );

    for( idx = 0; idx < kMaxNumOfSockets; idx++ )
    {
        // Create TCP socket
        payload[idx].sockfds = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8053 );
        if ( payload[idx].sockfds < 0 )
        {
            NN_LOG( "Failed to connect socket" );
            goto out;
        }

        // Set Non-Blocking Read() on FDS
        rc = nn::socket::Fcntl( payload[idx].sockfds, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
        ERROR_IF( rc < 0, "socket: %d, Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n",
                                payload[idx].sockfds, nn::socket::GetLastError() );

        // Everyone should Start
        g_workBool = true;

        // Create Reader Thread
        rc = g_pThreadedTest->CreateThread( &ReaderThread, (void *) &payload[idx], kThreadPriority );
        if ( rc < 0 )
        {
            NN_LOG( "Failed to create and start Reader (thread)!\n" );
            goto out;
        }

        // Create Writer Thread
        rc = g_pThreadedTest->CreateThread( &WriterThread, (void *) &payload[idx], kThreadPriority );
        if ( rc < 0 )
        {
            NN_LOG( "Failed to create and start Writer (thread)!\n" );
            goto out;
        }
    }

    // Sleep while threads do work
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

    NN_LOG( "Stopping Test...\n" );

    // Everyone should Stop
    g_workBool = false;

    // Sleep while threads die
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    for( idx = 0; idx < kMaxNumOfSockets; idx++ )
    {
        NN_LOG(
        "socket: %d, bytes sent: %ld, bytes recv: %ld, write block: %ld, read block: %ld write not %d: %d\n",
                        payload[idx].sockfds, payload[idx].bytesSent, payload[idx].bytesRecv,
                        payload[idx].writeBlocked, payload[idx].readBlocked,
                        kMaxTCPPacketSize, payload[idx].writeNotFull );
        NN_LOG(
        "bits sent: %ld, bites recv: %ld, bits sent /per sec: %d, bits recv /per sec: %d\n",
                        ( payload[idx].bytesSent * 8) ,
                        ( payload[idx].bytesRecv * 8) ,
                        ( ( payload[idx].bytesSent * 8) / 10  ),
                        ( ( payload[idx].bytesRecv * 8) / 10 ) );
        // Close Socket
        nn::socket::Close( payload[idx].sockfds );
    }

out:

     return;
}


TEST(ApiUnit,EchoClient_TCP_Fill_Write_Buffer)
{
    bool                      isSuccess = true;
    int                       idx, rc;
    SocketTestPayloadType     payload[kMaxNumOfSockets];
    unsigned char *           bufPtr = NULL;
    size_t                    bufPtrLen;

    NN_LOG( "In\n");

    // Tell Server to wait
    g_echoServer->setFalseEcho( true );

    // Initialize Payload Space
    memset( (char *) &payload, 0, sizeof( payload ) );

    // Make Random Buff
    bufPtrLen = 32767;
    rc = NATF::API::COMMON::MakeRandomNoise( &bufPtr, bufPtrLen, g_noiseVal );
    ERROR_IF( rc < 0, "Failed to Make Random Noise" );

    for( idx = 0; idx < kMaxNumOfSockets; idx++ )
    {
        // Create TCP socket
        payload[idx].sockfds = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8053 );
        if ( payload[idx].sockfds < 0 )
        {
            NN_LOG( "Failed to connect socket" );
            goto out;
        }

        // Tell Server to wait
        g_echoServer->setFalseEcho( true );

        // Write to socket until we block
        rc = 1;
        while ( rc > 0 )
        {
            rc = NATF::API::COMMON::WriteData( payload[idx].sockfds, bufPtr, bufPtrLen,
                                               payload[idx].bytesSent, payload[idx].writeBlocked );

            if ( rc < 0 ) break;

            payload[idx].bytesSent += rc;
        }
    }

    // Unblock the server
    g_echoServer->setFalseEcho( false );

    // Give Server time to servicd all writes
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(8));

    for( idx = 0; idx < kMaxNumOfSockets; idx++ )
    {
        // Read from socket until we block
        rc = 1;
        while ( rc > 0 )
        {
            rc = NATF::API::COMMON::ReadData( payload[idx].sockfds, bufPtr, bufPtrLen, payload[idx].bytesRecv);

            if ( rc < 0 ) break;

            payload[idx].bytesRecv += rc;
        }

        NN_LOG( "socket: %d, bytes sent: %ld, bytes recv: %ld, write block: %ld, read block: %ld write not %d: %d\n",
                        payload[idx].sockfds, payload[idx].bytesSent, payload[idx].bytesRecv,
                        payload[idx].writeBlocked, payload[idx].readBlocked, bufPtrLen, payload[idx].writeNotFull );

        NN_LOG( "bits sent: %ld, bites recv: %ld, bits sent /per sec: %d, bits recv /per sec: %d\n",
                        ( payload[idx].bytesSent * 8) ,
                        ( payload[idx].bytesRecv * 8) ,
                        ( ( payload[idx].bytesSent * 8) / 10  ),
                        ( ( payload[idx].bytesRecv * 8) / 10 ) );

        if ( payload[idx].bytesSent == payload[idx].bytesRecv )
        {
            NN_LOG( "==========================\n" );
        NN_LOG( "socket: %d - PASSED!\n", payload[idx].sockfds );
        }
        else
        {
            NN_LOG( "==========================\n" );
            NN_LOG( "socket: %d - FAILED!\n", payload[idx].sockfds );
        }

        // Close Socket
        if ( payload[idx].sockfds > -1 )
        {
            nn::socket::Close( payload[idx].sockfds );
            payload[idx].sockfds = -1;
        }
    }

    NN_LOG( "OUT\n");

out:

    // Free Memory
    if ( bufPtr != NULL )
    {
        free( bufPtr );
        bufPtr = NULL;
    }

    return;
}


TEST(ApiUnit,EchoClient_UDP_Fill_Write_Buffer)
{
    bool                      isSuccess = true;
    int                       idx, rc, count;
    SocketTestPayloadType     payload[1];
    unsigned char *           bufPtr = NULL;
    size_t                    bufPtrLen;
    uint64_t                  lastRecv = 0;

    NN_LOG( "In\n");

    // Tell Server to wait
    g_echoServer->setFalseEcho( true );

    // Initialize Payload Space
    memset( (char *) &payload, 0, sizeof( payload ) );

    // Create UDP socket
    payload[0].sockfds = NATF::API::COMMON::MakeUDPConnection(  g_echoServer->kInterfaceIPv4Addr, 8050,
                                                                g_echoServer->kInterfaceIPv4Addr, 8053 ); // UDP 8053
    if ( payload[0].sockfds < 0 )
    {
        NN_LOG( "Failed to connect socket" );
        goto out;
    }

    // Tell Server to wait
    g_echoServer->setFalseEcho( true );

    // Make Random Buff
    bufPtrLen = kMaxUDPPacketSize;  // MTU for UDP

    rc = NATF::API::COMMON::MakeRandomNoise( &bufPtr, bufPtrLen, g_noiseVal );
    ERROR_IF( rc < 0, "Failed to Make Random Noise" );

    // For ever...
    count = 1;
    for( ; ; )
    {
        // Tell Server to wait
        g_echoServer->setFalseEcho( true );

        for ( idx = 0; idx < count; idx++ )
        {
            rc = NATF::API::COMMON::WriteData( payload[0].sockfds, bufPtr, bufPtrLen,
                                               payload[0].bytesSent, payload[0].writeBlocked );

            if ( rc < 0 ) break;

            payload[0].bytesSent += rc;
        }

        // Unblock the server
        g_echoServer->setFalseEcho( false );

        // Give Server time to service all writes
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // Read from socket until we block
        rc = 1;
        while ( rc > 0 )
        {
            rc = NATF::API::COMMON::ReadData( payload[0].sockfds, bufPtr, bufPtrLen, payload[0].bytesRecv);

            if ( rc < 0 ) break;

           payload[0].bytesRecv += rc;
       }

       NN_LOG( "socket: %d, bytes sent: %ld, bytes recv: %ld, write block: %ld, read block: %ld write not %d: %d\n",
                        payload[0].sockfds, payload[0].bytesSent, payload[0].bytesRecv,
                        payload[0].writeBlocked, payload[0].readBlocked, bufPtrLen, payload[0].writeNotFull );

       NN_LOG( "bits sent: %ld, bites recv: %ld, bits sent /per sec: %d, bits recv /per sec: %d\n",
                        ( payload[0].bytesSent * 8) ,
                        ( payload[0].bytesRecv * 8) ,
                        ( ( payload[0].bytesSent * 8) / 10  ),
                        ( ( payload[0].bytesRecv * 8) / 10 ) );

       if ( payload[0].bytesSent > payload[0].bytesRecv )
       {
           NN_LOG( "==========================\n" );
           NN_LOG( "socket: %d - SUCCESS - MAXIMUM SUPPORTED UDP PAYLOAD is: %ld!\n", payload[0].sockfds, lastRecv );
           break;
       }
       else
       {
            NN_LOG( "==========================\n" );
            NN_LOG( "socket: %d - PASSED!\n", payload[0].sockfds );


            lastRecv = payload[0].bytesSent;

            count++;            // Add another Datagram
            continue;           // Jumop to Top
       }
    }

out:

    if ( bufPtr != NULL )
    {
        free( bufPtr );
        bufPtr = NULL;
    }

    if ( payload[0].sockfds > -1 )
    {
        nn::socket::Close( payload[0].sockfds );
        payload[0].sockfds = -1;
    }

    return;
}


TEST(ApiUnit,EchoClient_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,EchoClient_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
