﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_LoopbackDnsServer.h"     // Loopback DNS
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

#include <nn/socket/socket_ApiPrivate.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/socket/socket_ResolverOptionsPrivate.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#endif

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

// Anonymous
namespace {

ThreadedController * g_pThreadedTest      = NULL;
LoopbackDnsServer *  g_LoopbackDnsServer  = NULL;
LoopbackDnsServer *  g_LoopbackDnsServer2 = NULL;

int                  kMaxNumberOfThreads = 10;
int                  kThreadPriority = 10;

uint32_t             g_WorkVaruint32_t  = 0;
uint32_t             g_WorkVaruint32_t2 = 0;
uint32_t             g_WorkVaruint32_t3 = 0;

uint16_t             g_SavedTXID = 0;
uint16_t             g_SavedUDPPort  = 0;
uint16_t             g_SavedUDPPort2 = 0;

bool                 g_DifferentTXIDFound = false;
bool                 g_DifferentUDPPortFound = false;
bool                 g_SameTXIDFound = false;
bool                 g_SameUDPPortFound = false;

const int SERVER_A = 1;
const int SERVER_B = 2;

}  // Anonymous



// Standard Google Lookup
//
// ; <<>> DiG 9.10.3-P4 <<>> +noedns=0 google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 35193
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;; google.com.                    IN      A
//
// ;; ANSWER SECTION:
// ;; google.com.            150     IN      A       216.58.193.78
//
// ;; Query time: 31 msefalsec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:59:44 PDT 2016
// ;; MSG SIZE  rcvd: 44

namespace {

const unsigned char DnsRequest_Google_com_simple[] = {

0x96, 0xac, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
0x00, 0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e

};

// Standard: E-Root lookup
//
// $ dig -x 192.203.230.10 +noedns
//
// ; <<>> DiG 9.10.3-P4 <<>> -x 192.203.230.10 +noedns
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 30073
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;10.230.203.192.in-addr.arpa.   IN      PTR
//
// ;; ANSWER SECTION:
// 10.230.203.192.in-addr.arpa. 10066 IN   PTR     e.root-servers.net.
//
// ;; Query time: 2 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 23 12:27:05 PDT 2016
// ;; MSG SIZE  rcvd: 77
//
const unsigned char DnsRequest_ERoot_com_simple[] = {

0xf3, 0x78, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x02, 0x31, 0x30, 0x03, 0x32, 0x33, 0x30, 0x03, 0x32, 0x30, 0x33, 0x03,
0x31, 0x39, 0x32, 0x07, 0x69, 0x6e, 0x2d, 0x61, 0x64, 0x64, 0x72, 0x04,
0x61, 0x72, 0x70, 0x61, 0x00, 0x00, 0x0c, 0x00, 0x01, 0xc0, 0x0c, 0x00,
0x0c, 0x00, 0x01, 0x00, 0x00, 0x26, 0xaa, 0x00, 0x14, 0x01, 0x65, 0x0c,
0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x73,
0x03, 0x6e, 0x65, 0x74, 0x00

};


//; <<>> DiG 9.10.4-P4 <<>> alias.landsberger.com
//;; global options: +cmd
//;; Got answer:
//;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 10242
//;; flags: qr rd ra; QUERY: 1, ANSWER: 2, AUTHORITY: 0, ADDITIONAL: 1
//
//;; OPT PSEUDOSECTION:
//; EDNS: version: 0, flags:; udp: 4096
//;; QUESTION SECTION:
//;alias.landsberger.com. IN   A
//
//;; ANSWER SECTION:
//alias.landsberger.com.    2228    IN  CNAME landsberger.com.
//landsberger.com.  2228    IN  A   138.128.120.101
//
//;; Query time: 7 msec
//;; SERVER: 10.1.19.29#53(10.1.19.29)
//;; WHEN: Wed May 03 11:35:13 PST 2017
//;; MSG SIZE  rcvd: 80

const unsigned char DnsRequest_Alias_Landsbergber_com_CNAME[] = {

0x77, 0x91, 0x81, 0x80, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01,
0x05, 0x61, 0x6c, 0x69, 0x61, 0x73, 0x0b, 0x6c, 0x61, 0x6e, 0x64, 0x73,
0x62, 0x65, 0x72, 0x67, 0x65, 0x72, 0x03, 0x63, 0x6f, 0x6d, 0x00, 0x00,
0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x0d,
0x6a, 0x00, 0x02, 0xc0, 0x12, 0xc0, 0x12, 0x00, 0x01, 0x00, 0x01, 0x00,
0x00, 0x0d, 0x6a, 0x00, 0x04, 0x8a, 0x80, 0x78, 0x65, 0x00, 0x00, 0x29,
0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00

};


//; <<>> DiG 9.10.4-P4 <<>> alias2.landsberger.com
//;; global options: +cmd
//;; Got answer:
//;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 15384
//;; flags: qr rd ra; QUERY: 1, ANSWER: 2, AUTHORITY: 0, ADDITIONAL: 1
//
//;; OPT PSEUDOSECTION:
//; EDNS: version: 0, flags:; udp: 4096
//;; QUESTION SECTION:
//;alias2.landsberger.com.                IN      A
//
//;; ANSWER SECTION:
//alias2.landsberger.com. 3570    IN      CNAME   landsberger.com.
//landsberger.com.        3570    IN      A       138.128.120.101
//
//;; Query time: 1 msec
//;; SERVER: 10.1.19.29#53(10.1.19.29)
//;; WHEN: Thu May 04 10:11:16 PST 2017
//;; MSG SIZE  rcvd: 81

const unsigned char DnsRequest_Alias3_Landsbergber_com_CNAME[] = {

0xa7, 0x20, 0x81, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x06, 0x61, 0x6c, 0x69,
0x61, 0x73, 0x33, 0x0b, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x62, 0x65, 0x72, 0x67, 0x65, 0x72, 0x03,
0x63, 0x6f, 0x6d, 0x00, 0x00, 0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00,
0x0e, 0x10, 0x00, 0x4b, 0x3f, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
0x77, 0x77, 0x77, 0x77, 0x05, 0x63, 0x69, 0x73, 0x75, 0x6d, 0x03, 0x6f, 0x72, 0x67, 0x00, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x02, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x07, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x0e, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x09, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x14, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x15, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x08, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x0f, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x0a, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x04, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x13, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x0d, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x06, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x12, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x0b, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x03, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x10, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x17, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x0c, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x11, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x01, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x05, 0xc0,
0x34, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0e, 0x10, 0x00, 0x04, 0x01, 0x02, 0x03, 0x16


};

nn::socket::ResolverOption DisableDnsCache()
{
    nn::socket::ResolverOption   myOption;

    NN_LOG( "DisableDnsCache() - Begin\n" );
    myOption.key = static_cast<nn::socket::ResolverOptionKey>( nn::socket::ResolverOptionKey::RequestEnableDnsCacheBoolean );
    myOption.type = nn::socket::ResolverOptionType::Boolean;
    myOption.size = sizeof(bool);
    myOption.data.booleanValue = false;
    NN_LOG( "DisableDnsCache() - End\n" );

    return myOption;
}

void
StartLookbackDnsServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Loopback DNS server]: Thread starting\n" );

    LoopbackDnsServer * loopbackDnsServer = reinterpret_cast<LoopbackDnsServer *>( inArg );

    // Start the DNS Server Main Loop
    loopbackDnsServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Loopback DNS server]: Thread exiting\n" );
}


bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true, rval;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate DNS Server
    ///////////////////////////

    g_LoopbackDnsServer = new LoopbackDnsServer();
    if ( g_LoopbackDnsServer == NULL )
    {
        NN_LOG( "Failed to allocate a LoopbackDnsServer.  Can't start Loopback port DNS Server\n" );
        goto out;
    }

    ///////////////////////////
    ////  Change DNS Server Address - to (loopback)
    ///////////////////////////

    rval = g_LoopbackDnsServer->SetDnsResolver( "127.0.0.1", 8053 );
    ERROR_IF( rval != true, "Failed calling LoopbackDnsServer::SetDnsResolver" );


    ///////////////////////////
    //// Threaded DNS Server
    ///////////////////////////

    NN_LOG( "===> Starting [loopback] DNS Server\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumberOfThreads );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartLookbackDnsServer, reinterpret_cast<void *>( g_LoopbackDnsServer ), kThreadPriority );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started DNS Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_LoopbackDnsServer != NULL )
    {
        delete g_LoopbackDnsServer;
        g_LoopbackDnsServer = NULL;
    }

    return( false );
}


bool
CallGetHostByName( const char * in_domainname )
{
    nn::socket::HostEnt *   respHosts = nullptr;
    nn::socket::InAddr **   addr_list = nullptr;
    int                     idx;

    // Disable DNS Cache
    nn::socket::ResolverOption myOption = DisableDnsCache();

    respHosts = nn::socket::GetHostEntByName( in_domainname, &myOption, 1 );
    if ( respHosts == NULL )
    {
        NN_LOG( "nn::socket::GetHostByName() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
                 nn::socket::GetLastError(), *(nn::socket::GetHError() ),
                 nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }
    else
    {
        NN_LOG( "Host name       : [%s]\n", respHosts->h_name );
        NN_LOG( "Address Type    : [%d]\n", respHosts->h_addrtype );
        NN_LOG( "Len of Addr     : [%d]\n", respHosts->h_length );

        for(idx = 0; respHosts->h_aliases[idx] != NULL; idx++)
        {

           NN_LOG( "Alias: [%s]\n", respHosts->h_aliases[idx] );
        }

        addr_list = (nn::socket::InAddr **) respHosts->h_addr_list;
        for(idx = 0; addr_list[idx] != NULL; idx++)
        {

           NN_LOG( "IP Address: [%s]\n", nn::socket::InetNtoa( (*addr_list[idx] ) ) );
        }

        NN_LOG( "There are a total of [%d] IPv4 Addresses\n", idx );
    }

    return( true );

out:

    return( false );
}


bool
CallGetAddrInfo( const char *in_domainname, nn::socket::AddrInfo *out_resp, nn::socket::AiErrno * out_error )
{
    nn::socket::SockAddrIn      *ipv4;
    nn::socket::AddrInfo        *respIdx, hints;

    // Disable DNS Cache
    nn::socket::ResolverOption myOption = DisableDnsCache();

    // Set GAI Hints
    memset( (char *) &hints, 0, sizeof( hints ) );
    hints.ai_family   = nn::socket::Family::Af_Inet;

    hints.ai_family   = nn::socket::Family::Af_Inet;
    hints.ai_socktype = nn::socket::Type::Sock_Stream;
    hints.ai_flags    = nn::socket::AddrInfoFlag::Ai_CanonName;

    *out_error = nn::socket::GetAddrInfo( in_domainname, 0, &hints, &out_resp, &myOption, 1 );
    if ( nn::socket::AiErrno::EAi_Success != *out_error )
    {
        NN_LOG( "nn::socket::GetAddrInfo() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
        nn::socket::GetLastError(), *(nn::socket::GetHError() ),
        nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }

    for ( respIdx = out_resp; respIdx; respIdx = respIdx->ai_next)
    {
        ipv4 = (nn::socket::SockAddrIn *) (respIdx->ai_addr);

        NN_LOG( "Family: %d, Socktype: %d, Protocol: %d, Flags: 0x%0x, IPAddr: %s, Port: %d, Name: %s\n",
                 respIdx->ai_family, respIdx->ai_socktype, respIdx->ai_protocol,
                 respIdx->ai_flags, nn::socket::InetNtoa( ipv4->sin_addr ),
                 nn::socket::InetNtohs( ipv4->sin_port ), respIdx->ai_canonname );
    }

    return( true );

out:

    return( false );

}


bool
CallGetHostByAddr( const char *  dottedIP, nn::socket::SockLenT addrLen, nn::socket::Family family, nn::socket::HostEnt ** rtrnHosts )
{
    nn::socket::InAddr      ipv4Addr = {};
    nn::socket::HostEnt *   respHosts = nullptr;
    nn::socket::InAddr **   addr_list = nullptr;
    int                     idx, rc;

    // Disable DNS Cache
    nn::socket::ResolverOption myOption = DisableDnsCache();

    // Initialize Return Value
    *rtrnHosts = NULL;

    // Turn English IP Address into octets
    if ( addrLen > 3 )
    {
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, dottedIP, &ipv4Addr );
        if ( rc != 1 )
        {
            NN_LOG( "InetPton() failed - rc: %d errno: %d\n", rc, nn::socket::GetLastError() );
            goto out;
        }
    }

    NN_LOG( "Loopup IP Address: %d.%d.%d.%d, Len: %d, family: %d\n",
               (uint8_t) ( ipv4Addr.S_addr ), (uint8_t) ( ipv4Addr.S_addr >> 8 ),
               (uint8_t) ( ipv4Addr.S_addr >> 16 ), (uint8_t) ( ipv4Addr.S_addr >> 24 ),
               addrLen, family );

    respHosts = nn::socket::GetHostEntByAddr( &ipv4Addr, addrLen, family, &myOption, 1 );
    if ( respHosts == NULL )
    {
        NN_LOG( "nn::socket::GetHostByAddr() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
        nn::socket::GetLastError(), *(nn::socket::GetHError() ),
        nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }
    else
    {
        NN_LOG( "Host name       : [%s]\n", respHosts->h_name );
        NN_LOG( "Address Type    : [%d]\n", respHosts->h_addrtype );
        NN_LOG( "Len of Addr     : [%d]\n", respHosts->h_length );

        for(idx = 0; respHosts->h_aliases[idx] != NULL; idx++)
        {
            NN_LOG( "Alias: [%s]\n", respHosts->h_aliases[idx] );
        }

        addr_list = (nn::socket::InAddr **) respHosts->h_addr_list;
        for(idx = 0; addr_list[idx] != NULL; idx++)
        {
            NN_LOG( "IP Address: [%s]\n", nn::socket::InetNtoa( (*addr_list[idx] ) ) );
        }

        NN_LOG( "There are a total of [%d] IPv4 Addresses\n", idx );
    }

    // Save Return Pointer
    *rtrnHosts = respHosts;

    return( true );

out:

    return( false );
}

bool
CallGetNameInfo( const char *ipAddr )
{
    nn::socket::SockAddrIn      sa = {};
    nn::socket::SockLenT        saLen = 0;
    nn::socket::AiErrno         error = nn::socket::AiErrno::EAi_Success;
    char                        rtrnHost[80], rtrnService[80];
    size_t                      hostLen = 80, serviceLen = 80;
    nn::socket::NameInfoFlag    flags = nn::socket::NameInfoFlag::Ni_NameReqd;

    // Disable DNS Cache
    nn::socket::ResolverOption myOption = DisableDnsCache();

    // Initialize IP Address to lookup
    sa.sin_family = nn::socket::Family::Af_Inet;
    sa.sin_port   = nn::socket::InetHtons( 80 );

    if ( nn::socket::InetPton( nn::socket::Family::Af_Inet, ipAddr, &(sa.sin_addr)) != 1 )
    {
       NN_LOG( "CallGetNameInfo() - InetPton failed to convert IP Address %s!", ipAddr );
       goto out;
    }

    // Initialize Return Values
    memset( rtrnHost, 0, sizeof( rtrnHost ) );
    memset( rtrnService, 0, sizeof( rtrnService ) );

    // Call GetNameInfo()
    saLen = sizeof( sa );
    error = nn::socket::GetNameInfo( reinterpret_cast<nn::socket::SockAddr *>(&sa), saLen, rtrnHost, hostLen,
                                  rtrnService, serviceLen, flags, &myOption, 1 );
    if ( error == nn::socket::AiErrno::EAi_Success )
    {
        NN_LOG( "GNI: Good return Code, Returned, error: %d\n", error );
        NN_LOG( "Rtrn Host    : [%.*s]\n", hostLen, rtrnHost );
        NN_LOG( "Rtrn Service : [%.*s]\n", serviceLen, rtrnService );

        // Fall thru
    }
    else
    {
        NN_LOG( "nn::socket::CallGetNameInfo() Failed with: error: %d, errno: %d, h_errno: %d, hstrerror: %s\n",
                 error, nn::socket::GetLastError(), *(nn::socket::GetHError() ),
                 nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }

    return( true );

out:

    return( false );

}


} // Anonymous


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    /////////////////////////
    // Stop 2nd DNS Server
    /////////////////////////

    // If the (Second) loopback DNS server is allocated
    if ( g_LoopbackDnsServer2 != NULL )
    {
        NN_LOG( "Stopping 2nd DNS Server\n" );
        g_LoopbackDnsServer2->Stop();
        delete g_LoopbackDnsServer2;
        g_LoopbackDnsServer2 = NULL;
    }

    ///////////////////
    // Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping 1st DNS Server\n" );

    // If the loopback DNS server is allocated
    if ( g_LoopbackDnsServer != NULL )
    {
        g_LoopbackDnsServer->Stop();
        delete g_LoopbackDnsServer;
        g_LoopbackDnsServer = NULL;
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    NN_LOG( "DNS Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}


TEST(ApiUnit,DnsRequest_Initialize)
{
    InitializeTesting();
}


static uint64_t
GetMilliSeconds() NN_NOEXCEPT
{
    nn::os::Tick now_in_ticks = nn::os::GetSystemTick();
    return( nn::os::ConvertToTimeSpan( now_in_ticks ).GetMilliSeconds() );
}


// Anonymous Namespace
namespace {

struct DnsRequestTimer
{
    int64_t   startTime;
    int64_t   endTime;
};

static DnsRequestTimer myTimer[10];
static int myTimerIdx = 0;

}  // End anonymous




// Anonymous Namespace
namespace {


TEST(ApiUnit,DnsRequest_Test_CNAME)
{
    bool                    isSuccess = true, rval;
    nn::socket::AddrInfo    out_resp;
    nn::socket::AiErrno     out_error;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( g_LoopbackDnsServer->MakeDnsResponseCB );

    // Set alias.landsberger.com Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_Alias_Landsbergber_com_CNAME;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_Alias_Landsbergber_com_CNAME );

    {
        nn::socket::HostEnt *    respHosts;
        nn::socket::ResolverOption   myOption;

        myOption.key = static_cast<nn::socket::ResolverOptionKey>( nn::socket::ResolverOptionKey::RequestEnableDnsCacheBoolean );
        myOption.type = nn::socket::ResolverOptionType::Boolean;
        myOption.size = sizeof(bool);
        myOption.data.booleanValue = false;

        respHosts = nn::socket::GetHostEntByName(  "alias.landsberger.com", &myOption, 1 );
    }

    //`*********************
    //* GHBN
    //*********************
    NN_LOG( "Query: [GetHostByName][alias.landsberger.com]\n" );
    rval = CallGetHostByName( "alias.landsberger.com" );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );

    //*********************
    //* GETADDRINFO
    //*********************

    NN_LOG( "Query: [GetAddrInfo][alias.landsbergber.com]\n" );
    rval = CallGetAddrInfo( "alias.landsberger.com", &out_resp, &out_error );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );

    // Set alias3.landsbergber.com
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_Alias3_Landsbergber_com_CNAME;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_Alias3_Landsbergber_com_CNAME );

    //`*********************
    //* GHBN
    //*********************
    NN_LOG( "Query: [GetHostByName][alias3.landsberger.com]\n" );
    rval = CallGetHostByName( "alias3.landsberger.com" );
    ERROR_IF_AND_COUNT( rval == false, "GHBN failed but should have succeeded!" );


    //*********************
    //* GETADDRINFO
    //*********************

    NN_LOG( "Query: [GetAddrInfo][alias3.landsbergber.com]\n" );
    rval = CallGetAddrInfo( "alias3.landsberger.com", &out_resp, &out_error );
    ERROR_IF_AND_COUNT( rval == false, "GAI failed but should have succeeded!" );

out:

    return;
}


int
Count_Dns_Same_TXID( void *                  inArg,
                     unsigned char *         dnsRqst,
                     size_t &                dnsRqstLen,
                     unsigned char *         dnsResp,
                     size_t &                dnsRespLen,
                     bool *                  isUDP,
                     bool *                  shouldContinue )
{
    struct DNS_HEADER_REC *    pDnsHeader = NULL;
    uint16_t                   saveId = 0;
    uint16_t                   udpPort = 0;
    LoopbackDnsServer *        thisObj = reinterpret_cast<LoopbackDnsServer *>( inArg );

    // If FIRST Query..
    if ( g_WorkVaruint32_t == 0 )
    {
        // Save DNS Query Transaction Id
        pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
        g_SavedTXID = nn::socket::InetNtohs( pDnsHeader->Id );
        NN_LOG( "===> Saved Transaction ID [%d]\n", g_SavedTXID );


        // Save DNS (UDP Source Port)
        g_SavedUDPPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );
        NN_LOG( "===> Saved UDP Port [%d]\n", g_SavedUDPPort );

    }

    // Keep a running total
    g_WorkVaruint32_t++;

    // Compare DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Make sure Transaction ID is not different
    if ( saveId != g_SavedTXID )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different DNS TXN ID returned!  (This TXNID): %d, Previous TXNID %d\n", saveId,  g_SavedTXID );
        NN_LOG( "============================\n" );
        g_DifferentTXIDFound = true;
    }

    // Compare UDP Source Port
    udpPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );

    // Make sure UDP Port is not different
    if ( udpPort != g_SavedUDPPort )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different UDP Port!  (This Port): %d, Previous Port %d\n", udpPort,  g_SavedUDPPort );
        NN_LOG( "============================\n" );
        g_DifferentUDPPortFound = true;
    }


    // Save Finished timing
    if ( myTimerIdx < 10 )
    {
        myTimer[ myTimerIdx ].endTime = GetMilliSeconds();
    }

    // Start new timing
     myTimerIdx++;

    if ( myTimerIdx < 10 )
    {
        myTimer[ myTimerIdx ].startTime = GetMilliSeconds();
    }

    *shouldContinue = false;

    return( 0 );
}

} // Anonymous



TEST(ApiUnit,DnsRequest_Same_TXID)
{
    bool                    isSuccess = true, rval;
    int                     idx;
    int64_t                 delta;
    nn::socket::AddrInfo    out_resp;
    nn::socket::AiErrno     out_error;
    nn::socket::HostEnt *   respHosts;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( Count_Dns_Same_TXID );

    // Set Google DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_Google_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_Google_com_simple );

    //`*********************
    //* GHBN
    //*********************

    // Initialize Timers
    memset( (char *) &myTimer, 0, sizeof( myTimer ) );

    g_WorkVaruint32_t    = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // Start Timer
    myTimerIdx = 0;
    myTimer[ myTimerIdx ].startTime = GetMilliSeconds();

    NN_LOG( "Query: [GetHostByName][Google] Returns wrong answer\n" );
    rval = CallGetHostByName( "google.com" );
    NN_LOG( "There were: [%d] GHBN calls to the DNS server\n", myTimerIdx );
    ERROR_IF_AND_COUNT( rval == true, "GHBN received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    for( idx = 0; idx < 10; idx++ )
    {
        if ( myTimer[idx].endTime > 0 )
        {
            delta =  (myTimer[idx].endTime - myTimer[idx].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myTimer[idx].startTime,
                    myTimer[idx].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%ld>, Failed!!\n", delta );

        }
    }

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GHBN Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GHBN Different UDP Port found during processing!\n" );

    NN_LOG( "GHBN: There were a total of [%d] DNS Requests\n",  g_WorkVaruint32_t );
    ERROR_IF_AND_COUNT( g_WorkVaruint32_t < 8, "[%d] DNS Requests were seen - expected at least [8]!\n",
                                                                                      g_WorkVaruint32_t );

    //*********************
    //* GETADDRINFO
    //*********************

    // Initialize Timers
    memset( (char *) &myTimer, 0, sizeof( myTimer ) );

    g_WorkVaruint32_t    = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // Start Timer
    myTimerIdx = 0;
    myTimer[ myTimerIdx ].startTime = GetMilliSeconds();

    NN_LOG( "Query: [GetAddrInfo][Google] Returns wrong answer\n" );
    rval = CallGetAddrInfo( "google.com", &out_resp, &out_error );
    NN_LOG( "There were: [%d] GAI calls to the DNS server\n", myTimerIdx );
    ERROR_IF_AND_COUNT( rval == true, "GAI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    for( idx = 0; idx < 10; idx++ )
    {
        if ( myTimer[idx].endTime > 0 )
        {
            delta =  (myTimer[idx].endTime - myTimer[idx].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myTimer[idx].startTime,
                    myTimer[idx].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%ld>, Failed!!\n", delta );
        }
    }

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GAI Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GAI Different UDP Port found during processing!\n" );

    NN_LOG( "GAI: There were a total of [%d] DNS Requests\n",  g_WorkVaruint32_t );
    ERROR_IF_AND_COUNT( g_WorkVaruint32_t < 8, "[%d] DNS Requests were seen - expected at least [8]!\n",
                                                                                      g_WorkVaruint32_t );

    //*********************
    //* GHBA
    //*********************

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize Timers
    memset( (char *) &myTimer, 0, sizeof( myTimer ) );

    g_WorkVaruint32_t    = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // Start Timer
    myTimerIdx = 0;
    myTimer[ myTimerIdx ].startTime = GetMilliSeconds();

    NN_LOG( "Query: [GetHostByAddr][ERoot] Returns wrong answer\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &respHosts );
    NN_LOG( "There were: [%d] GHBA calls to the DNS server\n", myTimerIdx );
    ERROR_IF_AND_COUNT( rval == true, "GHBA received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    for( idx = 0; idx < 10; idx++ )
    {
        if ( myTimer[idx].endTime > 0 )
        {
            delta =  (myTimer[idx].endTime - myTimer[idx].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myTimer[idx].startTime,
                    myTimer[idx].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%ld>, Failed!!\n", delta );

        }
    }

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GHBA Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GHBA Different UDP Port found during processing!\n" );

    NN_LOG( "GHBA: There were a total of [%d] DNS Requests\n",  g_WorkVaruint32_t );
    ERROR_IF_AND_COUNT( g_WorkVaruint32_t < 8, "[%d] DNS Requests were seen - expected at least [8]!\n",
                                                                                      g_WorkVaruint32_t );

    //*********************
    //* GNI
    //*********************

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize Timers
    memset( (char *) &myTimer, 0, sizeof( myTimer ) );

    g_WorkVaruint32_t    = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // Start Timer
    myTimerIdx = 0;
    myTimer[ myTimerIdx ].startTime = GetMilliSeconds();

    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10" );
    NN_LOG( "There were: [%d] GNI calls to the DNS server\n", myTimerIdx );
    ERROR_IF_AND_COUNT( rval == true, "GNI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    for( idx = 0; idx < 10; idx++ )
    {
        if ( myTimer[idx].endTime > 0 )
        {
            delta =  (myTimer[idx].endTime - myTimer[idx].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myTimer[idx].startTime,
                    myTimer[idx].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%ld>, Failed!!\n", delta );

        }
    }

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GNI Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GNI Different UDP Port found during processing!\n" );

    NN_LOG( "GNI: There were a total of [%d] DNS Requests\n",  g_WorkVaruint32_t );
    ERROR_IF_AND_COUNT( g_WorkVaruint32_t < 8, "[%d] DNS Requests were seen - expected at least [8]!\n",
                                                                                      g_WorkVaruint32_t );

out:

    return;

}  // NOLINT(impl/function_size)


namespace {

int
Late_Dns_Response( void *                  inArg,
                   unsigned char *         dnsRqst,
                   size_t &                dnsRqstLen,
                   unsigned char *         dnsResp,
                   size_t &                dnsRespLen,
                   bool *                  isUDP,
                   bool *                  shouldContinue )
{
    struct DNS_HEADER_REC *    pDnsHeader = NULL;
    uint16_t                   saveId = 0;
    uint16_t                   udpPort = 0;
    LoopbackDnsServer *        thisObj = (LoopbackDnsServer *) inArg;

    // If FIRST Query.. Save the Transaction ID
    if ( g_WorkVaruint32_t == 0 )
    {
        // Save DNS Query Transaction Id
        pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
        g_SavedTXID = nn::socket::InetNtohs( pDnsHeader->Id );
        NN_LOG( "===> Saved Transaction ID [%d]\n", g_SavedTXID );


        // Save DNS (UDP Source Port)
        g_SavedUDPPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );
        NN_LOG( "===> Saved UDP Port [%d]\n", g_SavedUDPPort );
    }

    // Keep a running total
    g_WorkVaruint32_t++;

    // Compare DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Make sure Transaction ID is not different
    if ( saveId != g_SavedTXID )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different DNS TXN ID returned!  (This TXNID): %d, Previous TXNID %d\n", saveId,  g_SavedTXID );
        NN_LOG( "============================\n" );
        g_DifferentTXIDFound = true;
    }

    // Compare UDP Source Port
    udpPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );

    // Make sure UDP Port is not different
    if ( udpPort != g_SavedUDPPort )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different UDP Port!  (This Port): %d, Previous Port %d\n", udpPort,  g_SavedUDPPort );
        NN_LOG( "============================\n" );
        g_DifferentUDPPortFound = true;
    }

    if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
    {
        // Return a normal DNS Response
        return( g_LoopbackDnsServer->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
    }
    else
    {
        *shouldContinue = false;
    }


    return( 0 );
}

} // Anonymous


TEST(ApiUnit,DnsRequest_GHBN_Late_Dns_Response)
{
    bool                    isSuccess = true, rval = false;
    nn::socket::AddrInfo    out_resp;
    nn::socket::AiErrno     out_error;
    nn::socket::HostEnt *   respHosts;

    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( Late_Dns_Response );

    // Set Google DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_Google_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_Google_com_simple );


    //*********************
    //* GHBN
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // For I = 1 to 9
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "GHBN: Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear running total
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetHostByName][Google] Returns wrong answer\n" );
        rval = CallGetHostByName( "google.com" );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                NN_LOG( "===========> MATCHED  [%d] <=================\n", g_WorkVaruint32_t );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GHBN received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "Different DNS Transaction Id found during processing!\n" );
        ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "Different UDP Port found during processing!\n" );
    }

    NN_LOG( "GHBN: There were: [%d] calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GETADDRINFO
    //*********************

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // For I = 1 to 9
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "GAI: Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear running total
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetAddrInfo][Google] Returns wrong answer\n" );
        rval = CallGetAddrInfo( "google.com", &out_resp, &out_error );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                NN_LOG( "===========> MATCHED  [%d] <=================\n", g_WorkVaruint32_t );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GAI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GAI: Different DNS Transaction Id found during processing!\n" );
        ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GAI: Different UDP Port found during processing!\n" );
    }

    NN_LOG( "GAI: There were: [%d] calls to the DNS server\n", g_WorkVaruint32_t );

    //********************
    //* GHBA
    //*********************/

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // For I = 1 to 9
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "GHBA: Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear running total
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetHostByAddr][ERoot] Returns wrong answer\n" );
        rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &respHosts );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                NN_LOG( "===========> MATCHED  [%d] <=================\n", g_WorkVaruint32_t );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GHBA received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GHBA: Different DNS Transaction Id found during processing!\n" );
        ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GHBA: Different UDP Port found during processing!\n" );
    }

    NN_LOG( "GHBA: There were: [%d] calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GNI
    //*********************

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;

    // For I = 1 to 9
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "GNI: Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear running total
        g_WorkVaruint32_t = 0;

        // Make Call
        NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
        rval = CallGetNameInfo( "192.203.230.10" );
        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                NN_LOG( "===========> MATCHED  [%d] <=================\n", g_WorkVaruint32_t );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GNI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GNI: Different DNS Transaction Id found during processing!\n" );
        ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GNI: Different UDP Port found during processing!\n" );
    }

    NN_LOG( "GNI: There were: [%d] calls to the DNS server\n", g_WorkVaruint32_t );

out:

    return;

}    // NOLINT(impl/function_size)


// Anonymous
namespace {

int
No_Duplicate_Requests( void *                  inArg,
                       unsigned char *         dnsRqst,
                       size_t &                dnsRqstLen,
                       unsigned char *         dnsResp,
                       size_t &                dnsRespLen,
                       bool *                  isUDP,
                       bool *                  shouldContinue )
{
    struct DNS_HEADER_REC *    pDnsHeader = NULL;
    uint16_t                   saveId = 0;
    uint16_t                   udpPort = 0;
    LoopbackDnsServer *        thisObj = reinterpret_cast<LoopbackDnsServer *>( inArg );

    // If FIRST Query...
    if ( g_WorkVaruint32_t == 0 )
    {
        // Save DNS Query Transaction Id
        pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
        g_SavedTXID = nn::socket::InetNtohs( pDnsHeader->Id );
        NN_LOG( "===> Saved Transaction ID [%d]\n", g_SavedTXID );


        // Save DNS (UDP Source Port)
        g_SavedUDPPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );
        NN_LOG( "===> Saved UDP Port [%d]\n", g_SavedUDPPort );

        // Return a normal DNS Response
        return( g_LoopbackDnsServer->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
    }

    // If SECOND Query..

    // Compare Transaction ID
    if ( saveId == g_SavedTXID )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Same DNS TXN ID returned!  (This TXNID): %d, Previous TXNID %d\n", saveId,  g_SavedTXID );
        NN_LOG( "============================\n" );
        g_SameTXIDFound = true;
    }

    // Compare UDP Source Port
    udpPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );

    // Make sure UDP Port is not different
    if ( udpPort == g_SavedUDPPort )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Same UDP Port!  (This Port): %d, Previous Port %d\n", udpPort,  g_SavedUDPPort );
        NN_LOG( "============================\n" );
        g_SameUDPPortFound = true;
    }

    return( g_LoopbackDnsServer->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
}

} // Anonymous


TEST(ApiUnit,DnsRequest_No_Duplicate_Request)
{
    bool                    isSuccess = true, rval = false;
    nn::socket::AddrInfo    out_resp;
    nn::socket::AiErrno     out_error;
    nn::socket::HostEnt *   respHosts;


    // Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( No_Duplicate_Requests );

    // Set Google DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_Google_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_Google_com_simple );


    //*********************
    //* GHBN
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_SameTXIDFound = false;
    g_SameUDPPortFound = false;

    // Send FIRST Request
    rval = CallGetHostByName( "google.com" );
    ERROR_IF( rval == false, "GHBN: Failed to recieve FIRST answer to GHBN!" );

    // Send Second Request
    g_WorkVaruint32_t++;   // Next request
    rval = CallGetHostByName( "google.com" );
    ERROR_IF( rval == false, "GHBN: Failed to recieve SECOND answer to GHBN!" );

    ERROR_IF_AND_COUNT( g_SameTXIDFound == true, "GHBN: Same DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_SameUDPPortFound == true, "GHBN: Same UDP Port found during processing!\n" );


    //*********************
    //* GETADDRINFO
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_SameTXIDFound = false;
    g_SameUDPPortFound = false;

    // Send FIRST Request
    rval = CallGetAddrInfo( "google.com", &out_resp, &out_error );
    ERROR_IF( rval == false, "GAI: Failed to recieve FIRST answer to GHBN!" );

    // Send Second Request
    g_WorkVaruint32_t++;   // Next request
    rval = CallGetAddrInfo( "google.com", &out_resp, &out_error );
    ERROR_IF( rval == false, "GAI: Failed to recieve SECOND answer to GHBN!" );

    ERROR_IF_AND_COUNT( g_SameTXIDFound == true, "GAI: Same DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_SameUDPPortFound == true, "GAI: Same UDP Port found during processing!\n" );


    //*********************
    //* GHBA
    //*********************/

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_SameTXIDFound = false;
    g_SameUDPPortFound = false;

    // Send FIRST Request
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &respHosts );
    ERROR_IF( rval == false, "GHBA: Failed to recieve FIRST answer to GHBN!" );

    // Send Second Request
    g_WorkVaruint32_t++;   // Next request
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &respHosts );
    ERROR_IF( rval == false, "GHBA: Failed to recieve SECOND answer to GHBN!" );

    ERROR_IF_AND_COUNT( g_SameTXIDFound == true, "GHBA: Same DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_SameUDPPortFound == true, "GHBA: Same UDP Port found during processing!\n" );

    //*********************
    //* GNI
    //*********************

    // Set ERoot DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_SameTXIDFound = false;
    g_SameUDPPortFound = false;

    // Send FIRST Request
    rval = CallGetNameInfo( "192.203.230.10" );
    ERROR_IF( rval == false, "GNI: Failed to recieve FIRST answer to GHBN!" );

    // Send Second Request
    g_WorkVaruint32_t++;   // Next request
    rval = CallGetNameInfo( "192.203.230.10" );
    ERROR_IF( rval == false, "GNI: Failed to recieve SECOND answer to GHBN!" );

    ERROR_IF_AND_COUNT( g_SameTXIDFound == true, "GNI: Same DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_SameUDPPortFound == true, "GNI: Same UDP Port found during processing!\n" );

out:

    return;
}


//****************************************************************
//*
//*   T E S T   W I T H   T W O    D N S   S E R V E R s
//*
//****************************************************************

TEST(ApiUnit,DnsRequest_Create_Two_DNS_Servers)
{
    int    threadId = 0;
    bool   isSuccess = true;


    // Make sure Primary DNS is available
    ERROR_IF( g_LoopbackDnsServer == NULL, "Primary DNS Server is required!\n" );

    //////////////////////////////
    ////  Allocate DNS Server 2
    //////////////////////////////

    NN_LOG( "==============================\n" );
    NN_LOG( "==\n" );
    NN_LOG( "== STARTING SECOND DNS SERVER\n" );
    NN_LOG( "==\n" );
    NN_LOG( "== 127.0.0.1:9000\n" );
    NN_LOG( "==============================\n" );


    g_LoopbackDnsServer2 = new LoopbackDnsServer( "127.0.0.1", 9000, "127.0.0.1", 9000 );
    if ( g_LoopbackDnsServer2 == NULL )
    {
        NN_LOG( "Failed to allocate a server2 / LoopbackDnsServer\n" );
        goto out;
    }

    // Create a thread to run the Server in
    threadId = g_pThreadedTest->CreateThread( &StartLookbackDnsServer, (void *) g_LoopbackDnsServer2, kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }
    NN_LOG( "Successfully started [DNS-2] Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));


    //////////////////////////////
    //// Set NIFM to 2 DNS servers
    //////////////////////////////

    {
        nn::socket::SockAddrIn  localAddr[2];
        bool                isSuccess = true;
        int                 rc;

        // Tell Log what we are doing!
        NN_LOG( "=========================================\n" );
        NN_LOG( "==  Setting NIFM DNS Servers List to:\n" );
        NN_LOG( "==\n" );
        NN_LOG( "== 127.0.0.1:8053\n" );
        NN_LOG( "== 127.0.0.1:9000\n" );
        NN_LOG( "=========================================\n" );

        // Original DNS Server
        memset(&localAddr, 0, sizeof(localAddr));

        // [DNS1] Fill in Port and Family
        localAddr[0].sin_port    = nn::socket::InetHtons( 8053 );
        localAddr[0].sin_family     = nn::socket::Family::Af_Inet;

        // [DNS1] Turn IP Address into Network Address
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr[0].sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        // [DNS2] Fill in Port and Family
        localAddr[1].sin_port    = nn::socket::InetHtons( 9000 );
        localAddr[1].sin_family  = nn::socket::Family::Af_Inet;

        // [DNS2] Turn IP Address into Network Address
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr[1].sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );


        // Tell NIFM to use this new DNS Server Address
        nn::socket::PrivateDnsAddressArrayArgument arg;
        arg.count = 2;
        arg.addresses[0] = localAddr[0];
        arg.addresses[1] = localAddr[1];
        nn::socket::ResolverOption option = {
            .key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionPrivateKey::SetDnsServerAddressesPointer)),
            .type = nn::socket::ResolverOptionType::Pointer,
            .size = sizeof(arg),
            .data.pointerValue = reinterpret_cast<char*>(&arg)
        };
        rc = nn::socket::ResolverSetOption(option);
        ERROR_IF( rc != 0, "SetDnsAddressesPrivate() Failed but Success was expected!  rc = %d\n", rc );
    }

    return;

out:

    // If the (Second) loopback DNS server is allocated
    if ( g_LoopbackDnsServer2 != NULL )
    {
        g_LoopbackDnsServer2->Stop();
        delete g_LoopbackDnsServer2;
        g_LoopbackDnsServer2 = NULL;
    }

    return;
}


// Anonymous Namespace
namespace {

struct DnsRequestCount
{
    int64_t                   Acount;
    int64_t                   ATimerIdx;
    struct DnsRequestTimer    ATimer[20];
    int64_t                   Bcount;
    int64_t                   BTimerIdx;
    struct DnsRequestTimer    BTimer[20];
};

DnsRequestCount myDnsCount;

}; // End Anonymous Namespace



// Anonymous
namespace {

int
DNS_Callback_A( void *                  inArg,
                unsigned char *         dnsRqst,
                size_t &                dnsRqstLen,
                unsigned char *         dnsResp,
                size_t &                dnsRespLen,
                bool *                  isUDP,
                bool *                  shouldContinue )
{
    struct DNS_HEADER_REC *    pDnsHeader = NULL;
    uint16_t                   saveId = 0;
    uint16_t                   udpPort = 0;
    LoopbackDnsServer *        thisObj = reinterpret_cast<LoopbackDnsServer *>( inArg );

    NN_LOG( "DNS Server A hit !\n" );

    // If FIRST Query.. Save the Transaction ID
    if ( g_WorkVaruint32_t == 0 )
    {
        // Save DNS Query Transaction Id
        pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
        g_SavedTXID = nn::socket::InetNtohs( pDnsHeader->Id );
        NN_LOG( "===> Saved Transaction ID [%d]\n", g_SavedTXID );
    }

    // Keep a running total
    g_WorkVaruint32_t++;

    // Compare DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Make sure Transaction ID is not different
    if ( saveId != g_SavedTXID )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different DNS TXN ID returned!  (This TXNID): %d, Previous TXNID %d\n", saveId,  g_SavedTXID );
        NN_LOG( "============================\n" );
        g_DifferentTXIDFound = true;
    }

    // Get Senders UDP Source Port
    udpPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );

    // Initialize UDP Port if needed
    if ( g_SavedUDPPort == 0 )
    {
        g_SavedUDPPort = udpPort;
    }

    // Make sure UDP Port is not different
    if ( udpPort != g_SavedUDPPort )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different UDP Port!  (This Port): %d, Previous Port %d\n", udpPort,  g_SavedUDPPort );
        NN_LOG( "============================\n" );
        g_DifferentUDPPortFound = true;
    }

    myDnsCount.Acount++;

    // Save Finished timing
    if ( myDnsCount.ATimerIdx < 20 )
    {
        myDnsCount.ATimer[ myDnsCount.ATimerIdx ].endTime = GetMilliSeconds();
    }

    // Start new timing
    myDnsCount.ATimerIdx++;

    if ( myDnsCount.ATimerIdx < 20 )
    {
        myDnsCount.ATimer[ myDnsCount.ATimerIdx ].startTime = GetMilliSeconds();
    }

    *shouldContinue = false;

    return( 0 );
}


int
DNS_Callback_B( void *                  inArg,
                unsigned char *         dnsRqst,
                size_t &                dnsRqstLen,
                unsigned char *         dnsResp,
                size_t &                dnsRespLen,
                bool *                  isUDP,
                bool *                  shouldContinue )
{
    struct DNS_HEADER_REC *    pDnsHeader = NULL;
    uint16_t                   saveId = 0;
    uint16_t                   udpPort = 0;
    LoopbackDnsServer *        thisObj = reinterpret_cast<LoopbackDnsServer *>( inArg );

    NN_LOG( "DNS Server B hit !\n" );

    // If FIRST Query.. Save the Transaction ID
    if ( g_WorkVaruint32_t == 0 )
    {
        // Save DNS Query Transaction Id
        pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
        g_SavedTXID = nn::socket::InetNtohs( pDnsHeader->Id );
        NN_LOG( "===> Saved Transaction ID [%d]\n", g_SavedTXID );
    }

    // Keep a running total
    g_WorkVaruint32_t++;

    // Compare DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Make sure Transaction ID is not different
    if ( saveId != g_SavedTXID )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different DNS TXN ID returned!  (This TXNID): %d, Previous TXNID %d\n", saveId,  g_SavedTXID );
        NN_LOG( "============================\n" );
        g_DifferentTXIDFound = true;
    }

    // Compare UDP Source Port
    udpPort = nn::socket::InetNtohs( thisObj->recvAddr.sin_port );

    // Initialize UDP Port if needed
    if ( g_SavedUDPPort2 == 0 )
    {
        g_SavedUDPPort2 = udpPort;
    }

    // Make sure UDP Port is not different
    if ( udpPort != g_SavedUDPPort2 )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different UDP Port!  (This Port): %d, Previous Port %d\n", udpPort,  g_SavedUDPPort2 );
        NN_LOG( "============================\n" );
        g_DifferentUDPPortFound = true;
    }

    myDnsCount.Bcount++;

    // Save Finished timing
    if ( myDnsCount.BTimerIdx < 20 )
    {
        myDnsCount.BTimer[ myDnsCount.BTimerIdx ].endTime = GetMilliSeconds();
    }

    // Start new timing
    myDnsCount.BTimerIdx++;

    if ( myDnsCount.BTimerIdx < 20 )
    {
        myDnsCount.BTimer[ myDnsCount.BTimerIdx ].startTime = GetMilliSeconds();
    }

    *shouldContinue = false;

    return( 0 );
}

} // Anonymous


TEST(ApiUnit,DnsRequest_Two_Server_Timing)
{
    int                     idx = 0;
    bool                    isSuccess = true, rval = false;
    int64_t                 delta;
    nn::socket::AddrInfo    out_resp;
    nn::socket::AiErrno     out_error;
    nn::socket::HostEnt *   respHosts;

    // We require 2x DNS servers;
    ERROR_IF( g_LoopbackDnsServer2 == NULL, "This test requires 2x DNS Servers!  Failing\n" );

    // [DNS1] Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( DNS_Callback_A );

    // [DNS2] Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer2->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer2->SetDNSResponseCB( DNS_Callback_B );

    //*********************
    //* GHBN
    //*********************/

    memset( (char *) &myDnsCount, 0, sizeof( myDnsCount ) );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;
    g_SavedUDPPort  = 0;
    g_SavedUDPPort2 = 0;

    // Start Timer A
    myDnsCount.ATimerIdx = 0;
    myDnsCount.ATimer[ myDnsCount.ATimerIdx ].startTime = GetMilliSeconds();

    // Start Timer B
    myDnsCount.BTimerIdx = 0;
    myDnsCount.BTimer[ myDnsCount.BTimerIdx ].startTime = GetMilliSeconds();

    // Make Call
    NN_LOG( "Query: [GetHostByName][Google] Returns wrong answer\n" );
    rval = CallGetHostByName( "google.com" );
    ERROR_IF_AND_COUNT( rval == true, "GHBN received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    ////////////////
    // Results
    ///////////////

    NN_LOG( "=========================================\n" );
    NN_LOG( "A count    : %ld\n", myDnsCount.Acount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.ATimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.ATimer[ idx ].endTime -
                     myDnsCount.ATimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.ATimer[ idx ].startTime,
                    myDnsCount.ATimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "B count    : %ld\n", myDnsCount.Bcount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.BTimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.BTimer[ idx ].endTime -
                     myDnsCount.BTimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.BTimer[ idx ].startTime,
                    myDnsCount.BTimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "=========================================\n" );

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GHBN: Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GHBN: Different UDP Port found during processing!\n" );


    //*********************
    //* GETADDRINFO
    //*********************/

    memset( (char *) &myDnsCount, 0, sizeof( myDnsCount ) );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;
    g_SavedUDPPort  = 0;
    g_SavedUDPPort2 = 0;

    // Start Timer A
    myDnsCount.ATimerIdx = 0;
    myDnsCount.ATimer[ myDnsCount.ATimerIdx ].startTime = GetMilliSeconds();

    // Start Timer B
    myDnsCount.BTimerIdx = 0;
    myDnsCount.BTimer[ myDnsCount.BTimerIdx ].startTime = GetMilliSeconds();

    // Make Call
    NN_LOG( "Query: [GetAddrInfo][Google] Returns wrong answer\n" );
    rval = CallGetAddrInfo( "google.com", &out_resp, &out_error );
    ERROR_IF_AND_COUNT( rval == true, "GAI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    ////////////////
    // Results
    ///////////////

    NN_LOG( "=========================================\n" );
    NN_LOG( "A count    : %ld\n", myDnsCount.Acount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.ATimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.ATimer[ idx ].endTime -
                     myDnsCount.ATimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.ATimer[ idx ].startTime,
                    myDnsCount.ATimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "B count    : %ld\n", myDnsCount.Bcount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.BTimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.BTimer[ idx ].endTime -
                     myDnsCount.BTimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.BTimer[ idx ].startTime,
                    myDnsCount.BTimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "=========================================\n" );

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GAI: Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GAI: Different UDP Port found during processing!\n" );

    //*********************
    //* GHBA
    //*********************/

    memset( (char *) &myDnsCount, 0, sizeof( myDnsCount ) );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;
    g_SavedUDPPort  = 0;
    g_SavedUDPPort2 = 0;

    // Start Timer A
    myDnsCount.ATimerIdx = 0;
    myDnsCount.ATimer[ myDnsCount.ATimerIdx ].startTime = GetMilliSeconds();

    // Start Timer B
    myDnsCount.BTimerIdx = 0;
    myDnsCount.BTimer[ myDnsCount.BTimerIdx ].startTime = GetMilliSeconds();

    // Make Call
    NN_LOG( "Query: [GetHostByAddr][ERoot] Returns wrong answer\n" );
    rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &respHosts );
    ERROR_IF_AND_COUNT( rval == true, "GHBA received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    ////////////////
    // Results
    ///////////////

    NN_LOG( "=========================================\n" );
    NN_LOG( "A count    : %ld\n", myDnsCount.Acount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.ATimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.ATimer[ idx ].endTime -
                     myDnsCount.ATimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.ATimer[ idx ].startTime,
                    myDnsCount.ATimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "B count    : %ld\n", myDnsCount.Bcount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.BTimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.BTimer[ idx ].endTime -
                     myDnsCount.BTimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.BTimer[ idx ].startTime,
                    myDnsCount.BTimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "=========================================\n" );

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GHBA: Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GHBA: Different UDP Port found during processing!\n" );

    //*********************
    //* GNI
    //*********************/

    memset( (char *) &myDnsCount, 0, sizeof( myDnsCount ) );

    // Initialize running total
    g_WorkVaruint32_t = 0;
    g_DifferentTXIDFound = false;
    g_DifferentUDPPortFound = false;
    g_SavedUDPPort  = 0;
    g_SavedUDPPort2 = 0;

    // Start Timer A
    myDnsCount.ATimerIdx = 0;
    myDnsCount.ATimer[ myDnsCount.ATimerIdx ].startTime = GetMilliSeconds();

    // Start Timer B
    myDnsCount.BTimerIdx = 0;
    myDnsCount.BTimer[ myDnsCount.BTimerIdx ].startTime = GetMilliSeconds();

    // Make Call
    NN_LOG( "Query: [GetNameInfo][ERoot] Returns wrong answer\n" );
    rval = CallGetNameInfo( "192.203.230.10" );
    ERROR_IF_AND_COUNT( rval == true, "GNI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );

    ////////////////
    // Results
    ///////////////

    NN_LOG( "=========================================\n" );
    NN_LOG( "A count    : %ld\n", myDnsCount.Acount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.ATimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.ATimer[ idx ].endTime -
                     myDnsCount.ATimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.ATimer[ idx ].startTime,
                    myDnsCount.ATimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "B count    : %ld\n", myDnsCount.Bcount );

    for( idx = 0; idx < 20; idx++ )
    {
        if ( myDnsCount.BTimer[ idx ].endTime > 0 )
        {
            delta = (myDnsCount.BTimer[ idx ].endTime -
                     myDnsCount.BTimer[ idx ].startTime );
            NN_LOG( "[%d]: Start Timer: %ldms, End Timer: %ldms, Elasped: %dms\n",
                    idx,
                    myDnsCount.BTimer[ idx ].startTime,
                    myDnsCount.BTimer[ idx ].endTime,
                    delta );

            ERROR_IF_AND_COUNT( delta > 3000, "DNS Operation took longer than [3] seconds!  Actual=<%d>, Failed!!\n", delta );

        }
    }

    NN_LOG( "=========================================\n" );

    ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "GNI: Different DNS Transaction Id found during processing!\n" );
    ERROR_IF_AND_COUNT( g_DifferentUDPPortFound == true, "GNI: Different UDP Port found during processing!\n" );

out:

    return;

}   // NOLINT(impl/function_size)


// Anonymous
namespace {

int
DNS_Retry_Callback_A( void *                  inArg,
                      unsigned char *         dnsRqst,
                      size_t &                dnsRqstLen,
                      unsigned char *         dnsResp,
                      size_t &                dnsRespLen,
                      bool *                  isUDP,
                      bool *                  shouldContinue )
{
    LoopbackDnsServer *   thisObj = reinterpret_cast<LoopbackDnsServer *>( inArg );
    struct DNS_HEADER_REC *    pDnsHeader;
    uint16_t                   saveId;

    // If FIRST Query.. Save the Transaction ID
    if ( g_WorkVaruint32_t == 0 )
    {
        // Save DNS Query Transaction Id
        pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
        g_SavedTXID = nn::socket::InetNtohs( pDnsHeader->Id );
        NN_LOG( "===> Saved Transaction ID [%d]\n", g_SavedTXID );
    }

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Make sure Transaction ID is not different
    if ( saveId != g_SavedTXID )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different TXN ID returned!!  (This TXNID): %d, Previous TXNID: %d\n", saveId,  g_SavedTXID );
        NN_LOG( "============================\n" );
        g_DifferentTXIDFound = true;
    }

    // Keep a running total
    g_WorkVaruint32_t++;
    g_WorkVaruint32_t3 = SERVER_A;

    NN_LOG( "========> IN Callback A!  W1=%d, W2=%d <===========\n", g_WorkVaruint32_t, g_WorkVaruint32_t2);

    if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
    {
        return( g_LoopbackDnsServer->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
    }
    else
    {
        *shouldContinue = false;
    }

    return( 0 );
}


static int
DNS_Retry_Callback_B( void *                  inArg,
                      unsigned char *         dnsRqst,
                      size_t &                dnsRqstLen,
                      unsigned char *         dnsResp,
                      size_t &                dnsRespLen,
                      bool *                  isUDP,
                      bool *                  shouldContinue )
{
    LoopbackDnsServer *        thisObj = (LoopbackDnsServer *) inArg;
    struct DNS_HEADER_REC *    pDnsHeader;
    uint16_t                   saveId;

    // If FIRST Query.. Save the Transaction ID
    if ( g_WorkVaruint32_t == 0 )
    {
        // Save DNS Query Transaction Id
        pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
        g_SavedTXID = nn::socket::InetNtohs( pDnsHeader->Id );
        NN_LOG( "===> Saved Transaction ID [%d]\n", g_SavedTXID );
    }

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) &dnsRqst[0];
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Make sure Transaction ID is not different
    if ( saveId != g_SavedTXID )
    {
        NN_LOG( "============================\n" );
        NN_LOG( " FAILED!  Different TXN ID returned!!  (This TXNID): %d, Previous TXNID: %d\n", saveId,  g_SavedTXID );
        NN_LOG( "============================\n" );
        g_DifferentTXIDFound = true;
    }

    // Keep a running total
    g_WorkVaruint32_t++;
    g_WorkVaruint32_t3 = SERVER_B;

    NN_LOG( "========> IN Callback B!  W1=%d, W2=%d <===========\n", g_WorkVaruint32_t, g_WorkVaruint32_t2);

    if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
    {
        return( g_LoopbackDnsServer2->MakeDnsResponseCB( thisObj, dnsRqst, dnsRqstLen, dnsResp, dnsRespLen, isUDP, shouldContinue )  );
    }
    else
    {
        *shouldContinue = false;
    }

    return( 0 );
}

} // Anonymous



TEST(ApiUnit,DnsRequest_Two_Server_Retry)
{
    bool                    isSuccess = true, rval = false;
    nn::socket::AddrInfo    out_resp;
    nn::socket::AiErrno     out_error;
    nn::socket::HostEnt *   respHosts;

    // We require 2x DNS servers;
    ERROR_IF( g_LoopbackDnsServer2 == NULL, "This test requires 2x DNS Servers!  Failing\n" );

    // [DNS1] Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer->SetDNSResponseCB( DNS_Retry_Callback_A );

    // [DNS1] Set DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_Google_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_Google_com_simple );

    // [DNS2] Set DNS Rqst/Resp Parsers
    g_LoopbackDnsServer2->SetDNSRequestCB( g_LoopbackDnsServer->MakeDnsRequestCB );
    g_LoopbackDnsServer2->SetDNSResponseCB( DNS_Retry_Callback_B );

    // [DNS2] Set DNS Response
    g_LoopbackDnsServer2->UDPResponse = (unsigned char *) DnsRequest_Google_com_simple;
    g_LoopbackDnsServer2->UDPResponseLen = sizeof( DnsRequest_Google_com_simple );

    //*********************
    //* GHBN
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear Counter
        g_WorkVaruint32_t = 0;
        g_DifferentTXIDFound = false;

        // Make Call
        NN_LOG( "Query: [GetHostByName][Google] Returns wrong answer\n" );
        rval = CallGetHostByName( "google.com" );

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "Different DNS Transaction Id found during processing!\n" );

        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == SERVER_A )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == SERVER_B )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GHBN received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }
    }

    NN_LOG( "There were: [%d] GHBN calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GETADDRINFO
    //*********************/

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear Counter
        g_WorkVaruint32_t = 0;
        g_DifferentTXIDFound = false;

        // Make Call
        NN_LOG( "Query: [CallGetAddrInfo][Google] Returns wrong answer\n" );
        rval = CallGetAddrInfo( "google.com", &out_resp, &out_error );

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "Different DNS Transaction Id found during processing!\n" );

        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == SERVER_A )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == SERVER_B )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GAI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }
    }

    NN_LOG( "There were: [%d] GAI calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GHBA
    //*********************/

    // [DNS1] Set DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // [DNS2] Set DNS Response
    g_LoopbackDnsServer2->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer2->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear Counter
        g_WorkVaruint32_t = 0;
        g_DifferentTXIDFound = false;

        // Make Call
        NN_LOG( "Query: [GetHostByAddr][ERoot] Returns wrong answer\n" );
        rval = CallGetHostByAddr( "192.203.230.10", 4, nn::socket::Family::Af_Inet, &respHosts );

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "Different DNS Transaction Id found during processing!\n" );

        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == SERVER_A )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == SERVER_B )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GHBA received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }
    }

    NN_LOG( "There were: [%d] GHBA calls to the DNS server\n", g_WorkVaruint32_t );

    //*********************
    //* GNI
    //*********************/

    // [DNS1] Set DNS Response
    g_LoopbackDnsServer->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // [DNS2] Set DNS Response
    g_LoopbackDnsServer2->UDPResponse = (unsigned char *) DnsRequest_ERoot_com_simple;
    g_LoopbackDnsServer2->UDPResponseLen = sizeof( DnsRequest_ERoot_com_simple );

    // Initialize running total
    g_WorkVaruint32_t  = 0;
    g_WorkVaruint32_t2 = 0;

    // For I = 1 to 7
    for( g_WorkVaruint32_t2 = 1; g_WorkVaruint32_t2 < 9; g_WorkVaruint32_t2++ )
    {
        NN_LOG( "Testing Iteration: [%d]\n", g_WorkVaruint32_t2 );

        // Clear Counter
        g_WorkVaruint32_t = 0;
        g_DifferentTXIDFound = false;

        // Make Call
        NN_LOG( "Query: [GetHostByAddr][ERoot] Returns wrong answer\n" );
        rval = CallGetNameInfo( "192.203.230.10" );

        ERROR_IF_AND_COUNT( g_DifferentTXIDFound == true, "Different DNS Transaction Id found during processing!\n" );

        if ( rval == true )
        {
            if ( g_WorkVaruint32_t == g_WorkVaruint32_t2 )
            {
                if (  g_WorkVaruint32_t3 == SERVER_A )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 1: break;
                    case 3: break;
                    case 5: break;
                    case 7: break;
                    case 9: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server A processed a DNS record for Server B! W1=%d, W1=%d, w3=%d\n",
                                         g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                   }
                }
                else if ( g_WorkVaruint32_t3 == SERVER_B )
                {
                    switch( g_WorkVaruint32_t2 )
                    {
                    case 2: break;
                    case 4: break;
                    case 6: break;
                    case 8: break;
                    default:     NN_LOG( "====== FAILED:  Server DNS Server B processed a DNS record for Server A! W1=%d, W1=%d, w3=%d\n",
                                        g_WorkVaruint32_t,  g_WorkVaruint32_t2,  g_WorkVaruint32_t3 );

                                 ERROR_IF_AND_COUNT( rval == true, "Failed while processing!\n" );
                                 break;
                    }
                }

                NN_LOG( "=============> MATCHED <=============\n" );
            }
            else
            {
                ERROR_IF_AND_COUNT( rval == true, "GNI received a DNS ANSWER for the DNS Question - Succeeded but should have failed!" );
            }
        }
    }

    NN_LOG( "There were: [%d] GNI calls to the DNS server\n", g_WorkVaruint32_t );

out:

    return;

}  // NOLINT(impl/function_size)



TEST(ApiUnit,DnsRequest_Teardown)
{
    TeardownTesting();
}


#else // Windows

TEST(ApiUnit,DnsRequest_Window)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
