﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_EventCommon.h>
#include <nn/util/util_FormatString.h>

#include <mutex>
#include <atomic>
#include <cstdlib>
#include <cstring>      // size_t
#include <algorithm>    // std::min

#include "Complex/testNet_SelectUnitData.h" // declare SelectUnitData, Server, and Client; TODO: use interfaces
#include "Complex/testNet_UnitCommonSocket.h"
#include "Complex/testNet_SelectUnitServer.h"
#include "Complex/testNet_SelectUnitClient.h"
#include "Complex/testNet_UnitCommon.h"

namespace NATF {
namespace API {

char* StringDuplicate(const char* pStringIn)
{
    char* pDuplicatedString = NULL;
    size_t lengthOfString = 0;

    if ( NULL == pStringIn )
    {
        goto bail;
    };

    lengthOfString = strlen(pStringIn) + 1;
    if ( NULL == (pDuplicatedString = (char*) malloc(lengthOfString)))
    {
        goto bail;
    };

    memcpy(pDuplicatedString, pStringIn, lengthOfString);

bail:
    return pDuplicatedString;
}

std::mutex        g_traceLoggerLock;

TraceLogger::TraceLogger(const char* function, const char* file, unsigned line, const char* fmt, ...)
{
    std::lock_guard<std::mutex> lock(g_traceLoggerLock);

    // NO UNIT TEST TRACE
    char messageBuffer[512];
    va_list argptr;
    va_start(argptr, fmt);
    nn::util::VSNPrintf(messageBuffer, sizeof(messageBuffer), fmt, argptr);
    NN_LOG("%p %s %s\n", nn::os::GetCurrentThread(),
           function,
           messageBuffer);
    va_end(argptr);
};

TraceLogger::~TraceLogger()
{
    // NO UNIT TEST TRACE
    //printf("%p <\n", this);
};

/**
 * Reference Count Object implementation
 */
LockedReferenceCountObjectImpl::LockedReferenceCountObjectImpl(const char* name) NN_NOEXCEPT :
    m_lockedReferenceCount(1)
{
    //UNIT_TEST_TRACE("");
    nn::os::InitializeMutex(&m_lockedReferenceCountAccessLock, true, 0);

    memset(m_LockedReferenceCountObjectName,
           '\0',
           sizeof(m_LockedReferenceCountObjectName));

    if ( NULL != name )
    {
        memcpy(m_LockedReferenceCountObjectName,
               name,
               std::min<size_t>(strlen(name),
                                sizeof(m_LockedReferenceCountObjectName) - 1));
    };
}

LockedReferenceCountObjectImpl::~LockedReferenceCountObjectImpl()
{
    //UNIT_TEST_TRACE("");
    nn::os::FinalizeMutex(&m_lockedReferenceCountAccessLock);
    NN_ASSERT(0 == m_lockedReferenceCount);
}

int LockedReferenceCountObjectImpl::addReference()
{
    //UNIT_TEST_TRACE("");
    int rc = -1;
    nn::os::LockMutex(&m_lockedReferenceCountAccessLock);
    {
        rc = (++m_lockedReferenceCount);
    }
    nn::os::UnlockMutex(&m_lockedReferenceCountAccessLock);

    return rc;
}

int LockedReferenceCountObjectImpl::releaseReference()
{
    //UNIT_TEST_TRACE("");
    int rc = -1;
    nn::os::LockMutex(&m_lockedReferenceCountAccessLock);
    {
        rc = (--m_lockedReferenceCount);
    }
    nn::os::UnlockMutex(&m_lockedReferenceCountAccessLock);

    if ( 0 == rc )
    {
        delete this;
    };

    return rc;
}

/**
 * IValidator
 */
IValidator::IValidator()
{
    //UNIT_TEST_TRACE("");
}

IValidator::~IValidator()
{
    //UNIT_TEST_TRACE("");
}

/**
 * SimpleValidator class
 */

SimpleValidator::SimpleValidator() :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    m_DidFail(false),
    m_FileName(StringDuplicate("")),
    m_LineNumber(0),
    m_FunctionName(StringDuplicate("")),
    m_Message(StringDuplicate(""))
{
    //UNIT_TEST_TRACE("");
    nn::os::InitializeMutex(&m_DataAccessLock, true, 0);
}

bool SimpleValidator::TryFail(const char* fileNameIn,
                              unsigned int lineNumberIn,
                              const char* functionNameIn,
                              const bool conditionIn,
                              const char* messageFormatIn,
                              ... )
{
    //UNIT_TEST_TRACE("");
    bool rc = false;
    if (conditionIn == false)
    {
        return conditionIn;
    };

    nn::os::LockMutex(&m_DataAccessLock);
    {
        char messageBuffer[512];

        // first failure always wins
        if (m_DidFail == true)
        {
            rc = m_DidFail;
            goto bail;
        };

        if (NULL != m_FileName)
        {
            free(m_FileName);
            if (fileNameIn == NULL)
            {
                m_FileName = StringDuplicate("");
            }
            else
            {
                m_FileName = NULL;
            }
        };

        if (NULL != m_FunctionName)
        {
            free(m_FunctionName);
            if (functionNameIn == NULL)
            {
                m_FunctionName = StringDuplicate("");
            }
            else
            {
                m_FunctionName = NULL;
            }
        };

        if (NULL != m_Message)
        {
            free(m_Message);
            if (messageFormatIn == NULL)
            {
                m_Message = StringDuplicate("");
            }
            else
            {
                m_Message = NULL;
            };
        };

        if (NULL != fileNameIn)
        {
            m_FileName = StringDuplicate(fileNameIn);
        };

        m_LineNumber = lineNumberIn;

        if (NULL != functionNameIn)
        {
            m_FunctionName = StringDuplicate(functionNameIn);
        };

        if (NULL != messageFormatIn)
        {
            va_list argptr;
            va_start(argptr, messageFormatIn);
            nn::util::VSNPrintf(messageBuffer,
                                sizeof(messageBuffer),
                                messageFormatIn,
                                argptr);
            va_end(argptr);
            m_Message = StringDuplicate(messageBuffer);
        };


        NN_LOG("%p FAIL: %s:%d %s %s\n", nn::os::GetCurrentThread(),
               fileNameIn,
               lineNumberIn,
               functionNameIn,
               messageBuffer);

        rc = m_DidFail = conditionIn;
    };
bail:
    nn::os::UnlockMutex(&m_DataAccessLock);
    return rc;
}


bool SimpleValidator::DidSucceed() const
{
    bool rc = false;
    nn::os::LockMutex(&m_DataAccessLock);
    {
        rc = m_DidFail;
    }
    nn::os::UnlockMutex(&m_DataAccessLock);
    return !rc;
}

bool SimpleValidator::DidFail() const
{
    bool rc = false;
    nn::os::LockMutex(&m_DataAccessLock);
    {
        rc = m_DidFail;
    }
    nn::os::UnlockMutex(&m_DataAccessLock);
    return rc;
}

const char* SimpleValidator::FileName() const
{
    return m_FileName;
}

const int SimpleValidator::Line() const
{
    return m_LineNumber;
}

const char* SimpleValidator::FunctionName() const
{
    return m_FunctionName;
}

const char* SimpleValidator::Message() const
{
    return m_Message;
}

SimpleValidator::~SimpleValidator()
{
    if ( NULL != m_FileName )
    {
        free(m_FileName);
        m_FileName = NULL;
    };

    if ( NULL == m_FunctionName )
    {
        free(m_FunctionName);
        m_FunctionName = NULL;
    };

    if ( NULL == m_Message )
    {
        free(m_Message);
        m_Message = NULL;
    };

    nn::os::FinalizeMutex(&m_DataAccessLock);
}

/**
 * Unit Test Thread Base class implementation & helpers
 */

/**
 * @brief the UnitTestThreadBaseArgument class is used to wrap the argument along with a semaphore
 * this provides a simple mechanism whereby the caller is blocked until the thread is running safely.
 */
class UnitTestThreadBaseArgument :
        public virtual LockedReferenceCountObjectImpl
{
private:
    /** @brief semaphore */
    nn::os::SemaphoreType   m_Semaphore;

    /** the thread */
    UnitTestThreadBase* m_Thread;

    /** dtor */
    ~UnitTestThreadBaseArgument()
    {
        //UNIT_TEST_TRACE("");
        nn::os::FinalizeSemaphore(&m_Semaphore);
    };

protected:
    friend class UnitTestThreadBase;

    /** thread calls in to signal it is running */
    void SignalCallerToStart(UnitTestThreadBase* & pThreadOut)
    {
        //UNIT_TEST_TRACE("");
        NN_ASSERT(NULL != m_Thread);

        pThreadOut = m_Thread;
        // call early initialize
        if (false == m_Thread->EarlyBlockingInitialize(m_Semaphore))
        {
            nn::os::ReleaseSemaphore(&m_Semaphore);
        };
    }

public:
    /** create a thread and allow blocking */
    UnitTestThreadBaseArgument(const char* name, UnitTestThreadBase* pThread) :
        LockedReferenceCountObjectImpl(name),
        m_Thread(pThread)
    {
        //UNIT_TEST_TRACE(LockedReferenceCountObjectImpl::m_LockedReferenceCountObjectName);
        m_Thread->addReference();
        // initialize count to zero, max of 1; the
        // calling thread waits until the thread is started;
        // when it is it calls SignalCallerToStart
        nn::os::InitializeSemaphore( &m_Semaphore, 0, 1 );
    };

    /** wait until thread has started */
    void WaitUntilStarted()
    {
        //UNIT_TEST_TRACE(LockedReferenceCountObjectImpl::m_LockedReferenceCountObjectName);
        nn::os::AcquireSemaphore(&m_Semaphore);
        m_Thread->releaseReference();
        m_Thread = NULL;
        // caller needs to releaseReference
    };
};

bool UnitTestThreadBase::Start()
{
    //UNIT_TEST_TRACE(m_UnitTestThreadBaseName);
    bool rc = false;
    nn::Result result;
    UnitTestThreadBaseArgument* argument = NULL;

    nn::os::LockMutex(&m_BaseAccessLock);
    {
        if (m_State != STATE_INITIALIZED)
        {
            nn::os::UnlockMutex(&m_BaseAccessLock);
            goto bail;
        }
    }
    nn::os::UnlockMutex(&m_BaseAccessLock);

    if (NULL == (argument = new UnitTestThreadBaseArgument(m_UnitTestThreadBaseName, this)))
    {
        NN_ASSERT(false);
        goto bail;
    };

    m_ThreadStackAlignedPointer = m_ThreadStack;
    while (reinterpret_cast<uintptr_t>(m_ThreadStackAlignedPointer) % nn::os::ThreadStackAlignment!= 0)
    {
        m_ThreadStackAlignedPointer += 1;
    };

    nn::os::LockMutex(&m_BaseAccessLock);
    {
        m_State = STATE_STARTING;
    }
    nn::os::UnlockMutex(&m_BaseAccessLock);

    // create it first
    result = nn::os::CreateThread(&m_Thread,
                                  s_Run,
                                  argument,
                                  m_ThreadStackAlignedPointer,
                                  UnitTestThreadStackSize - 4096,
                                  nn::os::DefaultThreadPriority);
    if (result.IsFailure())
    {
        NN_ASSERT(false);
        goto bail;
    };

    // start it second
    nn::os::StartThread(&m_Thread);

    // wait until the thread has started to return to caller
    rc = true;
    argument->WaitUntilStarted();

bail:
    if (NULL != argument)
    {
        argument->releaseReference();
        argument = NULL;
    }
    return rc;
}

bool UnitTestThreadBase::Stop()
{
    //UNIT_TEST_TRACE(m_UnitTestThreadBaseName);
    bool rc = false;
    nn::os::LockMutex(&m_BaseAccessLock);
    {
        if (m_State != STATE_STARTING || m_State != STATE_RUNNING)
        {
            nn::os::UnlockMutex(&m_BaseAccessLock);
            goto bail;
        };
        m_State = STATE_FINISHING;
    }
    nn::os::UnlockMutex(&m_BaseAccessLock);

bail:
    return rc;
}

UnitTestThreadBase::State UnitTestThreadBase::GetState()
{
    //UNIT_TEST_TRACE(m_UnitTestThreadBaseName);
    State state;
    nn::os::LockMutex(&m_BaseAccessLock);
    {
        state = m_State;
    }
    nn::os::UnlockMutex(&m_BaseAccessLock);
    return state;
}
//
// Run implemented by client
//

bool UnitTestThreadBase::WaitForDone()
{
    //UNIT_TEST_TRACE(m_UnitTestThreadBaseName);
    bool rc = true;
    nn::os::WaitEvent(&m_finishedEvent);
    return rc;
}

bool UnitTestThreadBase::WaitForDoneWithTimeout()
{
    //UNIT_TEST_TRACE(m_UnitTestThreadBaseName);
    bool rc = true;
    rc = nn::os::TimedWaitEvent(&m_finishedEvent, nn::TimeSpan::FromMilliSeconds(m_WaitForDoneTimeout));
    return rc;
}

bool UnitTestThreadBase::EarlyBlockingInitialize(nn::os::SemaphoreType& blockingSemaphore)
{
    //UNIT_TEST_TRACE(m_UnitTestThreadBaseName);
    return false;
}

// the below code produces an erroneous warning which was fixed in later versions
// of visual studio (i.e. 2015 SP1)  but not others
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable : 4589)
#pragma warning(disable : 4702)
#endif

UnitTestThreadBase::UnitTestThreadBase(const char* name, uint64_t waitForDoneTimeout) NN_NOEXCEPT :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    m_State(STATE_INITIALIZED),
    m_WaitForDoneTimeout(waitForDoneTimeout)
{
    //UNIT_TEST_TRACE(name);
    memset(m_UnitTestThreadBaseName, '\0', sizeof(m_UnitTestThreadBaseName));
    if ( NULL != name )
    {
        memcpy(m_UnitTestThreadBaseName, name, std::min<size_t>(strlen(name), sizeof(m_UnitTestThreadBaseName) - 1));
    };

    memset(m_ThreadStack, '\0', sizeof(m_ThreadStack));
    nn::os::InitializeMutex(&m_BaseAccessLock, true, 0);
    nn::os::InitializeEvent(&m_finishedEvent, false, nn::os::EventClearMode_ManualClear);
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

UnitTestThreadBase::~UnitTestThreadBase()
{
    //UNIT_TEST_TRACE(m_UnitTestThreadBaseName);
    Stop();
    nn::os::FinalizeMutex(&m_BaseAccessLock);
    nn::os::DestroyThread(&m_Thread);
    nn::os::FinalizeEvent(&m_finishedEvent);
}

void UnitTestThreadBase::s_Run(void* pArgument)
{
    //UNIT_TEST_TRACE("");
    UnitTestThreadBaseArgument* threadArgument = static_cast<UnitTestThreadBaseArgument*>(pArgument);
    NN_ASSERT(NULL != threadArgument);

    UnitTestThreadBase* pThread = NULL;
    // get the pointer to the thread unblock start caller

    threadArgument->SignalCallerToStart(pThread);
    NN_ASSERT(NULL != pThread);

    nn::os::LockMutex(&pThread->m_BaseAccessLock);
    {
        pThread->m_State = STATE_RUNNING;
    }
    nn::os::UnlockMutex(&pThread->m_BaseAccessLock);

    //UNIT_TEST_TRACE(pThread->m_UnitTestThreadBaseName);

    // run the thread-provided run function
    pThread->Run();

    // broadcast that the thread is finished running
    nn::os::SignalEvent(&pThread->m_finishedEvent);

    pThread = NULL;
}

/**
 * Test timeout and cleanup thread class
 */
TestManagerThread::TestManagerThread(SelectUnitData* pUnitData, SelectUnitServer* pServer, SelectUnitClient* pClient) :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    UnitTestThreadBase(__FUNCTION__, (uint64_t)500),
    m_pUnitData(pUnitData),
    m_pServer(pServer),
    m_pClient(pClient)
{
    //UNIT_TEST_TRACE("");
    NN_ASSERT(m_pUnitData != NULL);
    NN_ASSERT(m_pServer != NULL);
    NN_ASSERT(m_pClient != NULL);
    m_pUnitData->addReference();
    m_pServer->addReference();
    m_pClient->addReference();

    nn::os::InitializeMutex(&m_managerAccessLock, true, 0);

    std::list<SocketContainer> socketContainerList;
    const SelectUnitData* pConstUnitData = m_pUnitData;
    pConstUnitData->GetSocketContainerListCopy(socketContainerList);
};

TestManagerThread::~TestManagerThread()
{
    //UNIT_TEST_TRACE("");

    nn::os::FinalizeMutex(&m_managerAccessLock);
    m_pClient->releaseReference();
    m_pClient = NULL;
    m_pServer->releaseReference();
    m_pServer = NULL;
    m_pUnitData->releaseReference();
    m_pUnitData = NULL;
};

bool TestManagerThread::WaitForDone()
{
    //UNIT_TEST_TRACE("");
    bool rc = true;
    nn::os::WaitEvent(&m_finishedEvent);
    //rc = nn::os::TimedWaitEvent(&m_finishedEvent, nn::TimeSpan::FromMilliSeconds(m_WaitForDoneTimeout));
    return rc;
}

void TestManagerThread::Run()
{
    //UNIT_TEST_TRACE("");
    bool rc = false;

    // start the server thread
    if (false == (rc = m_pServer->Start()))
    {
        goto bail;
    }
    else if (false == (rc = m_pClient->Start()))
    {
        m_pServer->Stop();
        goto bail;
    };

    if (false == (rc = m_pServer->WaitForDone()))
    {
        m_pClient->Stop();
    };
    UNIT_TEST_TRACE("Server done");

    if (false == (rc = m_pClient->WaitForDone()))
    {
        goto bail;
    };
    UNIT_TEST_TRACE("Client done");


bail:
    return;
}

}}
