﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_ApiPrivate.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nn/os/os_Thread.h>

#include <cstdlib> // For malloc
#include <new>     // For placement new
#include <string>

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"
#include "Complex/testNet_SocketShutdownAll.h"

 /**
   @brief This test ensures the SocketShutdownAll API and related API's are working correctly.

   @details This test create 4 threads, two clients and two servers which are paired together. one such pair are created with SocketExempt and are thus exempt from SocketShutdownAll.
            The clients and servers start to exchange ClientSendNum (5000) packets until half way through the exchange  SocketShutdownAll() is called and the non exempt sockets shut down.
            The non exempt client and server should shut down gracefully, the exempt client and server should continue sending until completion.
            The test ensures this process happens as expected.
 */

namespace
{
#define NN_LOG_CLIENT(format, ...) \
    do \
    {  \
        if (thread->GetExempt()) \
        { \
            NN_LOG("client exempt: " format, ##__VA_ARGS__); \
        } \
        else \
        { \
            NN_LOG("client: " format, ##__VA_ARGS__); \
        } \
    } while ( NN_STATIC_CONDITION(false) )
#define NN_LOG_SERVER(format, ...) \
    do \
    {  \
        if (thread->GetExempt()) \
        { \
            NN_LOG("server exempt: " format, ##__VA_ARGS__); \
        } \
        else \
        { \
            NN_LOG("server: " format, ##__VA_ARGS__); \
        } \
    } while ( NN_STATIC_CONDITION(false) )

    const int MessageBufferSize = 256;
    const int ClientSendNum = 5000;
    const char TestMessage[MessageBufferSize] = "Test Message";
    const char ServerAddr[] = "127.0.0.1";
    const char ExitMessage[] = "exit";
    int g_ExpectedShutdownSocketCount = 0;
    enum ThreadUsage : uint32_t
    {
        ThreadUsage_Server,
        ThreadUsage_Client,
        ThreadUsage_ServerExempt,
        ThreadUsage_ClientExempt,
    };

    void RunServerProcess(void * pData)
    {

        TestThread * thread = reinterpret_cast<TestThread *>(pData);
        nn::socket::SockAddrIn saServer = { 0 };
        char        clientMessage[MessageBufferSize] = { 0 };
        int         socketDescriptor = -1;
        int         clientSocketDescriptor = -1;
        ssize_t     readByteCount = 0;
        nn::socket::SockAddrIn sa = { 0 };
        nn::socket::SockLenT   saLen = sizeof(sa);


        NN_LOG_SERVER("startup\n");
        // Open a shutdown exempt socket if this is the shutdown exempt version of the server thread, otherwise open a non exempt socket
        if (thread->GetExempt())
        {
            socketDescriptor = nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        }
        else
        {
            socketDescriptor = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            ++g_ExpectedShutdownSocketCount;
        }

        if (socketDescriptor < 0)
        {
            NN_LOG_SERVER("Socket failed (error %d)\n", nn::socket::GetLastError());
            thread->SetExitReason(TestThread::ExitReason_Error);
            return;
        }

        saServer.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
        saServer.sin_port = nn::socket::InetHtons(0);
        saServer.sin_family = nn::socket::Family::Af_Inet;
        // bind the socket to an open port
        if (nn::socket::Bind(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saServer), sizeof(saServer)) < 0)
        {
            NN_LOG_SERVER("Bind failed (error %d)\n", nn::socket::GetLastError());
            thread->SetExitReason(TestThread::ExitReason_Error);
            nn::socket::Close(socketDescriptor);
            return;
        }
        // get the port the socket was bound to and set our server port to this port.
        // the main thread will need to grab this and provide the information to the clients.
        if (nn::socket::GetSockName(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&sa), &saLen) < 0)
        {
            NN_LOG_SERVER("GetSockName failed (error %d)\n", nn::socket::GetLastError());
            thread->SetExitReason(TestThread::ExitReason_Error);
            nn::socket::Close(socketDescriptor);
            return;
        }

        thread->SetServerPort(nn::socket::InetNtohs(sa.sin_port));
        NN_LOG_SERVER("listening for incoming messages on port %d\n", static_cast<int>(nn::socket::InetNtohs(sa.sin_port)));

        // start listening for incoming connections.
        nn::socket::Listen(socketDescriptor, 4);
        // server thread now ready for the client to start.
        thread->SetReady(true);

        nn::socket::SockAddrIn saClientAddress = { 0 };
        nn::socket::SockLenT clientAddressSize = sizeof(saClientAddress);

        // accept incoming client connection
        if ((clientSocketDescriptor = nn::socket::Accept(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saClientAddress), &clientAddressSize)) < 0)
        {
            NN_LOG_SERVER("Accept failed (error %d)\n", nn::socket::GetLastError());
            thread->SetExitReason(TestThread::ExitReason_Error);
            nn::socket::Close(socketDescriptor);
            return;
        }

        // if server is shutdown exempt, set the accepted socket as exempt as well.
        if (thread->GetExempt())
        {
            int exempt = (int)thread->GetExempt();
            nn::socket::SetSockOpt(clientSocketDescriptor, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Nn_Shutdown_Exempt, &exempt, sizeof(exempt));
        }

        NN_LOG_SERVER("connection accepted from %s\n", nn::socket::InetNtoa(saClientAddress.sin_addr));
        // continually receive messages from client until the socket is disconnected or receives the ExitMessage
        while (NN_STATIC_CONDITION(true))
        {
            if ((readByteCount = nn::socket::Recv(clientSocketDescriptor, clientMessage, sizeof(clientMessage) - 1, nn::socket::MsgFlag::Msg_None)) > 0)
            {
                clientMessage[readByteCount] = '\0';

                // look for the exit message
                if (strnlen(clientMessage, MessageBufferSize - 1) == strlen(ExitMessage))
                {
                    if (strncmp(clientMessage,ExitMessage,sizeof(ExitMessage)) == 0)
                    {
                        // exit with reason ExitReason_ReceivedExit if the ExitMessage is received
                        NN_LOG_SERVER("received exit message from client\n");
                        thread->SetExitReason(TestThread::ExitReason_ReceivedExit);
                        break;
                    }
                }
                // if it's not the exit message send back the received string
                if (nn::socket::Send(clientSocketDescriptor, clientMessage, strnlen(clientMessage, MessageBufferSize - 1), nn::socket::MsgFlag::Msg_None) < 0)
                {
                    NN_LOG_SERVER("Send failed (error %d)\n", nn::socket::GetLastError());
                    thread->SetExitReason((nn::socket::GetLastError()==nn::socket::Errno::EPipe) ?
                                          TestThread::ExitReason_Disconnected : TestThread::ExitReason_Error);
                    break;
                }
            }
            // if 0 bytes were received, that means the socket disconnected or was shut down.
            else if (readByteCount == 0 || (readByteCount == -1 && nn::socket::GetLastError() == nn::socket::Errno::ENetDown))
            {
                NN_LOG_SERVER("Client Disconnected.\n");
                // exit with reason ExitReason_Disconnected if this is the case.
                thread->SetExitReason(TestThread::ExitReason_Disconnected);
                break;
            }
            else
            {
                NN_LOG_SERVER("Recv failed (error %d)\n", nn::socket::GetLastError());
                thread->SetExitReason(TestThread::ExitReason_Error);
                break;
            }
        }
        //clean up
        nn::socket::Close(clientSocketDescriptor);
        nn::socket::Close(socketDescriptor);
        return;
    }

    void RunClientProcess(void * pData)
    {
        TestThread * thread = reinterpret_cast<TestThread *>(pData);
        nn::socket::SockAddrIn saServer = { 0 };
        char        serverMessage[MessageBufferSize] = { 0 };
        int         socketDescriptor = -1;
        ssize_t     readByteCount = 0;
        ssize_t     sent = 0;
        nn::socket::InAddr servAddr;
        nn::socket::InetAton(ServerAddr, &servAddr);
        thread->SetServerAddr(servAddr);

        NN_LOG_CLIENT("startup\n");
        // Open a shutdown exempt socket if this is the shutdown exempt version of the server thread, otherwise open a non exempt socket.
        if (thread->GetExempt())
        {
            socketDescriptor = nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        }
        else
        {
            socketDescriptor = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
            ++g_ExpectedShutdownSocketCount;
        }
        if (socketDescriptor < 0)
        {
            NN_LOG_CLIENT("Socket failed (error %d)\n", nn::socket::GetLastError());
            thread->SetExitReason(TestThread::ExitReason_Error);
            return;
        }


        NN_LOG_CLIENT("connecting to %s:%d\n", ServerAddr, thread->GetServerPort());

        saServer.sin_addr = thread->GetServerAddr();
        saServer.sin_port = nn::socket::InetHtons(thread->GetServerPort());
        saServer.sin_family = nn::socket::Family::Af_Inet;
        // connect to the server with the port provided by the main thread.
        if (nn::socket::Connect(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saServer), sizeof(saServer)) < 0)
        {
            NN_LOG_CLIENT("Connect failed (error %d)\n", nn::socket::GetLastError());
            thread->SetExitReason(TestThread::ExitReason_Error);
            nn::socket::Close(socketDescriptor);
            return;
        }

        NN_LOG_CLIENT("established connection to server\n");
        NN_LOG_CLIENT("start sending to server - '%s'\n", TestMessage);
        // send ClientSendNum packets to the server, and receive that many in return
        for (int i = 0; i < ClientSendNum; ++i)
        {
            // if we are halfway through sending and we are the exempt client, shutdown all sockets.
            // there should only be one exempt client in the test so this call is made only once
            if (i == (ClientSendNum / 2) && thread->GetExempt())
            {
                NN_LOG_CLIENT("shutdown all non-exempt sockets.\n");
                int shutdownSocketCount = nn::socket::ShutdownAllSockets(false);
                // make sure the non exempt sockets were counted and shutdown properly
                if (shutdownSocketCount < g_ExpectedShutdownSocketCount)
                {
                    NN_LOG_CLIENT("Error: ShutdownAll shutdown less than the expected number of sockets (only %d when %d at least %d were expected).\n", shutdownSocketCount, g_ExpectedShutdownSocketCount);
                    thread->SetExitReason(TestThread::ExitReason_Error);
                    nn::socket::Close(socketDescriptor);
                    break;
                }
                else
                {
                    NN_LOG_CLIENT("shutdown %d sockets.\n", shutdownSocketCount);
                }
            }

            // send a test message to the server
            if ((sent = nn::socket::Send(socketDescriptor, TestMessage, strnlen(TestMessage, MessageBufferSize), nn::socket::MsgFlag::Msg_None)) < 0)
            {
                NN_LOG_CLIENT("Send failed (error %d)\n", nn::socket::GetLastError());
                thread->SetExitReason((nn::socket::GetLastError()==nn::socket::Errno::EPipe) ?
                                      TestThread::ExitReason_Disconnected : TestThread::ExitReason_Error);
                nn::socket::Close(socketDescriptor);
                return;
            }

            // receive that same test message back
            if ((readByteCount = nn::socket::Recv(socketDescriptor, serverMessage, sizeof(serverMessage) - 1, nn::socket::MsgFlag::Msg_None)) < 0 && (nn::socket::GetLastError() != nn::socket::Errno::ENetDown))
            {
                NN_LOG_CLIENT("Recv failed (error %d)\n", nn::socket::GetLastError());
                thread->SetExitReason(TestThread::ExitReason_Error);
                nn::socket::Close(socketDescriptor);
                return;
            }

            // if we read 0 bytes, the socket is dissconnected. set exit reason to ExitReason_Disconnected and return.
            if (readByteCount == 0 || (readByteCount == -1 && nn::socket::GetLastError() == nn::socket::Errno::ENetDown))
            {
                NN_LOG_CLIENT("Server Disconnected.\n");
                thread->SetExitReason(TestThread::ExitReason_Disconnected);
                nn::socket::Close(socketDescriptor);
                return;
            }

            serverMessage[readByteCount] = '\0';
            // compare the string we sent to the one we received and make sure they are the same.
            if (strncmp(TestMessage, serverMessage, readByteCount) != 0)
            {
                NN_LOG_CLIENT("Messages did not match.");
                thread->SetExitReason(TestThread::ExitReason_Error);
                nn::socket::Close(socketDescriptor);
                return;
            }
        }
        // after we are done sending packets, send the server the exit message
        if ((sent = nn::socket::Send(socketDescriptor, ExitMessage, strnlen(ExitMessage, sizeof(ExitMessage)), nn::socket::MsgFlag::Msg_None)) < 0)
        {
            NN_LOG_CLIENT("Send failed (error %d)\n", nn::socket::GetLastError());
            thread->SetExitReason((nn::socket::GetLastError()==nn::socket::Errno::EPipe) ?
                                  TestThread::ExitReason_Disconnected : TestThread::ExitReason_Error);
            nn::socket::Close(socketDescriptor);
            return;
        }
        // we're done sending, set exit reason to ExitReason_FinishedSend and return;
        thread->SetExitReason(TestThread::ExitReason_FinishedSend);
        nn::socket::Close(socketDescriptor);
        return;
    }



} // namespace

// Main test thread
TEST(ApiUnit, SocketShutdownAll)
{
    bool isSuccess = true;
    nn::Result result;
    TestThread * server = nullptr;
    TestThread * client = nullptr;
    TestThread * serverExempt = nullptr;
    TestThread * clientExempt = nullptr;
    int linkedWaitHolderCount = 0;

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    //initialize MultiWait
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);


    //create threads and initialize their multiwait holders
    //create non-exempt server
    server = TestThread::CreateTestThread(RunServerProcess);
    if (!server)
    {
        ERROR_IF(!server, "\nError: failed to create server thread.\n");
    }


    nn::os::MultiWaitHolderType serverMultiWaitHolder;
    server->InitializeMultiWaitHolderWithThread(&serverMultiWaitHolder,ThreadUsage_Server);
    nn::os::LinkMultiWaitHolder(&multiWait, &serverMultiWaitHolder);
    ++linkedWaitHolderCount;
    //create non-exempt client
     client = TestThread::CreateTestThread(RunClientProcess);
    if (!client)
    {
        nn::os::UnlinkMultiWaitHolder(&serverMultiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&serverMultiWaitHolder);
        TestThread::FreeTestThread(server);
        ERROR_IF(!client, "\nError: failed to create client thread.\n");
    }

    nn::os::MultiWaitHolderType clientMultiWaitHolder;
    client->InitializeMultiWaitHolderWithThread(&clientMultiWaitHolder, ThreadUsage_Client);
    nn::os::LinkMultiWaitHolder(&multiWait, &clientMultiWaitHolder);
    ++linkedWaitHolderCount;

    //create exempt server
    serverExempt = TestThread::CreateTestThread(RunServerProcess);
    if (!serverExempt)
    {
        nn::os::UnlinkMultiWaitHolder(&serverMultiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&serverMultiWaitHolder);
        TestThread::FreeTestThread(server);
        nn::os::UnlinkMultiWaitHolder(&clientMultiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&clientMultiWaitHolder);
        TestThread::FreeTestThread(client);
        ERROR_IF(!serverExempt, "\nError: failed to create serverExempt thread.\n");
    }
    serverExempt->SetExempt(true);
    nn::os::MultiWaitHolderType serverExemptMultiWaitHolder;
    serverExempt->InitializeMultiWaitHolderWithThread(&serverExemptMultiWaitHolder, ThreadUsage_ServerExempt);
    nn::os::LinkMultiWaitHolder(&multiWait, &serverExemptMultiWaitHolder);
    ++linkedWaitHolderCount;

    //create exempt client
    clientExempt = TestThread::CreateTestThread(RunClientProcess);
    if (!clientExempt)
    {
        nn::os::UnlinkMultiWaitHolder(&serverMultiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&serverMultiWaitHolder);
        TestThread::FreeTestThread(server);
        nn::os::UnlinkMultiWaitHolder(&clientMultiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&clientMultiWaitHolder);
        TestThread::FreeTestThread(client);
        nn::os::UnlinkMultiWaitHolder(&serverExemptMultiWaitHolder);
        nn::os::FinalizeMultiWaitHolder(&serverExemptMultiWaitHolder);
        TestThread::FreeTestThread(serverExempt);
        ERROR_IF(!clientExempt, "\nError: failed to create clientExempt thread.\n");
    }
    clientExempt->SetExempt(true);
    nn::os::MultiWaitHolderType clientExemptMultiWaitHolder;
    clientExempt->InitializeMultiWaitHolderWithThread(&clientExemptMultiWaitHolder, ThreadUsage_ClientExempt);
    nn::os::LinkMultiWaitHolder(&multiWait, &clientExemptMultiWaitHolder);
    ++linkedWaitHolderCount;

    NN_LOG("Start Servers!\n");
    // start servers
    server->Start();
    serverExempt->Start();

    // wait for servers to be ready, should only be a tiny bit
    while (!(server->IsReady()) || !(serverExempt->IsReady()))
    {}
    NN_LOG("Start Clients!\n");
    // tell clients server ports and start clients
    client->SetServerPort(server->GetServerPort());
    client->Start();

    clientExempt->SetServerPort(serverExempt->GetServerPort());
    clientExempt->Start();

    // wait for each thread to return, check to make sure their exit reasons match with what is expected.
    for (int i = 0; i < linkedWaitHolderCount; ++i)
    {
        nn::os::MultiWaitHolderType * pThreadWaitHolder = nn::os::WaitAny(&multiWait);

        switch ((ThreadUsage)nn::os::GetMultiWaitHolderUserData(pThreadWaitHolder))
        {
            case ThreadUsage_Client:
                // non exempt client socket should be shut down, which should result in the socket disconnecting gracefully (read returns 0 bytes) and having exit reason ExitReason_Disconnected
                NN_LOG("Client thread returned.\n");
                if (client->GetExitReason() != TestThread::ExitReason_Disconnected)
                {
                    isSuccess = false;
                    NN_LOG("Error: Expected Client thread to return ExitReason_Disconnected, but it returned %s\n", client->GetExitReasonString());
                }
                break;
            case ThreadUsage_Server:
                // non exempt server socket should be shut down, which should result in the socket disconnecting gracefully (read returns 0 bytes) and having exit reason ExitReason_Disconnected
                NN_LOG("Server thread returned.\n");
                if (server->GetExitReason() != TestThread::ExitReason_Disconnected)
                {
                    isSuccess = false;
                    NN_LOG("Error: Expected Server thread to return ExitReason_Disconnected, but it returned %s\n", server->GetExitReasonString());
                }
                break;
            case ThreadUsage_ClientExempt:
                // exempt client socket should finish sending all packets, send an exit message to the server and have exit reason ExitReason_FinishedSend
                NN_LOG("ClientExempt thread returned.\n");
                if (clientExempt->GetExitReason() != TestThread::ExitReason_FinishedSend)
                {
                    isSuccess = false;
                    NN_LOG("Error: Expected ClientExempt thread to return ExitReason_FinishedSend, but it returned %s\n", clientExempt->GetExitReasonString());
                }
                break;
            case ThreadUsage_ServerExempt:
                // exempt server socket should remain open to receive the exit message from the client, and have exit reason ExitReason_ReceivedExit
                NN_LOG("ServerExempt thread returned.\n");
                if (serverExempt->GetExitReason() != TestThread::ExitReason_ReceivedExit)
                {
                    isSuccess = false;
                    NN_LOG("Error: Expected ServerExempt thread to return ExitReason_ReceivedExit, but it returned %s\n", serverExempt->GetExitReasonString());
                }
                break;
            default:
                isSuccess = false;
                NN_LOG("Error: Invalid Internal State, Unexpected Thread Usage.\n");
                break;
        }
        //clean up the returned threads multiwait holder

        nn::os::UnlinkMultiWaitHolder(pThreadWaitHolder);
        nn::os::FinalizeMultiWaitHolder(pThreadWaitHolder);
    }

    //clean up
    nn::os::FinalizeMultiWait(&multiWait);
    TestThread::FreeTestThread(server);
    TestThread::FreeTestThread(client);
    TestThread::FreeTestThread(serverExempt);
    TestThread::FreeTestThread(clientExempt);

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:
    EXPECT_EQ(isSuccess, true);
    return;
} //NOLINT(impl/function_size)

TestThread * TestThread::CreateTestThread(ThreadFunc* pFunc)
{
    unsigned char* pBuffer = (unsigned char*)malloc(sizeof(TestThread) + TestThread::StackAlign);

    unsigned char* pAlignedBuf = (pBuffer + TestThread::StackAlign - (reinterpret_cast<uintptr_t>(pBuffer) % TestThread::StackAlign));
    TestThread* pNewTestThread = new (pAlignedBuf) TestThread();
    pNewTestThread->m_pUnalignedBuf = pBuffer;

    if (nn::os::CreateThread(&(pNewTestThread->m_thread), pFunc, pNewTestThread, pNewTestThread->m_pStack, StackSize, 0, 0).IsFailure())
    {
        NN_LOG("Failed to create thread!\n\n");
        pNewTestThread->~TestThread();
        return NULL;
    }

    return pNewTestThread;

}

void TestThread::InitializeMultiWaitHolderWithThread(nn::os::MultiWaitHolderType * holder, uint32_t usage) NN_NOEXCEPT
{
    nn::os::InitializeMultiWaitHolder(holder, &m_thread);
    nn::os::SetMultiWaitHolderUserData(holder, usage);
}

void TestThread::Start() NN_NOEXCEPT
{
    nn::os::StartThread(&m_thread);
}

void TestThread::FreeTestThread(TestThread* pThread) NN_NOEXCEPT
{
    void * buf = pThread->m_pUnalignedBuf;
    pThread->~TestThread();
    free(buf);
}


void TestThread::SetClientAddr(nn::socket::InAddr addr) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    m_clientAddr = addr;
    nn::os::UnlockMutex(&m_mutex);
}

nn::socket::InAddr TestThread::GetClientAddr() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    nn::socket::InAddr addr = m_clientAddr;
    nn::os::UnlockMutex(&m_mutex);
    return addr;
}


void TestThread::SetServerAddr(nn::socket::InAddr addr) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    m_serverAddr = addr;
    nn::os::UnlockMutex(&m_mutex);
}


nn::socket::InAddr TestThread::GetServerAddr() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    nn::socket::InAddr addr = m_serverAddr;
    nn::os::UnlockMutex(&m_mutex);
    return addr;
}

void TestThread::SetServerPort(unsigned short port) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    m_serverPort = port;
    nn::os::UnlockMutex(&m_mutex);
}

unsigned short TestThread::GetServerPort() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    unsigned short port = m_serverPort;
    nn::os::UnlockMutex(&m_mutex);
    return port;
}

void TestThread::SetReady(bool ready) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    m_ready = ready;
    nn::os::UnlockMutex(&m_mutex);
}
bool TestThread::IsReady()
{
    nn::os::LockMutex(&m_mutex);
    bool ready = m_ready;
    nn::os::UnlockMutex(&m_mutex);
    return ready;
}

void TestThread::SetExitReason(ExitReason reason) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    m_reason = reason;
    nn::os::UnlockMutex(&m_mutex);
}

TestThread::ExitReason TestThread::GetExitReason() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    ExitReason reason = m_reason;
    nn::os::UnlockMutex(&m_mutex);
    return reason;
}

const char * TestThread::GetExitReasonString() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    char * reason;
    switch (m_reason)
    {
    case ExitReason_Error:
        return "ExitReason_Error";
    case ExitReason_Disconnected:
        return "ExitReason_Disconnected";
    case ExitReason_FinishedSend:
        return "ExitReason_FinishedSend";
    case ExitReason_ReceivedExit:
        return "ExitReason_ReceivedExit";
    default:
        return "UNKOWN INTERNAL ERROR";
    }
    nn::os::UnlockMutex(&m_mutex);
    return reason;
}

void TestThread::SetExempt(bool exempt) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    m_exempt = exempt;
    nn::os::UnlockMutex(&m_mutex);
}

bool TestThread::GetExempt() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_mutex);
    bool exempt = m_exempt;
    nn::os::UnlockMutex(&m_mutex);
    return exempt;
}

TestThread::TestThread() NN_NOEXCEPT
{
    nn::os::InitializeMutex(&m_mutex,false,0);
    m_clientAddr.S_addr = 0;
    m_serverAddr.S_addr = 0;
    m_serverPort = 0;
    m_ready = false;
    m_exempt = false;
    m_reason = ExitReason_Error;
}

TestThread::~TestThread() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_thread);
    nn::os::DestroyThread(&m_thread);
    nn::os::FinalizeMutex(&m_mutex);
}

