﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include "testNet_ApiCommon.h"
#include "natf/Utils/md5.h"
#include "natf/Utils/CommandLineParser.h"
#include "natf/Utils/InitApis.h"

namespace
{
    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}

namespace NATF {
namespace API {

TEST(SendRecv, Server)
{
    bool isSuccess = true;
    int clientSocket = -1;
    int listenSocket = -1;
    int rval;
    int argc;
    const char** pArgv;
    uint32_t transferSizeMb = 0;
    uint32_t defaultTransferSizeMb = 1;
    uint16_t portNum = 0;
    uint16_t defaultPortNum = DefaultPort;
    nn::socket::SockAddrIn localAddr;
    nn::socket::SockAddrIn clientAddr;
    nn::socket::SockLenT clientAddrLen = sizeof(clientAddr);
    Utils::ParserGroup parser;
    Utils::InitApi initApi(Utils::InitApiFlags_Nifm | Utils::InitApiFlags_Network | Utils::InitApiFlags_Socket);
    nn::Result result;

    NN_LOG("In\n\n");

    NETTEST_GET_ARGS(argc, pArgv);

    parser.AddParser(Utils::UInt32Parser("--TransferSizeMb", &defaultTransferSizeMb, transferSizeMb));
    parser.AddParser(Utils::UInt16Parser("--PortNum", &defaultPortNum, portNum));

    isSuccess = parser.Parse(argc, pArgv);
    ERROR_IF(!isSuccess, "Failed to parse command line args\n\n");

    // Passing InvalidUuid for default config profile
    isSuccess = initApi.Init(nn::util::InvalidUuid);

    // TODO: when SIGLONTD-3127 is fixed, check all platforms and remove workaround
    NX_ONLY_ERROR_IF(!isSuccess, "Failed to initialize NIFM api!\n\n");
    NIFM_INIT_FAILURE_WORKAROUND(isSuccess);

    // Create Listening Socket
    listenSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF(listenSocket < 0, "Failed to create socket. errno: %d\n\n", nn::socket::GetLastError());

    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
    localAddr.sin_port        = nn::socket::InetHtons(portNum);
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Bind to local port
    rval = nn::socket::Bind(listenSocket, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof(localAddr));
    ERROR_IF(rval < 0, "Failed to bind socket. errno: %d, port: %d\n\n", nn::socket::GetLastError(), portNum);

    // Set to listening socket
    rval = nn::socket::Listen(listenSocket, 10);
    ERROR_IF(rval < 0, "Failed to set socket to listening socket. errno: %d\n\n", nn::socket::GetLastError());

    memset(&clientAddr, 0, sizeof(clientAddr));

    // Wait for client connection
    clientSocket = nn::socket::Accept(listenSocket, reinterpret_cast<nn::socket::SockAddr *>(&clientAddr), &clientAddrLen);
    ERROR_IF(rval < 0, "Accept failed! errno: %d\n\n", nn::socket::GetLastError());

    NN_LOG("New client!\n");

    isSuccess = RecvData(clientSocket);
    ERROR_IF(!isSuccess, "RecvData failed!\n");

    isSuccess = SendData(clientSocket, transferSizeMb * 1024 * 1024);
    ERROR_IF(!isSuccess, "SendData failed!\n");

out:

    if( clientSocket >= 0 )
    {
        nn::socket::Shutdown(clientSocket, nn::socket::ShutdownMethod::Shut_RdWr);
        nn::socket::Close(clientSocket);
    }

    if( listenSocket >= 0 )
    {
        nn::socket::Shutdown(listenSocket, nn::socket::ShutdownMethod::Shut_RdWr);
        nn::socket::Close(listenSocket);
    }

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(SendRecv, Client)
{
    bool isSuccess = true;
    int socket = -1;
    int rval;
    int argc;
    const char** pArgv;
    uint32_t transferSizeMb = 0;
    uint32_t defaultTransferSizeMb = 1;
    uint16_t portNum = 0;
    uint16_t defaultPortNum = DefaultPort;
    char pIpAddr[MaxIpStrLen];
    nn::socket::SockAddrIn remoteAddr;
    NATF::Utils::ParserGroup parser;
    nn::Result result;

    NN_LOG("In\n\n");

    NETTEST_GET_ARGS(argc, pArgv);

    parser.AddParser(NATF::Utils::UInt32Parser("--TransferSizeMb", &defaultTransferSizeMb, transferSizeMb));
    parser.AddParser(NATF::Utils::UInt16Parser("--PortNum", &defaultPortNum, portNum));
    parser.AddParser(NATF::Utils::IpParser    ("--IpAddr", nullptr, pIpAddr, sizeof(pIpAddr)));

    isSuccess = parser.Parse(argc, pArgv);
    ERROR_IF(!isSuccess, "Failed to parse command line args\n\n");

    result = nn::nifm::Initialize();
    ERROR_IF(result.IsFailure(), "Failed to initialize NIFM Api. Desc: %d\n\n", result.GetDescription());

    nn::nifm::SubmitNetworkRequest();

    /* wait for network interface availability, while providing status */
    while( nn::nifm::IsNetworkRequestOnHold() )
    {
        NN_LOG("Waiting for network interface availability...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    ERROR_IF(!nn::nifm::IsNetworkAvailable(), "Network not available.\n\n");

    // Init socket api
    result = nn::socket::Initialize(reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                                    nn::socket::DefaultSocketMemoryPoolSize,
                                    nn::socket::DefaultSocketAllocatorSize,
                                    nn::socket::DefaultConcurrencyLimit);
    ERROR_IF(result.IsFailure(), "Failed to initialize socket Api. Desc: %d\n\n", result.GetDescription());

    // Create Listening Socket
    socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF(socket < 0, "Failed to create socket. errno: %d\n\n", nn::socket::GetLastError());

    memset(&remoteAddr, 0, sizeof(remoteAddr));
    rval = nn::socket::InetAton(pIpAddr, &remoteAddr.sin_addr);
    ERROR_IF(rval != 1, "Invalid ip string!\n\n");

    remoteAddr.sin_port   = nn::socket::InetHtons(portNum);
    remoteAddr.sin_family = nn::socket::Family::Af_Inet;

    rval = nn::socket::Connect(socket, reinterpret_cast<nn::socket::SockAddr*>(&remoteAddr), sizeof(remoteAddr));
    ERROR_IF(rval != 0, "Faled to connect to server! errno: %d\n\n", nn::socket::GetLastError());

    NN_LOG("Connected to server!\n");

    isSuccess = SendData(socket, transferSizeMb * 1024 * 1024);
    ERROR_IF(!isSuccess, "SendData failed!\n");

    isSuccess = RecvData(socket);
    ERROR_IF(!isSuccess, "RecvData failed!\n");

out:

    if( socket >= 0 )
    {
        nn::socket::Shutdown(socket, nn::socket::ShutdownMethod::Shut_RdWr);
        nn::socket::Close(socket);
    }

    nn::socket::Finalize();

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

}} // namespace NATF::API
