﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Complex/testNet_DnsBurn.h"
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/socket_TypesPrivate.h>
#include <nn/socket/socket_ResolverOptionsPrivate.h>
#include <nn/socket/resolver/resolver_Client.h>

namespace NATF {
namespace API {

/** utility function */
ssize_t WriteN(int sockfd, uint8_t* buf, size_t size)
{
    NN_LOG("WriteN: sockfd=%d, buf=%p, size=%u\n",
           sockfd, buf, size);

    ssize_t rc = -1;
    size_t nleft = size;
    uint8_t* ptr = buf;

    while (nleft > 0)
    {
        if ( 0 >= (rc = nn::socket::Write(sockfd,
                                          ptr,
                                          nleft)))
        {
            nn::socket::Errno errorNumber = nn::socket::GetLastError();
            if ( rc < 0 && (nn::socket::Errno::EIntr == errorNumber
                            || nn::socket::Errno::EAgain == errorNumber
                            || nn::socket::Errno::EWouldBlock != errorNumber ))
            {
                rc = 0;
                NN_LOG("WriteN sockfd: %d, error: %s (%d)\n",
                       sockfd, strerror(static_cast<int>(errorNumber)), errorNumber);
            }
            else
            {
                NN_LOG("WriteN sockfd: %d, error: %s (%d)\n",
                       sockfd, strerror(static_cast<int>(errorNumber)), errorNumber);
                return -1;
            };
        };

        nleft -= rc;
        ptr += rc;
    };

    return size;
};

/** test code */

DnsBurnTestThread::DnsBurnTestThread(bool isServer, bool shouldAcquireCancelHandle) :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    UnitTestThreadBase(__FUNCTION__, 10000),
    m_isInitialized(false),
    m_isServer(isServer),
    m_TotalResolveAttempts(0),
    m_NumberOfSuccessfulResolves(0),
    m_ShouldAcquireCancelHandle(shouldAcquireCancelHandle),
    m_CancelHandle(0)
{
};

void DnsBurnTestThread::InitializeServer(SimpleValidator* pSimpleValidator,
                                         unsigned serverDelay,
                                         unsigned numberOfListeners,
                                         bool forceServerTcp)
{
    NN_SDK_ASSERT(false == m_isInitialized);
    NN_SDK_ASSERT(NULL != pSimpleValidator);
    memset(m_HostName, '0', sizeof(m_HostName));

    m_pValidator = pSimpleValidator;
    m_pValidator->addReference();
    m_ServerDelay = serverDelay;
    m_NumberOfListeners = numberOfListeners;
    m_WorkerThreadNumber = 0;
    m_ForceServerTcp = forceServerTcp;

    m_isServer= true;
    m_isInitialized = true;

    m_ClientShouldUseGetaddrinfo = false;
    m_WasInterrupted = false;
    m_UdpServerSocket = -1;
};

void DnsBurnTestThread::InitializeClient(unsigned workerThreadNumber,
                                         SimpleValidator* pSimpleValidator,
                                         const char* hostName,
                                         bool shouldUseGetAddrInfo,
                                         unsigned listenBacklog)
{
    NN_SDK_ASSERT(false == m_isInitialized);
    NN_SDK_ASSERT(NULL != pSimpleValidator);
    NN_SDK_ASSERT(NULL != hostName);

    const unsigned MaximumBacklogSockets = 30;

    m_WorkerThreadNumber = workerThreadNumber;
    m_pValidator = pSimpleValidator;
    m_pValidator->addReference();
    memcpy(m_HostName, hostName, strlen(hostName) + 1);
    m_ClientShouldUseGetaddrinfo = shouldUseGetAddrInfo;
    m_isInitialized = true;

    m_ListenBacklog = listenBacklog > MaximumBacklogSockets ? MaximumBacklogSockets : listenBacklog;

    m_WasInterrupted = false;
    m_ServerDelay = 0;
    m_NumberOfListeners = 0;
    m_UdpServerSocket = -1;
};

DnsBurnTestThread::~DnsBurnTestThread()
{
    NN_LOG("Destroying Thread %d\n", m_WorkerThreadNumber);

    if ( -1 != m_UdpServerSocket )
    {
        nn::socket::Close(m_UdpServerSocket);
        m_UdpServerSocket = -1;
    };

    if ( -1 != m_TcpServerSocket )
    {
        nn::socket::Close(m_TcpServerSocket);
        m_TcpServerSocket = -1;

    };

    m_pValidator->releaseReference();
};

/** @brief run function, waits for testTimeout or signal */
void DnsBurnTestThread::RunClient()
{
    NN_LOG("Thread %d: Running\n", m_WorkerThreadNumber);

    while (true)
    {
        /** sleep for a little bit so we don't take over the core completely */
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

        if (true == m_ShouldAcquireCancelHandle)
        {
            nn::socket::ResolverOption option;
            nn::socket::resolver::ResolverGetOption(&option,
                                                    nn::socket::ResolverOptionKey::GetCancelHandleInteger);
            m_CancelHandle = option.data.integerValue;
            NN_LOG("nn::socket::RequestCancelHandle() returned: %08x, errno: %s (%d)\n",
                   m_CancelHandle, strerror(static_cast<int>(errno)), errno);
            m_CancelHandle = 0;
        };


        /**
         * If any of the threads then we bail
         */
        if (true == m_pValidator->DidFail())
        {
            NN_LOG("Thread %d: validator already failed: bailing\n", m_WorkerThreadNumber);
            goto bail;
        }
        /**
         * add code to get a cancel handle here
         * after SIGLONTD-7060
         */
        else if ( WasInterrupted() )
        {
            NN_LOG("Thread %d: Interrupted\n", m_WorkerThreadNumber);
            goto bail;
        };

        /** we increase the total number of resolve attempts */
        ++m_TotalResolveAttempts;

        /**
         * run test
         */
        if (true == m_ClientShouldUseGetaddrinfo)
        {
            if ( nn::socket::AiErrno::EAi_Success != BurnGetAddrInfo())
            {
                NN_LOG("BurnGetAddrInfo failed\n");
                goto bail;
            };
        }
        else if (-1 == BurnGetHostByName())
        {
            NN_LOG("BurnGetHostByName failed\n");
            goto bail;
        };

        if (true == m_ShouldAcquireCancelHandle)
        {
            AttemptCancel(m_CancelHandle, (true == m_ClientShouldUseGetaddrinfo ?
                                           "GetAddrInfo": "GetHostByName"));
        };

    }; // end while

bail:
    if (true == m_ShouldAcquireCancelHandle
        && 0 != m_CancelHandle)
    {
        AttemptCancel(m_CancelHandle, (true == m_ClientShouldUseGetaddrinfo ?
                                       "GetAddrInfo": "GetHostByName"));
    };

    return;
};

nn::socket::AiErrno DnsBurnTestThread::BurnGetAddrInfo()
{
    nn::socket::AiErrno rc = nn::socket::AiErrno::EAi_Success;
    nn::socket::Errno errorNumber = nn::socket::Errno::ESuccess;
    nn::socket::AddrInfo* pAddressInfoResult = NULL;
    const char* gaiErrorString = NULL;

    nn::socket::ResolverOption cancelOption;
    cancelOption.key = nn::socket::ResolverOptionKey::RequestCancelHandleInteger;
    cancelOption.type = nn::socket::ResolverOptionType::Integer;
    cancelOption.size = sizeof(int);
    cancelOption.data.integerValue = m_CancelHandle;

    nn::socket::AddrInfo hints;
    memset(&hints, 0, sizeof(nn::socket::AddrInfo));
    hints.ai_family = nn::socket::Family::Af_Inet;      /* Allow IPv4 or IPv6 */
    hints.ai_socktype = nn::socket::Type::Sock_Dgram; /* Datagram socket */
    hints.ai_flags = nn::socket::AddrInfoFlag::Ai_Passive;    /* For wildcard IP address */
    hints.ai_protocol = nn::socket::Protocol::IpProto_Ip;          /* Any protocol */
    hints.ai_canonname = NULL;
    hints.ai_addr = NULL;
    hints.ai_next = NULL;

    if (nn::socket::AiErrno::EAi_Success != (rc = nn::socket::GetAddrInfo(m_HostName,
                                                                     NULL,
                                                                     &hints,
                                                                     &pAddressInfoResult,
                                                                     &cancelOption,
                                                                     1)))
    {
        // todo: gaierrorstring might be NULL if
        // all service framework handles are in use
        errorNumber = nn::socket::GetLastError();
        gaiErrorString = nn::socket::GAIStrError(rc);

        // on error gai should always return a NULL
        if ( NULL != pAddressInfoResult )
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "Thread %d: GetAddrInfo rc: %s (%d), errno: %s (%d), but non-NULL result\n",
                           m_WorkerThreadNumber,
                           gaiErrorString, rc,
                           strerror(static_cast<int>(errorNumber)), errorNumber);
            goto bail;
        }
        else if (true == m_ShouldAcquireCancelHandle && nn::socket::Errno::ECanceled == errorNumber)
        {
            NN_LOG("%s Cancelled\n", __FUNCTION__);
            rc = nn::socket::AiErrno::EAi_Success;
            goto bail;
        }
        else if ( nn::socket::AiErrno::EAi_Memory == rc || nn::socket::AiErrno::EAi_Again == rc || (nn::socket::AiErrno::EAi_System == rc && nn::socket::Errno::EBusy == errorNumber))
        {
            rc = nn::socket::AiErrno::EAi_Success;
            goto bail;
        };

        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Thread %d: GetAddrInfo error rc: %s (%d), errno: %s (%d)\n",
                       m_WorkerThreadNumber,
                       gaiErrorString, rc,
                       strerror(static_cast<int>(errorNumber)),errorNumber);
        goto bail;
    }
    else if (NULL == pAddressInfoResult)
    {
        // on success we should always get a result
        errorNumber = nn::socket::GetLastError();
        gaiErrorString = nn::socket::GAIStrError(rc);
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Thread %d: GetAddrInfo rc: %s (%d), errno: %s (%d), but NULL result\n",
                       m_WorkerThreadNumber,
                       gaiErrorString, rc,
                       strerror(static_cast<int>(errorNumber)), errorNumber);
        goto bail;
    }
    else
    {
        ++m_NumberOfSuccessfulResolves;
        rc = nn::socket::AiErrno::EAi_Success;
    };

bail:
    if (NULL != pAddressInfoResult)
    {
        nn::socket::FreeAddrInfo(pAddressInfoResult);
        pAddressInfoResult = NULL;
    };

    return rc;
}

int DnsBurnTestThread::BurnGetHostByName()
{
    int rc = -1;
    nn::socket::Errno errorNumber = nn::socket::Errno::ESuccess;
    nn::socket::HErrno hostError = nn::socket::HErrno::Netdb_Success;
    const char* hostErrorString = nullptr;
    nn::socket::HostEnt *pHostEnt = nullptr;

    nn::socket::ResolverOption cancelOption;
    cancelOption.key = nn::socket::ResolverOptionKey::RequestCancelHandleInteger;
    cancelOption.type = nn::socket::ResolverOptionType::Integer;
    cancelOption.size = sizeof(int);
    cancelOption.data.integerValue = m_CancelHandle;

    if (nullptr == (pHostEnt = nn::socket::GetHostEntByName(m_HostName, &cancelOption,  1)))
    {
        // all service framework handles are in use
        errorNumber = nn::socket::GetLastError();
        hostError = *nn::socket::GetHError();
        hostErrorString = nn::socket::HStrError(hostError);

        if (true == m_ShouldAcquireCancelHandle && nn::socket::Errno::ECanceled == errorNumber)
        {
            NN_LOG("%s Cancelled\n", __FUNCTION__);
            rc = 0;
            goto bail;
        }
        else if (nn::socket::HErrno::Try_Again == hostError || nn::socket::Errno::EAgain == errorNumber)
        {
            rc = 0;
            goto bail;
        };

        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Thread %d: GetHostByName returned NULL, hostError: %s (%d), errno: %s (%d)\n",
                       m_WorkerThreadNumber,
                       hostErrorString, hostError,
                       strerror(static_cast<int>(errorNumber)), errorNumber);
    }
    else
    {
        ++m_NumberOfSuccessfulResolves;
        rc = 0;
    };

bail:
    return rc;
};

template <typename T>
T getData(void* buf, size_t byteOffset)
{
    return *(reinterpret_cast<T*>(reinterpret_cast<uint8_t*>(buf) + byteOffset));
}

template <typename T>
void putData(void* buf, size_t byteOffset, T value)
{
    T* pointer = reinterpret_cast<T*>(reinterpret_cast<uint8_t*>(buf) + byteOffset);
    *pointer = value;
}

/** @brief run function, waits for testTimeout or signal */
void DnsBurnTestThread::RunServer()
{
    int rc = -1;
    nn::socket::FdSet readfds;
    nn::socket::TimeVal tv;

    NN_LOG("Starting server; udpSocket: %d, tcpSocket: %d, forceTcp: %s\n",
           m_UdpServerSocket, m_TcpServerSocket, true == m_ForceServerTcp ? "true" : "false");

    while (true)
    {
        nn::socket::FdSetZero(&readfds);
        nn::socket::FdSetSet(m_UdpServerSocket, &readfds);
        nn::socket::FdSetSet(m_TcpServerSocket, &readfds);
        tv.tv_sec = 1;
        tv.tv_usec = 0;

        /**
         * If any of the threads failed then we bail
         */
        if (true == m_pValidator->DidFail())
        {
            NN_LOG("Validator failed, exiting server\n");
            goto bail;
        }
        /**
         * Test was interrupted
         */
        else if ( WasInterrupted() )
        {
            NN_LOG("Resolver server thread interrupted\n");
            goto bail;
        }
#define MAX(a,b)(a > b ? a : b)
        else if ( -1 == (rc = nn::socket::Select(MAX(m_TcpServerSocket, m_UdpServerSocket) + 1,
                                                 &readfds,
                                                 NULL,
                                                 NULL,
                                                 &tv)))
        {
            goto bail;
        }
#undef MAX
        else if ( 0 == rc )
        {
            continue;
        };

        /**
         * UDP response
         */
        if (nn::socket::FdSetIsSet(m_UdpServerSocket, &readfds))
        {
            if ( -1 == HandleServerUdp() )
            {
                goto bail;
            };

            // if the server is forced into TCP mode then we want the sleep on the TCP side
            if ( !m_ForceServerTcp )
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_ServerDelay));
            };

            goto bail;
        };

        /**
         * TCP response
         */
        if (nn::socket::FdSetIsSet(m_TcpServerSocket, &readfds))
        {
            if ( -1 == HandleServerTcp())
            {
                goto bail;
            };

            // since you only get to tcp mode after UDP (at least in this test)
            // we sleep here by default
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_ServerDelay));
        };
    }; // end while

bail:
    if ( -1 != m_UdpServerSocket)
    {
        nn::socket::Close(m_UdpServerSocket);
        m_UdpServerSocket = -1;
    };

    if ( -1 != m_TcpServerSocket)
    {
        nn::socket::Close(m_TcpServerSocket);
        m_TcpServerSocket = -1;
    };
};

int DnsBurnTestThread::HandleServerUdp()
{
    int rc = -1;
    nn::socket::SockAddrIn addr;
    nn::socket::SockLenT fromlen = sizeof(addr);
    nn::socket::Errno errorNumber = nn::socket::Errno::ESuccess;
    uint16_t txid = 0;
    uint16_t flags;
    char buffer[512];
    DnsBurnTestGetServerSocketAddress(addr);

    if ( -1 == (rc = nn::socket::RecvFrom(m_UdpServerSocket,
                                          buffer,
                                          sizeof(buffer),
                                          nn::socket::MsgFlag::Msg_None,
                                          reinterpret_cast<nn::socket::SockAddr*>(&addr),
                                          &fromlen)))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "UDP RecvFrom: socket: %d, rc: %d, errno: %s (%d)\n",
                       m_UdpServerSocket,
                       rc,
                       strerror(static_cast<int>(errorNumber)), errorNumber);
        goto bail;
    };

    txid = nn::socket::InetNtohs(getData<uint16_t>(buffer, TRANSACTION_ID_POSITION));
    flags = nn::socket::InetNtohs(getData<uint16_t>(buffer, FLAGS_POSITION));
    // put the transaction id into the out buffer
    putData<uint16_t>(g_DnsRawData, TRANSACTION_ID_POSITION, nn::socket::InetHtons(txid));

    // turn on / off truncate bit to force tcp
    flags = nn::socket::InetNtohs(getData<uint16_t>(g_DnsRawData, FLAGS_POSITION));
    if ( true == m_ForceServerTcp )
    {
        flags |= (1 << TC_BIT_SHIFT);
    }
    else
    {
        flags &= ~(1 << TC_BIT_SHIFT);
    };
    putData<uint16_t>(g_DnsRawData, FLAGS_POSITION, nn::socket::InetHtons(flags));

    // put number of answers for udp
    putData<uint16_t>(g_DnsRawData, NUM_ANSWERS_POSITION, nn::socket::InetHtons(NUM_UDP_ANSWERS));

    if ( -1 == (rc = nn::socket::SendTo(m_UdpServerSocket,
                                        g_DnsRawData,
                                        g_UdpDnsRawDataSize,
                                        nn::socket::MsgFlag::Msg_None,
                                        reinterpret_cast<nn::socket::SockAddr*>(&addr),
                                        fromlen)))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "UDP SendTo: socket: %d, rc: %d, errno: %s (%d)\n",
                       m_UdpServerSocket,
                       rc,
                       strerror(static_cast<int>(errorNumber)), errorNumber);
        goto bail;
    };

    rc = 0;

bail:
    return rc;
}

int DnsBurnTestThread::HandleServerTcp()
{
    int acceptedSocket = -1;
    int rc = -1;
    nn::socket::SockAddrIn addr;
    nn::socket::SockLenT fromlen;
    nn::socket::Errno errorNumber = nn::socket::Errno::ESuccess;
    uint16_t dnsLength = 0;
    uint16_t txid = 0;
    uint16_t flags;
    char buffer[512];

    acceptedSocket = nn::socket::Accept(m_TcpServerSocket,
                                        reinterpret_cast<nn::socket::SockAddr*>(&addr),
                                        &fromlen);
    if ( -1 == acceptedSocket )
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true,
                       "Accept: rc: %d, errno: %s (%d)\n",
                       rc,
                       strerror(static_cast<int>(errorNumber)), errorNumber);
        // nothing to close
        goto bail;
    }
    else if ( -1 == (rc = nn::socket::Read(acceptedSocket,
                                           reinterpret_cast<uint8_t*>(&dnsLength),
                                           sizeof(dnsLength)))
              || 0 == rc )
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator,
                       true, "TCP Read (1/2): socket: %d, rc: %d, errno: %s(%d)\n",
                       acceptedSocket,
                       rc,
                       strerror(static_cast<int>(errorNumber)),
                       errorNumber);
        nn::socket::Close(acceptedSocket);
        goto bail;
    };

    dnsLength = nn::socket::InetNtohs(dnsLength);

    if ( -1 == (rc = nn::socket::Read(acceptedSocket,
                                      buffer,
                                      dnsLength))
         || 0 == rc )
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "TCP Read (2/2): socket: %d, rc: %d, errno: %s (%d)\n",
                       acceptedSocket,
                       rc,
                       strerror(static_cast<int>(errorNumber)), errorNumber);
        nn::socket::Close(acceptedSocket);
        goto bail;
    };

    txid = nn::socket::InetNtohs(getData<uint16_t>(buffer, TRANSACTION_ID_POSITION));
    flags = nn::socket::InetNtohs(getData<uint16_t>(buffer, FLAGS_POSITION));

    // add the length to the raw data for the entire pdu minus the size of a 16 bit length
    putData<uint16_t>(g_DnsRawDataWithLength,
                      LENGTH_POSITION_RAW_ONLY,
                      nn::socket::InetHtons(g_DnsRawDataWithLengthSize - sizeof(uint16_t)));

    // put the transaction id into the out buffer
    putData<uint16_t>(g_DnsRawData, TRANSACTION_ID_POSITION, nn::socket::InetHtons(txid));

    // always turn off truncate bit to for tcp
    flags = nn::socket::InetNtohs(getData<uint16_t>(g_DnsRawData, FLAGS_POSITION));
    flags &= ~(1 << TC_BIT_SHIFT);
    putData<uint16_t>(g_DnsRawData, FLAGS_POSITION, nn::socket::InetHtons(flags));

    // put the number of answers for tcp
    putData<uint16_t>(g_DnsRawData, NUM_ANSWERS_POSITION, nn::socket::InetHtons(NUM_TCP_ANSWERS));

    if ( -1 == (rc = nn::socket::Write(acceptedSocket,
                                       g_DnsRawDataWithLength,
                                       g_DnsRawDataWithLengthSize)))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "TCP Write: socket: %d, rc: %d, errno: %s (%d)\n",
                       acceptedSocket,
                       rc,
                       strerror(static_cast<int>(errorNumber)), errorNumber);
        nn::socket::Close(acceptedSocket);
        goto bail;
    };

    rc = 0;

bail:
    //NN_LOG("TCP wrote %d bytes, txid=%04x, flags=%04x\n", rc, txid, flags);
    if ( -1 != acceptedSocket )
    {
        nn::socket::Close(acceptedSocket);
        acceptedSocket = -1;
    }
    return rc;
}


/** @brief run function, waits for testTimeout or signal */
void DnsBurnTestThread::Run()
{
    if ( m_isServer )
    {
        RunServer();
    }
    else
    {
        RunClient();
    };
};

void DnsBurnTestThread::Interrupt()
{
    std::lock_guard<std::mutex> lock(m_AccessLock);
    m_WasInterrupted  = true;
};

void DnsBurnTestThread::GetFinalStatistics(unsigned & total, unsigned & success, unsigned & failure) const
{
    total = m_TotalResolveAttempts;
    success = m_NumberOfSuccessfulResolves;
    failure = total - success;
}

int DnsBurnTestThread::WasInterrupted()
{
    std::lock_guard<std::mutex> lock(m_AccessLock);
    return m_WasInterrupted;
}

void* DnsBurnTestThread::operator new( std::size_t count )
{
    return malloc(count);
}

bool DnsBurnTestThread::EarlyBlockingInitialize(nn::os::SemaphoreType& blockingSemaphore)
{
    int rc = -1;
    nn::socket::Errno errorNumber = nn::socket::Errno::ESuccess;
    nn::socket::SockAddrIn addr;
    int yes = 1;

    if ( ! m_isServer )
    {
        goto bail;
    }
    DnsBurnTestGetServerSocketAddress(addr);

    NN_LOG("%s/%d: nn::socket::SockAddr for server: %s:%d\n",
           nn::socket::InetNtoa( addr.sin_addr ),
           nn::socket::InetNtohs( addr.sin_port ));

    // udp block
    if (-1 == (m_UdpServerSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)))
    {
        rc = -1;
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true, "UDP Socket: %d, errno: %d\n", rc, errorNumber);
        goto bail;
    }
    else if (-1 == (rc = nn::socket::SetSockOpt(m_UdpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &yes, sizeof(int))))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true, "UDP SetSockOpt: %d, errno: %d\n", rc, errorNumber);
        goto bail;
    }
    else if (-1 == (rc = nn::socket::Bind(m_UdpServerSocket,
                                          reinterpret_cast<nn::socket::SockAddr*>(&addr),
                                          sizeof(addr))))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true, "UDP Bind: %d, errno: %d\n", rc, errorNumber);
        goto bail;
    }
    // tcp block
    else if (-1 == (m_TcpServerSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)))
    {
        rc = -1;
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true, "TCP Socket: %d, errno: %d\n", rc, errorNumber);
        goto bail;
    }
    else if (-1 == (rc = nn::socket::SetSockOpt(m_TcpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &yes, sizeof(int))))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true, "TCP SetSockOpt: %d, errno: %d\n", rc, errorNumber);
        goto bail;
    }
    else if (-1 == (rc = nn::socket::Bind(m_TcpServerSocket,
                                          reinterpret_cast<nn::socket::SockAddr*>(&addr),
                                          sizeof(addr))))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true, "TCP Bind: %d, errno: %d\n", rc, errorNumber);
        goto bail;
    }
    else if (-1 == (rc = nn::socket::Listen(m_TcpServerSocket, m_ListenBacklog)))
    {
        errorNumber = nn::socket::GetLastError();
        SVALIDATE_FAIL(m_pValidator, true, "TCP Listen: %d, errno: %d\n", rc, errorNumber);
        goto bail;
    };

bail:
    return false; // we did not signal the semaphore
}

}; // namespace API
}; // namespace NATF
