﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest_Port.h"
#include "Modules/SoTestDataXferModule.h"

#include <cstring> // for memset
#include <cstdlib> // for malloc

//#define NET_VERBOSE

#ifdef NET_VERBOSE
#define NET_LOG_CONN_DROP(COND, LAST_FN) \
            Log("In: %s, Function: %s, Condition: %s, Ignored (CONN_DROP) : errno: %d\n",   \
                 __FUNCTION__, (LAST_FN), #COND, zzqq_err)
#define NET_LOG_EXPECTED(COND, LAST_FN) \
            Log("In: %s, Function: %s, Condition: %s, Ignored (EXPECTED) : errno: %d\n",   \
                 __FUNCTION__, (LAST_FN), #COND, zzqq_err)
#else
#define NET_LOG_CONN_DROP(COND, LAST_FN) {}
#define NET_LOG_EXPECTED(COND, LAST_FN) {}
#endif

#define NET_CHECK_CONN_DROP(err) \
            (((err) == nn::socket::Errno::EConnReset) ||    \
             ((err) == nn::socket::Errno::EShutDown) ||     \
             ((err) == nn::socket::Errno::EWouldBlock) ||   \
             ((err) == nn::socket::Errno::EPipe) ||         \
             ((err) == nn::socket::Errno::EConnAborted))

#define NET_EXPECT_OUT_RETURN_BASE(COND, LAST_FN, ABORT_ON_CONN_DROP) \
do                                                            \
{                                                             \
    if(!(COND))                                               \
    {                                                         \
        nn::socket::Errno zzqq_err = NetTest::GetLastError(); \
        if( !(NN_STATIC_CONDITION(ABORT_ON_CONN_DROP)) && \
            NET_CHECK_CONN_DROP(zzqq_err) ) \
        {                                                     \
            NET_LOG_CONN_DROP(COND, LAST_FN); \
            rval = 0;                                     \
            return AppErr_Continue;                       \
        }                                                     \
        else if (zzqq_err == expectedErr)                     \
        {                                                     \
            NET_LOG_EXPECTED(COND, LAST_FN);                            \
            rval = 0;                                     \
            return AppErr_Ok;                             \
        }                                                     \
        else                                                  \
        {                                                     \
            LogError("In: %s, Function: %s, Condition: %s, Failed : errno: %d\n",   \
                 __FUNCTION__, LAST_FN, #COND, zzqq_err);                            \
            rval = 0;                                     \
            return AppErr_Out;                            \
        }                                                     \
    }                                                         \
} while(NN_STATIC_CONDITION(false))

#define NET_EXPECT_OUT_GOTO_BASE(COND, LAST_FN, ABORT_ON_CONN_DROP) \
do                                                            \
{                                                             \
    if(!(COND))                                               \
    {                                                         \
        nn::socket::Errno zzqq_err = NetTest::GetLastError(); \
        if( !(NN_STATIC_CONDITION(ABORT_ON_CONN_DROP)) && \
            NET_CHECK_CONN_DROP(zzqq_err) ) \
        {                                                     \
            NET_LOG_CONN_DROP(COND, LAST_FN); \
        }                                                     \
        else if (zzqq_err == expectedErr)                     \
        {                                                     \
            NET_LOG_EXPECTED(COND, LAST_FN);                            \
        }                                                     \
        else                                                  \
        {                                                     \
            LogError("In: %s, Function: %s, Condition: %s, Failed : errno: %d\n",   \
                 __FUNCTION__, LAST_FN, #COND, zzqq_err);                            \
            isSuccess = false;                            \
            goto out;                                     \
        }                                                     \
    }                                                         \
} while(NN_STATIC_CONDITION(false))

#define NET_EXPECT_OUT_GOTO_ABORT(COND, LAST_FN)     NET_EXPECT_OUT_GOTO_BASE((COND), (LAST_FN), true)
#define NET_EXPECT_OUT_GOTO_GRACEFUL(COND, LAST_FN)  NET_EXPECT_OUT_GOTO_BASE((COND), (LAST_FN), false)
#define NET_EXPECT_OUT_RETURN_ABORT(COND, LAST_FN)     NET_EXPECT_OUT_RETURN_BASE((COND), (LAST_FN), true)
#define NET_EXPECT_OUT_RETURN_GRACEFUL(COND, LAST_FN)  NET_EXPECT_OUT_RETURN_BASE((COND), (LAST_FN), false)

namespace NATF {
namespace Modules {

    // ClientUdp
    SoTestDataXfer::AppErr SoTestDataXfer::ClientUdp(unsigned& totalSent, unsigned& totalRecvd, char* pSendBuffer, char* pRecvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        ssize_t rval = NetTest::SendTo(m_params.socketFd, pSendBuffer, m_params.bufferSize, m_params.xferOptions,
                (NetTest::SockAddr *)&m_params.remoteAddr, sizeof(m_params.remoteAddr));
        NET_EXPECT_OUT_RETURN_GRACEFUL( rval >= 0, "SendTo" );

        totalSent += static_cast<unsigned>(rval);
        rval = NetTest::RecvFrom(m_params.socketFd, pRecvBuffer, m_params.bufferSize, m_params.xferOptions, NULL, 0);
        NET_EXPECT_OUT_RETURN_GRACEFUL( rval >= 0, "RecvFrom" );

        if( rval > 0 )
        {
            totalRecvd += static_cast<unsigned>(rval);

            unsigned smallBuf = (static_cast<unsigned>(rval) < m_params.bufferSize) ? static_cast<unsigned>(rval) : m_params.bufferSize;
            if( memcmp(pSendBuffer, pRecvBuffer, smallBuf) != 0)
            {
                LogError("Received data does not match!\n");
                rval = -1;
                return AppErr_Out;
            }
        }
        return AppErr_Ok;
    }

    // ServerUdp
    SoTestDataXfer::AppErr SoTestDataXfer::ServerUdp(unsigned& totalSent, unsigned& totalRecvd, char* pSendBuffer, char* pRecvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        memset((void *)pRecvBuffer, 0x00, m_params.bufferSize);

        NetTest::SockAddrIn peer;
        NetTest::SockLen len = sizeof(peer);
        ssize_t rval = NetTest::RecvFrom(m_params.socketFd, pRecvBuffer, m_params.bufferSize,
                                    m_params.xferOptions, (NetTest::SockAddr *)&peer, &len);
        NET_EXPECT_OUT_RETURN_GRACEFUL( rval >= 0, "RecvFrom" );

        if( rval > 0 )
        {
            totalRecvd += static_cast<unsigned>(rval);

            if( nullptr != m_params.xferPattern )
            {
                unsigned smallBuf = (static_cast<unsigned>(rval) < m_params.bufferSize) ? static_cast<unsigned>(rval) : m_params.bufferSize;
                if( memcmp(m_params.xferPattern, pRecvBuffer, smallBuf) != 0)
                {
                    LogError("Received data does not match\n");
                    rval = -1;
                    return AppErr_Out;
                }
            }
            else
            {
                NETTEST_COPY_BUFFER_BYTES_POINTER(pSendBuffer, pRecvBuffer, MaxBufSize);
            }

            rval = NetTest::SendTo(m_params.socketFd, pSendBuffer, m_params.bufferSize, m_params.xferOptions,
                    (NetTest::SockAddr *)&peer, len);
            NET_EXPECT_OUT_RETURN_GRACEFUL( rval >= 0, "SendTo" );

            totalSent += static_cast<unsigned>(rval);
        }

        return AppErr_Ok;
    }

    // ClientTcp
    SoTestDataXfer::AppErr SoTestDataXfer::ClientTcp(unsigned& totalSent, unsigned& totalRecvd, char* pSendBuffer, char* pRecvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        uint32_t bytesSent = 0;
        do
        {
            ssize_t rval;
            if((rval = NetTest::Send(m_params.socketFd, &pSendBuffer[bytesSent], m_params.bufferSize - bytesSent, m_params.xferOptions)) >=0)
            {
                bytesSent += static_cast<uint32_t>(rval);
                totalSent += static_cast<unsigned>(rval);
            }
            else if(NetTest::GetLastError() == m_params.expectedXferErr)
            {
                // Need to allow some time for server to catch up before trying again.
                // Since we are all on the same core, we have to yield.
                NetTest::SleepMs(10);
            }
            else
            {
                LogError("SoTestDataXfer::ClientTcp Send() failed with errno = %d\n", NetTest::GetLastError());
                return AppErr_Out;
            }
        } while(bytesSent < m_params.bufferSize);

        uint32_t bytesRecvd = 0;
        do
        {
            ssize_t rval = NetTest::Recv(m_params.socketFd, &pRecvBuffer[bytesRecvd], m_params.bufferSize - bytesRecvd, m_params.xferOptions);
            if(rval >= 0)
            {
                totalRecvd += static_cast<unsigned>(rval);
                bytesRecvd += static_cast<uint32_t>(rval);
            }
        } while(bytesRecvd < m_params.bufferSize);

        if( memcmp(pSendBuffer, pRecvBuffer, m_params.bufferSize) != 0 )
        {
            LogError("Received data does not match: Recv: %*s, Send: %*s\n", m_params.bufferSize, pRecvBuffer, m_params.bufferSize, pSendBuffer);
            return AppErr_Out;
        }


        return AppErr_Ok;
    }

    // ServerTcp
    SoTestDataXfer::AppErr SoTestDataXfer::ServerTcp(unsigned& totalSent, unsigned& totalRecvd, char* pSendBuffer, char* pRecvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        memset((void *)pRecvBuffer, 0x00, m_params.bufferSize);

        ssize_t rval = NetTest::Recv(m_params.socketFd, pRecvBuffer, m_params.bufferSize, m_params.xferOptions);
        NET_EXPECT_OUT_RETURN_GRACEFUL( rval >= 0, "Recv" );

        if( rval > 0 )
        {
            totalRecvd += static_cast<unsigned>(rval);

            // Accomodate rare case where xfer pattern is incrementally received across multiple recvs
            // in order to avoid intermittent failures with send/recv data mismatch
            unsigned compareLen = ((m_params.lastRecvIdx + static_cast<unsigned>(rval)) < m_params.bufferSize) ? static_cast<unsigned>(rval) : m_params.bufferSize - m_params.lastRecvIdx;
            if( nullptr != m_params.xferPattern )
            {
                if( memcmp(&m_params.xferPattern[m_params.lastRecvIdx], pRecvBuffer, compareLen) != 0 )
                {
                    LogError("Received data does not match. rval: %d bufsize: %d c1: %c, c2: %c, %c %c\n", (int)rval, m_params.bufferSize, pSendBuffer[0], pRecvBuffer[0], pSendBuffer[m_params.bufferSize - 1], pRecvBuffer[m_params.bufferSize - 1]);
                    rval = -1;
                    return AppErr_Out;
                }
            }
            else
            {
                NETTEST_COPY_BUFFER_BYTES_POINTER(&pSendBuffer[m_params.lastRecvIdx], pRecvBuffer, compareLen);
            }

            m_params.lastRecvIdx = (m_params.lastRecvIdx + static_cast<unsigned>(rval)) % m_params.bufferSize;

            uint32_t bytesSent = 0;
            do
            {
                if((rval = NetTest::Send(m_params.socketFd, &pSendBuffer[bytesSent], m_params.bufferSize - bytesSent, m_params.xferOptions)) >=0)
                {
                    bytesSent += static_cast<uint32_t>(rval);
                    totalSent += static_cast<unsigned>(rval);
                }
                else if(NetTest::GetLastError() == m_params.expectedXferErr)
                {
                    // Need to allow some time for server to catch up before trying again.
                    // Since we are all on the same core, we have to yield.
                    NetTest::SleepMs(10);
                }
                else
                {
                    LogError("SoTestDataXfer::ServerTcp Send() failed with errno = %d\n", NetTest::GetLastError());
                }
            } while(bytesSent < m_params.bufferSize);
        }

        return AppErr_Ok;
    }

    // SetOpt
    bool SoTestDataXfer::SetOpt() NN_NOEXCEPT
    {
        bool isSuccess = true;
        int rval = 0;
        int value = 0;

        if( m_params.isNonBlocking )
        {
            rval = NetTest::Fcntl(m_params.socketFd, static_cast<int>(nn::socket::FcntlCommand::F_SetFl), nn::socket::FcntlFlag::O_NonBlock);
            if (rval != 0)
            {
                LogError("Error: Fcntl(nn::socket::FcntlFlag::O_NonBlock) failed. rval: %d\n", rval);
                isSuccess = false;
                goto out;
            }
        }

        if( m_params.socketSendSize > 0 )
        {
            value = m_params.socketSendSize;
            rval = NetTest::SetSockOpt(m_params.socketFd, nn::socket::Level::Sol_Socket,
                   nn::socket::Option::So_SndBuf, &value, sizeof(int));
            if (rval < 0)
            {
               Log("Warning: setsockopt(nn::socket::Option::So_SndBuf) failed! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
            }
        }

        if( m_params.socketRecvSize > 0 )
        {
            value = m_params.socketRecvSize;
            rval = NetTest::SetSockOpt(m_params.socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &value, sizeof(int));
            if( rval < 0 )
            {
               Log("Warning: setsockopt(nn::socket::Option::So_RcvBuf) failed! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
            }
        }

    out:
        return isSuccess;
    }

    // Init
    bool SoTestDataXfer::Init(nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        bool isSuccess = true;

        if( (m_params.initType == InitType_All) || (m_params.initType == InitType_Init) || (m_params.initType == InitType_AllExempt))
        {
            if( !NetTest::Init() )
            {
                if( nn::socket::Errno::ESuccess != expectedErr )
                {
                    Log("Expected NetTest::Init() failure.\n");
                } else
                {
                    LogError("Error: NetTest::Init!\n");
                    isSuccess = false;
                    goto out;
                }
            }
        }
        if( (m_params.initType == InitType_All) || (m_params.initType == InitType_Socket) )
        {
            int rval = NetTest::Socket(m_params.socketFamily, m_params.socketType, m_params.socketProtocol);
            NET_EXPECT_OUT_GOTO_ABORT( rval >= 0, "Socket" );

            Log("Socket created: %d\n", rval);
            m_params.socketFd = rval;
            SetOpt();
        }
        if ((m_params.initType == InitType_AllExempt) || (m_params.initType == InitType_SocketExempt))
        {
            int rval = NetTest::SocketExempt(m_params.socketFamily, m_params.socketType, m_params.socketProtocol);
            NET_EXPECT_OUT_GOTO_ABORT(rval >= 0, "Socket");

            Log("Exempt Socket created: %d\n", rval);
            m_params.socketFd = rval;
            SetOpt();
        }

    out:

        return isSuccess;
    }

    // Setup
    bool SoTestDataXfer::Setup(nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        bool isSuccess = true;
        int rval;

        memset((void *) &m_params.remoteAddr, 0, sizeof(m_params.remoteAddr));
        m_params.addrLen = sizeof(m_params.remoteAddr);

        if( m_params.appType == AppType_Client )
        {
            m_params.remoteAddr.sin_family = nn::socket::Family::Af_Inet;
            m_params.remoteAddr.sin_port = NetTest::Htons(m_params.port);
            rval = NetTest::InetAddrPton(nn::socket::Family::Af_Inet, m_params.pIp, &m_params.remoteAddr.sin_addr);
            NET_EXPECT_OUT_GOTO_ABORT( rval >= 0, "InetAddrPton" );

            if( m_params.socketType == nn::socket::Type::Sock_Stream )
            {
                // Wait for server to start
                Log("Connecting...\n");
                rval = NetTest::Connect (m_params.socketFd, (NetTest::SockAddr *)&m_params.remoteAddr,  sizeof(m_params.remoteAddr));
                NET_EXPECT_OUT_GOTO_ABORT( rval >= 0, "Connect" );
            }
        }
        else
        {
            NetTest::SockAddrIn localAddr;
            memset ((void *) &localAddr, 0, sizeof(NetTest::SockAddrIn));
            localAddr.sin_family = nn::socket::Family::Af_Inet;
            localAddr.sin_port = NetTest::Htons(m_params.port);
            localAddr.sin_addr.S_addr = nn::socket::InAddr_Any;

            rval = NetTest::Bind(m_params.socketFd, (NetTest::SockAddr *)&localAddr, sizeof(localAddr));
            NET_EXPECT_OUT_GOTO_ABORT( rval == 0, "Bind" );

            if( m_params.socketType == nn::socket::Type::Sock_Stream )
            {
                rval = NetTest::Listen (m_params.socketFd, m_params.backLog);
                NET_EXPECT_OUT_GOTO_ABORT( rval == 0, "Listen" );

                Log("Listening for connection on %d. backlog: %d\n", m_params.socketFd, m_params.backLog);
                m_params.addrLen = sizeof(m_params.remoteAddr);

                Log("Accepting...\n");
                rval = NetTest::Accept(m_params.socketFd, (NetTest::SockAddr *)&m_params.remoteAddr, &m_params.addrLen);
                NET_EXPECT_OUT_GOTO_ABORT( rval >= 0, "Accept" );

                NetTest::Close(m_params.socketFd);
                m_params.socketFd = rval;

                Log("Accept socket(%d) over port %d successful\n", rval, NetTest::Ntohs(m_params.remoteAddr.sin_port));
            }
        }
    out:

        return isSuccess;
    }

    // Teardown
    bool SoTestDataXfer::Teardown(nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        bool isSuccess = true;
        ssize_t rval;

        if( (m_params.teardownType == TeardownType_All) || (m_params.teardownType == TeardownType_Graceful ) )
        {
            rval = NetTest::Shutdown(m_params.socketFd, nn::socket::ShutdownMethod::Shut_Wr );
            NET_EXPECT_OUT_GOTO_GRACEFUL( rval == 0, "Shutdown" );

            unsigned bytesUntilNoData = 0;
            char recvBuffer[MaxBufSize];
            memset((void *)recvBuffer, 0x00, m_params.bufferSize);
            while((rval = NetTest::RecvFrom(m_params.socketFd, recvBuffer, m_params.bufferSize, m_params.xferOptions, NULL, 0)) != 0)
            {
                NET_EXPECT_OUT_GOTO_GRACEFUL( rval > 0, "RecvFrom" );
                if (rval < 0)
                {
                    break;
                }
                else // (rval > 0)
                {
                    bytesUntilNoData += static_cast<unsigned>(rval);
                }
            }

            Log(" Received %d bytes after shutdown\n", bytesUntilNoData);

            rval = NetTest::Shutdown(m_params.socketFd, nn::socket::ShutdownMethod::Shut_Rd );
            NET_EXPECT_OUT_GOTO_GRACEFUL( rval == 0, "Shutdown" );
        }
        else if( m_params.teardownType == TeardownType_Shutdown )
        {
            rval = NetTest::Shutdown(m_params.socketFd, m_params.shutdownType);
            NET_EXPECT_OUT_GOTO_GRACEFUL( rval == 0, "Shutdown" );
        }
        if( (m_params.teardownType == TeardownType_All) || (m_params.teardownType == TeardownType_Close) )
        {
            Log("Close\n");
            rval = NetTest::Close(m_params.socketFd);
            NET_EXPECT_OUT_GOTO_GRACEFUL( rval == 0, "Close" );
        }
    out:

        return isSuccess;
    }

    // DataXfer
    bool SoTestDataXfer::DataXfer(nn::socket::Errno expectedErr) NN_NOEXCEPT
    {
        bool isSuccess = true;
        char sendBuffer[MaxBufSize];
        char recvBuffer[MaxBufSize];
        NetTest::Tick end, start;
        NetTest::Time duration;
        unsigned totalSent  = 0;
        unsigned totalRecvd = 0;
        unsigned milSec     = 0;
        float sentThroughputMbitSec  = 0.0f;
        float recvdThroughputMbitSec = 0.0f;
        int rval = 0;
        nn::socket::Errno err  = nn::socket::Errno::ESuccess;

        if( m_params.bufferSize > MaxBufSize )
        {
            LogError("bufferSize is too large, Max size: %d\n", MaxBufSize);
            isSuccess = false;
            goto out;
        }

        memset((void *)sendBuffer, '\0', sizeof(sendBuffer));
        memset((void *)recvBuffer, '\0', sizeof(recvBuffer));

        if( nullptr != m_params.xferPattern )
        {
            NETTEST_COPY_BUFFER_BYTES_ARRAY(sendBuffer, m_params.xferPattern);
        }

        rval = NetTest::Fcntl(m_params.socketFd, static_cast<int>(nn::socket::FcntlCommand::F_SetFl), nn::socket::FcntlFlag::O_NonBlock);
        if( rval != 0 )
        {
            err = NetTest::GetLastError();
            LogError("Error: fcntl(nn::socket::FcntlCommand::F_SetFl) failed! rval: %d errno: %d\n", rval, err);
            isSuccess = false;
            goto out;
        }

        start = NetTest::GetTick();
        while( NetTest::TickToTime(NetTest::GetTick() - start).GetSeconds() <= m_params.duration )
        {
            AppErr appErr = AppErr_Ok;
            if( m_params.socketType == nn::socket::Type::Sock_Dgram )
            {
                if( m_params.appType == AppType_Client )
                {
                    appErr = ClientUdp(totalSent, totalRecvd, sendBuffer, recvBuffer, expectedErr);
                }
                else if( m_params.appType == AppType_Server )
                { // UDP Server
                    appErr = ServerUdp(totalSent, totalRecvd, sendBuffer, recvBuffer, expectedErr);
                }
                else
                {
                    LogError("Error: Unknown AppType! AppType: %d\n", m_params.appType);
                    isSuccess = false;
                    goto out;
                }
            } else // TCP
            {
                if( m_params.appType == AppType_Client )
                {
                    appErr = ClientTcp(totalSent, totalRecvd, sendBuffer, recvBuffer, expectedErr);
                }
                else if( m_params.appType == AppType_Server )
                { // TCP Server
                    appErr = ServerTcp(totalSent, totalRecvd, sendBuffer, recvBuffer, expectedErr);
                }
                else
                {
                    LogError("Error: Unknown AppType! AppType: %i\n", m_params.appType);
                    isSuccess = false;
                    goto out;
                }
            }

            switch( appErr )
            {
            case AppErr_Ok: break;
            case AppErr_Continue:
                goto out_continue;
            case AppErr_Out:
                isSuccess = false;
                goto out;
            default:
                LogError("Error! Unknown return error!\n");
                isSuccess = false;
                goto out;
            }
    out_continue:

            NetTest::SleepMs(1);
        }

        end = NetTest::GetTick();
        duration = NetTest::TickToTime(end - start);
        milSec = static_cast<int>(duration.GetMilliSeconds());
        sentThroughputMbitSec  = totalSent  * 8.0f / (milSec / 1000.0f) / 1000000.0f;
        recvdThroughputMbitSec = totalRecvd * 8.0f / (milSec / 1000.0f) / 1000000.0f;
        Log(" Total sent bytes: %d, Throughput: %d.%d Mbits/Sec\n", static_cast<int>(totalSent),
                                                                    static_cast<int>(sentThroughputMbitSec),
                                                                    static_cast<int>(sentThroughputMbitSec  * 100.0f) % 100 );

        Log(" Total received bytes: %d, Throughput: %d.%d Mbits/Sec\n", static_cast<int>(totalRecvd),
                                                                        static_cast<int>(recvdThroughputMbitSec),
                                                                        static_cast<int>(recvdThroughputMbitSec * 100.0f) % 100 );

    out:

        return isSuccess;
    } // NOLINT(impl/function_size)

    // Constructor
    SoTestDataXfer::SoTestDataXfer(AppType appType, const char* pIp, const char* xferPattern, unsigned short port, nn::socket::Protocol protoType, InitType initType, int durationSec) NN_NOEXCEPT
    {
        memset(&m_params, 0, sizeof(m_params));

        m_params.isNonBlocking       = false;
        m_params.socketRecvSize      = 1024 * 64 - 1;
        m_params.socketSendSize      = 32 * 1024;
        m_params.appType             = appType;
        m_params.port                = port;
        m_params.backLog             = 2;
        m_params.shutdownType        = nn::socket::ShutdownMethod::Shut_RdWr;

        // The client needs to stop running first for this to be reliable for blocking TCP sockets
        m_params.duration            = (appType == AppType_Client) ? durationSec : ((durationSec * 125) / 100);

        m_params.bufferSize          = MaxBufSize; // Used to be 1024
        m_params.initType            = initType;

        m_params.socketFd            = -1;
        m_params.socketProtocol      = protoType;
        m_params.socketFamily        = nn::socket::Family::Af_Inet;
        m_params.xferOptions         = nn::socket::MsgFlag::Msg_None;
        m_params.expectedInitErr     = nn::socket::Errno::ESuccess;
        m_params.expectedSetupErr    = nn::socket::Errno::ESuccess;
        m_params.expectedXferErr     = nn::socket::Errno::EWouldBlock;
        m_params.expectedTeardownErr = nn::socket::Errno::ENotConn;
        m_params.lastRecvIdx         = 0;

        NETTEST_COPY_BUFFER_STRING_ARRAY(m_params.pIp, pIp);

        if( 0 != strncmp(xferPattern, "Echo", 4) )
        {
            m_params.xferPattern = (char*)malloc(MaxBufSize);
            NETTEST_COPY_BUFFER_BYTES_POINTER(m_params.xferPattern, xferPattern, MaxBufSize);
        }
        else
        {
            m_params.xferPattern = nullptr;
        }

        if( protoType == nn::socket::Protocol::IpProto_Tcp )
        {
            m_params.socketType   = nn::socket::Type::Sock_Stream;
            m_params.teardownType = TeardownType_All;
        }
        else if( protoType == nn::socket::Protocol::IpProto_Udp )
        {
            m_params.socketType   = nn::socket::Type::Sock_Dgram;
            m_params.teardownType = TeardownType_Close;
        }
        else
        {
            LogError("Error: Unknown protocol passed.\n\n");
        }
    }

    // Destructor
    SoTestDataXfer::~SoTestDataXfer() NN_NOEXCEPT
    {
        if( nullptr != m_params.xferPattern )
        {
            free(m_params.xferPattern);
            m_params.xferPattern = nullptr;
        }
    }

    // Run
    bool SoTestDataXfer::Run() NN_NOEXCEPT
    {
        bool isSuccess = true;
        const int sleepCountSec = 2;

        if( m_params.appType == AppType_Client )
        {
            Log("Sleeping %d Seconds...\n", sleepCountSec);
            NetTest::SleepMs(sleepCountSec * 1000);
            Log("Done Sleeping!\n");
        }

        Log("Calling Init...\n");
        if( !Init(m_params.expectedInitErr) )
        {
            isSuccess = false;
            LogError("SoTestInit failed\n");
            goto out;
        }

        Log("Calling Setup...\n");
        if( !Setup(m_params.expectedSetupErr) )
        {
            isSuccess = false;
            LogError("SoTestSetup\n");
            goto out;
        }

        Log("Calling DataXfer...\n");
        if( !DataXfer(m_params.expectedXferErr) )
        {
            isSuccess = false;
            LogError("SoTestDataXfer, errno: %d\n", NetTest::GetLastError());
            goto out;
        }
        Log("Data xfer complete\n");

        Log("Calling Teardown...\n");
        if( !Teardown(m_params.expectedTeardownErr) )
        {
            LogError("Error: SoTestTeardown\n");
            isSuccess = false;
        }
    out:

        return isSuccess;
    }

    // CreateTestNode
    SoTestDataXfer* SoTestDataXfer::CreateTestNode(TestThread* pThread, const char* appTypeName, const char* ipAddr, int portNumber, const char* protocolTypeName, int durationSec, const char* xferPattern) NN_NOEXCEPT
    {
        return SoTestDataXfer::CreateTestNode(pThread, appTypeName, ipAddr, portNumber, protocolTypeName, durationSec, xferPattern, false);
    }
    SoTestDataXfer* SoTestDataXfer::CreateTestNode(TestThread* pThread, const char* appTypeName, const char* ipAddr, int portNumber, const char* protocolTypeName, int durationSec, const char* xferPattern, bool socketExempt) NN_NOEXCEPT
    {
        AppType appType;
        nn::socket::Protocol protocolType;
        InitType initType;

        // Validate command line options for constructor
        // and return nullptr if any options are invalid

        if( nullptr == pThread )
        {
            return nullptr;
        }

        if( 0 == strcmp(appTypeName, "Server") )
        {
            appType = AppType_Server;
        }
        else if( 0 == strcmp(appTypeName, "Client") )
        {
            appType = AppType_Client;
        }
        else
        {
            pThread->LogError(" * Invalid command line arguments: appTypeName = %s\n\n", appTypeName);
            return nullptr;
        }

        if( 0 == strcmp(protocolTypeName, "TCP") )
        {
            protocolType = nn::socket::Protocol::IpProto_Tcp;
        }
        else if( 0 == strcmp(protocolTypeName, "UDP") )
        {
            protocolType = nn::socket::Protocol::IpProto_Udp;
        }
        else
        {
            pThread->LogError(" * Invalid command line arguments: protocolTypeName = %s\n\n",
                protocolTypeName);
            return nullptr;
        }

        if( MaxPortNumber < portNumber )
        {
            pThread->LogError(" * Invalid command line arguments: portNumber = %d\n\n", portNumber);
            return nullptr;
        }
        if (socketExempt)
        {
            initType = InitType_SocketExempt;
        }
        else
        {
            initType = InitType_Socket;
        }
        // Validation is completed
        // so call constructor and return the new object

        if( appType == AppType_Server )
        {
            return new SoTestDataXfer(appType, nullptr, xferPattern, static_cast<unsigned short>(portNumber), protocolType, initType, durationSec);
        }
        else
        {
            return new SoTestDataXfer(appType, ipAddr, xferPattern, static_cast<unsigned short>(portNumber), protocolType, initType, durationSec);
        }
    }

    // GetName
    const char* SoTestDataXfer::GetName() const NN_NOEXCEPT
    { return "SoTestDataXfer"; }
}
}
