﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "NetTest_Port.h"

#include "Utils/InitApis.h"
#include "Utils/ExchangeInfo.h"

#define LogError(...) LogErrorManual(__FILE__, __FUNCTION__, __LINE__, __VA_ARGS__)

namespace NATF
{
    class BaseTest;
    class TestThread;

    enum ModuleResult
    {
        ModuleResult_Pass       = 0,
        ModuleResult_Fail       = 1,
        ModuleResult_Incomplete = 2
    };

    enum ThreadState
    {
        ThreadState_Null       = 0,
        ThreadState_Created    = 1,
        ThreadState_Running    = 2,
        ThreadState_Finished   = 3,
        ThreadState_Terminated = 4
    };

    // Derive from this class to create a module.
    class BaseModule
    {
        NN_DISALLOW_COPY(BaseModule);

    public:
        explicit BaseModule(bool useSelect = true) NN_NOEXCEPT;
        virtual ~BaseModule() NN_NOEXCEPT {}

        virtual bool Run() NN_NOEXCEPT = 0;
        virtual const char* GetName() const NN_NOEXCEPT = 0;
        const char* GetThreadName() const NN_NOEXCEPT;

        // This function will only be called on slave modules.
        // A module is a slave when added to a test via AddSlaveModule()
        virtual void StopSlave() NN_NOEXCEPT {}

        void Log(const char* pFormat, ...) const NN_NOEXCEPT;
        void LogErrorManual(const char* pFileName, const char* pFunctionName, unsigned lineNumber, const char* pFormat, ...) const NN_NOEXCEPT;

    protected:
        // For ease of conversion, we will use Polls signiture.
        int SelectOrPoll(NetTest::PollFd* pFds, nn::socket::NfdsT fdCount, int timeoutMs) const NN_NOEXCEPT;
        const char* GetRemoteIp() const NN_NOEXCEPT;
        bool WriteHostIp(char* pStr, uint32_t bufferLen) const NN_NOEXCEPT;

    private:
        ModuleResult GetResult() const NN_NOEXCEPT;

        static int SelectWithPollSignature(NetTest::PollFd* pFds, nn::socket::NfdsT fdCount, int timeoutMs) NN_NOEXCEPT;

        typedef int (*SelectOrPollFnPtr)(NetTest::PollFd* pFds, nn::socket::NfdsT fdCount, int timeoutMs);

        SelectOrPollFnPtr m_pSelectOrPollFn;
        BaseModule* m_pNext;
        ModuleResult m_result;
        TestThread* m_pThread;

        friend TestThread;
    };

    class TestThread
    {
        NN_DISALLOW_COPY(TestThread);

    private:
        TestThread(const BaseTest*, const char* pName, unsigned timeoutSec) NN_NOEXCEPT;
        ~TestThread() NN_NOEXCEPT;

    public:

        static void ThreadFunc(void* pParam) NN_NOEXCEPT;

        void AddModule(BaseModule& module) NN_NOEXCEPT;
        void Destroy() NN_NOEXCEPT;
        void Log(const char* pLog, ...) const NN_NOEXCEPT;
        void LogErrorManual(const char* pFileName, const char* pFunctionName, unsigned lineNumber, const char* pFormat, ...) const NN_NOEXCEPT;
        bool IsSlave() const NN_NOEXCEPT;
        const char* GetName() const NN_NOEXCEPT;
        const char* GetTestName() const NN_NOEXCEPT;

        static const unsigned StackSize       = 1024 * 64;
        static const unsigned StackAlign      = 4096;
        static const unsigned StrBufLen       = 64;
        static const unsigned SlaveTimeoutSec = 5;

    private:

        void Start() NN_NOEXCEPT;
        bool LogResults() const NN_NOEXCEPT;

        unsigned GetTimeout() const NN_NOEXCEPT;
        ThreadState GetState() const NN_NOEXCEPT;

        unsigned char m_pStack[StackSize];
        unsigned char* m_pUnalignedBuf;

        BaseModule* m_pHead;
        BaseModule* m_pTail;
        TestThread* m_pNext;

        NetTest::Thread m_thread;
        bool m_isSlaveThread;
        unsigned m_timeoutSec;
        char m_pName[StrBufLen];
        ThreadState m_state;
        const BaseTest* const m_pTest;

        friend BaseTest;
        friend BaseModule;
    };

    // Derive from this class to create a test
    class BaseTest
    {
        NN_DISALLOW_COPY(BaseTest);

    public:
        BaseTest(const char* pTestName, bool doExchangeIpInfo, Utils::InitApiFlags initFlags, const nn::util::Uuid& netProfile = nn::util::InvalidUuid) NN_NOEXCEPT;
        virtual ~BaseTest() NN_NOEXCEPT;

        bool Run(bool initApis = true) NN_NOEXCEPT;
        const char* GetName() const NN_NOEXCEPT;

        void Log(const char* pFormat, ...) const NN_NOEXCEPT;
        void LogErrorManual(const char* pFileName, const char* pFunctionName, unsigned lineNumber, const char* pFormat, ...) const NN_NOEXCEPT;

        const char* GetRemoteIp() const NN_NOEXCEPT;
        bool WriteHostIp(char* pStr, uint32_t bufferLen) const NN_NOEXCEPT;

        static const uint32_t NameBufferLen = 64;
        static const char* const RemoteHostName;

    protected:

        bool AddSlaveModule(const char* pThreadName, BaseModule& module) NN_NOEXCEPT;
        TestThread* CreateTestThread(const char* pName, unsigned timeoutSec) NN_NOEXCEPT;
        void SetName(const char* pTestName) NN_NOEXCEPT;

    private:

        virtual bool Config() NN_NOEXCEPT = 0;
        virtual bool Init() NN_NOEXCEPT
        { return true; }

        virtual bool Cleanup() NN_NOEXCEPT
        { return true; }

        void CleanupThreads() NN_NOEXCEPT;
        bool PrivateInit(Utils::InitApi* pInitApi) NN_NOEXCEPT;

        bool m_doExchangeInfo;
        Utils::InitApiFlags m_initFlags;
        nn::util::Uuid m_netProfile;
        Utils::ExchangeInfo m_exchangeInfo;
        TestThread* m_pHead;
        TestThread* m_pTail;
        NetTest::Tick m_startTime;
        char m_pNameBuffer[NameBufferLen];
    };

} // namespace NATF
