﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "natf.h"
#include "Utils/md5.h"
#include "nn/ssl/ssl_Connection.h"

namespace NATF {
namespace Modules {

class LibCurl : public BaseModule
{
private:
    NN_DISALLOW_COPY(LibCurl);

public:

    enum HttpMethod
    {
        HttpMethod_Get  = 0,
        HttpMethod_Put  = 1,
        HttpMethod_Post = 2,
    };

    struct Params;

    explicit LibCurl(const Params& params) NN_NOEXCEPT;
    virtual ~LibCurl() NN_NOEXCEPT NN_OVERRIDE {}

    virtual bool Run() NN_NOEXCEPT NN_OVERRIDE;
    virtual const char* GetName() const NN_NOEXCEPT NN_OVERRIDE;

    static const uint32_t MaxPollTimeoutSec = 10;
    static const uint32_t ServerCertBufLen = 4 * 1024;
    static const uint32_t UrlBufferLen = 128;
    static const uint32_t PathBufLen = 256;
    static const uint32_t LargePathBufLen = 1024;
    static const uint32_t HostNameBufLen = 32;
    static const uint32_t ProxyUsrPwdBufLen = 32;
    static const uint32_t HttpHeaderOptionBufLen = 128;
    static const uint32_t CipherNameBufLen = nn::ssl::Connection::CipherInfo::NameBufLen;
    static const uint32_t SslVersionBufLen = nn::ssl::Connection::CipherInfo::VersionBufLen;

    struct Params
    {
        Params() NN_NOEXCEPT;

        // If multiApiConnectionCount is zero, use the curl_easy api will be used
        uint32_t multiApiConnectionCount;
        int32_t expectedCurlReturn;
        HttpMethod httpMethod;
        int32_t expectedHttpResponse;
        int expectedAuthReturn;
        MD5Hash::Result expectedHash;
        uint64_t uploadSize;
        bool useSelect;
        bool doVerifyPeer;
        bool doVerifyHostname;
        bool doVerifyDate;
        bool isAutoUrlEnabled;
        uint16_t proxyPort;
        int proxyAuthMethod;
        bool useInternalPki;
        bool publishDownSpeedToTeamCity;
        bool publishUpSpeedToTeamCity;
        nn::ssl::Connection::SessionCacheMode sessionCacheMode;
        uint32_t verifyResultArrayLen;

        int32_t sendBuffer;
        int32_t recvBuffer;
        int32_t curlHandleLoop;
        int32_t curlPerformLoop;
        int32_t curlHostErrRetryCount;
        int32_t timeBetweenHostErrRetryMs;
        bool checkHash;
        bool isExpectingBufferTooShort;

        const char* pUploadData;
        char pUrl[UrlBufferLen];
        char pClientCertPath[PathBufLen];

        // pServerCertPath may contain a list of server certs.
        // This is why we use a larger path len than pClientCertPath
        char pServerCertPath[LargePathBufLen];
        char pProxyServer[HostNameBufLen];
        char pProxyUserPwd[ProxyUsrPwdBufLen];
        char pHttpHeaderOption[HttpHeaderOptionBufLen];
        char pExpectedCipherName[CipherNameBufLen];
        char pExpectedSslVersion[SslVersionBufLen];
    };

private:

    class Helper;
    class CurlEasyData;

    bool CheckHashValues(CurlEasyData* pEasyData) NN_NOEXCEPT;
    bool VerifyParams() NN_NOEXCEPT;

    Params m_params;

    bool m_isBufferTooShortError;
    char m_pServerCertBuffer[ServerCertBufLen];
};

}
}
